/*
 * Affiliation.java:  a Java implementation of Affiliation class
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Affiliation.java,v 1.10 2002/11/19 02:05:17 nozomi Exp $
 * $Log: Affiliation.java,v $
 * Revision 1.10  2002/11/19 02:05:17  nozomi
 * remove unused linktable
 *
 * Revision 1.9  2002/09/17 04:52:45  nozomi
 * support indirect reference, and construction using XML
 *
 * Revision 1.8  2002/04/09 03:17:57  nozomi
 * add some set methods
 *
 * Revision 1.7  2002/04/09 03:01:40  nozomi
 * change emptyPersistentID handling
 *
 * Revision 1.6  2002/04/09 01:59:34  nozomi
 * use Author instead of String to call non-trivial constructor
 *
 * Revision 1.5  2002/04/08 01:45:30  nozomi
 * Change StringDelegate to Name
 *
 * Revision 1.4  2002/03/25 11:31:11  okawa
 * no message
 *	
 * Revision 1.3  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 *
 * Revision 1.2  2002/02/14 08:30:02  okawa
 * add set/getFrom(), set/getUntil()
 *	
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 *
 * Revision 1.1  1999/10/24 08:06:29  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.util.Date;

import org.nomencurator.Author;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An implementation of <code>Affiliation</code> in Nomencurator
 * data model.
 *
 * @see org.nomencurator.NamedObject
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org</A>
 *
 * @version 	19 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class Affiliation
    extends NamedObject
    implements Serializable
{
    /** <code>Author</code> of which affiliation is represented by this object<code> */
    protected Author author;

    /** <code>String</code> representing name of insitute */
    String institute;

    /** <coce>Date</code> when the <code>author</code> affiliateted to the institute etc. */
    protected Date from;

    /** <coce>Date</code> when the <code>author</code> stopped to be affiliateted to the institute etc. */
    protected Date until;

    /** Overrides superclas' <CODE>entity</CODE>
    protected Affiliation entity;

    /** Constructs an "empty" <code>Affiliation</code> */
    public Affiliation()
    {
	this(null, null, null, null);
    }
    
    /**
     * Constructs a <code>Affiliation</code> object having
     * <code>persistentID</code> as its representation,
     *
     */
    public Affiliation(String name)
    {
	super(name);
    }
    
    /**
     * Constructs a <code>Affiliation</code> based on
     * <code>name</code>
     */
    public Affiliation(Name name)
    {
	//don't use NamedObject(String) constructor
	this();
	
	if(name instanceof Affiliation){
	    Affiliation affiliation = (Affiliation)name;
	    author = affiliation.author;
	    institute = affiliation.institute;
	    from   = affiliation.from;
	    until  = affiliation.until;
	}
	else
	    setPersistentID(name.getName());
    }
    
    /**
     * Constructs a non-trivial <code>Affiliation</code>.
     *
     * @param author <code>Author</code> of which affiliation should be represented by this object
     * @param institute name of institute where <code>author</code> affiliated
     * @param from  <code>Date</code> when <code>author</code> affiliated to the <code>institute</code>
     * @param until <code>Date</code> when <code>author</code> unaffiliated from the <code>institute</code>
     */
    public Affiliation(Author author, String institute,
		       Date from, Date until)
    {
	super();
	
	this.author = author;
	this.institute = institute;
	this.from   = from;
	this.until  = until;
    }

    /**
     * Constructs an <CODE>Affiliation</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying <CODE>Affiliation</CODE>
     *
     */
    public Affiliation(Element xml)
    {
	super();
	NodeList childNodes = xml.getChildNodes();
	int subChildCount = childNodes.getLength();
	
	String persistentID = null;
	for (int j = 0; j < subChildCount; j++) {
	    Node node = childNodes.item(j);
	    if(node.getNodeType () == Node.ELEMENT_NODE) {
		Element element = (Element)node;
		String tagName = element.getTagName();

		if (tagName.equals ("oid"))
		    persistentID = getString(element);
		else if(tagName.equals ("institution")) {
		    // TBD
		    // AffiliationInstitution.getInstance().addLinkRecord(rec);
		}
		else if(tagName.equals ("from"))
		    from = java.sql.Date.valueOf(getString(element));
		else if(tagName.equals ("until"))
		    until = java.sql.Date.valueOf(getString(element));
		else{}
            }
        }
	if(persistentID != null &&
	   !persistentID.equals(getName()))
	    setName(persistentID); //i.e. other key data are empty

    }
    
    /**
     * Returns <code>String</code> representing persistend ID of this object
     *
     * @return <code>String</code> representing persistend ID of this object
     */ 
    public String persistentID()
    {
	StringBuffer buffer = getClassNameHeaderBuffer();
	if(author == null) 
	    buffer.append(fieldSeparator).append(fieldSeparator);
	else {
	    String pid = author.persistentID();
	    buffer.append(pid.substring(pid.indexOf(author.getClassNameHeader())));
	    buffer.append(fieldSeparator).append(from).append(fieldSeparator).append(until);
	}

	return buffer.toString();
    }
    
    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public int getFieldSepartorsCount()
    {
	return 2 + new Author().getFieldSepartorsCount();
    }

    /**
     * Merges <code>namedObject</code> with this <code>Affiliation</code>
     * if possible, but not yet implemented.
     * It returns true if merged.
     *
     * @param namedObject a <code>NamedObject</code> to be merged
     *
     * @return true if merged, or false if not mergiable
     */
    public boolean merge(NamedObject namedObject)
    {
	if(!getClassNameHeader().equals(namedObject.getClassNameHeader()))
	    return false;
	return false; //not yet implemented
    }
    
    /**
     * Parses a <code>line</code> and sets values of this object accordint to it
     *
     * @param line <code>String</code> containing fragment of data to be set
     */
    public void parseLine(String line)
    {
    }
    
    /**
     * Returns <code>Date</code> when the author affiliated to the institute
     *
     * @return Date when the author affiliated to the institute
     */
    public Date getFrom()
    {
	return from;
    }
    
    /**
     * Sets <code>from</code> as <code>Date</code> when the author affiliated to the institute
     *
     * @param from <code>Date</code> when the author affiliated to the institute
     */
    public void setFrom(Date from)
    {
	if(this.from == from)
	    return;

	this.from = from;
    }
    
    /**
     * Returns <code>Date</code> when the author unaffiliated from the institute
     *
     * @return Date when the author unaffiliated from the institute
     */
    public Date getUntil()
    {
	return until;
    }
    
    /**
     * Sets <code>until</code> as <code>Date</code> when the author unaffiliated from the institute
     *
     * @param until <code>Date</code> when the author unaffiliated from the institute
     */
    public void setUntil(Date until)
    {
	if(this.until == until)
	    return;

	this.until = until;
    }
    
    /**
     * Returns <code>Author</code> whose affiliation is represented by this
     *
     * @return Author whose affiliation is represented by this
     */
    public Author getAuthor()
    {
	return author;
    }
    
    /**
     * Sets <code>author</code> whose affiliation is represented by this
     *
     * @param author <code>Author</code> whose affiliation is represented by this
     */
    public void setAuthor(Author author)
    {
	if(this.author == author)
	    return;

	this.author = author;
    }
    
    /**
     * Returns <code>String</code> representing name of the institution
     *
     * @return String representing name of the institution
     */
    public String getInstitute()
    {
	return institute;
    }
    
    /**
     * Sets <code>institute</code> as name of the institution
     *
     * @param institute <code>String</code> representing name of the institution
     */
    public void setInstitute(String institute)
    {
	if(this.institute == institute)
	    return;

	this.institute = institute;
    }
    
    /**
     * Returns XML <code>String</code> representing this object
     *
     * @return XML <code>String</code> representing this object
     */
    public String toXMLString()
    {
	StringBuffer buf = new StringBuffer();
	
	buf.append("<Affiliation>\n<oid>");
	buf.append(persistentID());
	buf.append("</oid>\n");
	if(getAuthor() != null) {
    	    buf.append("<author>").append(getAuthor().persistentID()).append("</author>\n");
    	}
	buf.append("<institute>").append(getInstitute()).append("</institute>\n");
	if(getFrom() != null) {
    	    buf.append("<from>").append(getFrom().toString()).append("</from>\n");
    	}
    	if(getUntil() != null) {
    	    buf.append("<until>").append(getUntil().toString()).append("</until>\n");
    	}
	buf.append("</Affiliation>\n");
	
        return buf.toString();
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	if(!(object instanceof Affiliation))
	    throw new IllegalArgumentException(object.getClass().getName() + " can't be an entity of " + getClass().getName());
	entity = (Affiliation)object;
	super.setEntity(object);
    }

}
