/*
 * Annotation.java:  a Java implementation of Annotation class
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Annotation.java,v 1.17 2002/11/12 03:16:20 nozomi Exp $
 * $Log: Annotation.java,v $
 * Revision 1.17  2002/11/12 03:16:20  nozomi
 * use Nomencurator as an Object pool with name resolver
 *
 * Revision 1.16  2002/10/01 18:27:40  nozomi
 * update PID separator count
 *
 * Revision 1.15  2002/10/01 17:13:21  nozomi
 * fix perisistentID() to use first annotator
 *
 * Revision 1.14  2002/09/27 20:11:47  nozomi
 * accessor methods parse persistentID if isNominal()
 *
 * Revision 1.13  2002/09/17 04:57:59  nozomi
 * support indirect reference
 *
 * Revision 1.12  2002/09/10 20:06:02  nozomi
 * add Annotation(Element)
 *
 * Revision 1.11  2002/09/05 05:15:30  nozomi
 * add DEFAULT_LINK_TYPE
 *
 * Revision 1.10  2002/05/17 12:20:37  ryo
 * change function name from toXML() to toXMLString()
 *
 * Revision 1.9  2002/05/16 09:03:48  ryo
 * fix bug in persistentID()
 *
 * Revision 1.8  2002/04/16 03:53:37  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.7  2002/04/09 03:26:17  nozomi
 * annotator and annotatant can be add only once to these lists
 *
 * Revision 1.6  2002/04/09 03:01:40  nozomi
 * change emptyPersistentID handling
 *
 * Revision 1.5  2002/04/08 01:45:30  nozomi
 * Change StringDelegate to Name
 *
 * Revision 1.4  2002/04/01 07:03:18  nozomi
 * add setAnnotators() and addAnnotatants()
 *
 * Revision 1.3  2002/04/01 06:48:23  nozomi
 * add toXML
 *
 * Revision 1.2  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 *
 * Revision 1.1  1999/10/24 08:06:30  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import jp.kyasu.util.Sorter;

import org.nomencurator.NamedObject;
import org.nomencurator.Appearance;

import org.nomencurator.util.Comparer;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * <code>Annotation</code> implements Annotation data structure in Nomencurator data model
 *
 * @see org.nomencurator.Appearance
 * @see org.nomencurator.NameUsage
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org/</A>
 *
 * @version 	12 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class Annotation
    extends NamedObject
    implements Serializable
{
    public static final String DEFAULT_LINK_TYPE = "refer";
    
    /** type of linkage between <code>NameUsage</code>s */
    protected String linkType;

    /** Vector containing annotator <code>NameUsage</code>s */
    protected Vector annotators;

    /** Vector containing annotatant <code>NameUsage</code>s */
    protected Vector annotatants;

    /** <code>Appearance</code> encoding this <code>Annotation</code> */
    protected Appearance appearance;

    protected static Sorter sorter = new Sorter(new Comparer());

    /**
     * Constructs an empty <code>Annotation</code> object
     */
    public Annotation()
    {
	this(DEFAULT_LINK_TYPE, null, new Vector(), new Vector());
    }
    
    /**
     * Constructs an <code>Annotation</code> having
     * <code>name</code> as its name, i.e. perisitent ID
     *
     * @param name <code>String</code> representing its name,
     * i.e. perisitent ID
     */
    public Annotation(String name)
    {
	this();
	setPersistentID(name);
    }
    

    /**
     * Constructs a copy of given <code>name</code>
     *
     * @param name <code>Name</code> representation of an <code>Annotation</code>
     */
    public Annotation(Name name)
    {
	//don't use NamedObject(String) constructor
	this();
	
	if(name instanceof Annotation) {
	    Annotation a = (Annotation)name;
	    linkType    = a.linkType;
	    appearance  = a.appearance;
	    annotators  = a.annotators;
	    annotatants = a.annotatants;
	}
	else
	    setPersistentID(name.getName());
    }
    
    /**
     * Constructs a non-trivial <code>Annotation</code>.
     *
     * @param type <code>String</code> representing linkage type
     * @param citation <code>Apperance</code> encoding this <code>Annotation</code>
     * @param annotators <code>Vector</code> containing annotating <code>NameUsage</code>s
     * @param annotatants <code>Vector</code> containing annotated <code>NameUsage</code>s
     */
    public Annotation(String type, Appearance citation,
		      Vector annotators, Vector annotatants)
    {
	super();
	linkType = type;
	appearance = citation;
	
	this.annotators = annotators;
	this.annotatants = annotatants;
    }

    /**
     * Constructs an <CODE>Annotaion</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying an <CODE>Annotation</CODE>
     *
     */
    public Annotation(Element xml)
    {
	this(xml, null);
    }

    /**
     * Constructs an <CODE>Annotaion</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying an <CODE>Annotation</CODE>
     *
     */
    public Annotation(Element xml, Appearance ap)
    {
	super();
	NodeList nodeList = xml.getChildNodes();
	int nodeCount = nodeList.getLength();
	String persistentID = null;
	int authorCount = 0;
	boolean toBeResolved = false;

	for (int j = 0; j < nodeCount; j++) {
	    Node node = nodeList.item(j);
	    if(node.getNodeType () == Node.ELEMENT_NODE) {
		Element element = (Element)node;
		String tagName = element.getTagName();

		if (tagName.equals ("oid"))
		    persistentID = getString(element);
		else if(tagName.equals ("linktype"))
		    linkType =  getString(element);
		else if(tagName.equals ("appearance")) {
		    String pID = getString(element);
		    if(ap != null &&
		       ap.getName().equals(pID)) {
			appearance = ap;
			toBeResolved = true;
		    }
		    else{
			ap =
			    (Appearance)curator.get(pID);
			if(ap != null) {
			    appearance = ap;
			    toBeResolved = true;
			}
			else {
			    appearance = new Appearance();
			    appearance.setName(pID);
			    curator.put(appearance);
			    appearance.addAnnotation(this);
			}
		    }
		    /*
		    LinkRecord rec = new LinkRecord(getString(ele), annotation.persistentID());
		    AppearanceAnnotation.getInstance().addLinkRecord(rec);
		    */
		}
		else if(tagName.equals ("from")) {
		    String pID = getString(element);
		    NameUsage n =
			(NameUsage)curator.get(pID);
		    if(n != null) {
			if(annotators == null)
			    annotators = new Vector();
			annotators.addElement(n);
			toBeResolved = true;
		    }
		    else {
			n = new NameUsage();
			n.setName(pID);
			curator.put(n);
			addAnnotator(n);
		    }

		    /*
		    LinkRecord rec = new LinkRecord(annotation.persistentID(), getString(element);
		    AnnotationFrom.getInstance().addLinkRecord(rec);
		    */
		}
		else if(tagName.equals ("to")) {
		    String pID = getString(element);
		    NameUsage n =
			(NameUsage)curator.get(pID);
		    if(n != null) {
			if(annotatants == null)
			    annotatants = new Vector();
			annotatants.addElement(n);
			toBeResolved = true;
		    }
		    else {
			n = new NameUsage();
			n.setName(pID);
			curator.put(n);
			addAnnotatant(n);
		    }

		    /*
		    LinkRecord rec = new LinkRecord(annotation.persistentID(), getString(element);
		    AnnotationTo.getInstance().addLinkRecord(rec);
		    */
		}
		else{}
            }
        }

	if(persistentID != null &&
	   !persistentID.equals(getName()))
	    setName(persistentID); //i.e. other key data are empty

	if(toBeResolved)
	    curator.resolve(this);
    }
    
    /**
     * Returns a persistent ID representing this <code>Annotation</code>
     *
     * @return String representing this <code>Annotation</code>
     */
    public String persistentID()
    {
	if(entity != null)
	    return ((Annotation)getEntity()).persistentID();

	if(string != null && string.length() != 0)
	    return string;
	
	StringBuffer buffer = getClassNameHeaderBuffer();
	buffer.append(linkType);
	buffer.append(fieldSeparator);

	if(annotators != null && !annotators.isEmpty()) {
	    int size = annotators.size();
	    String[] annotatorNames = new String[size];
	    for(int i = 0; i < size; i++) {
		annotatorNames[i] =
		    ((NameUsage)annotators.elementAt(i)).getAscribedName();
	    }
	    sorter.quicksort(annotatorNames, 0, annotatorNames.length - 1);
	    buffer.append(annotatorNames[0]);
	}
	buffer.append(fieldSeparator);

	if(appearance == null){
	    //request code here;
	}
	if(appearance == null) {
		buffer.append(new Appearance().getPersistentIDContents());
	}
	else{
	    buffer.append(appearance.getPersistentIDContents());
	}

	return buffer.toString();
    }
    
    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public int getFieldSepartorsCount()
    {
	return 10; //linktype, annotator, + 8 from Appearance
    }

    /**
     * Merges <code>namedObject</code> with this <code>Annotation</code>
     * if possible, but not yet implemented.
     * It returns true if merged.
     *
     * @param namedObject a <code>NamedObject</code> to be merged
     *
     * @return true if merged, or false if not mergiable
     */
    public boolean merge(NamedObject namedObject)
    {
	if(!(namedObject instanceof Annotation))
	    return false;
	return false; //not yet implemented
    }
    
    /**
     * Parses a <code>line</code> and sets values of this object accordint to it
     *
     * @param line <code>String</code> containing fragment of data to be set
     */
    public void parseLine(String line)
    {
	if(entity != null)
	    ((Annotation)getEntity()).parseLine(line);

	String s = peal(line);
	if(line.startsWith("reason"))
	    linkType = s;
	else if(line.startsWith("annotators"))
	    annotators.addElement(new NameUsage(s));
	else if(line.startsWith("annotatants"))
	    annotatants.addElement(new NameUsage(s));
	else if(line.startsWith("appearance"))
	    appearance  = new Appearance(s);
    }

    /**
     * Returns <code>String</code> representing link type
     *
     * @return String representing link type
     */
    public String getLinkType()
    {
	if(entity != null)
	    return ((Annotation)getEntity()).getLinkType();

	if(!isNominal())
	    return linkType;

	return getPersistentIDComponent(0);
    }

    /**
     * Sets <code>type</code> as linkage type of this <code>Annotation</code>
     * if it is different from current link type.
     *
     * @param type <code>String</code> representing linkage type to be set
     */
    public void setLinkType(String type)
    {
	if(entity != null)
	    ((Annotation)getEntity()).setLinkType(type);

	if(type != null &&  type.equals(linkType))
	    return;

	linkType = type;
    }

    /**
     * Returns <code>Appearance</code> encoding this <code>Appearance</code>
     *
     * @return <code>Appearance</code> encoding this <code>Appearance</code>
     */
    public Appearance getAppearance()
    {
	if(entity != null)
	    return ((Annotation)getEntity()).getAppearance();

	return appearance;
    }

    /**
     * Sets <code>appearance</code> as <code>Appearance</code> encoding this <code>Appearance</code>
     * if it is different from what currently stored.
     *
     * @param <code>appearance</code> <code>Appearance</code> to be set as encoder of this <code>Appearance</code>
     */
    public void setAppearance(Appearance appearance)
    {
	if(entity != null)
	    ((Annotation)getEntity()).setAppearance(appearance);

	if(this.appearance == appearance)
	    return;

	this.appearance =  appearance;
    }

    /**
     * Returns a <code>Vector</code> of annotator <code>NameUsage</code>s
     *
     * @return <code>Vector</code> containing annotator <code>NameUsage</code>s
     */ 
    public Vector getAnnotators()
    {
	if(entity != null)
	    return ((Annotation)getEntity()).getAnnotators();

	return annotators;
    }
    
    /**
     * Sets <code>nameUsages</code> as the <code>Vector</code> of annotator <code>NameUsage</code>s
     *
     * @param nameUsages <code>Vector</code> containing annotator <code>NameUsage</code>s
     */ 
    public void setAnnotators(Vector nameUsages)
    {
	if(entity != null)
	    ((Annotation)getEntity()).setAnnotators(nameUsages);

	if(annotators == nameUsages)
	    return;
	annotators = nameUsages;
    }
    
    /**
     * Adds a <code>NameUsage</code> as an annotator of this <code>Annotation</code>s
     * with returning true if it is not contained in the annotator list.
     * Otherwise it returs false without re-addition of 
     * <code>nameUsage</code> to the list
     *
     * @param <code>nameUsage</code> <code>NameUsage</code> to be an annotator of this <code>Annotation</code>s
     *
     * @return true if <code>nameUsage</code> added to the list successfully,
     * or false if not.
     */ 
    public boolean addAnnotator(NameUsage nameUsage)
    {
	if(entity != null)
	    return ((Annotation)getEntity()).addAnnotator(nameUsage);

	if(nameUsage == null)
	    return false;

	if(annotators == null)
	    annotators = new Vector();

	if(annotators.contains(nameUsage))
	    return false;
	
	annotators.addElement(nameUsage);
	nameUsage.addAnnotation(this);
	
	return true;
    }
    
    /**
     * Removes a <code>nameUsage</code> from annotator list of this <code>Annotation</code>s
     *
     * @param <code>nameUsage</code> <code>NameUsage</code> to be removed from annotator list of this <code>Annotation</code>s
     */ 
    public void removeAnnotator(NameUsage nameUsage)
    {
	if(entity != null)
	    ((Annotation)getEntity()).removeAnnotator(nameUsage);

	if(nameUsage == null ||
	   annotators == null ||
	   !annotators.contains(nameUsage))
	    return;

	annotators.removeElement(nameUsage);
	nameUsage.removeAnnotation(this);
    }
    
    /**
     * Returns a <code>Vector</code> of annotatant <code>NameUsage</code>s
     *
     * @return <code>Vector</code> containing annotatant <code>NameUsage</code>s
     */ 
    public Vector getAnnotatants()
    {
	if(entity != null)
	    return ((Annotation)getEntity()).getAnnotatants();

	return annotatants;
    }
    
    /**
     * Sets <code>nameUsages</code> as the <code>Vector</code> of annotatant <code>NameUsage</code>s
     *
     * @param nameUsages <code>Vector</code> containing annotatant <code>NameUsage</code>s
     */ 
    public void setAnnotatants(Vector nameUsages)
    {
	if(entity != null)
	    ((Annotation)getEntity()).setAnnotatants(nameUsages);

	if(annotatants == nameUsages)
	    return;
	annotatants = nameUsages;
    }
    
    /**
     * Adds a <code>NameUsage</code> as an annotatant of this <code>Annotation</code>s
     * with returning true if it is not contained in the annotatant list.
     * Otherwise it returs false without re-addition of 
     * <code>nameUsage</code> to the list
     *
     * @param <code>nameUsage</code> <code>NameUsage</code> to be an annotatant of this <code>Annotation</code>s
     *
     * @return true if <code>nameUsage</code> added to the list successfully,
     * or false if not.
     */ 
    public boolean addAnnotatant(NameUsage nameUsage)
    {
	if(nameUsage == null)
	    return false;

	if(entity != null)
	    return ((Annotation)getEntity()).addAnnotatant(nameUsage);

	if(annotatants == null)
	    annotatants = new Vector();

	if(annotatants.contains(nameUsage))
	    return false;

	annotatants.addElement(nameUsage);
	return true;
    }
    
    /**
     * Removes a <code>nameUsage</code> from annotatant list of this <code>Annotation</code>s
     *
     * @param <code>nameUsage</code> <code>NameUsage</code> to be removed from annotatant list of this <code>Annotation</code>s
     */ 
    public void removeAnnotatant(NameUsage nameUsage)
    {
	if(entity != null)
	    ((Annotation)getEntity()).removeAnnotatant(nameUsage);

	annotatants.removeElement(nameUsage);
    }
    
    /**
     * Returns an expression of this <code>Annotation</code> as XML <code>String</code>
     *
     * @return XML String of this <code>Annotation</code>
     */
    public String toXMLString()
    {
	if(entity != null)
	    return ((Annotation)getEntity()).toXMLString();

	StringBuffer buf = new StringBuffer();
	    
	buf.append("<Annotation>\n<oid>");
	buf.append(persistentID());
	buf.append("</oid>\n<linktype>");
	buf.append(linkType);
	buf.append("</linktype>\n<appearance>");
	buf.append(appearance);
	buf.append("</appearance>\n");
	if(annotators != null) {
            Enumeration e = annotators.elements();
	    while(e.hasMoreElements()) {
    	        buf.append("<from>");
		buf.append(((NameUsage)(e.nextElement())).persistentID());
		buf.append("</from>\n");
	    }
	}
	if(annotatants != null) {
            Enumeration e = annotatants.elements();
	    while(e.hasMoreElements()) {
    	        buf.append("<to>");
		buf.append(((NameUsage)(e.nextElement())).persistentID());
		buf.append("</to>\n");
	    }
	}
	buf.append("</Annotation>\n");
	
        return buf.toString();
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	if(object == this ||
	   (object != null && !(object instanceof Annotation)))
	    throw new IllegalArgumentException(object.getClass().getName() + " can't be an entity of " + getClass().getName());
	super.setEntity(object);
    }

};
