/*
 * Appearance.java:  a Java implementation of Appearance class
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Appearance.java,v 1.20 2002/11/12 03:16:20 nozomi Exp $
 * $Log: Appearance.java,v $
 * Revision 1.20  2002/11/12 03:16:20  nozomi
 * use Nomencurator as an Object pool with name resolver
 *
 * Revision 1.19  2002/10/01 18:27:40  nozomi
 * update PID separator count
 *
 * Revision 1.18  2002/09/27 19:57:03  nozomi
 * accessor methods parse persistentID if isNominal()
 *
 * Revision 1.17  2002/09/17 04:57:59  nozomi
 * support indirect reference
 *
 * Revision 1.16  2002/09/10 19:53:29  nozomi
 * add Appearance(Element)
 *
 * Revision 1.15  2002/09/09 05:40:37  nozomi
 * use ANONYMOUS insetad of nullString
 *
 * Revision 1.14  2002/09/04 07:05:42  t.okada
 * fieldSeparator is added to persistentID
 *
 * Revision 1.13  2002/09/04 06:30:56  t.okada
 * lines item is added to persistentID
 *
 * Revision 1.12  2002/08/27 01:57:50  nozomi
 * add Apparance(Publication) constructor
 *
 * Revision 1.11  2002/08/01 07:19:32  ryo
 * migration to NameUsage from NameRecord
 *
 * Revision 1.10  2002/05/10 13:30:15  ryo
 * comment out debug message
 *
 * Revision 1.9  2002/04/16 23:43:51  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.8  2002/04/16 03:53:37  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.7  2002/04/10 05:08:27  nozomi
 * use nullString as empty value
 *
 * Revision 1.6  2002/04/09 03:02:02  nozomi
 * change emptyPersistentID handling
 *
 * Revision 1.5  2002/04/08 01:45:30  nozomi
 * Change StringDelegate to Name
 *
 * Revision 1.4  2002/04/01 06:48:30  nozomi
 * use java.util.Vector
 *
 * Revision 1.3  2002/03/08 13:53:47  okawa
 * add setPublication()
 *
 * Revision 1.2  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 *
 * Revision 1.1  1999/10/24 08:06:32  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.Enumeration;
import java.util.Vector;

import org.nomencurator.NamedObject;
import org.nomencurator.Annotation;
import org.nomencurator.Publication;
import org.nomencurator.NameUsage;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An implementation of Appearance data structure of Nomencurator data model
 *
 * @see 	org.nomencurator.NamedObject
 * @see 	org.nomencurator.NameUsage
 * @see 	org.nomencurator.Annotation
 * @see 	org.nomencurator.Publication
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org/</A>
 *
 * @version 	12 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class Appearance
    extends NamedObject
    implements Serializable
{
    /**
     * <code>Vector</code> of <code>NameUsage</code>s
     * encoded by this <code>Appearance</code> object</code>
     */
    protected Vector nameUsages;

    /**
     * <code>Vector</code> of <code>Annotation</code>s
     * encoded by this <code>Appearance</code> object</code>
     */
    protected Vector annotations;

    /** <code>Publication</code> where the appearance appeared */
    protected Publication publication;

    /** Index of page(s) in persistentID */
    public static final int PAGE = 0;

    /** Index of lines in persistentID */
    public static final int LINES = PAGE + 1;

    /** Index of publication in persistentID */
    public static final int PUBLICATION = LINES + 1;

    /** Index of author(s) in persistentID */
    public static final int AUTHOR = PUBLICATION + Publication.AUTHOR;
    /** Index of year in persistentID*/
    public static final int YEAR = PUBLICATION + Publication.YEAR;
    /** Index of title or ISXN in persistentID*/
    public static final int TITLE = PUBLICATION + Publication.TITLE;
    /** Index of volume in persistentID*/
    public static final int VOLUME = PUBLICATION + Publication.VOLUME;
    /** Index of number in persistentID*/
    public static final int NUMBER = PUBLICATION + Publication.NUMBER;
    /** Index of first page in persistentID*/
    public static final int FIRST_PAGE = PUBLICATION + Publication.FIRST_PAGE;
    /** Index of last page in persistentID*/
    public static final int LAST_PAGE = PUBLICATION + Publication.LAST_PAGE;
    /** number of items in the persistentID*/
    public static final int PID_ITEMS = LAST_PAGE;

    /** 
     * <code>String</code> represents the page where
     * the appearance appeared in the publication
     */
    protected String page;
     
    /**
     * <code>String</code> represents the lines on the page
     * where the appearance appeared.
     * It may represent either single line or a range.
     */
    protected String lines;

    /** <code>String</code> represents apperance of names itself */
    protected String appearance;

    /**
     * Constructs an empty <code>Appearance</code> object
     */
    public Appearance()
    {
	super();
	/*
	appearance = ANONYMOUS;
	page       = ANONYMOUS;
	lines      = ANONYMOUS;

	nameUsages = new Vector();
	annotations = new Vector();
	*/
    }

    /**
     * Constructs an <code>Appearance</code> having
     * <code>name</code> as its name, i.e. perisitent ID
     *
     * @param name <code>String</code> representing its name,
     * i.e. perisitent ID
     */
    public Appearance(String name)
    {
	super(name);
    }

    /**
     * Constructs a copy of given <code>name</code>
     *
     * @param name <code>Name</code> representation of an <code>Appearance</code>
     */
    public Appearance(Name name)
    {
	//don't use NamedObject(String) constructor
	this();
	
	if(name instanceof Appearance){
	    Appearance a = (Appearance)name;

	    publication = a.publication;
	    page        = a.page;
	    lines       = a.lines;

	    appearance  = a.appearance;

	    nameUsages = (Vector)a.nameUsages.clone();
	    annotations = (Vector)a.annotations.clone();
	    
	}
	else
	    setPersistentID(name.getName());
    }

    /**
     * Constructs an <code>Appearance</code> in
     * <CODE>publication</CODE>
     *
     * @param publication <code>Publication</code> where
     * this appeared
     */
    public Appearance(Publication publication)
    {
	this();
	setPublication(publication);
    }

    /**
     * Constructs a non-trivial <code>Appearance</code>.
     *
     * @param publication <code>Publication</code> where
     * this appeared
     */
    public Appearance(Publication publication,
		      String page,
		      String lines,
		      String appearance,
		      Vector nameUsages,
		      Vector annotations)
    {
	this();
	
	this.publication = publication;

	this.page        = page;

	this.lines = lines;

	this.appearance  = appearance;
	
	this.nameUsages = (Vector)nameUsages.clone();
	this.annotations = (Vector)annotations.clone();
    }

    /**
     * Constructs an <CODE>Appearance</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying an <CODE>Appearance</CODE>
     *
     */
    public Appearance(Element xml)
    {
	this(xml, null);
    }

    /**
     * Constructs an <CODE>Appearance</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying an <CODE>Appearance</CODE>
     *
     */
    public Appearance(Element xml, Publication pub)
    {
	super();
	NodeList nodeList = xml.getChildNodes();
	int nodeCount = nodeList.getLength();
	String persistentID = null;
	int authorCount = 0;
	boolean toBeResolved = false;

	for (int j = 0; j < nodeCount; j++) {
	    Node node = nodeList.item(j);
	    if(node.getNodeType () == Node.ELEMENT_NODE) {
		Element element = (Element)node;
		String tagName = element.getTagName();

		if (tagName.equals ("oid"))
		    persistentID = getString(element);
		else if(tagName.equals ("page"))
		    page =  getString(element);
		else if(tagName.equals ("lines"))
		    lines = getString(element);
		else if(tagName.equals ("appearance"))
		    appearance = getString(element);
		else if(tagName.equals ("publication")) {
		    String pID = getString(element);
		    Publication p = null;
		    if(pub != null &&
		       pub.getName().equals(pID))
			p = pub;
		    else{
			p = (Publication)curator.get(pID);
			if(p != null) {
			    publication = p;
			    toBeResolved = true;
			}
			else {
			    p = new Publication();
			    p.setName(pID);
			    curator.put(p);
			    p.addAppearance(this);
			}
		    }
		}
		else if(tagName.equals ("annotation")) {
		    String pID = getString(element);
		    Annotation a =
			(Annotation)curator.get(pID);
		    if(a != null) {
			if(annotations == null)
			    annotations = new Vector();
			annotations.addElement(a);
			toBeResolved = true;
		    }
		    else {
			a = new Annotation();
			a.setName(pID);
			curator.put(a);
			addAnnotation(a);
		    }
		    /*
		    LinkRecord rec = new LinkRecord(appearance.persistentID(), getString(element));
		    AppearanceAnnotation.getInstance().addLinkRecord(rec);
		    */
		}
		else if(tagName.equals ("nameusage")) {
		    String pID = getString(element);
		    NameUsage n =
			(NameUsage)curator.get(pID);
		    if(n != null) {
			if(nameUsages == null)
			    nameUsages = new Vector();
			nameUsages.addElement(n);
			//addNameUsage(n);
			toBeResolved = true;
		    }
		    else {
			n = new NameUsage();
			n.setName(pID);
			curator.put(n);
			addNameUsage(n);
		    }

		    /*
		    LinkRecord rec = new LinkRecord(appearance.persistentID(), getString(element));
		    AppearanceNameUsage.getInstance().addLinkRecord(rec);
		    */
		}
		else{}
            }
        }

	if(persistentID != null &&
	   !persistentID.equals(getName()))
	    setName(persistentID); //i.e. other key data are empty

	if(toBeResolved)
	    curator.resolve(this);
    }

    /**
     * Returns a persistent ID representing this <code>Appearance</code>
     *
     * @return String representing this <code>Appearance</code>
     */
    public String persistentID()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).persistentID();

	if(string != null && string.length() != 0)
	    return string;
	
	StringBuffer pid = getClassNameHeaderBuffer();

	if(page != null)
	    pid.append(page);
	pid.append(fieldSeparator);
	if(lines != null)
	    pid.append(lines);
	pid.append(fieldSeparator);
	if(publication == null){
	    //request code here;
	}
	if(publication == null)
	    pid.append(publicationString());
	else{
	    pid.append(publication.persistentID());
	}
	return pid.toString();
    }
    
    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public int getFieldSepartorsCount()
    {
	return 8; //after page, line and 6 from Publication
    }

    /**
     * Merges <code>namedObject</code> with this <code>Appearance</code>
     * if possible, but not yet implemented.
     * It returns true if merged.
     *
     * @param namedObject a <code>NamedObject</code> to be merged
     *
     * @return true if merged, or false if not mergiable
     */
    public boolean merge(NamedObject namedObject)
    {
	if(!getClassNameHeader().equals(namedObject.getClassNameHeader()))
	    return false;
	return false; //not yet implemented
    }
    
    /**
     * Returns <code>String</code> encoding publication.
     * It is equivalent to persistent ID of a <code>Publication</code>
     * without "class name header".
     *
     * @return <code>String</code> encoding publication
     */
    public String publicationString()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).publicationString();

	String ret = "";
	String classHeader;
	if(publication != null) {
	    ret = publication.persistentID();
	    classHeader = publication.getClassNameHeader();
	}
	else {
	    if(string == null || string.length() == 0)
		return ret;
	    
	    classHeader = new Publication().getClassNameHeader();
	    ret = string;
	}
	
	return ret.substring(ret.indexOf(classHeader) + classHeader.length());
    }
    
    /**
     * Parses a <code>line</code> and sets values of this object accordint to it
     *
     * @param line <code>String</code> containing fragment of data to be set
     */
    public void parseLine(String line)
    {
	String s = peal(line);
	if(line.startsWith("page"))
	    page = s;
	else if(line.startsWith("lines"))
	    lines = s;
	else if(line.startsWith("appearance"))
	    appearance = s;
	else if(line.startsWith("Publication"))
	    publication = new Publication(s);
	else if(line.startsWith("NameUsage"))
	    nameUsages.addElement(new NameUsage(s));
	else if(line.startsWith("Annotation"))
	    annotations.addElement(new Annotation(s));
    }
    
    /**
     * Returns <code>String</code> representing appearance.

     * appearance extractor
     * when apperance is not yet given,
     * extract it from persistentID() automatically.
     * once extracted, it will be used in later request
     */
    public String getAppearance()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).getAppearance();
	/*
	if(appearance == null) {
	    appearance = persistentID().substring(getClassNameHeader().length(), string.indexOf(fieldSeparator));
	}
	*/
	return appearance;
    }

    /**
     * Sets <code>appearance</code> as name appearance
     *
     * @param appearance <code>String</code> to be set
     */
    public void setAppearance(String appearance)
    {
	if(entity != null)
	    ((Appearance)getEntity()).setAppearance(appearance);

    	if(this.appearance == appearance ||
	   (appearance != null && appearance.equals(this.appearance)))
	    return;

   	this.appearance = appearance;
    }
    

    
    /**
     * Returns page.
     * It extracts page from persistentID() automatically
     * when page is not yet given.
     * Once extracted, it will be used in later request.
     *
     * @return page <code>String</code> representing page
     */
    /*
    public String page()
    {
    }
    */

    /**
     * Returns page.
     * It extracts page from persistentID() automatically
     * when page is not yet given.
     * Once extracted, it will be used in later request.
     *
     * @return page <code>String</code> representing page
     */
    public String getPage()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).getPage();

	/*
	if(page != null)
	    return page;

	String p = getPersistentIDComponent(0);

	if(p.length() > 0)
	    page = p;

	return page;
	*/

	if(!isNominal())
	    return page;

	return getPersistentIDComponent(0);
    }

    /**
     * Sets <code>page</code>
     *
     * @param page <code>String</code> to be set as page
     */
    public void setPage(String page)
    {
	if(entity != null)
	    ((Appearance)getEntity()).setPage(page);

    	if(this.page == page ||
	   (page != null && page.equals(this.page)))
	    return;

    	this.page = page;
    }
    

    /**
     * Returns <code>String</code> representing lines where names are appeared.
     *
     * @return <code>String</code> representing lines where names are appeared.
     */
    public String getLines()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).getLines();

	/*
	if(lines != null)
	    return lines;

	String l = getPersistentIDComponent(1);

	if(l.length() > 0)
	    lines = l;

	return lines;
	*/

	if(!isNominal())
	    return lines;

	return getPersistentIDComponent(1);
    }

    /**
     * Sets <code>lines</code>
     *
     * @param lines <code>String</code> to be set as lines
     */
    public void setLines(String lines)
    {
	if(entity != null)
	    ((Appearance)getEntity()).setLines(lines);

    	if(this.lines == lines ||
	   (lines != null && lines.equals(this.lines)))
	    return;

    	this.lines = lines;
    }

    /**
     * Returns <code>Publication</code> containing this appearance
     * @return Publication containing this appearance
     */
    public Publication getPublication()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).getPublication();

	return publication;
    }
    
    /**
     * Sets publication as the container of this appearance.
     *
     * @param publication <code>Publication</code> to be set as the container of this appearance
     */
    public void setPublication(Publication publication)
    {
	if(entity != null)
	    ((Appearance)getEntity()).setPublication(publication);

	if(this.publication == publication)
	    return;

	if(this.publication != null) {
	    //set null to avoid infinity loop
	    Publication p = this.publication;
	    this.publication =  null;
	    p.removeAppearance(this);
	}

	this.publication =  publication;
	if(this.publication != null)
	    this.publication.addAppearance(this);
    }
    
    /**
     * Returns <code>Vectocr</code> representing <code>Annotation</code> list
     *
     * @return Vector representing <code>Annotation</code> list
     */
    public Vector getAnnotations()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).getAnnotations();

	return annotations;
    }
    
    /**
     * Sets annotations as the <code>Annotation</code> list
     * @param annotation <code>Vector</code> representing <code>Annotation</code> list
     */
    public void setAnnotations(Vector annotations)
    {
	if(entity != null)
	    ((Appearance)getEntity()).setAnnotations(annotations);

	if(this.annotations == annotations)
	    return;

	this.annotations = annotations;
    }
  
    /**
     * Add <code>annotation</code> to <code>Annotation</code> list
     * if it is not contained in the list.
     *
     * @param annotation <code>Annotation</code> to be added
     */
    public boolean addAnnotation(Annotation annotation)
    {
	if(entity != null)
	    return ((Appearance)getEntity()).addAnnotation(annotation);

	if(annotations == null)
	    annotations = new Vector();

	if(annotations.contains(annotation))
	    return false;

	annotations.addElement(annotation);
	return true;
    }
  
    /**
     * Returns <code>Vector</code> representing <code>NameUsage</code> list
     *
     * @return <code>Vector</code> representing <code>NameUsage</code> list
     */
    public Vector getNameUsages()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).getNameUsages();

	return nameUsages;
    }
    
    /**
     * Sets nameUsages as <code>NameUsage</code> list
     *
     * @param nameRecrods <code>Vector</code> representing <code>NameUsage</code> List
     */
    public void setNameUsages(Vector nameUsages)
    {
	if(entity != null)
	    ((Appearance)getEntity()).setNameUsages(nameUsages);

	if(this.nameUsages == nameUsages)
	    return;

	this.nameUsages = nameUsages;
    }

    /**
     * Add <code>nameUsage</code> to <code>NameUsage</code> list
     * with returning true if it is not contained in the list.
     * Otherwise it returs false without re-addition of 
     * <code>nameUsage</code> to the list
     *
     * @param nameUsage <code>NameUsage</code> to be added
     *
     * @return true if <code>nameUsage</code> added to the list successfully,
     * or false if not.
     */
    public boolean addNameUsage(NameUsage nameUsage)
    {
	if(entity != null)
	    return ((Appearance)getEntity()).addNameUsage(nameUsage);

	if(nameUsages == null)
	    nameUsages = new Vector();

	if(nameUsages.contains(nameUsage))
	    return false;

	nameUsages.addElement(nameUsage);
	nameUsage.setAppearance(this);
	return true;
    }
  
    /**
     * Returns an expression of this <code>Appearance</code> as XML <code>String</code>
     *
     * @return XML String of this <code>Appearance</code>
     */
    public String toXMLString()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).toXMLString();

	StringBuffer buf = new StringBuffer();
	buf.append("<Appearance>\n");
	buf.append("<oid>" + persistentID() + "</oid>\n");

	if(page != null && page.length() > 0)
	    buf.append("<page>" + page + "</page>\n");

	if(lines != null && lines.length() > 0)
	    buf.append("<lines>" + lines + "</lines>\n");

	if(appearance != null && appearance.length() > 0)
	    buf.append("<appearance>" + appearance + "</appearance>\n");
	
        Publication publication = getPublication();
        if(publication != null) {
	    buf.append("<publication>" + publication.persistentID() + "</publication>\n");
        }

	if(nameUsages != null) {
		for (Enumeration e = nameUsages.elements(); e.hasMoreElements(); ) {
			buf.append("<nameusage>" + ((NameUsage)e.nextElement()).persistentID() + "</nameusage>\n");
		}
	}

	if(annotations != null) {
		for (Enumeration e = annotations.elements(); e.hasMoreElements(); ) {
			buf.append("<annotation>" + ((Annotation)e.nextElement()).persistentID() + "</annotation>\n");
		}
	}

	buf.append("</Appearance>\n");
	    
        return buf.toString();
    }
    
    /**
     * Returns XML <code>String</code> of the all related <code>NamedObject</code>s
     *
     * @return XML <code>String</code> representing all <code>NamedObject</code>s
     * relationg to this <code>Appearance</code>
     */
    public String toRelatedXMLString()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).toRelatedXMLString();

	// create XML String of the Appearance itself
	StringBuffer buf = new StringBuffer();
	buf.append(toXMLString());

	// create XML of the Publication
	if(publication != null) {
	    buf.append(publication.toXMLString());
	}
	
	// create XML of the NameUsages
	if(nameUsages != null) {
            Enumeration e = nameUsages.elements();
	    while(e.hasMoreElements()) {
                NameUsage nameUsage = (NameUsage)e.nextElement();
    	        buf.append(nameUsage.toXMLString());
	    }
        }
	
	// create XML of the Annotations
	if(annotations != null) {
	    Enumeration e = annotations.elements();
	    while(e.hasMoreElements()) {
		Annotation annotation = (Annotation)e.nextElement();
		buf.append(annotation.toXMLString());
	    }
	}
	return buf.toString();
    }

    public String[] toSQL()
    {
	if(entity != null)
	    return ((Appearance)getEntity()).toSQL();

	String[] sql = new String[3];
	int i = 0;
	sql[i++] = persistentID();
	sql[i++] = page;
	//	sql[i++] = lines;
	sql[i++] = appearance;

	return sql;
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	if(object == this ||
	   (object != null &&
	    !(object instanceof Appearance)
	    ))
	    throw new IllegalArgumentException(object.getClass().getName() + " can't be an entity of " + getClass().getName());
	super.setEntity(object);
    }

    //cheating methods to return Publication's data
}
