/*
 * Author.java:  a Java implementation of Author class
 * for Nomencurator data strucutre
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Author.java,v 1.28 2002/11/19 02:05:17 nozomi Exp $
 * $Log: Author.java,v $
 * Revision 1.28  2002/11/19 02:05:17  nozomi
 * remove unused linktable
 *
 * Revision 1.27  2002/11/12 03:16:20  nozomi
 * use Nomencurator as an Object pool with name resolver
 *
 * Revision 1.26  2002/10/18 07:45:50  nozomi
 * remove debug infro printing line
 *
 * Revision 1.25  2002/10/18 02:04:31  nozomi
 * modify getFullname() to retun if only suranme
 *
 * Revision 1.24  2002/09/27 15:05:00  nozomi
 * getSurname/FirstName/MiddleName() parses persistentID if isNominal()
 *
 * Revision 1.23  2002/09/17 04:57:59  nozomi
 * support indirect reference
 *
 * Revision 1.22  2002/09/10 18:10:09  nozomi
 * fix mistype in Author(Element)
 *
 * Revision 1.21  2002/09/10 18:07:24  nozomi
 * modify puttin publiation in Author(Element)
 *
 * Revision 1.20  2002/09/10 07:43:33  nozomi
 * use resolver instad of link table
 *
 * Revision 1.19  2002/09/09 16:15:32  nozomi
 * add XML parsing support methods; modify getFullname() method
 *
 * Revision 1.18  2002/09/09 05:40:37  nozomi
 * use ANONYMOUS insetad of nullString
 *
 * Revision 1.17  2002/09/06 02:13:27  nozomi
 * add getFullSurname() method
 *
 * Revision 1.16  2002/09/02 21:43:25  nozomi
 * support getAuthorViewName()
 *
 * Revision 1.15  2002/05/10 14:11:26  t.okada
 * modify persistentID()
 *
 * Revision 1.14  2002/05/08 10:45:58  ryo
 * append feudality tag in toXmlString()
 *
 * Revision 1.13  2002/04/19 21:40:33  nozomi
 * clear and remove methods for Publication and Affiliation
 *
 * Revision 1.12  2002/04/14 23:57:33  nozomi
 * fix minor mistake in the comment
 *
 * Revision 1.11  2002/04/11 02:55:25  nozomi
 * XML handring methods were moved to data classes
 *
 * Revision 1.10  2002/04/10 07:44:48  nozomi
 * members are protected, change some access methods' name
 *
 * Revision 1.9  2002/04/09 03:01:40  nozomi
 * change emptyPersistentID handling
 *
 * Revision 1.8  2002/04/08 01:45:30  nozomi
 * Change StringDelegate to Name
 *
 * Revision 1.7  2002/04/01 06:48:31  nozomi
 * use java.util.Vector
 *
 * Revision 1.6  2002/03/25 11:30:49  okawa
 * addAffiliation()
 *
 * Revision 1.5  2002/03/20 15:11:38  okawa
 * add getFeudality()
 *
 * Revision 1.4  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 *
 * Revision 1.3  2002/02/14 08:29:27  okawa
 * add set/getPublications(), set/getAffiliations()
 *
 * Revision 1.2  2002/02/01 11:18:14  nozomi
 * Make it Cloneable
 *
 * Revision 1.1  1999/10/24 08:06:31  nozomi
 * Initial revision
 *	
 * Revision 1.1  1999/10/24 08:06:31  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.util.Date;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

import org.nomencurator.Affiliation;
import org.nomencurator.Publication;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An implementation of Author data structure of Nomencurator data model
 *
 * @see 	org.nomencurator.NamedObject
 * @see 	org.nomencurator.Affiliation
 * @see 	org.nomencurator.Publication
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org/</A>
 *
 * @version 	19 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class Author
    extends NamedObject
    implements Serializable, Cloneable
{
    
    public static final boolean ABBREVIATE = true;

    /** <code>Vector</code> representing affiliation list of the <code>Author</code> */
    protected Vector affiliations;

    /** <code>Vector</code> representing publication list of the <code>Author</code> */
    protected Vector publications;

    /** Title part of author's name, e.g. Sir or Load */
    protected String title;

    /** Author's first name */
    protected String firstName;

    /** Author's middle name */
    protected String middleName;

    /** Feudality part of author's name, e.g. van, von or de */
    protected String feudality;

    /** Author's surname */
    protected String surname;

    /** Epithet part of author's name, e.g. Jr. or Sen. */
    protected String epithet;

    /** <code>Date</code> of birth */
    protected Date birth;

    /** <code>Date</code> of death */
    protected Date death;

    /** Year of author's first publication */
    protected int firstPublication;

    /** Year of author's last publication */
    protected int lastPublication;

    /**
     * Constructs an empty <code>Author</code> object
     */
    public Author(){
	super();
	author();
    }
    
    /**
     * Constructs an <code>Auahot</code> having
     * <code>author</code> as its name, i.e. perisitent ID
     *
     * @param author <code>String</code> representing its name,
     * i.e. perisitent ID
     */
    public Author(String author)
    {
	super(author);
    }
    
    /**
     * Constructs a copy of given <code>name</code>
     *
     * @param name <code>Name</code> representation of an <code>Author</code>
     */
    public Author(Name name)
    {
	//don't use NamedObject(String) constructor
	this();
	
	if(name instanceof Author){
	    author((Author)name);
	}
	else
	    setPersistentID(name.getName());
    }
    
    /**
     * Constructs a non-trivial <code>Author</code> object.
     *
     * @param title <code>String</code> represents title of author's name
     * @param firstName <code>String</code> represents author's first name
     * @param middleName <code>String</code> represents author's middle name
     * @param surname <code>String</code> represents author's surname
     * @param epithet <code>String</code> represents epithet of author's name
     * @param birth <code>Date</code> of birth
     * @param death <code>Date</code> of death
     * @param firstPublication year of (known) first publication in <code>int</code>
     * @param lastPublication year of (known) last publication in <code>int</code>
     * @param afiiliations affiliation list as a <code>Vector</code> 
     * @param publications publication list as a <code>Vector</code> 
     */
    public Author(String title, String firstName, String middleName,
		  String feudality, String  surname, String epithet,
		  Date birth, Date death,
		  int firstPublication, int lastPublication,
		  Vector affiliations, Vector publications)
    {
	this();
	author(title, firstName, middleName, feudality, surname, epithet,
	       birth, death, firstPublication, lastPublication,
	       affiliations, publications);
    }
    
    /**
     * Constructs copy of <code>author</code>
     *
     * @param author <code>Author</code>
     */
    public Author(Author author)
    {
	super();
	author(author);
    }

    /**
     * Constructs an <CODE>Author</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying <CODE>Author</CODE>
     *
     */
    public Author(Element xml)
    {
	super();
	NodeList childNodes = xml.getChildNodes();
	int subChildCount = childNodes.getLength();

	boolean toBeResolved = false;

	String persistentID = null;
	for (int j = 0; j < subChildCount; j++) {
	    Node child = childNodes.item(j);
	    if(child.getNodeType () == Node.ELEMENT_NODE) {
		Element element = (Element)child;
		String tagName = element.getTagName();

		if (tagName.equals ("oid"))
		    persistentID = getString(element);
		else if(tagName.equals ("surname"))
		    surname = getString(element);
		else if(tagName.equals ("firstName"))
		    firstName = getString(element);
		else if(tagName.equals ("middleName"))
		    middleName =getString(element);
		else if(tagName.equals ("feudality"))
		    feudality = getString(element);
		else if(tagName.equals ("title"))
		    title = getString(element);
		else if(tagName.equals ("epithet"))
		    epithet = getString(element);
		else if(tagName.equals ("birth"))
		    birth = getDate(getString(element));
		else if(tagName.equals ("death"))
		    death = getDate(getString(element));
		else if(tagName.equals ("publication")) {
		    String pID = getString(element);
		    Publication p =
			(Publication)curator.get(pID);
		    if(p != null) {
			publications.addElement(p);
			toBeResolved = true;
		    }
		    else {
			p = new Publication();
			p.setName(pID);
			curator.put(p);
			addPublication(p);
		    }
		}
		else if(tagName.equals ("affiliation")) {
		    String pID = getString(element);
		    Affiliation a =
			(Affiliation)curator.get(pID);
		    if(a == null) {
			a = new Affiliation();
			a.setName(pID);
			curator.put(a);
			addAffiliation(a);
		    }
		    addAffiliation(a);

		}
		else{}
            }
	    
	}

	if(persistentID != null &&
	   !persistentID.equals(getName()))
	    setName(persistentID); //i.e. other key data are empty

	if(toBeResolved)
	    curator.resolve(this);

    }

    /*
     * Sets empty values to attributes.
     */
    private void author()
    {
	title      = ANONYMOUS;
	firstName  = ANONYMOUS;
	middleName = ANONYMOUS;
	surname    = ANONYMOUS;
	epithet    = ANONYMOUS;
	birth      = null;
	death      = null;
	firstPublication  = 0;
	lastPublication   = 0;
	affiliations = new Vector();
	publications = new Vector();
    }
    
    /*
     * Sets attribute values as in <code>author</code>
     *
     * @param author <code>Author</code> of which values to be set
     */
    private void author(Author author)
    {
	author(author.title, author.firstName, author.middleName,
	       author.feudality, author.surname, author.epithet, 
	       author.birth, author.death, author.firstPublication, author.lastPublication,
	       author.affiliations, author.publications);
    }
    
    /*
     * Sets given paratemeters to attributes
     *
     * @param title <code>String</code> represents title of author's name
     * @param firstName <code>String</code> represents author's first name
     * @param middleName <code>String</code> represents author's middle name
     * @param surname <code>String</code> represents author's surname
     * @param epithet <code>String</code> represents epithet of author's name
     * @param birth <code>Date</code> of birth
     * @param death <code>Date</code> of death
     * @param firstPublication year of (known) first publication in <code>int</code>
     * @param lastPublication year of (known) last publication in <code>int</code>
     * @param afiiliations affiliation list as a <code>Vector</code> 
     * @param publications publication list as a <code>Vector</code> 
     */
    private void author(String title, String firstName, String middleName,
			String feudality, String  surname, String epithet,
			Date birth, Date death,
			int firstPublication, int lastPublication,
			Vector affiliations, Vector publications)
    {
	this.title      = title;
	this.firstName  = firstName;
	this.middleName = middleName;
	this.feudality  = feudality;
	this.surname    = surname;
	this.epithet    = epithet;
	this.birth      = birth;
	this.death      = death;
	this.firstPublication = firstPublication;
	this.lastPublication  = lastPublication;
	this.affiliations = affiliations;
	this.publications = publications;
    }

    /**
     * Returns a persistent ID representing this <code>Author</code>
     *
     * @return String representing this <code>Author</code>
     */
    public String getPersistentID()
    {
	if(entity != null)
	    return ((Author)getEntity()).getPersistentID();

	return persistentID();
    }
    
    /**
     * Returns a persistent ID representing this <code>Author</code>
     *
     * @return String representing this <code>Author</code>
     */
    public String persistentID()
    {
	if(entity != null)
	    return ((Author)getEntity()).persistentID();

	if(string != null && string.length() != 0)
	    return string;

	StringBuffer pid = getClassNameHeaderBuffer();

	pid.append(getFullname());
	//chenge it as locale independent!
	pid.append(fieldSeparator).append(birth);
	pid.append(fieldSeparator).append(death);
	return pid.toString();
    }
    
    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public int getFieldSepartorsCount()
    {
	return 2;
    }

    /**
     * Merges <code>namedObject</code> with this <code>Author</code>
     * if possible, but not yet implemented.
     * It returns true if merged.
     *
     * @param namedObject a <code>NamedObject</code> to be merged
     *
     * @return true if merged, or false if not mergiable
     */
    public boolean merge(NamedObject nr){
	if(!getClassNameHeader().equals(nr.getClassNameHeader()))
	    return false;
	return false; //not yet implemented
    }

    /**
     * Returns full name of author in <code>String</code>
     *
     * @return full name of author in <code>String</code>
     */
    public String getFullname()
    {
	if(entity != null)
	    return ((Author)getEntity()).getFullname();

	return getFullname(" ");
    }
    
    /**
     * Returns full name of author in <code>String</code>
     *
     * @return full name of author in <code>String</code>
     */
    public String getFullname(boolean abbreviate)
    {
	if(entity != null)
	    return ((Author)getEntity()).getFullname(abbreviate);

	return getFullname(" ", abbreviate);
    }
    
    /**
     * Returns full name of author in <code>String</code>.
     * Each name components is separated by <code>nameSeparator</code>
     *
     * @param nameSeparator <code>String</code> to be used to separate name components
     *
     * @return full name of author in <code>String</code>
     */
    public String getFullname(String nameSeparator)
    {
	if(entity != null)
	    return ((Author)getEntity()).getFullname(nameSeparator);

	return getFullname(nameSeparator, false);
    }


    /**
     * Returns full name of author in <code>String</code>.
     * Each name components is separated by <code>nameSeparator</code>
     *
     * @param nameSeparator <code>String</code> to be used to separate name components
     *
     * @return full name of author in <code>String</code>
     */
    public String getFullname(String nameSeparator, boolean abbreviate)
    {
	if(entity != null)
	    return ((Author)getEntity()).getFullname(nameSeparator, abbreviate);

	StringBuffer s = new StringBuffer();

	if(isNominal()) {
	    StringTokenizer tokens = 
		new StringTokenizer(getPersistentIDComponent(0), ",");
	    if(tokens.countTokens() < 1)
		return null;

	    s.append(tokens.nextToken());
	    if(!tokens.hasMoreTokens())
		return s.toString();

	    if(s.length() > 0)
		s.append(",").append(nameSeparator);
	    tokens = new StringTokenizer(tokens.nextToken());
	    int i = 0;
	    if(tokens.countTokens() > 0) {
		while(tokens.hasMoreTokens()) {
		    if(i > 0)
			s.append(nameSeparator);
		    if(abbreviate) {
			s.append(tokens.nextToken().charAt(0));
			s.append('.');
		    }
		    else
			s.append(tokens.nextToken());
		    i++;
		}
	    }
	    return s.toString();
	}

	if(feudality != null && feudality.length() > 0) {
	    s.append(nameSeparator).append(feudality);
	    if(surname != null && surname.length() > 0)
		s.append(nameSeparator);
	}

	if(surname != null)
	    s.append(surname);

	if(epithet != null && epithet.length() > 0)
	    s.append(nameSeparator).append(epithet);

	if(firstName != null && firstName.length() > 0) {
	    s.append(",").append(nameSeparator);
	    if(abbreviate)
		s.append(firstName.charAt(0)).append(".");
	    else
		s.append(firstName);
	}

	if(middleName != null && middleName.length() > 0) {
	    s.append(",").append(nameSeparator);
	    StringTokenizer tokens = new StringTokenizer(middleName);
	    int i = 0;
	    while (tokens.hasMoreTokens()) {
		String m = tokens.nextToken();
		if(m.length() > 0) {
		    if(i > 0)
			s.append(nameSeparator);
		    if(abbreviate) {
			s.append(m.charAt(0));
			s.append('.');
		    }
		    else
			s.append(m);
		    i++;
		}
	    }
	}

	if(title != null && title.length() > 0)
	    s.append(",").append(nameSeparator).append(title);

	return s.toString();
    }
    
    /**
     * Parses a <code>line</code> and sets values of this object accordint to it
     *
     * @param line <code>String</code> containing fragment of data to be set
     */
    public void parseLine(String line){
    }

    /**
     * Sets <code>title</code> as title of author's name
     *
     * @param title <code>String</code> containing title of author's name to be set
     */
    public void setTitle(String title)
    {
	if(entity != null)
	    ((Author)getEntity()).setTitle(title);

	if(this.title == title ||
	   (title != null && title.equals(this.title)))
	    return;

	this.title = title;
    }

    /**
     * Returns title of author's name
     *
     * @return <code>String</code> representing title of author's name
     */
    public String getTitle()
    {
	if(entity != null)
	    return ((Author)getEntity()).title;

	return title;
    }

    /**
     * Sets <code>firstName</code> as author's first name
     *
     * @param firstName <code>String</code> containing author's first name to be set
     */
    public void setFirstName(String firstName)
    {
	if(entity != null)
	    ((Author)getEntity()).setFirstName(firstName);

	if(this.firstName == firstName ||
	   (firstName != null && firstName.equals(this.firstName)))
	    return;

	this.firstName = firstName;
    }

    /**
     * Returns author's first name
     *
     * @return <code>String</code> representing author's first name
     */
    public String getFirstName()
    {
	if(entity != null)
	    return ((Author)getEntity()).firstName;

	if(!isNominal())
	    return firstName;

	StringTokenizer tokens = 
	    new StringTokenizer(getPersistentIDComponent(0), ",");
	if(tokens.countTokens() < 2)
	    return null;

	tokens.nextToken(); // discard surname
	tokens =  new StringTokenizer(tokens.nextToken());

	if(tokens.countTokens() < 1)
	    return null;
	return tokens.nextToken();
    }

    /**
     * Sets <code>middleName</code> as author's middle name
     *
     * @param middleName <code>String</code> containing author's middle name to be set
     */
    public void setMiddleName(String middleName)
    {
	if(entity != null)
	    ((Author)getEntity()).setMiddleName(middleName);

	if(this.middleName == middleName ||
	   (middleName != null && middleName.equals(this.middleName)))
	    return;

	this.middleName = middleName;
    }

    /**
     * Returns author's middle name
     *
     * @return <code>String</code> representing author's middle name
     */
    public String getMiddleName()
    {
	if(entity != null)
	    return ((Author)getEntity()).middleName;

	if(!isNominal())
	    return middleName;

	StringTokenizer tokens = 
	    new StringTokenizer(getPersistentIDComponent(0), ",");
	if(tokens.countTokens() < 2)
	    return null;

	tokens.nextToken(); // discard surname
	tokens =  new StringTokenizer(tokens.nextToken());

	if(tokens.countTokens() < 2)
	    return null;
	tokens.nextToken(); // discard first name
	return tokens.nextToken();
    }

    /**
     * Sets <code>surname</code> as author's surname
     *
     * @param surname <code>String</code> containing author's surname to be set
     */
    public void setSurname(String surname)
    {
	if(entity != null)
	    ((Author)getEntity()).setSurname(surname);

	if(this.surname == surname ||
	   (surname != null && surname.equals(this.surname)))
	    return;

	this.surname = surname;
    }

    /**
     * Returns author's surname
     *
     * @return <code>String</code> representing author's surname
     */
    public String getSurname()
    {
	if(entity != null)
	    return ((Author)getEntity()).surname;

	if(!isNominal())
	    return surname;

	StringTokenizer tokens = 
	    new StringTokenizer(getPersistentIDComponent(0), ",");
	if(tokens.countTokens() < 1)
	    return null;

	return tokens.nextToken();
    }

    /**
     * Returns author's surname
     *
     * @return <code>String</code> representing author's surname
     */
    public String getFullSurname()
    {
	if(entity != null)
	    return ((Author)getEntity()).getFullSurname();

	StringBuffer buffer = new StringBuffer();
	boolean appended = false;
	if(feudality != null &&
	   feudality.length() > 0) {
	    buffer.append(feudality);
	    buffer.append(' ');
	    appended = true;
	}
	if(surname != null &&
	   surname.length() > 0) {
	    buffer.append(surname);
	    appended = true;
	}
	if(epithet != null &&
	   epithet.length() > 0) {
	    if(appended)
		buffer.append(' ');
	    buffer.append(epithet);
	    appended = true;
	}

	return buffer.toString();
    }

    
    /**
     * Sets <code>epithet</code> of author's surname
     *
     * @param epithet <code>String</code> containing epithet of author's surname to be set
     */
    public void setEpithet(String epithet)
    {
	if(entity != null)
	    ((Author)getEntity()).setEpithet(epithet);

	if(this.epithet == epithet ||
	   (epithet != null && epithet.equals(this.epithet)))
	    return;

	this.epithet = epithet;
    }

    /**
     * Returns epithet of author's name
     *
     * @return <code>String</code> representing epithet of author's name
     */
    public String getEpithet()
    {
	if(entity != null)
	    return ((Author)getEntity()).epithet;

	return epithet;
    }

    /**
     * Sets <code>feudality</code> of author's surname
     *
     * @param feudality <code>String</code> containing feudality of author's surname to be set
     */
    public void setFeudality(String feudality)
    {
	if(entity != null)
	    ((Author)getEntity()).setFeudality(feudality);

	if(this.feudality == feudality ||
	   (feudality != null && feudality.equals(this.feudality)))
	    return;

	this.feudality = feudality;
    }

    /**
     * Returns feudality of author's name
     *
     * @return <code>String</code> representing feudality of author's name
     */
    public String getFeudality()
    {
	if(entity != null)
	    return ((Author)getEntity()).feudality;

	return feudality;
    }

    /**
     * Returns full name of author in <code>String</code> with abbreviated 
     * first and middle names.
     * Each name components is separated by a space
     *
     * @return full name of author with abbreviated first and middle names in <code>String</code>
     */
    public String getInitializedName()
    {
	if(entity != null)
	    return ((Author)getEntity()).getInitializedName();

	return getInitializedName(" ");
    }

    /**
     * Returns full name of author in <code>String</code> with abbreviated 
     * first and middle names.
     * Each name components is separated by <code>nameSeparator</code>
     *
     * @param nameSeparator <code>String</code> to be used to separate name components
     *
     * @return full name of author with abbreviated first and middle names in <code>String</code>
     */
    public String getInitializedName(String nameSeparator)
    {
	if(entity != null)
	    return ((Author)getEntity()).getInitializedName(nameSeparator);

	StringBuffer name = new StringBuffer();
	if(feudality != null && feudality.length() > 0) {
	    name.append(nameSeparator).append(feudality);
	    if(surname != null && surname.length() > 0)
		name.append(nameSeparator);
	}

	if(surname != null)
	    name.append(surname);

	if(epithet != null && epithet.length() > 0)
	    name.append(nameSeparator).append(epithet);

	if((firstName != null && firstName.length() > 0) ||
	   (middleName != null && middleName.length() > 0)) {
	    name.append(',').append(nameSeparator);
	    if(firstName != null && firstName.length() > 0)
		name.append(initialize(firstName, nameSeparator));
	    
	    if(middleName != null && middleName.length() > 0)
		name.append(nameSeparator).append(initialize(middleName, nameSeparator));
	}
	
	return name.toString();
    }

    /**
     * Returns abbreviated expression of <code>name</code> separated
     * by <code>nameSeparator</code> if <code>name</code> consists of
     * multiple parts.
     *
     * @param name <code>String</code> representing name to be abbreviated
     * @param nameSeparator <code>String</code> to be used to separate name components
     *
     * @returns abbreviated expression of <code>name</code>
     */ 
    public static String initialize(String name, String nameSeparator)
    {
	StringTokenizer st = new StringTokenizer(name);
	StringBuffer initials = new StringBuffer();
	int tokens = 0;
	while (st.hasMoreTokens()) {
	    if(tokens != 0)
		initials.append(nameSeparator);
	    initials.append(st.nextToken().charAt(0));
	    initials.append(".");
	}
	return initials.toString();
    }

    /**
     * Returns birth <code>Date</code> of the author
     *
     * @return birth <code>Date</code> of the author
     */
    public Date getBirthDate()
    {
	if(entity != null)
	    return ((Author)getEntity()).birth;

	return birth;
    }

    /**
     * Sets <code>birthDate</code> as birth <code>Date</code> of the author
     *
     * @param birthDate author's birth day in <code>Date</code>
     */
    public void setBirthDate(Date birthDate)
    {
	if(entity != null)
	    ((Author)getEntity()).setBirthDate(birthDate);

	if(birth == birthDate)
	    return;

	birth = birthDate;
    }

    /**
     * Returns death <code>Date</code> of the author
     *
     * @return death <code>Date</code> of the author
     */
    public Date getDeathDate()
    {
	if(entity != null)
	    return ((Author)getEntity()).death;

	return death;
    }

    /**
     * Sets <code>deathDate</code> as death <code>Date</code> of the author
     *
     * @param deathDate author's death day in <code>Date</code>
     */
    public void setDeathDate(Date deathDate)
    {
	if(entity != null)
	    ((Author)getEntity()).setDeathDate(deathDate);

	if(death == deathDate)
	    return;

	death = deathDate;
    }

    /**
     * Returns <code>Vector</code> representing author's publication list
     *
     * @return <code>Vector</code> representing author's publication list
     */
    public Vector getPublications()
    {
	if(entity != null)
	    return ((Author)getEntity()).publications;

	return publications;
    }

    /**
     * Sets <code>publications</code> as <code>Vector</code> representing author's publication list
     *
     * @param publications <code>Vector</code> representing author's publication list
     */
    public void setPublications(Vector publications)
    {
	if(entity != null)
	    ((Author)getEntity()).setPublications(publications);

	if(this.publications == publications)
	    return;

	this.publications = publications;
    }

    /**
     * Adds <code>publication</code> to author's publication list
     * with returning true if it is added successfully, or false if not.
     *
     * @param publication <code>Publication</code> to be added to author's publication list
     *
     * @return true if <code>publication</code> was added to the publication list successfully, or false if not.
     */
    public boolean addPublication(Publication publication)
    {
	if(entity != null)
	    return ((Author)getEntity()).addPublication(publication);

	if(publications == null)
	    publications = new Vector();

	if(publications.contains(publication))
	    return false;

	publications.addElement(publication);
	return true;
    }

    /**
     * Removes <code>publication</code> from author's publication list.
     *
     * @param publication <code>Publication</code> to be removed from author's publication list
     */
    public void removePublication(Publication publication)
    {
	if(entity != null)
	    ((Author)getEntity()).removePublication(publication);

	if(publications == null ||
	   ! publications.contains(publication))
	    return;

	publications.removeElement(publication);
    }

    /**
     * Clears author's publication list.
     */
    public void clearPublication()
    {
	if(entity != null)
	    ((Author)getEntity()).clearPublication();

	if(publications == null ||
	   publications.isEmpty())
	    return;

	publications.clear();
    }

    /**
     * Returns <code>Vector</code> representing author's affiliation list
     *
     * @return <code>Vector</code> representing author's affiliation list
     */
    public Vector getAffiliations()
    {
	if(entity != null)
	    return ((Author)getEntity()).affiliations;

	return affiliations;
    }

    /**
     * Sets <code>affiliations</code> as <code>Vector</code> representing author's affiliation list
     *
     * @param affiliations <code>Vector</code> representing author's affiliation list
     */
    public void setAffiliations(Vector affiliations)
    {
	if(entity != null)
	    ((Author)getEntity()).setAffiliations(affiliations);

	this.affiliations = affiliations;
    }

    /**
     * Adds <code>affiliation</code> to author's affiliation list
     * with returning true if it is added successfully, or false if not.
     *
     * @param affiliation <code>Affiliation</code> to be added to author's affiliation list
     *
     * @return true if <code>affiliation</code> was added to the affiliation list successfully, or false if not.
     */
    public boolean addAffiliation(Affiliation affiliation)
    {
	if(entity != null)
	    return ((Author)getEntity()).addAffiliation(affiliation);

	if(affiliations == null) this.affiliations = new Vector();
	if(affiliations.contains(affiliation))
	    return false;

	affiliations.addElement(affiliation);
	return true;
    }

    /**
     * Removes <code>affiliation</code> from author's affiliation list.
     *
     * @param affiliation <code>Affiliation</code> to be removed from author's affiliation list
     */
    public void removeAffiliation(Affiliation affiliation)
    {
	if(entity != null)
	    ((Author)getEntity()).removeAffiliation(affiliation);

	if(affiliations == null ||
	   ! affiliations.contains(affiliation))
	    return;

	affiliations.removeElement(affiliation);
    }

    /**
     * Clears author's affiliation list.
     *
     */
    public void clearAffiliations()
    {
	if(entity != null)
	    ((Author)getEntity()).clearAffiliations();

	if(affiliations == null ||
	   affiliations.isEmpty())
	    return;

	affiliations.clear();
    }

    public Object clone()
    {
	/*
	Author author = new Author(this);
	author.affiliations = (Vector)affiliations.clone();
	author.publications = (Vector)publications.clone();
	return author;
	*/
	return new Author(this);
    }

    /**
     * Returns XML <code>String</code> representing this objet 
     *
     * @return XML <code>String</code> representing this objet
     */
    public String toXMLString()
    {
	if(entity != null)
	    return ((Author)getEntity()).toXMLString();

	StringBuffer buf = new StringBuffer();
	
	buf.append("<Author>\n");
	buf.append("<oid>").append(persistentID()).append("</oid>\n");
	buf.append("<surname>").append(getSurname()).append("</surname>\n");
	buf.append("<firstName>").append(getFirstName()).append("</firstName>\n");
	buf.append("<middleName>").append(getMiddleName()).append("</middleName>\n");
	buf.append("<title>").append(getTitle()).append("</title>\n");
	buf.append("<epithet>").append(getEpithet()).append("</epithet>\n");
	buf.append("<feudality>").append(getFeudality()).append("</feudality>\n");
	if(getBirthDate() != null) {
    	    buf.append("<birth>").append(getBirthDate().toString()).append("</birth>\n");
    	}
    	if(getDeathDate() != null) {
    	    buf.append("<death>").append(getDeathDate().toString()).append("</death>\n");
    	}
    	if(getPublications() != null) {
	    Enumeration e = getPublications().elements();
	    while(e.hasMoreElements()) {
    	        buf.append("<publication>").append(((Publication)e.nextElement()).persistentID()).append("</publication>\n");
	    }
	}
	if(getAffiliations() != null) {
            Enumeration e = getAffiliations().elements();
	    while(e.hasMoreElements()) {
    	        buf.append("<affiliation>").append(((Affiliation)e.nextElement()).persistentID()).append("</affiliation>\n");
	    }
	}
	
	// buf.append("<previous>").append(getPrevious().persistentID()).append("</previous>");
	// buf.append("<next>").append(getNext().persistentID()).append("</next>");
	buf.append("</Author>\n");
	
        return buf.toString();
    }	
	
    
    /**
     * create XML String of the all Related NamedObject
     *
     * @return XML String of this <code>Author</code> object
     */
    public String toRelatedXMLString()
    {
	if(entity != null)
	    return ((Author)getEntity()).toRelatedXMLString();

	// create XML String of the Author itself
	StringBuffer buf = new StringBuffer();
	buf.append(toXMLString());
	
	// create XML of the Publications
	if(getPublications() != null) {
            Enumeration e = getPublications().elements();
	    while(e.hasMoreElements()) {
                Publication publication = (Publication)e.nextElement();
    	        buf.append(publication.toXMLString());
	    }
    	}
	// create XML of the Affiliation
	if(getAffiliations() != null) {
            Enumeration e = getAffiliations().elements();
	    while(e.hasMoreElements()) {
                Affiliation affiliation = (Affiliation)e.nextElement();
    	        buf.append(affiliation.toXMLString());
	    }
    	}
	
	return buf.toString();
    }

    /**
     * Returns author's name in format appropriate to use as a part of
     * view name
     *
     * @return String representing author's name
     */
    public String getAuthorViewName()
    {
	if(entity != null)
	    return ((Author)getEntity()).getAuthorViewName();

	StringBuffer buffer = new StringBuffer();
	getAuthorViewName(buffer);
	return buffer.toString();
    }

    /**
     * Returns author's name in format appropriate to use as a part of
     * view name
     *
     * @param buffer <CODE>StringBuffer</CODE> to append author's name
     *
     * @return StringBufer containing author's name
     */
    public boolean getAuthorViewName(StringBuffer buffer)
    {
	if(entity != null)
	    return ((Author)getEntity()).getAuthorViewName(buffer);

	boolean appended = false;
	if(feudality != null && feudality.length() != 0) {
	    buffer.append(feudality);
	    buffer.append(' ');
	    appended = true;
	}

	if(surname != null && surname.length() != 0) {
	    buffer.append(surname);
	    appended = true;
	}

	if(epithet != null && epithet.length() != 0) {
	    if(appended)
		buffer.append(' ');
	    buffer.append(epithet);
	    appended = true;
	}

	return appended;
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	if(object == this ||
	   (object != null && !(object instanceof Author)))
	    throw new IllegalArgumentException(object.getClass().getName() + " can't be an entity of " + getClass().getName());
	entity = (Author)object;
	super.setEntity(object);
    }
}
