/*
 * Name.java:  a class provaides a extendable string
 * for Java implementation of the Nomencurator, 
 * a Nomenclature Heuristic Model.  For restriction of "final class String"
 * in Java, it uses Name instead of String.  It spoiles simplicity
 * and conceptual importance of name commutability.
 *
 * The same thing can be implemented using nameing mechanism of CORBA.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Name.java,v 1.16 2002/11/12 03:16:20 nozomi Exp $
 * $Log: Name.java,v $
 * Revision 1.16  2002/11/12 03:16:20  nozomi
 * use Nomencurator as an Object pool with name resolver
 *
 * Revision 1.15  2002/10/01 02:51:52  nozomi
 * fix equals()
 *
 * Revision 1.14  2002/10/01 02:50:51  nozomi
 * equals() examines objects by getEntity()
 *
 * Revision 1.13  2002/09/17 05:00:11  nozomi
 * modify getName(), add getSynonym()
 *
 * Revision 1.12  2002/09/10 07:37:18  nozomi
 * modify definition of pedantic functions
 *
 * Revision 1.11  2002/09/10 01:10:28  nozomi
 * getEntity() call itself recursively if the entity is a Name
 *
 * Revision 1.10  2002/09/09 23:04:48  nozomi
 * add isSynonym() and isNonsense()
 *
 * Revision 1.9  2002/09/09 16:26:56  nozomi
 * fix misspelling in tautology
 *
 * Revision 1.8  2002/09/07 22:12:23  nozomi
 * add entity to provide an indirect reference
 *
 * Revision 1.7  2002/08/29 19:46:38  nozomi
 * implements interface Nameable
 *
 * Revision 1.6  2002/04/15 23:38:04  nozomi
 * make nullString public
 *
 * Revision 1.5  2002/04/09 09:55:31  nozomi
 * string is protected; add public nullString
 *
 * Revision 1.4  2002/04/08 01:38:44  nozomi
 * Change StringDelegate to Name
 *
 * Revision 1.3  2002/03/28 03:47:05  nozomi
 * make constructor to be protected
 *
 * Revision 1.2  2002/03/28 03:45:15  nozomi
 * extends Observable
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * 
 * Revision 1.1  1999/10/24 08:06:33  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.lang.String;

import java.util.Locale;
import java.util.Observable;

/**
 * A <CODE>Name</CODE> provides a name, a tagged indirect reference.
 * It "extends" the final <code>String</code> class, 
 * which makes an <code>Object</code> "compatible" with <code>String</code>,
 * i.e. make namable.
 *
 * @version 	22 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class Name
    extends Observable
    implements Nameable, Serializable
{
    /** string entitiy */
    protected String string;

    /** object entitiy */
    protected Object entity;

    /** null string */
    public static final String EMPTY_STRING = "";
    public static final String ANONYMOUS = EMPTY_STRING;

    protected static Nomencurator curator = Nomencurator.getInstance();

    public Name()
    {
	this(ANONYMOUS);
    }
    
    public Name(String name)
    {
	this(name, null);
    }

    public Name(String name, Object object)
    {
	string = name;
	entity = object;
    }
    
    /**
     * Gets name in <CODE>String</CODE>
     *
     * @return String representing a name
     */
    public String getName()
    {
	if(entity != null && entity instanceof Name)
	    return ((Name)entity).getName();
	return string;
    }

    /**
     * Gives a name as a <CODE>String</CODE>
     *
     * @param String representing a name
     */
    public void setName(String name)
    {
	string = name;
	setChanged();
	notifyObservers(string);
    }

    /**
     * Gets synonym in <CODE>String</CODE>,
     * or null if it is not a synonym
     *
     * @return String representing a name
     */
    public String getSynonym()
    {
	if(isSynonym())
	    return string;
	return null;
    }


    /**
     * Gets entity of this <CODE>Name</CODE>.
     * Returns itself if it doesn't have entity separately,
     * or entity if the entity non-null.
     * If the entity is an instance of <CODE>Name</CODE>,
     * the method returns the result of recursive apply of 
     * this method
     *
     * @return Object enitity representied by the name.  It is never null.
     */
    public Object getEntity()
    {
	if(entity == null)
	    return this;

	if(!(entity instanceof Name))
	    return entity;

	return ((Name)entity).getEntity();
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	entity = object;
	setChanged();
	notifyObservers(entity);
    }

    /**
     * Returns true if this name is a tautology.
     * A pedantic method.
     *
     * @return true if the name refers to itself
     */
    public boolean isTautology()
    {
	if(string != null)
	    return string.equals(entity);
	return string == entity;
    }

    /**
     * Returns true if the entity is unnamed
     * A pedantic method.
     *
     * @return true if the entity is not named
     */
    public boolean isAnonymous()
    {
	return ((string == null || string.equals(ANONYMOUS)) && entity != null);
    }

    /**
     * Returns true if this name is nominal, i.e. no entity
     * A pedantic method.
     *
     * @return true if the entity is null
     */
    public boolean isNominal()
    {
	return (string != null && !string.equals(ANONYMOUS) &&  (entity == null));
    }

    /**
     * Returns true if this name is a synonym, i.e. both string
     * and entity are non-null.
     * A pedantic method?  Could be...
     *
     * @return true if both string and entity are non-null
     */
    public boolean isSynonym()
    {
	return (string != null && !string.equals(ANONYMOUS) &&  (entity != null));
    }

    /**
     * Returns true if <CODE>name</CODE> is a synonym of this
     * <CODE>Name</CODE>, i.e. both have the same entity
     * but different in spelling.
     *
     * @param name <CODE>Name</CODE> to be examined
     *
     * @return true if both string and entity are non-null
     */
    public boolean isSynonym(Name name)
    {
	if(name == null || name == this ||
	   !getName().equals(name.getName()))
	    return false;

	return (getEntity() == name.getEntity());
    }

    /**
     * Returns true if <CODE>name</CODE> is a homonym of this
     * <CODE>Name</CODE>, i.e. they have the same spelling
     * but different entity
     *
     * @param name <CODE>Name</CODE> to be examined
     *
     * @return true if <CODE>name</CODE> is a homonym of this <CODE>Name</CODE>
     */
    public boolean isHomonym(Name name)
    {
	if(name == null || name == this ||
	   getEntity() != name.getEntity())
	    return false;

	return (getName().equals(name.getName()));
    }

    /**
     * Returns true if the entity is unnamed
     * A pedantic method.
     *
     * @return true if it does not have independent entity nor named
     */
    public boolean isSelf()
    {
	return ((string == null || string.equals(ANONYMOUS)) && entity == null);
    }


    /**
     * Returns true if <CODE>object</CODE> equals to this <CODE>Name</CODE>
     *
     * @param object <CODE>Object</CODE> to be compared
     *
     * @return true if <CODE>object</CODE> equals to this <CODE>Name</CODE> 
     */
    public boolean equals(Object object)
    {
	if(this == object)
	    return true;

	if(object == null)
	    return false;

	if(!(object instanceof Name)) {
	    if(string != null)
		string.equals(object);
	    else
		return false;
	}

	return getEntity() == ((Name)object).getEntity();
    }

    /*
     * interfaces to String class
     */
    public char charAt(int index) { return string.charAt(index); }
    
    public int compareTo(Object o) { return string.compareTo(o); }
    
    public int compareTo(String anotherString) { return string.compareTo(anotherString); }
    
    public int compareToIgnoreCase(String str) { return string.compareToIgnoreCase(str); }
    
    public String concat(String str) { return string. concat(str); }
    
    public static String copyValueOf(char[] data) { return String.copyValueOf(data); }
    
    public static String copyValueOf(char[] data, int offset, int count) { return String.copyValueOf(data, offset, count); }
    
    public boolean endsWith(String suffix) { return string.endsWith(suffix); }
    
    //    public boolean equals(Object anObject) { return string.equals(anObject); }
    
    public boolean equalsIgnoreCase(String anotherString) { return string.equalsIgnoreCase(anotherString); }
    
    public byte[] getBytes() { return string.getBytes(); }
    
    public void getBytes(int srcBegin, int srcEnd, byte[] dst, int dstBegin) { string.getBytes(srcBegin, srcEnd, dst, dstBegin); }
    
    public byte[] getBytes(String enc) throws java.io.UnsupportedEncodingException { return string.getBytes(enc); }
    
    public void getChars(int srcBegin, int srcEnd, char[] dst, int dstBegin) { string.getChars(srcBegin, srcEnd, dst, dstBegin); }
    
    public int hashCode() { return string.hashCode(); }
    
    public int indexOf(int ch) { return string.indexOf(ch); }
    
    public int indexOf(int ch, int fromIndex) { return string.indexOf(ch, fromIndex); }
    
    public int indexOf(String str) { return string.indexOf(str); }
    
    public int indexOf(String str, int fromIndex) { return string.indexOf(str, fromIndex); }
    
    public String intern() { return string.intern(); }
    
    public int lastIndexOf(int ch) { return string.lastIndexOf(ch); }
    
    public int lastIndexOf(int ch, int fromIndex) { return string.lastIndexOf(ch, fromIndex); }
    
    public int lastIndexOf(String str) { return string.lastIndexOf(str); }
    
    public int lastIndexOf(String str, int fromIndex) { return string.lastIndexOf(str, fromIndex); }
    
    public int length() { return string.length(); }
    
    public boolean regionMatches(boolean ignoreCase, int toffset, String other, int ooffset, int len) {
	return string.regionMatches(ignoreCase, toffset, other, ooffset, len); }
    
    public boolean regionMatches(int toffset, String other, int ooffset, int len) {
	return string.regionMatches(toffset, other, ooffset, len); }
    
    public String replace(char oldChar, char newChar) { return string.replace(oldChar, newChar); }
    
    public boolean startsWith(String prefix) { return string.startsWith(prefix); }
    
    public boolean startsWith(String prefix, int toffset) { return string.startsWith(prefix, toffset); }
    
    public String substring(int beginIndex) { return string.substring(beginIndex); }
    
    public String substring(int beginIndex, int endIndex) { return string.substring(beginIndex, endIndex); }
    
    public char[] toCharArray() { return string.toCharArray(); }
    
    public String toLowerCase() { return string.toLowerCase(); }
    
    public String toLowerCase(Locale locale) { return string.toLowerCase(locale); }
    
    public String toString() { return string.toString(); }
    
    public String toUpperCase() { return string.toUpperCase(); }
    
    public String toUpperCase(Locale locale) { return string.toUpperCase(locale); }
    
    public String trim() { return string.trim(); }
    
    public static String valueOf(boolean b) { return String.valueOf(b); }
    
    public static String valueOf(char c) { return String.valueOf(c); }
    
    public static String valueOf(char[] data) { return String.valueOf(data); }
    
    public static String valueOf(char[] data, int offset, int count) {
	return String.valueOf(data, offset, count); }
    
    public static String valueOf(double d) { return String.valueOf(d); }
    
    public static String valueOf(float f) { return String.valueOf(f); }
    
    public static String valueOf(int i) { return String.valueOf(i); }
    
    public static String valueOf(long l) { return String.valueOf(l); }
    
    public static String valueOf(Object obj) { return String.valueOf(obj); }
    
}
