/*
 * NameUsage.java:  a Java implementation of NameUsage class
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: NameUsage.java,v 1.36 2002/11/20 05:20:32 nozomi Exp $
 * $Log: NameUsage.java,v $
 * Revision 1.36  2002/11/20 05:20:32  nozomi
 * remove unused import line
 *
 * Revision 1.35  2002/11/12 03:16:20  nozomi
 * use Nomencurator as an Object pool with name resolver
 *
 * Revision 1.34  2002/10/02 06:58:12  nozomi
 * improve typeOfType handling
 *
 * Revision 1.33  2002/10/02 02:47:24  nozomi
 * add typeOfType accesor
 *
 * Revision 1.32  2002/10/01 04:10:26  nozomi
 * add comment on getViewName(), getYear() and getRoot()
 *
 * Revision 1.31  2002/10/01 03:44:07  nozomi
 * implement add/removeTypeDesignator() to handle cases where multiple names share single type
 *
 * Revision 1.30  2002/10/01 00:53:15  nozomi
 * modify type relating member
 *
 * Revision 1.29  2002/09/27 20:08:16  nozomi
 * accessor methods parse persistentID if isNominal()
 *
 * Revision 1.28  2002/09/20 14:27:55  nozomi
 * fix null Publication handling
 *
 * Revision 1.27  2002/09/17 04:57:59  nozomi
 * support indirect reference
 *
 * Revision 1.26  2002/09/10 22:10:50  nozomi
 * add NameUsage(Element)
 *
 * Revision 1.25  2002/09/09 05:41:16  nozomi
 * use immidiate value insetad of nullString
 *
 * Revision 1.24  2002/09/07 01:43:14  nozomi
 * null pointer handling in getAuthors()
 *
 * Revision 1.23  2002/09/06 02:14:06  nozomi
 * add getYear() method; change getAuthors() method
 *
 * Revision 1.22  2002/09/02 21:45:06  nozomi
 * modify getViewName()
 *
 * Revision 1.21  2002/09/01 10:46:37  nozomi
 * change method name from set/setName to set/getUsedName
 *
 * Revision 1.20  2002/08/28 00:33:46  nozomi
 * support getViewName()
 *
 * Revision 1.19  2002/08/27 08:12:49  ryo
 * change variable type of type into boolean
 *
 * Revision 1.18  2002/08/22 13:27:51  t.okada
 * It corrects for a locale colum addition.
 *
 * Revision 1.17  2002/08/22 09:35:01  nozomi
 * support isType()
 *
 * Revision 1.16  2002/07/18 14:38:55  t.okada
 * note item is added to nameusage
 *
 * Revision 1.15  2002/05/15 06:33:52  ryo
 * modify setHigherTaxon()
 *
 * Revision 1.14  2002/05/14 03:16:21  t.okada
 * change XML tag name recorder to appearance
 *
 * Revision 1.13  2002/05/13 12:22:57  t.okada
 * bugfix toXMLString()
 *
 * Revision 1.12  2002/05/10 15:50:59  t.okada
 * modify setAppearance()
 *
 * Revision 1.11  2002/04/15 23:32:11  nozomi
 * add toSQL() method
 *
 * Revision 1.10  2002/04/15 04:26:39  nozomi
 * make names symetric for get/set higher taxon method
 *
 * Revision 1.9  2002/04/15 03:32:51  nozomi
 * fix Locale relating bugs
 *
 * Revision 1.8  2002/04/15 03:30:59  nozomi
 * make constants static
 *
 * Revision 1.7  2002/04/14 23:54:57  nozomi
 * add Locale
 *
 * Revision 1.6  2002/04/09 03:01:40  nozomi
 * change emptyPersistentID handling
 *
 * Revision 1.5  2002/04/08 01:41:51  nozomi
 * Change NameRecord to NameUsage
 *
 * Revision 1.4  2002/04/01 06:48:31  nozomi
 * use java.util.Vector
 *	
 * Revision 1.3  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 *	
 * Revision 1.2  2002/02/14 08:28:15  okawa
 * add annotationList property
 *	
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 *	
 * Revision 1.1  1999/10/24 08:06:35  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.util.Enumeration;
import java.util.Locale;
import java.util.Vector;

import org.nomencurator.Author;
import org.nomencurator.Appearance;
import org.nomencurator.NamedObject;
import org.nomencurator.Nomencurator;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An implementation of <code>NameUsage</code> in Nomencurator
 * data model.
 * It was referred to as NameRecord in the original publication.
 *
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org</A>
 *
 * @version 	20 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class NameUsage
    extends NamedObject
    implements Serializable
	       
{
    /** <code>Locale</code> of this name */
    protected Locale locale;

    /** Langage name for ICBN locale */
    public static final String ICBN_LANG = "icbn";

    /** Locale for ICBN */
    public static final Locale ICBN = new Locale(ICBN_LANG, "");

    /** Langage name for ICZN locale */
    public static final String ICZN_LANG = "iczn";

    /** Locale for ICZN */
    public static final Locale ICZN = new Locale(ICZN_LANG, "");

    /** The name used by this usage */
    protected String ascribedName;

    /** <code>String</code> representing rank of this name usage.  It may be null if rank-less system */
    protected String rank;

    /** <code>Appearance</code> where this usage was recorded */
    protected Appearance appearance;

    /** Authority of this <code>NameUsage</code>, i.e. the first usage of the name */
    protected NameUsage authority;

    /** <code>Vector</code> containing authors of this <code>NameUsage</code> */
    protected Vector authors;

    /** year of authority indicated in the original publication */
    protected int year;

    /** pointer to higher <code>NameUsage</code> */
    protected NameUsage higherTaxon;

    /** boolean indeicating assignment to the higher taxon is incertae sedis or not */
    protected boolean incertaeSedis;

    /** note */
    protected String note;
    
    /**
     * <code>Vector</code> containing either lower <code>NameUsage</code>s
     * or <code>Annotation</code> pointing to them.
     */
    protected Vector lowerTaxa;

    /** boolean indicating this taxon is type or not */
    protected boolean isType;

    /** <CODE>NameUsage</CODE> designated as type of this <CODE>NameUsage</CODE> */
    protected NameUsage type;

    /**
     * <CODE>Vector</CODE> contains <CODE>NameUsage</CODE> when multiple
     * <CODE>NameUsage</CODE>s appeared in single <CODE>Appearance</CODE>
     * designate the same <CODE>NameUsage</CODE> as their type, e.g. 
     * species names in ICBN and ICZN shares the same type specimen.
     */
    protected Vector typeDesignators;

    /**
     * <CODE>String</CODE> representing type status of this <CODE>NameUsage</CODE>.
     * If this <CODE>NameUsage</CODE> represents a taxon, the value is NOMINOTYPE.
     * If this <CODE>NameUsage</CODE> represents a specimen ID, the valume is one of
     * "holotype", 
     */
    protected String typeOfType;

    /**
     * A constant indicates non-specimen type, i.e. type taxon.
     * The word "nominotype" was taken from ICZN 4th ed. 
     * Art. 37, 44, 47 and 61.2.
     */
    public static final String NOMINOTYPE = "nominotype";

    /**
     * A constatns indicates allotype.
     *
     */
    public static final String ALLOTYPE = "allotype";

    public static final String COTYPE = "cotype";
    public static final String GENOTYPE = "genotype";
    public static final String HAPANTOTYPE = "hapantotype";
    public static final String HOLOTYPE = "holotype";
    public static final String LECTOTYPE = "lectotype";
    public static final String NEOTYPE = "neotype";
    public static final String PARALECTOTYPE = "paralectotype";
    public static final String PARATYPE = "paratype";
    public static final String SYNTYPE = "syntype";
    public static final String TOPOTYPE = "topotype";
    public static final String TYPE_SERIES = "type series";
    public static final String TYPE_HOST = "type host";

    /**
     * <code>Vector</code> containing <code>Annotation</code>s made by this <code>NameUsage</code>
     */
    protected Vector annotations;

    /** Constructs an "empty" <code>NameUsage</code> */
    public NameUsage()
    {
	super();
    }

    /** Constructs an "empty" <code>NameUsage</code> */
    public NameUsage(Appearance apperance)
    {
	this();
	setAppearance(apperance);
    }

    /**
     * Constructs a <code>NameUsage</code> object having
     * <code>persistentID</code> as its representation,
     */
    public NameUsage(String persistentID)
    {
	super(persistentID);
    }

    /**
     * Constructs a <code>NameUsage</code> based on
     * <code>nameUsage</code>
     */
    public NameUsage(Name nameUsage)
    {
	//don't use NamedObject(String) constructor
	this();
	
	if(nameUsage instanceof NamedObject){
	    if(nameUsage instanceof NameUsage){
		//copy code
	    }
	    else //incompatible
		string = "";
	}
	else{
	    if(this.isA(nameUsage))
		string = nameUsage.string;
	    else
		string = "";	 
	}
    }

    public NameUsage(String rank, String name,
		     Name auth, Name rec,
		     boolean type,
		     Name higher, Name [] lower)
    {
	this();
	
	this.rank = rank;
	ascribedName     = name;
	authority = new NameUsage(auth);
	appearance  = new Appearance(rec);
	this.isType = type;
	higherTaxon    = new NameUsage(higher);
	
	for(int i = 0; i < lower.length; i++){
	    NameUsage nr = new NameUsage(lower[i]);
	    this.lowerTaxa.addElement(nr);
	}
	
    }

    /**
     * Constructs an <CODE>NameUage</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying a <CODE>NameUsage</CODE>
     *
     */
    public NameUsage(Element xml)
    {
	this(xml, null);
    }

    /**
     * Constructs an <CODE>NameUage</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying a <CODE>NameUsage</CODE>
     *
     */
    public NameUsage(Element xml, Appearance ap)
    {
	super();
	NodeList nodeList = xml.getChildNodes();
	int nodeCount = nodeList.getLength();
	String persistentID = null;
	int authorCount = 0;
	boolean toBeResolved = false;

	for (int j = 0; j < nodeCount; j++) {
	    Node node = nodeList.item(j);
	    if(node.getNodeType () == Node.ELEMENT_NODE) {
		Element element = (Element)node;
		String tagName = element.getTagName();
		
		if     (tagName.equals ("oid"))
		    persistentID = getString(element);
		else if(tagName.equals ("rank"))
		    rank =  getString(element);
		else if(tagName.equals ("name"))
		    ascribedName =  getString(element);
		else if(tagName.equals ("authority")) {
		    String pID = getString(element);
		    authority = (NameUsage)curator.get(pID);
		    if(authority == null) {
			authority = new NameUsage();
			authority.setName(pID);
			curator.put(authority);
		    }
		    /*
		    if(linkFlag) {
			LinkRecord rec = new LinkRecord(nameUsage.persistentID(), getString(element));
			NameUsageAuthority.getInstance().addLinkRecord(rec);
		    } else {
			nameUsage.parseLine("authority{" + getString(element) + "}");
		    }
		    */
		}
		else if(tagName.equals("note"))
		    note = getString(element);
		else if(tagName.equals("locale")) {
		    String localeID = getString(element);

		    if(localeID == null || localeID.length() <= 2) {
			locale = new Locale(" ", "");
		    } else {
			String lang = localeID.substring(0, localeID.indexOf("_"));
			String coun = localeID.substring(localeID.indexOf("_") + 1) ;
			locale = new Locale(lang, coun);
		    }
		}
		else if(tagName.equals ("appearance")) {
		    String pID = getString(element);
		    if(ap != null &&
		       ap.getName().equals(pID)) {
			appearance = ap;
			toBeResolved = true;
		    }
		    else {
			appearance =
			    (Appearance)curator.get(pID);
			if(appearance != null) {
			    toBeResolved = true;
			}
			else {
			    appearance = new Appearance();
			    appearance.setName(pID);
			    curator.put(appearance);
			    appearance.addNameUsage(this);
			}

		    /*
		    if(linkFlag) {
			LinkRecord rec = new LinkRecord(nameUsage.persistentID(), getString(element);
			NameUsageRecorder.getInstance().addLinkRecord(rec);
		    } else {
			nameUsage.parseLine("appearance{" + getString(element) + "}");
                    }
		    */
		    }
		}
		else if(tagName.equals ("higher")) {
		    String pID = getString(element);
		    higherTaxon = (NameUsage)curator.get(pID);
		    if(higherTaxon != null) {
			toBeResolved = true;
		    }
		    else {
			higherTaxon = new NameUsage();
			higherTaxon.setName(pID);
			curator.put(higherTaxon);
			higherTaxon.lowerTaxa = new Vector();
			higherTaxon.lowerTaxa.addElement(this);
		    }
		    /*
		    if(linkFlag) {
			LinkRecord rec = new LinkRecord(nameUsage.persistentID(), getString(element);
			NameUsageHigher.getInstance().addLinkRecord(rec);
		    } else {
			nameUsage.parseLine("higher{" + getString(element) + "}");
		    }
		    */
		}
		else if(tagName.equals ("lowerTaxon")) {
		    String pID = getString(element);
		    NameUsage n = (NameUsage)curator.get(pID);
		    if(n != null) {
			if(lowerTaxa == null)
			    lowerTaxa = new Vector();
			lowerTaxa.addElement(this);
			toBeResolved = true;
		    }
		    else {
			n = new NameUsage();
			n.setName(pID);
			curator.put(n);
			addLowerTaxon(n);
		    }
		    /*
		    if(linkFlag) {
			LinkRecord rec = new LinkRecord(getString(element), nameUsage.persistentID());
			NameUsageHigher.getInstance().addLinkRecord(rec);
		    } else {
			nameUsage.parseLine("lower{" + getString(element) + "}");
		    }
		    */
		}
		else if(tagName.equals ("isType")) {
		    isType = Boolean.valueOf(getString(element)).booleanValue();
		}
		else if(tagName.equals ("type")) {
		    typeOfType = element.getAttribute("type");
		    String pID = getString(element);
		    NameUsage n = (NameUsage)curator.get(pID);
		    if(n != null) {
			toBeResolved = true;
		    }
		    else {
			n = new NameUsage();
			n.setName(pID);
			curator.put(n);
		    }
		    type = n;
		}
		else if(tagName.equals("typeDesignator")) {
		    String pID = getString(element);
		    NameUsage n = (NameUsage)curator.get(pID);
		    if(n != null) {
			toBeResolved = true;
		    }
		    else {
			n = new NameUsage();
			n.setName(pID);
			curator.put(n);
		    }
		    addTypeDesignator(n);
		}
		else if(tagName.equals ("annotation")) {
		    String pID = getString(element);
		    Annotation a =
			(Annotation)curator.get(pID);
		    if(a != null) {
			if(annotations == null)
			    annotations = new Vector();
			annotations.addElement(a);
			toBeResolved = true;
		    }
		    else {
			a = new Annotation();
			a.setName(pID);
			curator.put(a);
			addAnnotation(a);
		    }

		    /*
		    if(linkFlag) {
			LinkRecord rec = new LinkRecord(getString(element), nameUsage.persistentID());
			AnnotationFrom.getInstance().addLinkRecord(rec);
		    } else {
			nameUsage.parseLine("annotation{" + getString(element) + "}");
		    }
		    */
		}
		else{}
            }
        }

	if(persistentID != null &&
	   !persistentID.equals(getName()))
	    setName(persistentID); //i.e. other key data are empty

	if(isType)
	    setType(true);

	if(toBeResolved)
	    curator.resolve(this);
    }
    
    public String persistentID()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).persistentID();

	if(string != null && string.length() != 0)
	    return string;
	
	StringBuffer buffer = getClassNameHeaderBuffer();
	if(rank != null)
	     buffer.append(rank);
	buffer.append('_');
	buffer.append(ascribedName);
	buffer.append('_');

	if(appearance == null){
	    //request code here;
	}
	if(appearance == null)
	    buffer.append("_______");
	else{
	    String pid = appearance.persistentID();
	    buffer.append(pid.substring(pid.indexOf(appearance.getClassNameHeader().toString())));
	}
	return buffer.toString();
    }
    
    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public int getFieldSepartorsCount()
    {
	return 21;
    }

    public void parseLine(String line)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).parseLine(line);

	String s = peal(line);
	if(line.startsWith("rank"))
	    rank = s;
	else if(line.startsWith("group"))
	    rank = s;
	else if(line.startsWith("majorname"))
	    //	    major = s;
	    ascribedName = s;
	else if(line.startsWith("minorname"))
	    //	    minor = s;
	    ascribedName = s;
	else if(line.startsWith("authority"))
	    //      authority = new Publication(s.substring(s.indexOf("Publication::")));
	    //	    authority = new Appearance(s);
	    authority = new NameUsage(s);
	else if(line.startsWith("recorder"))
	    appearance = new Appearance(s);
	else if(line.startsWith("lower"))
	    addLowerTaxon(new NameUsage(s));
	else if(line.startsWith("higher"))
	    higherTaxon  = new NameUsage(s);
	else if(line.startsWith("type"))
	    isType = Boolean.valueOf(s).booleanValue();
	else{
	    if(appearance == null){
		appearance = new Appearance();
	    }
	    appearance.addAnnotation(new Annotation(s));
	}
    }
    
    public boolean merge(NamedObject nr)
    {
	if(!isSameClass(nr))
	    return false;
	return false; //not yet implemented
    }

    /**
     * Returns <code>String</code> representing rank name
     *
     * @return <code>String</code> representing rank name
     */ 
    public String getRank()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getRank();

	if(!isNominal())
	    return rank;

	return getPersistentIDComponent(0);
    }
    
    /**
     * Sets <code>rank</code> as rank name
     *
     * @param rank <code>String</code> representing rank name
     */ 
    public void setRank(String rank)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setRank(rank);

	if(rank == this.rank ||
	   (rank != null && rank.equals(this.rank)))
	    return;

	this.rank = rank;
    }
    
    /**
     * Returns <code>String</code> representing the name used
     *
     * @return <code>String</code> representing the name used
     */ 
    public String getUsedName()
    {
	return getAscribedName();
    }

    /**
     * Returns <code>String</code> representing the name used
     *
     * @return <code>String</code> representing the name used
     */ 
    public String getAscribedName()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).ascribedName;

	if(!isNominal())
	    return ascribedName;

	return getPersistentIDComponent(1);
    }
    
    /**
     * Sets <code>name</code> as the name used
     *
     * @param name <code>String</code> representing the name used
     */ 
    public void setUsedName(String name)
    {
	setAscribedName(name);
    }
    /**
     * Sets <code>name</code> as the name used
     *
     * @param name <code>String</code> representing the name used
     */ 
    public void setAscribedName(String name)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setUsedName(name);


	if(name == ascribedName ||
	   (name != null && name.equals(ascribedName)))
	    return;

	ascribedName = name;
    }
    
    /**
     * Returns <code>Locale</code> representing locale
     *
     * @return <code>Locale</code> representing locale
     */ 
    public Locale getLocale()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getLocale();

	return locale;
    }
    
    /**
     * Sets <code>locale</code> as locale of this <code>NameUsage</code>
     *
     * @param locale <code>Locale</code> to be set
     */ 
    public void setLocale(Locale locale)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setLocale(locale);


	if(this.locale == locale ||
	   (locale != null && locale.equals(this.locale)))
	    return;

	this.locale = locale;
    }
    
    /*
     * Returns <code>Appearance</code> encoding this <code>NameUsage</code>
     *
     * @return <code>Appearance</code> encoding this <code>NameUsage</code>
     */
    public Appearance getAppearance()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getAppearance();

	return appearance;
    }
    
    /*
     * Sets <code>appearance</code> as <code>Apperance</code> encoding this <code>NameUsage</code>
     *
     * @param appearance <code>Appearance</code> encoding this <code>NameUsage</code>
     */
    public void setAppearance(Appearance app)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setAppearance(app);

	if(app == this.appearance)
	    return;

	if(this.appearance != null) {
	    Appearance ap = this.appearance;
	    this.appearance = null;
	    //	    ap.removeNameUsage(this);
	}

	this.appearance = app;
	if(app != null)
	    app.addNameUsage(this);
    }
    
    /**
     * Returns <code>String</code> representing the note used
     *
     * @return <code>String</code> representing the note used
     */ 
    public String getNote()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getNote();

	return note;
    }
    
    /**
     * Sets <code>note</code> as the note used
     *
     * @param note <code>String</code> representing the note used
     */ 
    public void setNote(String note)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setNote(note);


    	this.note = note;
    }
    
   
    /**
     * Returns <code>NameUsage</code> representing higher taxon
     *
     * @return <code>NameUsage</code> representing higher taxon
     */
    public NameUsage getHigherTaxon()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getHigherTaxon();

	return higherTaxon;
    }
    
    /**
     * Sets <code>nameUsage</code> as higher taxon
     *
     * @param nameUsage <code>NameUsage</code> representing higher taxon
     */
    public void setHigherTaxon(NameUsage higherTaxon)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setHigherTaxon(higherTaxon);

	if(higherTaxon == this.higherTaxon)
	    return;

	if(this.higherTaxon != null) {
	    NameUsage n = this.higherTaxon;
	    this.higherTaxon = null;
	    n.removeLowerTaxon(this);
	}
		
	this.higherTaxon = higherTaxon;
	if(this.higherTaxon != null)
	    this.higherTaxon.addLowerTaxon(this);
    }

    /**
     * Returns true if the assignment to the higher taxon
     * should be considerd as incertaes sedis
     *
     * @return true if the assignment to the higher taxon
     * should be considerd as incertae sedis
     */
    public boolean isIncertaeSedis()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).isIncertaeSedis();

	return incertaeSedis;
    }

    /**
     * Sets incertaeSedis to indicate whether the assignment to the higher taxon should be considerd as incertae sedis
     *
     * @param incertaeCeis <code>boolean</code> true if the assignment to the higher taxon
     * should be considerd as incertae sedis
     */
    public void setIncertaeSedis(boolean incertaeSedis)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setIncertaeSedis(incertaeSedis);


	if(this.incertaeSedis == incertaeSedis)
	    return;

	this.incertaeSedis = incertaeSedis;
    }

    /**
     * Returns <code>Vector</code> representing the list of lower taxa
     *
     * @return <code>Vector</code> representing the list of lower taxa
     */
    public Vector getLowerTaxa()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getLowerTaxa();

	return lowerTaxa;
    }

    /**
     * Sets <code>lowerTaxa</code> as the list of lower taxa
     *
     * @param lowerTaxa <code>Vector</code> representing the list of lower taxa
     */
    public void setLowerTaxa(Vector lowerTaxa)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setLowerTaxa(lowerTaxa);


	if(this.lowerTaxa == lowerTaxa)
	    return;

	this.lowerTaxa = lowerTaxa;
    }

    /**
     * Adds <code>nameUsage</code> to the list of lower taxa
     * It returns true if the <code>nameUsage</code> added to the
     * list successfuly, or false if the <code>nameUsage</code> is
     * already in the list.
     *
     * @param nameUsage <code>NameUsage</code> to be added to the list of lower taxa
     *
     * @return true if <code>nameUsage</code> was appended to the list of lower taxa
     * successfully, or false if <code>nameUsage</code> is already in the list
     */
    public boolean addLowerTaxon(NameUsage nameUsage)
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).addLowerTaxon(nameUsage);

	if(lowerTaxa == null)
	    lowerTaxa = new Vector();

	if(lowerTaxa.contains(nameUsage))
	   return false;

	lowerTaxa.addElement(nameUsage);
	nameUsage.setHigherTaxon(this);
	return true;
    }

    /**
     * Removes <code>nameUsage</code> from the list of lower taxa
     *
     * @param nameUsage <code>NameUsage</code> to be removed fromthe list of lower taxa
     */
    public void removeLowerTaxon(NameUsage nameUsage)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).removeLowerTaxon(nameUsage);

	if(lowerTaxa == null ||
	   !lowerTaxa.contains(nameUsage))
	    return;

	lowerTaxa.removeElement(nameUsage);
	nameUsage.setHigherTaxon(null);
    }

    /**
     * Returns <CODE>NameUsage</CODE> representing type taxon of this
     * <CODE>NameUsage</CODE>, or null if no type is designated
     *
     * @return NamedObject representing type taxon of this
     * <CODE>NameUsage</CODE>, or null if no type is designated
     */
    public NameUsage getType()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getType();

	return type;
    }

    /**
     * Returns <CODE>NameUsage</CODE> representing type taxon of this
     * <CODE>NameUsage</CODE>, or null if no type is designated
     *
     * @return NamedObject representing type taxon of this
     * <CODE>NameUsage</CODE>, or null if no type is designated
     */
    public String getTypeOfType()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getTypeOfType();

	return typeOfType;
    }

    /**
     * Returns <CODE>NameUsage</CODE> representing type taxon of this
     * <CODE>NameUsage</CODE>, or null if no type is designated
     *
     * @return NamedObject representing type taxon of this
     * <CODE>NameUsage</CODE>, or null if no type is designated
     */
    public void setTypeOfType(String typeOfType)
    {
	if(entity != null) {
	    ((NameUsage)getEntity()).setTypeOfType(typeOfType);
	    return;
	}

	if(typeOfType == this.typeOfType ||
	   (this.typeOfType != null &&
	    this.typeOfType.equals(typeOfType)))
	    return;

	if(typeOfType == null)
	    setType(false);
	else {
	    setType(true);
	    this.typeOfType = typeOfType;
	}

    }

    /**
     * Returns true if this <code>NamedObject</code> is a type
     *
     * @return true if this <code>NamedObject</code> is a type
     */
    public boolean isType()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).isType();

	if(typeDesignators != null)
	    return true;
	else if(higherTaxon != null && higherTaxon.type != null)
	    return higherTaxon.type.getEntity() == getEntity();
	else
	    return false;
    }
    
    /**
     * Sets boolean as type of taxon
     *
     * @param type boolean true if this taxon is a nominal type
     */
    public void setType(boolean type)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setType(type);

	if(higherTaxon == null)
	    return;

	if(type)
	    higherTaxon.setType(this);
	else if(higherTaxon.getType() == this)
	    higherTaxon.setType(null, null);
    }

    /**
     * Sets boolean as type of taxon
     *
     * @param type boolean true if this taxon is a nominal type
     */
    public void setType(String typeOfType)
    {
	if(typeOfType == null)
	    return;
	if(higherTaxon != null)
	    higherTaxon.setType(this, typeOfType);
    }

    /**
     * Sets <CODE>type</CODE> as type of this <CODE>NameUsage</CODE>.
     * The  <CODE>type</CODE> may be null.
     *
     * @param type <CODE>NameUsage</CODE> to be designated as the type of
     * this <CODE>NameUsage</CODE>
     */
    public void setType(NameUsage type)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setType(type);

	if(type == this.type || type == this)
	    return;

	this.type = type;
    }

    /**
     * Sets <CODE>type</CODE> as <CODE>typeOfType</CODE> of this <CODE>NameUsage</CODE>
     * The  <CODE>type</CODE> may be null.  If <CODE>type</CODE> is null,
     * <CODE>typeOfType</CODE> is ignored.
     *
     * @param type <CODE>NameUsage</CODE> to be designated as the type of
     * this <CODE>NameUsage</CODE>
     * @param typeOfType <CODE>String</CODE> represents type of type, e.g.
     * holotype.
     * 
     */
    public void setType(NameUsage type, String typeOfType)
    {
	setType(type);
	if(type != null)
	    this.typeOfType = typeOfType;
	else
	    this.typeOfType = null;
    }

    /**
     * Adds <CODE>designator</CODE> as one of <CODE>NameUsage</CODE>s
     * designating this <CODE>NameUsage</CODE> as its type.
     *
     * @param designator <CODE>NameUsage</CODE> designating this <CODE>NameUsage</CODE>
     * as its type
     */
    public void addTypeDesignator(NameUsage designator)
    {
	if(higherTaxon == null &&
	   typeDesignators == null) {
	    setHigherTaxon(designator);
	    return;
	}
	
	if(typeDesignators == null) {
	    typeDesignators = new Vector();
	}

	if(higherTaxon != null) {
	    typeDesignators.addElement(higherTaxon);
	    higherTaxon = null;
	}

	if(typeDesignators.contains(designator))
	    return;

	typeDesignators.addElement(designator);
	designator.setType(this);
    }

    /**
     * Removes <CODE>designator</CODE> from the list of <CODE>NameUsage</CODE>s
     * designating this <CODE>NameUsage</CODE> as its type.
     *
     * @param designator <CODE>NameUsage</CODE> to be remvoed from designator list
     * of this <CODE>NameUsage</CODE>
     */
    public void removeTypeDesignator(NameUsage designator)
    {
	if(typeDesignators == null) {
	    if(higherTaxon == designator) {
		higherTaxon.setType(null, null);
		higherTaxon = null;
	    }
	    return;
	}

	if(!typeDesignators.contains(designator))
	    return;

	typeDesignators.removeElement(designator);
	designator.setType(null, null);

	if(typeDesignators.size() == 1) {
	    higherTaxon = (NameUsage)typeDesignators.elementAt(0);
	    typeDesignators.clear();
	    typeDesignators = null;
	}
    }

    /**
     * Returns <code>Vector</code> representing the list of <code>Annotatoin</code>s
     *
     * @return <code>Vector</code> representing the list of <code>Annotatoin</code>s
     */
    public Vector getAnnotations()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getAnnotations();

	return annotations;
    }

    /**
     * Sets <code>annotations</code> as the list of <code>Annotation</code>s made by this <code>NameUsage</code>
     *
     * @param annotations <code>Vector</code> representing the list of <code>Annotation</code>s made by this <code>NameUsage</code>
     */
    public void setAnnotations(Vector annotations)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setAnnotations(annotations);

	if(this.annotations == annotations)
	    return;

	this.annotations = annotations;
    }

    /**
     * Adds <code>Annotation</code> to the list of <code>Annotation</code>s made by this <code>NameUsage</code>
     * It returns true if the <code>annotation</code> added to the
     * list successfuly, or false if the <code>annotation</code> is
     * already in the list.
     *
     * @param annotation <code>Annotation</code> to be added to the list of <code>Annotation</code>s made by this <code>NameUsage</code>
     *
     * @return true if <code>annotation</code> was appended to the list of <code>Annotation</code>s made by this <code>NameUsage</code>
     * successfully, or false if <code>annotation</code> is already in the list
     */
    public boolean addAnnotation(Annotation annotation)
    {
	if(annotation == null)
	    return false;

	if(entity != null)
	    return ((NameUsage)getEntity()).addAnnotation(annotation);

	if(annotations == null)
	    annotations = new Vector();

	if(annotations.contains(annotation))
	   return false;

	annotations.addElement(annotation);
	return true;
    }

    /**
     * Removes <code>annotation</code> from the list of <code>Annotation</code>s made by this <code>NameUsage</code>
     *
     * @param annotation <code>NameUsage</code> to be removed fromthe list of <code>Annotation</code>s made by this <code>NameUsage</code>
     */
    public void removeAnnotation(Annotation annotation)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).removeAnnotation(annotation);

	annotations.removeElement(annotation);
	annotation.removeAnnotator(this);
    }

    /**
     * Returns <code>NameUsage</code> representing authority of this <code>NameUsage</code>,
     * i.e. first usage of the name
     *
     * @return <code>NameUsage</code> representing authority of this <code>NameUsage</code>
     */
    public NameUsage getAuthority()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getAuthority();

	return authority;
    }

    /**
     * Sets <code>authority</code> as authority <code>NameUsage</code> of this object.
     *
     * @param authority <code>NameUsage</code> representing authority of this <code>NameUsage</code>
     */
    public void setAuthority(NameUsage authority)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setAuthority(authority);

	if(this.authority == authority)
	    return;

	this.authority = authority;
    }

    /**
     * Returns <code>Vector</code> representing the list of author of the name.
     * It shall be null if authors  of thename matche to authors of autority
     * <code>NameUsage</code>.
     *
     * @return <code>Vector</code> representing the list of author of the name
     */
    public Vector getAuthors()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getAuthors();

	//return authors;
	if(appearance == null ||
	   appearance.getPublication() == null)
	    return null;
	return appearance.getPublication().getAuthors();
    }

    /**
     * Sets <code>authors</code> as the list of <code>Author</code>s of the name.
     * Is shall not be called if authors  of thename matche to authors of autority
     * <code>NameUsage</code>.
     *
     * @param authors <code>Vector</code> representing the list of <code>Author</code>s of the name
     */
    public void setAuthors(Vector authors)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setAuthors(authors);

	if(this.authors == authors)
	    return;

	this.authors = authors;
    }

    /**
     * Adds <code>Author</code> to the list of <code>Author</code>s of the name.
     * It returns true if the <code>author</code> added to the
     * list successfuly, or false if the <code>author</code> is
     * already in the list.
     * <P>
     * Is shall not be called if authors  of thename matche to authors of autority
     * <code>NameUsage</code>.
     *
     * @param author <code>Author</code> to be added to the list of <code>Author</code>s of the name
     *
     * @return true if <code>author</code> was appended to the list
     * successfully, or false if <code>author</code> is already in the list
     */
    public boolean addAuthor(Author author)
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).addAuthor(author);

	if(authors == null)
	    authors = new Vector();

	if(authors.contains(author))
	   return false;

	authors.addElement(author);
	return true;
    }

    /**
     * Removes <code>author</code> from the list of <code>Author</code>s of the name
     *
     * @param author <code>NameUsage</code> to be removed fromthe list of <code>Author</code>s of the name
     */
    public void removeAuthor(Author author)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).removeAuthor(author);

	authors.removeElement(author);
    }

    /**
     * Returns authority year in <code>int</code>.
     * It must be zero if authors of authority <code>NameUsage</code>
     * are nominal authority of the name.
     *
     * @return int representing authority year.
     * i.e. first usage record of the name
     */
    public int getAuthoriyYear()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getAuthoriyYear();

	return year;
    }

    /**
     * Sets year as authority year.
     * It shall be used only if authority <code>NameUsage</code>
     * gives aurthority author names difference from authors
     * of the authority <code>NameUsage</code>, or if
     * the authority <code>NameUsage</code> is unknown but
     * author names and year are known.
     *
     * @param year <code>int</code> representing authority year.
     */
    public void setAuthoriyYear(int year)
    {
	if(entity != null)
	    ((NameUsage)getEntity()).setAuthoriyYear(year);

	if(this.year == year)
	    return;

	this.year = year;
    }

    /**
     * create XML String 
     *
     * @return XML String representing this NameUsage object
     */
    public String toXMLString()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).toXMLString();

	
	StringBuffer buf = new StringBuffer();
	buf.append("<NameUsage>\n");
	buf.append("<oid>").append(persistentID()).append("</oid>\n");
	if(rank != null)
	    buf.append("<rank>").append(rank).append("</rank>\n");
	if(ascribedName != null)
	    buf.append("<name>").append(ascribedName).append("</name>\n");

	if(locale != null)
	    buf.append("<locale>").append(locale.toString()).append("</locale>\n");

	if(appearance != null)
    	    buf.append("<appearance>").append(appearance.persistentID()).append("</appearance>\n");
	
	if(authority != null)
    	    buf.append("<authority>").append(authority.persistentID()).append("</authority>\n");

	if(authors != null) {
	    Enumeration e = authors.elements();
	    while(e.hasMoreElements()) {
		buf.append("<author>").append(((Author)e.nextElement()).persistentID()).append("</author>\n");
	    }
	}

	if(year != 0)
    	    buf.append("<year>").append(year).append("</year>\n");

    	if(higherTaxon != null) {
    	    buf.append("<higher");
	    if(incertaeSedis)
		buf.append(" attribute=incertaeSedis");
	    buf.append('>').append(higherTaxon.persistentID()).append("</higher>\n");
    	}
	
	if(lowerTaxa != null) {
	    Enumeration e = lowerTaxa.elements();
	    while(e.hasMoreElements()) {
		buf.append("<lowerTaxon>").append(((NameUsage)e.nextElement()).persistentID()).append("</lowerTaxon>\n");
	    }
	}

	if(annotations != null) {
	    Enumeration e = annotations.elements();
	    while(e.hasMoreElements()) {
		buf.append("<annotation>").append(((Annotation)e.nextElement()).persistentID()).append("</annotation>\n");
	    }
	}

	if(note != null)
	    buf.append("<note>").append(note).append("</note>\n");
	
	if(isType())
	    buf.append("<isType>true</isType>\n");

	if(type != null) {
	    buf.append("<type");
	    if(typeOfType != null)
		buf.append(" type=\"").append(typeOfType).append("\"");
	    buf.append(">").append(type.persistentID()).append("</type>\n");
	}

	if(typeDesignators != null) {
	    Enumeration e = typeDesignators.elements();
	    while(e.hasMoreElements())
		buf.append("<typeDesignator>").append(((NameUsage)e.nextElement()).getName()).append("</typeDesignator>");
	}

	buf.append("</NameUsage>\n");
	
        return buf.toString();
    }

    /**
     * create XML String of the all related <code>NamedObject</code>s
     *
     * @return XML String representing this <code>NameUsage</code> object
     */
    public String toRelatedXMLString()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).toRelatedXMLString();

	// create XML String of the NameUsage itself
	StringBuffer buf = new StringBuffer();
	buf.append(toXMLString());
	
	// create XML of the higher NameUsage
	if(higherTaxon != null) {
    	    buf.append(higherTaxon.toXMLString());
        }

	// create XML of the lower NameUsage
	if(lowerTaxa != null) {
            Enumeration e = lowerTaxa.elements();
	    while(e.hasMoreElements()) {
    	        buf.append(((NameUsage)e.nextElement()).toXMLString());
	    }
        }

	// create XML of the authority NameUsage
	if(authority != null) {
    	    buf.append(authority.toXMLString());
    	}

	if(authors != null) {
	    Enumeration e = authors.elements();
	    while(e.hasMoreElements()) {
		buf.append(((Author)e.nextElement()).toXMLString());
	    }
	}

	// create XML of the recorder Appearance
	if(appearance != null) {
    	    buf.append(appearance.toXMLString());
    	}
	
	// create XML of the related Annotation
	if(annotations != null) {
            Enumeration e = annotations.elements();
	    while(e.hasMoreElements()) {
    	        buf.append(((Annotation)e.nextElement()).toXMLString());
	    }
    	}

//	if(type != null)
//	    buf.append(type.toXMLString());
//	
//	if(typeOf != null)
//	    buf.append(typeOf.toXMLString());
	
	return buf.toString();
    }

    public String[] toSQL()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).toSQL();

	String[] sql = new String[8];
	int i = 0;
	sql[i++] = persistentID();
	sql[i++] = rank;
	sql[i++] = ascribedName;
	sql[i++] = (authority == null)?   "null" : authority.persistentID();
	sql[i++] = (appearance == null)?    "null" : appearance.persistentID();
	sql[i++] = (higherTaxon == null)?      "null" : higherTaxon.persistentID();
	sql[i++] = new Boolean(isType).toString();

	return sql;
    }

    /**
     * Returns name of taxonomic view where this <CODE>NameUsage</CODE>
     * belongs to.
     *
     * @return String representing name of the taxonomic view
     */
    public String getViewName()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getViewName();

	return getViewName(false);
    }

    /**
     * Returns name of taxonomic view where this <CODE>NameUsage</CODE>
     * belongs to.  The parameter <CODE>toSort</CODE> determines
     * words order in the name.  If <CODE>toSort</CODE> is true,
     * the first word represents the year of name publciation.
     *
     * @param toSort boolean to determine words order
     *
     * @return String representing name of the taxonomic view
     */
    public String getViewName(boolean toSort)
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getViewName(toSort);

	if(appearance == null ||
	   appearance.getPublication() == null) {
	    return "";
	}

	Publication p = appearance.getPublication();
	String viewName = appearance.getPublication().getViewName();
	if(!toSort)
	    return viewName;
	else
	    return viewName;
    }

    /**
     * Returns yaer of publication as <CODE>String</CODE>
     *
     * @return String representing the year of name publication
     */
    public String getYear()
    {
	if(entity != null)
	    return ((NameUsage)getEntity()).getYear();
	
	if(appearance != null) {
	    Publication p = appearance.getPublication();
	    if(p != null)
		return p.getYear();
	}
	return null;
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	if(object == null ||
	   (object != null && !(object instanceof NameUsage)))
	    throw new IllegalArgumentException(object.getClass().getName() + " can't be an entity of " + getClass().getName());
	super.setEntity(object);
    }

    /**
     * Returns root <CODE>NameUsage</CODE> of the name hierarchy
     * where this <CODE>NameUsage</CODE> belongs to.
     *
     * @return root <CODE>NameUsage</CODE> of the name hierarhcy.
     */
    public NameUsage getRoot()
    {
	NameUsage root = this;
	while(root.higherTaxon != null)
	    root = root.higherTaxon;
	return root;
    }
}
