/*
 * NamedObject.java: an abstract class provaides a "name-commutable"
 * mechanism for Java implementation of the Nomencurator, a
 * Nomenclature Heuristic Model.  For restriction of "final class
 * String" in Java, it uses Name instead of String.  It
 * spoiles simplicity and conceptual importance of name commutability.
 *
 * The same thing can be implemented using nameing mechanism of CORBA.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: NamedObject.java,v 1.19 2002/11/19 01:56:30 nozomi Exp $
 * $Log: NamedObject.java,v $
 * Revision 1.19  2002/11/19 01:56:30  nozomi
 * remove createObjectId()
 *
 * Revision 1.18  2002/09/28 00:49:52  nozomi
 * fix index counting bug in getPersistentIDComponent(String, int)
 *
 * Revision 1.17  2002/09/27 11:27:40  nozomi
 * fix mistype in getPeresistentIDComponent() method name
 *
 * Revision 1.16  2002/09/27 11:27:00  nozomi
 * fix mistype in getPeresistentIDComponent() method name
 *
 * Revision 1.15  2002/09/27 09:40:34  nozomi
 * mdify getPeresistentIDComponent() method
 *
 * Revision 1.14  2002/09/09 17:22:17  nozomi
 * fix mistyping in getPseudoHTML()
 *
 * Revision 1.13  2002/09/09 16:52:56  nozomi
 * fix mistakes in comment
 *
 * Revision 1.12  2002/09/09 16:13:57  nozomi
 * add XML parsing support methods; is it right place?
 *
 * Revision 1.11  2002/09/08 14:22:12  nozomi
 * support mutual add/remove
 *
 * Revision 1.10  2002/08/29 19:48:02  nozomi
 * implements interface Nameable
 *
 * Revision 1.9  2002/06/24 13:09:15  ryo
 * fix reflexive bug in equals()
 *
 * Revision 1.8  2002/04/19 21:41:28  nozomi
 * incorporate peal fix by ryo
 *
 * Revision 1.7  2002/04/15 23:37:17  nozomi
 * make createObjectId() public
 *
 * Revision 1.6  2002/04/10 07:47:58  nozomi
 * persistentID is misused in ParseNomXml and NameRecordEditPanel, but they'll be fixed later
 *
 * Revision 1.5  2002/04/09 03:01:40  nozomi
 * change emptyPersistentID handling
 *
 * Revision 1.4  2002/04/08 04:58:18  nozomi
 * get partial class name from getClass().getName() call
 *
 * Revision 1.3  2002/04/08 01:38:26  nozomi
 * Change StringDelegate to Name
 *
 * Revision 1.2  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 *	
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 *	
 * Revision 1.1  1999/10/24 08:06:34  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.text.ParsePosition;

import java.util.Date;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.Vector;

import org.nomencurator.Name;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * <code>NamedObject</code> provides an <code>Object</code>
 * commutable with its name represented not by <code>String</code>
 * but by <code>Name</code> because <code>java.lang.String</code>
 * is final.
 *
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org/</A>
 * @see org.nomencurator.Name
 * @see java.lang.String
 *
 * @version 	19 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class NamedObject
    extends Name
    implements Serializable
{
    /*
     * Object class of Java has toString() method, so why do we need NamedObject?
     * If you mean concrete object, Object class is sufficient.  A name is only
     * a handle of a concrete object, i.e. phiysical entity. 
     * It is insufficient, however, if you mean abstract object because abstract 
     * object must be commutable with its name.  You can store String to place for Object, 
     * but you can't store Object to place for String.  Extending String class is therefore 
     * streightforward solution.  However, String class is final class in Java,
     * so we can't extend String class.  Sigh.
     */

    protected static String classNameSeparator = "::";
    
    //    protected char fieldSeparator = '_';
    protected static String fieldSeparator = "_";
    
    protected static char[] openers = {'{', '>'};
    protected static char[] closers = {'}', '<'};

    protected final static int BIBTEX = 0;
    protected final static int XML    = BIBTEX + 1;

    protected static char contentsOpen  = openers[BIBTEX];
    protected static char contentsClose = closers[BIBTEX];

    
    /**
     * Constructs an empty <code>NamedObject</code>
     * Only the subclasses may call this constructor.
     */
    protected NamedObject()
    {
	this("");
    }
    
    /**
     * Constructs an <code>NamedObject</code> having
     * given <code>name</code> as <code>String</code> representation.
     */
    protected NamedObject(String name)
    {
	super();
	setPersistentID(name);
    }
    
    /**
     * Returns persistent ID representing this <code>NamedObject</code>.
     * The subclasses must provide this method.
     *
     * @return String representing persistent ID of this <code>NamedObject</code>.
     */
    public abstract String persistentID();

    /**
     * Returns persistent ID representing an empty <code>NamedObject</code>.
     * The subclasses must provide this method.
     *
     * @return String representing persistent ID of an empty <code>NamedObject</code>.
     */
    public String emptyPersistentID()
    {
	return emptyPersistentID(getFieldSepartorsCount());
    }

    /**
     * Returns persistent ID having <code>seprators</code> underscores
     * after class name header.
     *
     * @return String representing persistent ID of an empty <code>NamedObject</code>.
     */
    protected String emptyPersistentID(int separators)
    {
	StringBuffer pid = getClassNameHeaderBuffer();
	for(int i = 0; i < separators; i++)
	    pid.append(fieldSeparator);
	return pid.toString();
    }

    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public abstract int getFieldSepartorsCount();
    
    /**
     * Returns persistent ID representing this <code>NamedObject</code>.
     *
     * @return String representing persistent ID of this <code>NamedObject</code>.
     */
    public String getPersistentID()
    {
	return persistentID();
    }

    /**
     * Returns persistent ID representing this <code>NamedObject</code>.
     *
     * @return String representing persistent ID of this <code>NamedObject</code>.
     */
    public /*protected*/  void setPersistentID(String pid)
    {
	if(isA(pid))
	    string = pid;
	else
	    string = null;
    }

    /**
     * Reteruns class name without package part
     *
     * @reterun class name without package part
     */
    protected String getClassName()
    {
	String className = getClass().getName();
	return className.substring(className.lastIndexOf(".") + 1);
    }

    /**
     * Returns <code>String</code> containing class name and
     * class name separator, i.e. header of persistent ID.
     *
     * @return String persistentID header
     */
    protected String getClassNameHeader()
    {
	return getClassNameHeaderBuffer().toString();
    }

    /**
     * Returns <code>StringBuffer</code> containing class name and
     * class name separator as a seed of persistent ID.
     *
     * @return StringBuffer containing persistentID header
     */
    protected StringBuffer getClassNameHeaderBuffer()
    {
	StringBuffer buffer = new StringBuffer(getClassName());
	buffer.append(classNameSeparator);
	return buffer;
    }
    
    /**
     * Returns <code>String</code> representation of this <code>Object</code>
     * 
     * @return <code>String</code> representing this <code>Object</code>
     */
    public String toString()
    {
	return persistentID();
    }
    
    /**
     * Returns true if <code>ojbectName</code> rpresents an instance of
     * the class, or false if it isn't, including null. 
     *
     * @param objectName "name" <code>String</code> representation of
     * a <code>NamedObject</code>
     *
     * @return true if <code>ojbectName</code> rpresents an instance of
     * the class, or false if it isn't, including null. 
     */  
    protected boolean isA(String objectName)
    {
	if(objectName == null)
	    return false;

	return objectName.startsWith(getClassNameHeader().toString());
    }
    
    /**
     * Returns true if <code>ojbectName</code> rpresents an instance of
     * the class, or false if it isn't, including null. 
     *
     * @param objectName "name" <code>DelegateString</code> representation of
     * a <code>NamedObject</code>
     *
     * @return true if <code>ojbectName</code> rpresents an instance of
     * the class, or false if it isn't, including null. 
     */  
    protected boolean isA(Name objectName)
    {
	if(objectName == null)
	    return false;

	return isA(objectName.string);
    }
    
    /**
     * Parse <code>line</code> and set values of this object
     * if the <code>line</code> represents them appropriately.
     *
     * @param line <code>String</code> representing values of this object
     */
    public abstract void parseLine(String line);
    
    /**
     * Returns contents of a <code>line</code>
     *
     * @param line <code>String</code> containing values of this object
     *
     * @return <code>String</code> representing values of this object
     */
    protected String peal(String line)
    {
	return line.substring(line.indexOf(contentsOpen) + 1, line.lastIndexOf(contentsClose));
    }
    
    /**
     * Returns true if this object is resolved, i.e. has contents
     *
     * @return true if this object is resolved, i.e. has contents
     */
    public boolean isSovled()
    {
	return (string == null || string.length() == 0);
    }
    
    /**
     * Puts this object to <code>nomencurator</code>, an object repository.
     * Returns this if equivalent object is not in <code>nomencurator</code>,
     * or the equivalent object found in <code>nomencurator</code>.
     *
     * @param nomencurator an object repository
     *
     * @return this if no equivalent object is not in <code>nomencurator</code>,
     * or the equivalent object found in <code>nomencurator</code>
     */
    public NamedObject putTo(Nomencurator nomencurator)
    {
	
	String pid = persistentID();
	
	//the name is not recorded in the Nomencurator
	if(!nomencurator.containsKey(pid)){
	    nomencurator.put(pid, this);
	    return this;
	}
	
	NamedObject tnr = (NamedObject)nomencurator.get(pid);
	if(tnr == this)
	    return this;
	
	//it is named
	if(pid.length() > emptyPersistentID().length()){
	    
	    // but it knows nothing better than its name
	    if(string.length() > 0)
		return tnr;
	    
	    // or recorded one knows nothing better than its name
	    else if(tnr.string.length() > 0){
		nomencurator.remove(pid);
		nomencurator.put(pid, this);
		return this;
	    }
	    
	    // or neithr...needs a negciation
	    else if(tnr.merge(this))
		return tnr;
	}
	return this;  //?
    }

    public boolean isSameClass(NamedObject namedObject)
    {
	if(this == namedObject)
	    return true;
	return getClassNameHeader().equals(namedObject.getClassNameHeader());
    }
    
    //it should be abstract
    public abstract boolean merge(NamedObject nobj);
    
    public char charAt(int index) { return persistentID().charAt(index); }
    
    public int compareTo(Object o) { return persistentID().compareTo(o); }
    
    public int compareTo(String anotherString) { return persistentID().compareTo(anotherString); }
    
    public int compareToIgnoreCase(String str) { return persistentID().compareToIgnoreCase(str); }
    
    public String concat(String str) { return persistentID(). concat(str); }
    
    public boolean endsWith(String suffix) { return persistentID().endsWith(suffix); }
    
	public boolean equals(Object anObject) {
		if (anObject != null && anObject instanceof NamedObject)
			return persistentID().equals(((NamedObject)anObject).persistentID());
		return false;
	}

    public boolean equalsIgnoreCase(String anotherString) { return persistentID().equalsIgnoreCase(anotherString); }
    
    public byte[] getBytes() { return persistentID().getBytes(); }
    
    public void getBytes(int srcBegin, int srcEnd, byte[] dst, int dstBegin) { persistentID().getBytes(srcBegin, srcEnd, dst, dstBegin); }
    
    public byte[] getBytes(String enc) throws java.io.UnsupportedEncodingException { return persistentID().getBytes(enc); }
    
    public void getChars(int srcBegin, int srcEnd, char[] dst, int dstBegin) { persistentID().getChars(srcBegin, srcEnd, dst, dstBegin); }
    
    public int hashCode() { return persistentID().hashCode(); }
    
    public int indexOf(int ch) { return persistentID().indexOf(ch); }
    
    public int indexOf(int ch, int fromIndex) { return persistentID().indexOf(ch, fromIndex); }
    
    public int indexOf(String str) { return persistentID().indexOf(str); }
    
    public int indexOf(String str, int fromIndex) { return persistentID().indexOf(str, fromIndex); }
    
    public String intern() { return persistentID().intern(); }
    
    public int lastIndexOf(int ch) { return persistentID().lastIndexOf(ch); }
    
    public int lastIndexOf(int ch, int fromIndex) { return persistentID().lastIndexOf(ch, fromIndex); }
    
    public int lastIndexOf(String str) { return persistentID().lastIndexOf(str); }
    
    public int lastIndexOf(String str, int fromIndex) { return persistentID().lastIndexOf(str, fromIndex); }
    
    public int length() { return persistentID().length(); }
    
    public boolean regionMatches(boolean ignoreCase, int toffset, String other, int ooffset, int len) {
	return persistentID().regionMatches(ignoreCase, toffset, other, ooffset, len); }
    
    public boolean regionMatches(int toffset, String other, int ooffset, int len) {
	return persistentID().regionMatches(toffset, other, ooffset, len); }
    
    public String replace(char oldChar, char newChar) { return persistentID().replace(oldChar, newChar); }
    
    public boolean startsWith(String prefix) { return persistentID().startsWith(prefix); }
    
    public boolean startsWith(String prefix, int toffset) { return persistentID().startsWith(prefix, toffset); }
    
    public String substring(int beginIndex) { return persistentID().substring(beginIndex); }
    
    public String substring(int beginIndex, int endIndex) { return persistentID().substring(beginIndex, endIndex); }
    
    public char[] toCharArray() { return persistentID().toCharArray(); }
    
    public String toLowerCase() { return persistentID().toLowerCase(); }
    
    public String toLowerCase(Locale locale) { return persistentID().toLowerCase(locale); }
    
    public String toUpperCase() { return persistentID().toUpperCase(); }
    
    public String toUpperCase(Locale locale) { return persistentID().toUpperCase(locale); }
    
    public String trim() { return persistentID().trim(); }

    public String getPersistentIDContents()
    {
	String pid = persistentID();
	return pid.substring(pid.indexOf(classNameSeparator) + classNameSeparator.length() + 1);
    }

    /*
    public NamedObjectUtility getUtilityInstance()
    {
	return NamedObjectUtility.getInstance();
    }
    */

    /**
     * Returns XML <code>String</code> representing this object
     *
     * @return XML <code>String</code> representing this object
     */
    public String toXMLString()
    {
	return null;
    }


    /**
     * Returns XML <code>String</code> of the all related <code>NamedObject</code>s
     *
     * @return XML <code>String</code> representing all <code>NamedObject</code>s 
     * relationg to the <code>NamedObject</code>
     */
    public String toRelatedXMLString()
    {
        return toXMLString();
    }

    /**
     * Gets name in <CODE>String</CODE>
     *
     * @return String representing a name
     */
    public String getName(){ return persistentID();}

    /**
     * Establishes mutual linkage between <CODE>NamedObject</CODE>s.
     *
     * @param o1 a <CODE>NamedObject</CODE> to be linked
     * @param v1 a <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * to contain <CODE>o1</CODE>.  It is assumed to owned by <CODE>o2</CODE>
     * @param o2 a <CODE>NamedObject</CODE> to be linked
     * @param v2 a <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * to contain <CODE>o2</CODE>.  It is assumed to owned by <CODE>o1</CODE>
     *
     * @return int result code where 0 indicates both <CODE>NamedObject</CODE>s
     * were put to <CODE>Vector</CODE>s, 1 or 2 indicates eithor <CODE>o1</CODE>
     * or <CODE>o2</CODE> was not addetd to the <CODE>Vector</CODE>, 3 indicates
     * no <CODE>NamedObject</CODE> was added, -1 indicates both <CODE>Vector</CODE>s
     * are null.
     */
    int mutualAdd(NamedObject o1, Vector v1, NamedObject o2, Vector v2) 
    {
	if(v1 == null && v2 == null)
	    return -1;

	int resultCode = 0;

	if(v1 == null || v1.contains(o1))
	    resultCode |= 1;
	else {
	    v1.addElement(o1);
	    o2.setChanged();
	}

	if(v2 == null | v2.contains(o2))
	    resultCode |= 2;
	else {
	    v2.addElement(o2);
	    o1.setChanged();
	}

	return resultCode;
    }

    /**
     * Releases mutual linkage between <CODE>NamedObject</CODE>s.
     *
     * @param o1 a <CODE>NamedObject</CODE> to be unlinked
     * @param v1 a <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * to remove <CODE>o1</CODE> from it.  It is assumed to owned by <CODE>o2</CODE>
     * @param o2 a <CODE>NamedObject</CODE> to be unlinked
     * @param v2 a <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * to remove <CODE>o2</CODE> from it.  It is assumed to owned by <CODE>o1</CODE>
     *
     * @return int result code where 0 indicates both <CODE>NamedObject</CODE>s
     * were removed from <CODE>Vector</CODE>s, 1 or 2 indicates eithor <CODE>o1</CODE>
     * or <CODE>o2</CODE> was not removed from the <CODE>Vector</CODE>, 3 indicates
     * no <CODE>NamedObject</CODE> was removed, -1 indicates both <CODE>Vector</CODE>s
     * are null.
     */
    int mutualRemove(NamedObject o1, Vector v1, NamedObject o2, Vector v2) 
    {
	if(v1 == null && v2 == null)
	    return -1;

	int resultCode = 0;

	if(v1 == null || !v1.contains(o1))
	    resultCode |= 1;
	else {
	    v1.remove(o1);
	    o2.setChanged();
	}

	if(v2 == null | !v2.contains(o2))
	    resultCode |= 2;
	else {
	    v2.remove(o2);
	    o1.setChanged();
	}

	return resultCode;
    }

    /**
     * get String Data
     * @param  xmlElement an <CODE>Element</CODE> of XML document
     * @return String representing contents of <CODE>xmlElement</CODE>
     */
    protected String getString(Element xmlElement)
    {
	Node node = xmlElement.getFirstChild();
	if(node == null)
	    return "";
	if(node instanceof Text) {
	    Text text = (Text)node;
	    if(text.getData() == null)
		return "";
	    else
		return text.getData().trim();
        }
	return getPseudoHTML(xmlElement);
    }

    protected String getPseudoHTML(Element xmlElement)
    {
	StringBuffer html = new StringBuffer();
	if(xmlElement.hasChildNodes()) {
	    NodeList nodes = xmlElement.getChildNodes();
	    int children = nodes.getLength();
	    for(int i = 0; i < children; i++)
		parsePseudoHTML(nodes.item(i), html);
	}
	return html.toString();
    }

    protected void parsePseudoHTML(Node e, StringBuffer html)
    {
	String nodeName = e.getNodeName();
	boolean toAppendTag = !(e instanceof Text);
	boolean isBR = nodeName.equals("BR");
	if(toAppendTag & !isBR){
	    html.append('<');
	    html.append(nodeName);
	    html.append('>');
	}
	if(!e.hasChildNodes() && !isBR) {
	    html.append(e.getNodeValue());
	}
	else {
	    NodeList nodes = e.getChildNodes();
	    int children = nodes.getLength();
	    for(int i = 0; i < children; i++)
		parsePseudoHTML(nodes.item(i), html);
	}
	if(toAppendTag){
	    if(isBR){
		html.append("<BR>");
	    }
	    else {
		html.append("</");
		html.append(nodeName);
		html.append(">\n");
	    }
	}
    }

    /**
     * get Date Data
     * @param  s String
     * @return Date
     */
    protected Date getDate(String s)
    {
	return new SimpleDateFormat().parse(s, new ParsePosition(0));
    }

    /*
    public String getPersistentIDComponent(int counts)
    {
	String pid = getPersistentIDContents();

	int index = 0;
	for(int i = 0; index < pid.length() && i < counts; i++)
	    index = pid.indexOf(fieldSeparator, index);

	if(index < pid.length()) {
	    index++;
	    int next = pid.indexOf(fieldSeparator, index);
	    if(next == -1)
		next = pid.length();
	    pid = pid.substring(index, next);
	}

	return pid;
    }
    */    

    /**
     * Returns the component of the persistentID at <CODE>index</CODE>,
     * or null if <CODE>index</CODE> is out or range.
     *
     * @param index position of the component in the persistentID.
     *
     * @return String representing the component 
     */
    protected String getPersistentIDComponent(int index)
    {
	return getPersistentIDComponent(getName(), index);
    }

    /**
     * Returns the component of the persistentID at <CODE>index</CODE>,
     * or null if <CODE>index</CODE> is out or range.
     *
     */
    public static String getPersistentIDComponent(String pid, int index)
    {
	StringTokenizer tokens = 
	    new StringTokenizer(pid.substring(classNameSeparator.length() + pid.indexOf(classNameSeparator)),
				fieldSeparator);
	if(index > tokens.countTokens())
	    return null;
	int i = -1;
	String token = null;
	while(i < index && tokens.hasMoreTokens()) {
	    token = tokens.nextToken();
	    i++;
	}
	if(i < index)
	    token = null;

	return token;
    }

}
