/*
 * Preference.java: 
 *
 * Copyright (c) 2002 Hiroyuki Okawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Preference.java,v 1.11 2002/08/20 06:13:10 ryo Exp $
 * $Log: Preference.java,v $
 * Revision 1.11  2002/08/20 06:13:10  ryo
 * modify the processing for extending Choice
 *
 * Revision 1.10  2002/08/15 12:57:35  ryo
 * add the processing for extending Choice
 *
 * Revision 1.9  2002/08/13 02:33:35  ryo
 * add save/load processing of user added ranks
 *
 * Revision 1.8  2002/08/01 08:04:45  ryo
 * rename to "enable" from "valid"
 *
 * Revision 1.7  2002/07/12 16:52:29  ryo
 * add name attribute
 *
 * Revision 1.6  2002/07/03 12:06:42  ryo
 * create preference file when not found
 *
 * Revision 1.5  2002/07/01 11:27:33  ryo
 * return the zero length string if the node has no value
 *
 * Revision 1.4  2002/06/21 11:33:39  ryo
 * modify preference file format from object binary data to XML
 *
 * Revision 1.3  2002/06/04 13:45:03  ryo
 * add preference for localdb path
 *
 * Revision 1.2  2002/03/28 06:30:05  okawa
 * add copyright
 *
 * Revision 1.1  2002/02/01 10:01:20  okawa
 * initial import into CVS
 *
 */

package org.nomencurator;

import java.io.File;
import java.io.Serializable;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.net.URL;
import java.util.Vector;
import java.util.Enumeration;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.apache.xerces.parsers.DOMParser;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.nomencurator.controller.PreferenceManager;
import org.nomencurator.broker.AbstractConnection;
import org.nomencurator.broker.HttpConnection;
import org.nomencurator.broker.JdbcConnection;
import org.nomencurator.broker.LocaldbConnection;
import org.nomencurator.broker.NamedObjectBroker;
import org.nomencurator.editor.model.AuthorEditModel;

public class Preference
implements Serializable
{
    private int higherCount;
    private int lowerCount;

	/**
	 * Constructor
	 */
    public Preference() {
		// create preference instance with the default value
        this(1, 3, System.getProperty("user.dir") + System.getProperty("file.separator") + "localdb");
    }

	/**
	 * Constructor
	 * @param int higherCount
	 * @param int lowerCount
	 * @param int searchPriority
	 */
    public Preference(int higherCount, int lowerCount, String localdbPath) {
        this.higherCount = higherCount;
        this.lowerCount = lowerCount;
    }

	/**
	 * @return higherCount
	 */
	public int getHigherCount() {
	    return higherCount;
    }

	/**
	 * @return int lowerCount
	 */
	public int getLowerCount() {
	    return lowerCount;
    }
    
   	/**
	 * @param int higherCount
	 */
	public void setHigherCount(int higherCount) {
	    this.higherCount = higherCount;
    }

   	/**
	 * @param int lowerCount
	 */
	public void setLowerCount(int lowerCount) {
	    this.lowerCount = lowerCount;
    }

	/**
	 * Write preference to XML file
	 * @param file
	 */
	public synchronized void writeXML(File file)
		throws IOException {
		BufferedWriter out = new BufferedWriter(new FileWriter(file));
		out.write("<?xml version=\"1.0\" ?>\n");
		out.write("<property>\n");
		Vector v = NamedObjectBroker.getInstance().getConnectionInterfaces();
		if (v != null && v.size() > 0) {
			for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
				Object o = e.nextElement();
				if (o instanceof JdbcConnection) {
					out.write("\t<connection type=\"JDBC\" ");
					out.write(" name=\"" + ((JdbcConnection)o).getName() + "\"");
					out.write(" writable=\"" + ((JdbcConnection)o).getWritable() + "\"");
					out.write(" enable=\"" + ((JdbcConnection)o).getValid() + "\"");
					out.write(">\n");
					out.write("\t\t<driverClassName>" + ((JdbcConnection)o).getDriverName() + "</driverClassName>\n");
					out.write("\t\t<url>" + ((JdbcConnection)o).getUrl() + "</url>\n");
					out.write("\t\t<username>" + ((JdbcConnection)o).getUsername() + "</username>\n");
					out.write("\t\t<password>" + ((JdbcConnection)o).getPassword() + "</password>\n");
					out.write("\t</connection>\n");
				} else if (o instanceof HttpConnection) {
					out.write("\t<connection type=\"HTTP\" ");
					out.write(" name=\"" + ((HttpConnection)o).getName() + "\"");
					out.write(" writable=\"" + ((HttpConnection)o).getWritable() + "\"");
					out.write(" enable=\"" + ((HttpConnection)o).getValid() + "\"");
					out.write(">\n");
					out.write("\t\t<url>" + ((HttpConnection)o).getUrl() + "</url>\n");
					out.write("\t</connection>\n");
				} else if (o instanceof LocaldbConnection) {
					out.write("\t<connection type=\"LOCAL\" ");
					out.write(" name=\"" + ((LocaldbConnection)o).getName() + "\"");
					out.write(" writable=\"" + ((LocaldbConnection)o).getWritable() + "\"");
					out.write(" enable=\"" + ((LocaldbConnection)o).getValid() + "\"");
					out.write(">\n");
					out.write("\t\t<path>" + ((LocaldbConnection)o).getPath() + "</path>\n");
					out.write("\t</connection>\n");
				}
			}
		}
		out.write("\t<getTreeNest higher=\"" + new Integer(higherCount).toString()
			+ "\" lower=\"" + new Integer(lowerCount).toString() + "\"/>\n");
		// write the user added <code>Rank</code>s
		v = Rank.getUserAddedRanks();
		if (v != null && v.size() > 0) {
			for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
				Rank rank = (Rank)e.nextElement();
				out.write("\t<rank name=\"" + rank.getName() + "\" ");
				if (rank.getHigher() != null) {
					out.write("higher=\"" + rank.getHigher().getName() + "\" ");
				}
				if (rank.getLower() != null) {
					out.write("lower=\"" + rank.getLower().getName() + "\" ");
				}
				out.write("/>\n");
			}
		}
		// write the user added contents of title
		v = AuthorEditModel.getExtendedTitles();
		if (v != null && v.size() > 0) {
			for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
				String str = (String)e.nextElement();
				out.write("\t<authorPanel name=\"title\" value=\"" + str + "\" />\n");
			}
		}
		// write the user added contents of feudality
		v = AuthorEditModel.getExtendedFeudalities();
		if (v != null && v.size() > 0) {
			for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
				String str = (String)e.nextElement();
				out.write("\t<authorPanel name=\"feudality\" value=\"" + str + "\" />\n");
			}
		}
		// write the user added contents of epithet
		v = AuthorEditModel.getExtendedEpithets();
		if (v != null && v.size() > 0) {
			for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
				String str = (String)e.nextElement();
				out.write("\t<authorPanel name=\"epithet\" value=\"" + str + "\" />\n");
			}
		}
		out.write("</property>\n");
		out.flush();
		out.close();
	}

	/**
	 * Read preference from XML file
	 * @param file
	 */
	public void readXML(File file)
		throws IOException, FileNotFoundException {
		DOMParser parser = new DOMParser();
		try {
			parser.parse(new InputSource(file.getAbsolutePath()));
		} catch (SAXException e) {
			e.printStackTrace();
		}
		Document doc = parser.getDocument();
		AbstractConnection co;

		// get connections
		NodeList nl = doc.getElementsByTagName("connection");
		int matches = nl.getLength();
		int i;
		Vector v = new Vector();
		for (i = 0; i < matches; i++) {
			Element e = (Element) nl.item(i);
			String type = e.getAttribute("type");
			co = null;
			if (type.equals("JDBC")) {
				co =
					new JdbcConnection(
						new String(getElementContent(e, "driverClassName")),
						new String(getElementContent(e, "url")),
						new String(getElementContent(e, "username")),
						new String(getElementContent(e, "password")));
			} else if (type.equals("HTTP")) {
				co =
					new HttpConnection(new String(getElementContent(e, "url")));
			} else if (type.equals("LOCAL")) {
				co =
					new LocaldbConnection(
						new String(getElementContent(e, "path")));
			}
			boolean writable = false; // default value.
			boolean enable = true; // default value.
			String str;
			str = e.getAttribute("name");
			if (str != null && str.length() > 0) {
				co.setName(str);
			}
			str = e.getAttribute("writable");
			if (str != null && str.length() > 0) {
				writable = new Boolean(str).booleanValue();
			}
			str = e.getAttribute("enable");
			if (str != null && str.length() > 0) {
				enable = new Boolean(str).booleanValue();
			}
			if (co != null) {
				co.setWritable(writable);
				co.setValid(enable);
				v.add(co);
			}
		}
		NamedObjectBroker.getInstance().setConnectionInterfaces(v);
		NamedObjectBroker.getInstance().connectionAll();

		// get tree nest
		nl = doc.getElementsByTagName("getTreeNest");
		matches = nl.getLength();
		for (i = 0; i < matches; i++) {
			Element e = (Element) nl.item(i);
			int higher = new Integer(e.getAttribute("higher")).intValue();
			int lower = new Integer(e.getAttribute("lower")).intValue();
			if (higher < 1)
				higher = 1;
			if (lower < 1)
				lower = 1;
			this.setHigherCount(higher);
			this.setLowerCount(lower);
		}
		
		// get user added rank
		nl = doc.getElementsByTagName("rank");
		matches = nl.getLength();
		for (i = 0; i < matches; i++) {
			Element e = (Element) nl.item(i);
			String name   = e.getAttribute("name");
			String higher = e.getAttribute("higher");
			String lower  = e.getAttribute("lower");
			Rank rank = new Rank(name);
			if (higher != null)
				rank.setHigher(Rank.get(higher));
			if (lower != null)
				rank.setLower(Rank.get(lower));
			Rank.put(rank);
		}
		
		// get the user added contents of title, contents of feudality, contents of epithet
		nl = doc.getElementsByTagName("authorPanel");
		matches = nl.getLength();
		for (i = 0; i < matches; i++) {
			Element e = (Element) nl.item(i);
			String name  = e.getAttribute("name");
			String value = e.getAttribute("value");
			if (name.equals("title")) {
				AuthorEditModel.addTitle(value);
			} else if (name.equals("feudality")) {
				AuthorEditModel.addFeudality(value);
			} else if (name.equals("epithet")) {
				AuthorEditModel.addEpithet(value);
			} else {
			}
		}
		
	}

	private String getElementContent(Element e, String TagName) {
		NodeList nl = e.getElementsByTagName(TagName);
		Node n = nl.item(0);
		Node content = n.getFirstChild();
		return (content == null) ? "" : content.getNodeValue();
	}

}
