/*
 * Publication.java:  a Java implementation of Publication class
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 1999, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Publication.java,v 1.36 2002/11/19 02:05:17 nozomi Exp $
 * $Log: Publication.java,v $
 * Revision 1.36  2002/11/19 02:05:17  nozomi
 * remove unused linktable
 *
 * Revision 1.35  2002/11/12 03:18:27  nozomi
 * use Nomencurator as an Object pool with name resolver
 *
 * Revision 1.34  2002/10/19 08:10:35  nozomi
 * decode persistentID if nominal
 *
 * Revision 1.33  2002/09/27 19:57:03  nozomi
 * accessor methods parse persistentID if isNominal()
 *
 * Revision 1.32  2002/09/17 04:57:59  nozomi
 * support indirect reference
 *
 * Revision 1.31  2002/09/10 18:10:54  nozomi
 * add Publication(Element)
 *
 * Revision 1.30  2002/09/09 16:50:32  nozomi
 * use abbreviation mode to get author summary
 *
 * Revision 1.29  2002/09/06 07:44:29  nozomi
 * improved Publication#getAuthorListSummary()
 *
 * Revision 1.28  2002/09/03 03:16:45  nozomi
 * add getAuthorsViewName()
 *
 * Revision 1.27  2002/09/02 21:50:49  nozomi
 * fix getViewName()
 *
 * Revision 1.26  2002/09/02 21:43:09  nozomi
 * support getViewName()
 *
 * Revision 1.25  2002/08/29 11:55:01  ryo
 * don't set the citation title in decomposePID()
 *
 * Revision 1.24  2002/08/27 02:33:35  nozomi
 * null pointer handling in getOmittedCitationTitle()
 *
 * Revision 1.23  2002/08/26 09:23:54  t.okada
 * add publisher and place to publication
 *
 * Revision 1.22  2002/08/21 12:01:36  ryo
 * modify getOmittedCitationTitle(), add the processing for d', l' and umlaut
 *
 * Revision 1.21  2002/08/20 13:16:43  t.okada
 * add notes column
 *
 * Revision 1.20  2002/08/20 12:38:47  ryo
 * modify getOmittedCitationTitle(), return one word of the title when the title which removed the article and the preposition is one word
 *
 * Revision 1.19  2002/08/20 12:09:19  ryo
 * add getOmittedCitationTitle() and use in persistentID()
 *
 * Revision 1.18  2002/08/15 05:16:55  ryo
 * change type of year into String
 *
 * Revision 1.17  2002/06/07 14:09:24  ryo
 * add authorListSummary which is automatically created from authorList
 *
 * Revision 1.16  2002/05/15 10:14:56  ryo
 * fix bug in parseLine()
 *
 * Revision 1.15  2002/05/14 14:03:00  ryo
 * fix bug of setContentsTitle()
 *
 * Revision 1.14  2002/05/10 14:11:12  t.okada
 * modify persistentID()
 *
 * Revision 1.13  2002/05/08 10:46:19  ryo
 * append citationTitle and contentsTitle tag in toXmlString()
 *
 * Revision 1.12  2002/04/19 21:42:21  nozomi
 * method name change from getAppearanceList to getAppearances
 *
 * Revision 1.11  2002/04/15 23:39:22  nozomi
 * SQL friendly
 *
 * Revision 1.10  2002/04/14 23:58:18  nozomi
 * fix minor mistake in the comment
 *
 * Revision 1.9  2002/04/11 02:55:25  nozomi
 * XML handring methods were moved to data classes
 *
 * Revision 1.8  2002/04/09 03:01:40  nozomi
 * change emptyPersistentID handling
 * 
 * Revision 1.7  2002/04/08 01:45:30  nozomi
 * Change StringDelegate to Name
 * 
 * Revision 1.6  2002/04/01 06:48:31  nozomi
 * use java.util.Vector
 * 
 * Revision 1.5  2002/03/08 13:53:19  okawa
 * avoid NullPointerException
 * 
 * Revision 1.4  2002/03/07 16:41:01  okawa
 * change decomposePID()
 * 
 * Revision 1.3  2002/02/22 06:59:14  okawa
 * add getAppearanceList()
 * 
 * Revision 1.2  2002/02/21 02:15:29  okawa
 * get utility instance of the NamedObject
 * 
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 * 
 * Revision 1.1  1999/10/24 08:06:39  nozomi
 * Initial revision
 */

package org.nomencurator;

import java.io.Serializable;

import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import org.nomencurator.Author;
import org.nomencurator.Appearance;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An implementation of Publication data structure of Nomencurator data model
 *
 * @see 	org.nomencurator.NamedObject
 * @see 	org.nomencurator.Author
 * @see 	org.nomencurator.Appearance
 * @see <A HREF="http://www.nomencurator.org/">http://www.nomencurator.org/</A>
 *
 * @version 	19 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
public class Publication
    extends NamedObject
    implements Serializable
{
    /** Constant indicating ISBN */
    public  static final int ISBN = 0;

    /** Constant indicating ISSN */
    public  static final int ISSN = ISBN + 1;

    /** Constant indicating neithr ISBN nor ISSN*/
    public  static final int NEITHER = ISBN - 1;

    /** Index of author(s) in persistentID */
    public static final int AUTHOR = 0;
    /** Index of year in persistentID*/
    public static final int YEAR = AUTHOR + 1;
    /** Index of title or ISXN in persistentID*/
    public static final int TITLE = YEAR + 1;
    /** Index of volume in persistentID*/
    public static final int VOLUME = TITLE + 1;
    /** Index of number in persistentID*/
    public static final int NUMBER = VOLUME + 1;
    /** Index of first page in persistentID*/
    public static final int FIRST_PAGE = NUMBER + 1;
    /** Index of last page in persistentID*/
    public static final int LAST_PAGE = FIRST_PAGE + 1;
    /** number of items in the persistentID*/
    public static final int PID_ITEMS = LAST_PAGE;


    /** <code>Vector</code> representing the list of <code>Author</code>s */
    protected Vector authors;

    /**
     * <code>Vector</code> representing the list of <code>Appearance</code>s 
     * appeared in the publication
     */
    protected Vector appearances;

    /** <code>String</code> representing author list as apperared in the publiation */
    protected String authorNames;

    /** <code>String</code> representing affiliation list as apperared in the publiation */
    protected String affiliations;
    
    /** <code>String</code> representing citation title, e.g. journal name */
    protected String citationTitle;

    /** <code>String</code> representing contents title, e.g. title of an article */
    protected String contentsTitle;

    /** <code>String</code> representing ISBN (in case of book) or ISSN (in case of periodicals) */
    protected String isxn;

    /** <code>String</code> representing publication year */
    protected String year;

    /** <code>String</code> representing volume of the publication for articles, periodicals etc. */
    protected String volume;

    /** <code>String</code> representing number of the publication for articles, periodicals etc. */
    protected String number;

    /** <code>String</code> representing first page the publication */
    protected String firstPage;

    /** <code>String</code> representing last page the publication */
    protected String lastPage;

    /** <code>String</code> representing notes the publication */
    protected String notes;

    /** <code>Date</code> when the manuscript was received */
    protected Date   received;

    /** <code>Date</code> when the revised manuscript was received */
    protected Date   revised;

    /** <code>Date</code> when the manuscript was acceppted for publication */
    protected Date   accepted;

    /** <code>String</code> representing publisher of the publication */
    protected String publisher;

    /** <code>String</code> representing place where the publication was published */
    protected String place;

    /** <code>Publication</code> containging this publication, e.g. edited book containing a chapter */
    protected Publication container;

    /** <code>Vector</code> represents a list of <code>Publication</code>s contained in this <code>Publication</code>,
     * e.g. chapters in a edited book */
    protected Vector contents;

    protected String authorListSummary;

    private static final int SQL_PID = 1;
    private static final int SQL_AUTHORNAMES = SQL_PID + 1;
    private static final int SQL_AFFILIATIONS = SQL_AUTHORNAMES + 1;
    private static final int SQL_CITATIONTITLE = SQL_AFFILIATIONS + 1;
    private static final int SQL_ISXN = SQL_CITATIONTITLE + 1;
    private static final int SQL_YEAR = SQL_ISXN + 1; //int
    private static final int SQL_VOLUME = SQL_YEAR + 1;
    private static final int SQL_NUMBER = SQL_VOLUME + 1;
    private static final int SQL_FIRSTPAGE = SQL_NUMBER + 1;
    private static final int SQL_LATPAGE = SQL_FIRSTPAGE + 1;
    private static final int SQL_NOTES = SQL_LATPAGE + 1;
    private static final int SQL_RECEIVED = SQL_NOTES + 1; //Date
    private static final int SQL_REVISED = SQL_RECEIVED + 1; //Date
    private static final int SQL_ACCEPTED = SQL_REVISED + 1; //Date
    private static final int SQL_COLUMNS  = SQL_ACCEPTED + 1; //number of columns

    /**
     * list of an article and a preposition
     */
    private static String[] articleAndPrepositionList = {
	"a", "about", "af", "along", "an", "and", "at", "au",
	"aus", "avec", "behind", "bei", "beyond", "by",
	"d'", "da", "dans", "das", "de",
	"del", "dell'", "della", "delle", "dem",
	"den", "der", "des", "det", "di",
	"die", "do", "du", "e", "ein",
	"einem", "einer", "eines", "el", "en",
	"et", "for",
	new String(new char[] {'f', 0xf6, 'r'}), // O-umlaut
	"for", "foer",
	"fra", "from",
	new String(new char[] {'f', 0xfc, 'r'}), // U-umlaut
	"fur", "fuer",
	"i", "il", "im", "in", "l'",
	"la", "le", "les", "los", "mit",
	"of", "on", "os", "pour", "pro",
	"som", "sur", "the", "till", "to",
	new String(new char[] {0xfc, 'b', 'e', 'r'}), // U-umlaut
	"uber", "ueber",
	"uma", "un", "una", "und",
	"une", "van", "von", "with", "y", "zu", "zur"
    };
    
    protected final static  String[][] abbreviationMap =  {
	{"Journal", "J."}, {"International", "Int."}, {"Science", "Sci."},
	{"Adavance", "Adv."}, {"Adavances", "Adv."}, {"Microbiology", "Microb."},
	{"Biochemistory", "Biochem."} 
    };

    protected static Hashtable titleAbbreviation = new Hashtable(abbreviationMap.length);
    
    /**
     * Constructs an empty <code>Publication</code> object
     */
    public Publication()
    {
	super();
    }
    
    /**
     * Constructs an <code>Auahot</code> having
     * <code>name</code> as its name, i.e. perisitent ID
     *
     * @param name <code>String</code> representing its name,
     * i.e. perisitent ID
     */
    public Publication(String name)
    {
	super(name);
    }
    
    /**
     * Constructs a copy of given <code>name</code>
     *
     * @param name <code>Name</code> representation of an <code>Publication</code>
     */
    public Publication(Name name)
    {
	//don't use NamedObject(String) constructor
	this();
	
	if(name instanceof Publication)
	    publication((Publication)name);
	else
	    setPersistentID(name.getName());
    }

    /**
     * Constructs a <code>Publication</code> object using
     * based on SQL <code>resultSet</code>
     *
     * @param resultSet <code>java.sql.ResultSet</code> representing
     * SQL result
     */
    public Publication(ResultSet resultSet)
    throws java.sql.SQLException
    {
	this(resultSet.getString(SQL_AUTHORNAMES),
	     resultSet.getString(SQL_AFFILIATIONS),
	     resultSet.getString(SQL_CITATIONTITLE),
	     null, //contentsTitle
	     resultSet.getString(SQL_ISXN),
	     resultSet.getString(SQL_YEAR),
	     resultSet.getString(SQL_VOLUME),
	     resultSet.getString(SQL_NUMBER),
	     resultSet.getString(SQL_FIRSTPAGE),
	     resultSet.getString(SQL_LATPAGE),
	     resultSet.getString(SQL_NOTES),
	     resultSet.getDate(SQL_RECEIVED),
	     resultSet.getDate(SQL_REVISED),
	     resultSet.getDate(SQL_ACCEPTED),
	     null, //authors
	     null, //appearances
	     null, //container
	     null //contents
	     );

	setPersistentID(resultSet.getString(SQL_PID));     
    }

    /**
     * Constructs a non-trivial <code>Publication</code> object.
     *
     * @param authorNames <code>String</code> represents author names as appeared in the publication
     * @param affiliations <code>String</code> represents affiliations of authors as appeared in the publication 
     * @param citationTitle <code>String</code> represents citational title of the publication
     * @param contentsTitle <code>String</code> represents title of the article, chapter etc.
     * @param isxn <code>String</code> represents ISBN or ISSN of the publication if available
     * @param year <code>int</code> represents year of publication
     * @param volume <code>String</code> represents volume of the publication
     * @param number <code>String</code> represents number of the publication
     * @param firstPage <code>String</code> represents first page of the publication 
     * @param lastPage <code>String</code> represents last page of the publication
     * @param notes <code>String</code> represents notes of the publication
     * @param received <code>Date</code> when the manuscript was received
     * @param revised <code>Date</code> when the revised manuscript was received
     * @param accepted <code>Date</code> when the manuscript was acceppted to publish
     * @param authors <code>Vector</code> representing the list of <code>Author</code>s
     * @param appearances <code>Vector</code> representing the list of <code>Appearance</code>s 
     * @param container <code>Publication</code> containging this publication, e.g. edited book containing a chapter
     * @param  contents<code>Vector</code> represents a list of <code>Publication</code>s contained in this <code>Publication</code>,
     * e.g. chapters in a edited book
     */    
    public Publication(String authorNames,
		       String affiliations,
		       String citationTitle,
		       String contentsTitle,
		       String isxn,
		       String year,
		       String volume,
		       String number,
		       String firstPage,
		       String lastPage,
		       String notes,
		       Date received,
		       Date revised,
		       Date accepted,
		       Vector authors,
		       Vector appearances,
		       Publication container,
		       Vector contents)
    {
	this();
	publication(false, authorNames, affiliations, citationTitle, contentsTitle, isxn, 
		    year, volume, number, firstPage, lastPage, notes,
		    received, revised, accepted,
		    authors, appearances, container, contents);
    }

    /**
     * Constructs a deep copy of <code>publication</code>
     *
     * @paran publication <code>Publication</code> to be copied deeply
     */
    public Publication(Publication publication)
    {
	this();
	publication(publication);
    }


    /**
     * Constructs an <CODE>Publication</CODE> object using XML data
     * given by <CODE>xml</CODE>
     *
     * @param xml <CODE>Element</CODE> specifying <CODE>Publication</CODE>
     *
     */
    public Publication(Element xml)
    {
	super();
	NodeList nodeList = xml.getChildNodes();
	int subChildCount = nodeList.getLength();
	String persistentID = null;
	Hashtable hash = new Hashtable();
	int authorCount = 0;
	boolean toBeResolved = false;

	for (int j = 0; j < subChildCount; j++) {
	    Node node = nodeList.item(j);
	
	    if(node.getNodeType () == Node.ELEMENT_NODE) {
		Element element = (Element)node;
		String tagName = element.getTagName();
		String id = element.getAttribute("id");

		if (tagName.equals ("oid")) 
		    persistentID = getString(element);
		else if(tagName.equals ("authors"))
		    authorNames = getString(element);
		else if(tagName.equals ("affiliations"))
		    affiliations = getString(element);
		else if(tagName.equals ("author")) {
		    String pID = getString(element);
		    hash.put(id, pID);
		    authorCount++;
		}
		else if(tagName.equals ("citationTitle"))
		    citationTitle = getString(element);
		else if(tagName.equals ("contentsTitle"))
		    contentsTitle = getString(element);
		else if(tagName.equals ("isxn"))
		    isxn = getString(element);
		else if(tagName.equals ("year"))
		    year = getString(element);
		else if(tagName.equals ("volume"))
		    volume = getString(element);
		else if(tagName.equals ("number"))
		    number = getString(element);
		else if(tagName.equals ("startpage"))
		    firstPage = getString(element);
		else if(tagName.equals ("lastpage"))
		    lastPage = getString(element);
		else if(tagName.equals ("notes"))
		    notes = getString(element);
		else if(tagName.equals ("publisher"))
		    publisher = getString(element);
		else if(tagName.equals ("place"))
		    place = getString(element);
		else if(tagName.equals ("appearance")) {
		    String pID = getString(element);
		    Appearance ap =
			(Appearance)curator.get(pID);
		    if(ap != null) {
			if(appearances == null)
			    appearances = new Vector();
			appearances.addElement(ap);
			toBeResolved = true;
		    }
		    else {
			ap = new Appearance();
			ap.setName(pID);
			curator.put(ap);
			addAppearance(ap);
		    }
		}
		else{}
            }
        }
	
	if(persistentID != null &&
	   !persistentID.equals(getName()))
	    setName(persistentID); //i.e. other key data are empty

	if(authorCount > 0) {
	    authors = new Vector(authorCount);
	    String pID = null;
	    for(int i = 0; i < authorCount; i++) {
		pID = (String)hash.get(Integer.toString(i));
		Author au =
		    (Author)curator.get(pID);
		if(au != null) {
		    authors.addElement(au);
		    toBeResolved = true;
		}
		else {
		    au = new Author();
		    au.setName(pID);
		    curator.put(au);
		    addAuthor(au);
		}
	    }
	}
	/*
	if(toBeResolved)
	    curator.resolve(this);
	*/
    }

    /**
     * Copies <code>Publication</code> <code>p</code> to this object deeply
     *
     * @paran p <code>Publication</code> to be copied deeply
     */
    protected void publication(Publication p)
    {
	publication(true,
		    p.authorNames,
		    p.affiliations,
		    p.citationTitle,
		    p.contentsTitle,
		    p.isxn,
		    p.year,
		    p.volume,
		    p.number,
		    p.firstPage,
		    p.lastPage,
		    p.notes,
		    p.received,
		    p.revised,
		    p.accepted,
		    p.authors,
		    p.appearances,
		    p.container,
		    p.contents);
    }
    
    /**
     * Copy parameters to this object in deep or shallow depending on <code>deepCopy</code>
     *
     * @param deepCopy boolean indicating wheter deep or shallow copy to be done
     * @param authorNames <code>String</code> represents author names as appeared in the publication
     * @param affiliations <code>String</code> represents affiliations of authors as appeared in the publication 
     * @param citationTitle <code>String</code> represents citational title of the publication
     * @param contentsTitle <code>String</code> represents title of the article, chapter etc.
     * @param isxn <code>String</code> represents ISBN or ISSN of the publication if available
     * @param year <code>int</code> represents year of publication
     * @param volume <code>String</code> represents volume of the publication
     * @param number <code>String</code> represents number of the publication
     * @param firstPage <code>String</code> represents first page of the publication 
     * @param lastPage <code>String</code> represents last page of the publication
     * @param notes <code>String</code> represents notes of the publication
     * @param received <code>Date</code> when the manuscript was received
     * @param revised <code>Date</code> when the revised manuscript was received
     * @param accepted <code>Date</code> when the manuscript was acceppted to publish
     * @param authors <code>Vector</code> representing the list of <code>Author</code>s
     * @param appearances <code>Vector</code> representing the list of <code>Appearance</code>s 
     * @param container <code>Publication</code> containging this publication, e.g. edited book containing a chapter
     * @param  contents<code>Vector</code> represents a list of <code>Publication</code>s contained in this <code>Publication</code>,
     * e.g. chapters in a edited book
     */    
    protected void publication(boolean deepCopy,
			       String authorNames,
			       String affiliations,
			       String citationTitle,
			       String contentsTitle,
			       String isxn,
			       String year,
			       String volume,
			       String number,
			       String firstPage,
			       String lastPage,
			       String notes,
			       Date received,
			       Date revised,
			       Date accepted,
			       Vector authors,
			       Vector appearances,
			       Publication container,
			       Vector contents)
    {
	if(deepCopy && authorNames != null)
	    this.authorNames = new String(authorNames);
	else
	    this.authorNames = authorNames;

	if(deepCopy && affiliations != null)
	    this.affiliations = new String(affiliations);
	else
	    this.affiliations = affiliations;

	if(deepCopy && citationTitle != null)
	    this.citationTitle = new String(citationTitle);
	else
	    this.citationTitle = citationTitle;

	if(deepCopy && contentsTitle != null)
	    this.contentsTitle = new String(contentsTitle);
	else
	    this.contentsTitle = contentsTitle;

	if(deepCopy && isxn != null)
	    this.isxn = new String(isxn);
	else
	    this.isxn = isxn;

	this.year = year;

	if(deepCopy && volume != null)
	    this.volume = new String(volume);
	else
	    this.volume = volume;

	if(deepCopy && number != null)
	    this.number = new String(number);
	else
	    this.number = number;

	if(deepCopy && firstPage != null)
	    this.firstPage = new String(firstPage);
	else
	    this.firstPage = firstPage;

	if(deepCopy && lastPage != null)
	    this.lastPage = new String(lastPage);
	else
	    this.lastPage = lastPage;

	if(deepCopy && notes != null)
		this.notes = new String(notes);
	else
		this.notes = notes;

	if(deepCopy && received != null)
	    this.received = (Date)received.clone();
	else
	    this.received = received;

	if(deepCopy && revised != null)
	    this.revised = (Date)revised.clone();
	else
	    this.revised = revised;

	if(deepCopy && accepted != null)
	    this.accepted = (Date)accepted.clone();
	else
	    this.accepted = accepted;

	if(deepCopy && authors != null)
	    this.authors = (Vector)authors.clone();
	else
	    this.authors = authors;

	if(deepCopy && appearances != null)
	    this.appearances = (Vector)appearances.clone();
	else
	    this.appearances = appearances;

	this.container = container;

	if(deepCopy && contents != null)
	    this.contents = (Vector)contents.clone();
	else
	    this.contents = contents;
    }

    /**
     * Returns a persistent ID representing this <code>Publication</code>
     *
     * @return String representing this <code>Publication</code>
     */
    public String persistentID()
    {
	if(entity != null)
	    return ((Publication)getEntity()).persistentID();
	return persistentID(fieldSeparator);
    }
    
    /**
     * Returns a persistent ID representing this <code>Publication</code>
     * with specified  <code>separator</code>
     *
     * @param separator <code>char</code> to be used as the field separator
     *
     * @return String representing this <code>Publication</code>
     */
    //    public String persistentID(char separator)
    public String persistentID(String separator)
    {
	if(entity != null)
	    return ((Publication)getEntity()).persistentID(separator);

	return persistentID(separator, true);
    }
    
    /**
     * Returns a persistent ID representing this <code>Publication</code>
     * with specified  <code>separator</code>.  It contains class name header
     * if <code>withClassName</code> true.
     *
     * @param separator <code>char</code> to be used as the field separator
     * @param withClassName <code>boolean</code> specifying with or without
     * class name header
     *
     * @return String representing this <code>Publication</code>
     */
    //    public String persistentID(char separator, boolean withClassName)
    public String persistentID(String separator, boolean withClassName)
    {
	if(entity != null)
	    return ((Publication)getEntity()).persistentID(separator, withClassName);

	if(string != null && string.length() != 0) {
	    if(withClassName)
		return string;
	    else 
		return string.substring(getClassNameHeader().length());
	}

	StringBuffer ret;

	if(withClassName)
	    ret = getClassNameHeaderBuffer();
	else
	    ret = new StringBuffer();
	if(authorNames != null)
	    ret.append(authorNames);
	ret.append(separator);
	if(year != null)
	    ret.append(year);
	ret.append(separator);
	if (getISXN() != null && getISXN().length() != 0) {
		ret.append(getISXN());
	} else {
	    String t = getOmittedCitationTitle();
	    if(t != null)
		ret.append(t);
	}
	ret.append(separator);
	if(volume != null)
	    ret.append(volume);
	ret.append(separator);
	if(number != null)
	    ret.append(number);
	ret.append(separator);
	if(firstPage != null)
	    ret.append(firstPage);
	ret.append(separator);
	if(lastPage != null)
	    ret.append(lastPage);
	
	return ret.toString();
    }
    
    /**
     * Returnes number of fields separators in persistent ID
     *
     * @returnes int representing number of fields separators in persistent ID
     */ 
    public int getFieldSepartorsCount()
    {
	return PID_ITEMS; //after author, year, journal, volume, number, firstPage
    }

    /**
     * Merges <code>namedObject</code> with this <code>Publication</code>
     * if possible, but not yet implemented.
     * It returns true if merged.
     *
     * @param namedObject a <code>NamedObject</code> to be merged
     *
     * @return true if merged, or false if not mergiable
     */
    public boolean merge(NamedObject nr){
	if(!getClassNameHeader().equals(nr.getClassNameHeader()))	
	    return false;
	return false; //not yet implemented
    }
    
    boolean decomposed = false;

    /**
     * Decomposes given persistent ID
     */
    private void decomposePID()
    {
	if(string == null || string.length() == 0 || decomposed)
	    return;

	decomposed = true;
	
	String tmp = string.substring(getClassNameHeader().length());

	if(tmp.indexOf(fieldSeparator) < 0)
	    return;

	authorNames = tmp.substring(0, tmp.indexOf(fieldSeparator));

	tmp = tmp.substring(tmp.indexOf(fieldSeparator) + 1);
    year = tmp.substring(0, tmp.indexOf(fieldSeparator));

	tmp = tmp.substring(tmp.indexOf(fieldSeparator) + 1);
//	citationTitle = tmp.substring(0, tmp.indexOf(fieldSeparator));

	tmp = tmp.substring(tmp.indexOf(fieldSeparator) + 1);
	volume = tmp.substring(0, tmp.indexOf(fieldSeparator));

	tmp = tmp.substring(tmp.indexOf(fieldSeparator) + 1);
	number = tmp.substring(0, tmp.indexOf(fieldSeparator));

	tmp = tmp.substring(tmp.indexOf(fieldSeparator) + 1);
	firstPage = tmp.substring(0, tmp.indexOf(fieldSeparator));

	tmp = tmp.substring(tmp.indexOf(fieldSeparator) + 1);
	//	if(tmp.length() > 0 && tmp.charAt(0) != fieldSeparator)
	if(tmp.length() > 0 && !tmp.startsWith(fieldSeparator))
	    lastPage = tmp;
    
    }

    /**
     * Returns authors' names
     *
     * @return <code>String</code> authors' names as appeared in the publication
     */
    public String getAuthorNames()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAuthorNames();

	if(!isNominal())
	    return authorNames;

	return getPersistentIDComponent(AUTHOR);
    }

    /**
     * Sets <code>authorName</code> as authors' names appeared in the publication
     *
     * @param authorName <code>String</code> representing authors' names as appeared in the publication
     */ 
    public void setAuthorNames(String authorNames)
    {
	if(entity != null)
	    ((Publication)getEntity()).setAuthorNames(authorNames);

	if(this.authorNames == authorNames ||
	   (authorNames != null && authorNames.equals(this.authorNames)))
	    return;
	this.authorNames = authorNames;
    }
    
    /**
     * Returns affiliation list
     *
     * @return <code>String</code> affiliation list as appeared in the publication
     */
    public String getAffiliations()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAffiliations();

	if(affiliations == null || affiliations.length() == 0) 
	    decomposePID();
	return affiliations;
    }

    /**
     * Sets <code>affiliations</code> appeared in the publication
     *
     * @param affiliations <code>String</code> representing affiliation list appeared in the publication
     */ 
    public void setAffiliationsString(String affiliations)
    {
	if(entity != null)
	    ((Publication)getEntity()).setAffiliationsString(affiliations);

	if(this.affiliations == affiliations ||
	   (affiliations != null && affiliations.equals(this.affiliations)))
	   return;

	this.affiliations = affiliations;
    }

    /**
     * Returns citation title of the publication, e.g. journal name
     *
     * @return <code>String</code> representing citation title of the publication, e.g. journal name
     */
    public String getCitationTitle()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getCitationTitle();

	/*
	if(citationTitle == null || citationTitle.length() == 0) 
	    decomposePID();
	*/

	if(!isNominal())
	    return citationTitle;

	return getPersistentIDComponent(2);
    }
    
    /**
     * Sets <code>citationTitle</code> of the publication
     *
     * @param citationTitle <code>String</code> representing citation title of the publication, e.g. journal name
     */
    public void setCitationTitle(String citationTitle)
    {
	if(entity != null)
	    ((Publication)getEntity()).setCitationTitle(citationTitle);

	if(this.citationTitle == citationTitle ||
	   (citationTitle != null && citationTitle.equals(this.citationTitle)))
	    return;

	this.citationTitle = citationTitle;
    }

    /**
     * Returns contents title of the publication, e.g. article's title
     *
     * @return <code>String</code> representing contents title of the publication, e.g. article's title
     */
    public String getTitle()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getTitle();

	if(!isNominal()) {
	    if(contentsTitle == null || contentsTitle.length() == 0) 
		return contentsTitle;
	    if(isxn == null || isxn.length() == 0) 
		return isxn;
	}

	return getPersistentIDComponent(TITLE);
    }

    /**
     * Returns contents title of the publication, e.g. article's title
     *
     * @return <code>String</code> representing contents title of the publication, e.g. article's title
     */
    public String getContentsTitle()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getContentsTitle();

	if(!isNominal())
	    return contentsTitle;

	return getPersistentIDComponent(TITLE);
    }

    /**
     * Sets <code>contentsTitle</code> of the publication
     *
     * @param contentsTitle <code>String</code> representing contents title of the publication, e.g. article's title
     */
    public void setContentsTitle(String contentsTitle)
    {
	if(entity != null)
	    ((Publication)getEntity()).setContentsTitle(contentsTitle);

	if(this.contentsTitle == contentsTitle ||
	   (contentsTitle != null && contentsTitle.equals(this.contentsTitle)))
	    return;
	this.contentsTitle = contentsTitle;
    }

    /**
     * Returns ISBN or ISXN 
     *
     * @return <code>String</code>
     */
    public String getISXN()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getISXN();

	if(isxn == null || isxn.length() == 0) 
	    decomposePID();
	return isxn;
    }

    /**
     * Returns <code>String</code> representing ISBN, or 
     * null if it is not ISBN
     *
     * @return <code>String</code> representing ISBN or null if it is not ISBN
     */
    public String getISBN()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getISBN();
	return getISXN("ISBN");
    }
    
    /**
     * Returns <code>String</code> representing ISSN, or 
     * null if it is not ISSN
     *
     * @return <code>String</code> representing ISSN or null if it is not ISSN
     */
    public String getISSN()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getISSN();

	return getISXN("ISSN");
    }
    
    /**
     * Returns <code>String</code> it <code>getISXN</code> starts with
     * <code>heading</code> or null if no.
     *
     * @param heading <code>String</code> specifying ISBN or ISSN.
     *
     * @return <code>String</code> representing ISXN or null if it does not start with <code>heading</code>
     */
    public String getISXN(String heading)
    {
	if(entity != null)
	    return ((Publication)getEntity()).getISXN(heading);

	String value = getISXN();
	if(value.startsWith(heading))
	   return value;
	else
	    return null;
    }

    /**
     * Sets <code>isxn</code> as ISBN or ISSN of the publication
     *
     * @param isxn <code>String</code> to be set as ISBN or ISSN.
     */
    public void setISXN(String isxn)
    {
	if(entity != null)
	    ((Publication)getEntity()).setISXN(isxn);
	if(this.isxn == isxn ||
	   (isxn != null && isxn.equals(this.isxn)))
	    return;

	this.isxn = isxn;
    }
    
    /**
     * Returns year of publication in <code>int</code>
     *
     * @return <code>int</code> representing year of publication
     */
    public String getYear()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getYear();
	/*
	if(year == null || year.length() == 0) 
	    decomposePID();
	*/
	if(!isNominal())
	    return year;

	return getPersistentIDComponent(YEAR);
    }
    
    /**
     * Sets <code>year</code> of publication
     *
     * @param year <code>int</code> representing year of publication
     */ 
    public void setYear(String year)
    {
	if(entity != null)
	    ((Publication)getEntity()).setYear(year);

	if(this.year == year ||
	   (year != null && year.equals(this.year)))
	    return;

	this.year = year;
    }
    
    /**
     * Returns volume of periodical containing the publication
     *
     * @return <code>String</code> representing volume of periodical containing the publication
     */
    public String getVolume()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getVolume();

	if(!isNominal())
	    return volume;

	return getPersistentIDComponent(VOLUME);
    }
    
    /**
     * Sets <code>volume</code> of the publication
     *
     * @param volume <code>String</code> representing volume of the publication
     */
    public void setVolume(String volume)
    {
	if(entity != null)
	    ((Publication)getEntity()).setVolume(volume);

	if(volume == this.volume ||
	   (volume != null && volume.equals(this.volume)))
	    return;

	this.volume = volume;
    }
    
    /**
     * Returns number of journal issue etc. containing the publication
     *
     * @return String representing number of journal issue etc. containing the publication
     */
    public String getNumber ()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getNumber();

	if(!isNominal())
	    return number;

	return getPersistentIDComponent(NUMBER);
    }
    
    /**
     * Sets <code>number</code> as number of journal issue etc. containing the publication
     *
     * @param number <code>String</code> representing number of journal issue etc. containing the publication
     */
    public void setNumber(String number)
    {
	if(entity != null)
	    ((Publication)getEntity()).setNumber(number);

	if(number == this.number ||
	   (number != null && number.equals(this.number)))
	    return;

	this.number = number;
    }
    
    /**
     * Returns <code>String</code> representing first page of the publication
     *
     * @return <code>String</code> representing first page of the publication
     */
    public String getFirstPage()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getFirstPage();

	if(!isNominal())
	    return firstPage;

	return getPersistentIDComponent(FIRST_PAGE);
    }
    
    /**
     * Sets <code>page</code> as the first page of the publication
     *
     * @param page <code>String</code> representing the first page of the publication
     */
    public void setFirstPage(String page)
    {
	if(entity != null)
	    ((Publication)getEntity()).setFirstPage(page);

	if(page == firstPage ||
	   (page != null && page.equals(firstPage)))
	    return;

	firstPage = page;
    }
    
    /**
     * Returns <code>String</code> representing last page of the publication
     *
     * @return <code>String</code> representing last page of the publication
     */
    public String getLastPage()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getLastPage();

	if(!isNominal())
	    return lastPage;

	return getPersistentIDComponent(LAST_PAGE);
    }
    
    /**
     * Sets <code>page</code> as the last page of the publication
     *
     * @param page <code>String</code> representing the last page of the publication
     */
    public void setLastPage(String page)
    {
	if(entity != null)
	    ((Publication)getEntity()).setLastPage(page);

	if(entity != null)
	    ((Publication)getEntity()).setLastPage(page);

	if(page == lastPage ||
	   (page != null && page.equals(lastPage)))
	    return;

	lastPage = page;
    }
    
    /**
     * Returns <code>String</code> representing notes of the publication
     *
     * @return <code>String</code> representing notes of the publication
     */
    public String getNotes()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getNotes();

	if(notes == null || notes.length() == 0) 
	    decomposePID();
	return notes;
    }
    
    /**
     * Sets <code>page</code> as the last page of the publication
     *
     * @param page <code>String</code> representing the last page of the publication
     */
    public void setNotes(String notes)
    {
	if(entity != null)
	    ((Publication)getEntity()).setNotes(notes);

	if(notes == this.notes ||
	   (notes != null && notes.equals(this.notes)))
	    return;

	this.notes = notes;
    }
    
    /**
     * Parses a <code>line</code> and sets values of this object accordint to it
     *
     * @param line <code>String</code> containing fragment of data to be set
     */
    public void parseLine(String line)
    {
	if(entity != null)
	    ((Publication)getEntity()).parseLine(line);

	String s = peal(line);
	if(line.startsWith("authors"))
	    authorNames = s;
	else if(line.startsWith("affiliations"))
	    affiliations = s;
	else if(line.startsWith("year"))
	    year = s;
	else if(line.startsWith("ISXN"))
	    isxn = s;
	else if(line.startsWith("title"))
	    citationTitle    = s;
	else if(line.startsWith("citationTitle"))
	    citationTitle    = s;
	else if(line.startsWith("contentsTitle"))
	    contentsTitle    = s;
	else if(line.startsWith("volume"))
	    volume   = s;
	else if(line.startsWith("number"))
	    number   = s;
	else if(line.startsWith("from"))
	    firstPage     = s;
	else if(line.startsWith("to"))
	    lastPage = s;
	else if(line.startsWith("notes"))
	    notes     = s;
	else if(line.startsWith("publisher"))
	    publisher = s;
	else if(line.startsWith("place"))
	    place     = s;
	/*
	  else if(line.startsWith("received"))
	  received = s;
	  else if(line.startsWith("revised"))
	  revised  = s;
	  else if(line.startsWith("accepted"))
	  accepted = s;
	*/
	else if(line.startsWith("Author"))
	    addAuthor(new Author(s));
	else if(line.startsWith("Appearance"))
	    addAppearance(new Appearance(s));
    }

    /**
     * Returns <code>Publication</code> containing the pulication,
     * e.g. an edited book containing this publciation as a chapter
     *
     * @return <code>Publication</code> containing the pulication
     */
    public Publication getContainer()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getContainer();

	return container;
    }

    /**
     * Sets <code>publication</code> as container of this publication,
     * e.g. an edited book containing this publciation as a chapter
     *
     * @return <code>Publication</code> to be set as container of the pulication
     */ 
    public void setContainer(Publication publication)
    {
	if(entity != null)
	    ((Publication)getEntity()).setContainer(publication);

	if(container == publication)
	    return;
	if(container != null)
	    container.removeContents(this);
	container = publication;
	if(container != null)
	    container.addContents(this);
    }

    /**
     * Returns name of publisher in <code>String</code>
     *
     * @return Strings represnting name of publisher
     */
    public String getPublisher()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getPublisher();

	return publisher;
    }

    /**
     * Sets <code>publisher</code> as name of the publisher
     *
     * @param publisher <code>String</code> representing name of the publisher
     *
     */
    public void setPublisher(String publisher)
    {
	if(entity != null)
	    ((Publication)getEntity()).setPublisher(publisher);

	if(publisher == this.publisher ||
	   (publisher != null && publisher.equals(this.publisher)))
	    return;

	this.publisher = publisher;
    }

    /**
     * Returns <code>String</code> representing place of publication
     *
     * @return String representing place of publication
     */
    public String getPlace()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getPlace();
	return place;
    }

    /**
     * Sets <code>place</code> as place of publication
     *
     * @param place <code>String</code> representing place of publication
     */
    public void setPlace(String place)
    {
	if(entity != null)
	    ((Publication)getEntity()).setPlace(place);

	if(place == this.place ||
	   (place != null && place.equals(this.place)))
	    return;

	this.place = place;
    }

    /**
     * Returns <code>Date</code> when the manuscript was received
     *
     * @returns <code>Date</code> when the manuscript was received
     */
    public Date getReceived()
    {
	return received;
    }

    /**
     * Sets <code>date</code> when the manuscript was received
     *
     * @param date <code>Date</code> when the manuscript was received
     */
    public void setReceived(Date date)
    {
	if(entity != null)
	    ((Publication)getEntity()).setReceived(date);

	if(date == received)
	    return;
	received = date;
    }

    /**
     * Returns <code>Date</code> when the manuscript was revised
     *
     * @returns <code>Date</code> when the manuscript was revised
     */
    public Date getRevised()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getReceived();
	return revised;
    }

    /**
     * Sets <code>date</code> when the manuscript was reviseed
     *
     * @param date <code>Date</code> when the manuscript was revised
     */
    public void setRevised(Date date)
    {
	if(entity != null)
	    ((Publication)getEntity()).setReceived(date);

	if(revised == date)
	    return;

	revised = date;
    }

    /**
     * Returns <code>Date</code> when the manuscript was accepted for publication
     *
     * @returns <code>Date</code> when the manuscript was accepted for publication
     */
    public Date getAccepted()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAccepted();
	return accepted;
    }

    /**
     * Sets <code>date</code> when the manuscript was accepted for publication
     *
     * @param date <code>Date</code> when the manuscript was accepted for publication
     */
    public void setAccepted(Date date)
    {
	if(entity != null)
	    ((Publication)getEntity()).setAccepted(date);

	if(accepted == date)
	    return;
	accepted = date;
    }

    /**
     * Returns <code>Vector</code> representing <code>Author</code>s list of this publication
     *
     * @return Vector representing <code>Author</code>s list of this publication
     */
    public Vector getAuthors()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAuthors();
	return authors;
    }
    
    /**
     * Sets <code>authors</code> as <code>Author</code>s list of this publication
     *
     * @param authors <code>Vector</code> representing <code>Author</code>s list of this publication
     */
    public void setAuthors(Vector authors)
    {
	if(entity != null)
	    ((Publication)getEntity()).setAuthors(authors);
	if(this.authors == authors)
	    return;
	this.authors = authors;
    }

    /**
     * Adds <code>author</code> to the author list
     *
     * @param author <code>Author</code> to be added to the author list
     */
    public boolean addAuthor(Author author)
    {
	if(entity != null)
	    return ((Publication)getEntity()).addAuthor(author);

	if(authors == null)
	    authors = new Vector();
	if(authors.contains(author))
	    return false;

	authors.addElement(author);
	author.addPublication(this);
	return true;
    }

    /**
     * Removes <code>author</code> from the author list
     *
     * @param author <code>Author</code> to be removed form the author list
     */
    public void removeAuthor(Author author)
    {
	if(entity != null)
	    ((Publication)getEntity()).removeAuthor(author);

	if(authors == null)
	    return;

	authors.removeElement(author);
	author.removePublication(this);
	if(authors.isEmpty())
	    authors = null;
    }

    /**
     * Returns <code>Vector</code> representing table of contents
     *
     * @return Vector representing table of contents
     */
    public Vector getContents()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getContents();
	return contents;
    }

    /**
     * Sets <code>contents</code> as <code>Vector</code> representing table of contents
     *
     * @param contents <code>Vector</code> representing table of contents
     */
    public void setContents(Vector contents)
    {
	if(entity != null)
	    ((Publication)getEntity()).setContents(contents);
	if(this.contents == contents)
	    return;
	this.contents = contents;
    }

    /**
     * Adds <code>publication</code> to the table of contents
     *
     * @param publication <code>Publication</code> to be added to the table of contents
     */
    public boolean addContents(Publication publication)
    {
	if(entity != null)
	    return ((Publication)getEntity()).addContents(publication);
	if(publication == null)
	   return false;

	if(contents == null)
	    contents = new Vector();
	if(contents.contains(publication))
	   return false;

	contents.addElement(publication);
	publication.setContainer(this);
	return true;
    }

    /**
     * Removes <code>publication</code> from the table of contents
     *
     * @param publication <code>Publication</code> to be removed from the table of contents
     */
    public void removeContents(Publication publication)
    {
	if(entity != null)
	    ((Publication)getEntity()).removeContents(publication);

	if(contents == null ||
	   publication == null ||
	   !contents.contains(publication))
	    return;

	contents.removeElement(publication);
	publication.setContainer(null);

	if(contents.isEmpty())
	    contents = null;
    }

    /**
     * Returns <code>Vector</code> representing appearance list
     *
     * @return <code>Vector</code> representing appearance list
     */
    public Vector getAppearances()
    {
	if(entity != null)
	    ((Publication)getEntity()).getAppearances();
	return appearances;
    }

    /**
     * Sets <code>appearances</code> as <code>Vector</code> representing list of <code>Appearance</code>s
     *
     * @param appearances <code>Vector</code> representing table of <code>Appearance</code>s
     */
    public void setAppearances(Vector appearances)
    {
	if(entity != null)
	    ((Publication)getEntity()).setAppearances(appearances);

	if(this.appearances == appearances)
	    return;

	if ( this.appearances != null &&
	     appearances != null ) {
	    Object[] a = this.appearances.toArray();
	    for(int i = 0; i < a.length; i++) {
		if(appearances.contains(a[i]))
		    removeAppearance((Appearance)a[i]);
	    }
	}
	this.appearances = appearances;

	if ( this.appearances != null) {
	    Object[] a = this.appearances.toArray();
	    for(int i = 0; i < a.length; i++) {
		((Appearance)a[i]).setPublication(this);
	    }
	}
    }

    /**
     * Adds <code>appearance</code> to the list of <code>Appearance</code>s
     *
     * @param appearance <code>Appearance</code> to be added to the list of <code>Appearance</code>s
     */
    public void addAppearance(Appearance appearance)
    {
	if(entity != null)
	    ((Publication)getEntity()).addAppearance(appearance);

	if(appearance == null)
	    return;

	if(appearances == null)
	    appearances = new Vector();
	if(appearances.contains(appearance))
	    return;

	appearances.addElement(appearance);
	appearance.setPublication(this);
    }

    /**
     * Removes <code>appearance</code> from the list of <code>Appearance</code>s
     *
     * @param appearance <code>Appearance</code> to be removed from the list of <code>Appearance</code>s
     */
    public void removeAppearance(Appearance appearance)
    {
	if(entity != null)
	    ((Publication)getEntity()).removeAppearance(appearance);

	if(appearances == null ||
	   appearance == null ||
	   !appearances.contains(appearance)
	   )
	    return;

	appearances.removeElement(appearance);
	appearance.setPublication(null);

	if(appearances.isEmpty())
	    appearances = null;
    }


    public String getAuthorListSummary()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAuthorListSummary();

	return getAuthorListSummary(authors);
    }

    public static String getAuthorListSummary(Vector authors)
    {
	StringBuffer buffer = new StringBuffer();
	
	if(authors != null && authors.size() > 0) {
	    buffer.append(((Author)authors.elementAt(0)).getFullname(Author.ABBREVIATE));
	    switch(authors.size()) {
	    case 1:
		break;
	    case 2:
		buffer.append(" & ");
		buffer.append(((Author)authors.elementAt(1)).getFullname(Author.ABBREVIATE));
		break;
	    default:
		buffer.append(" et al.");
	    }
	}

	return buffer.toString();
    }
	
    public void setAuthorListSummary(String authorListSummary)
    {
	if(entity != null)
	    ((Publication)getEntity()).setAuthorListSummary(authorListSummary);

	this.authorListSummary = authorListSummary;
    }

    /**
     * create XML String 
     *
     * @return XML String of this <code>Publcation</code> object
     */
    public String toXMLString()
    {
	if(entity != null)
	    return ((Publication)getEntity()).toXMLString();

	StringBuffer buf = new StringBuffer();
	buf.append("<Publication>\n");
	buf.append("<oid>").append(persistentID()).append("</oid>\n");
	if(authorNames != null)
	    buf.append("<authors>").append(authorNames).append("</authors>\n");
	if(authorListSummary != null)
	    buf.append("<authorListSummary>").append(authorListSummary).append("</authorListSummary>\n");
	String s = getAffiliations();
	if(s != null)
	    buf.append("<affiliation>").append(s).append("</affiliation>\n");
        if(authors != null) {
	    for (int i=0; i<authors.size(); i++) {
		buf.append("<author id=\"").append(i).append("\">").append(((Author)authors.elementAt(i)).persistentID()).append("</author>\n");
	    }
	}

	s = getCitationTitle();
	if(s != null)
	    buf.append("<citationTitle>").append(s).append("</citationTitle>\n");

	s = getContentsTitle();
	if(s != null)
	    buf.append("<contentsTitle>").append(s).append("</contentsTitle>\n");

	s = getISXN();
	if(s != null)
	    buf.append("<isxn>").append(s).append("</isxn>\n");

	s = getYear();
	if(s != null)
	    buf.append("<year>").append(s).append("</year>\n");

	s = getVolume();
	if(s != null)
	    buf.append("<volume>").append(s).append("</volume>\n");

	s = getNumber();
	if(s != null)
	    buf.append("<number>").append(s).append("</number>\n");

	s = getFirstPage();
	if(s != null)
	    buf.append("<startpage>").append(s).append("</startpage>\n");

	s = getLastPage();
	if(s != null)
	    buf.append("<lastpage>").append(s).append("</lastpage>\n");

	s = getNotes();
	if(s != null)
	    buf.append("<notes>").append(s).append("</notes>\n");

	if(getReceived() != null)
	    buf.append("<received>").append(getReceived()).append("</received>\n");

	if(getRevised() != null)
	    buf.append("<revised>").append(getRevised()).append("</revised>\n");

	if(getAccepted() != null)
	    buf.append("<accepted>").append(getAccepted()).append("</accepted>\n");

	s = getPublisher();
	if(s != null)
	    buf.append("<publisher>").append(s).append("</publisher>\n");

	s = getPlace();
	if(s != null)
	    buf.append("<place>").append(s).append("</place>\n");

	if(s != null)
	// buf.append("<partof>").append(getPartOf()).append("</partof>\n");
	// buf.append("<parts>").append(getParts()).append("</parts>\n");
        if(getAppearances() != null) {
            Enumeration e = getAppearances().elements();
	    while(e.hasMoreElements()) {
    	        buf.append("<appearance>").append(((Appearance)e.nextElement()).persistentID()).append("</appearance>\n");
	    }
	}
	buf.append("</Publication>\n");
	
        return buf.toString();
    }
    
    /**
     * create XML String of the all Related NamedObject
     *
     * @return XML String of this <code>Publication</code> object and required objects for PublicationPanel
     */
    public String toRelatedXMLString()
    {
	if(entity != null)
	    return ((Publication)getEntity()).toRelatedXMLString();

	// create XML String of the Publication itself
	StringBuffer buf = new StringBuffer();
	buf.append(toXMLString());
	
	// create XML of the Authors
	if(authors != null) {
            Enumeration e = authors.elements();
	    while(e.hasMoreElements()) {
                Author author = (Author)e.nextElement();
    	        buf.append(author.toXMLString());
	    }
    	}
	// create XML of the Appearances
	if(getAppearances() != null) {
            Enumeration e = getAppearances().elements();
	    while(e.hasMoreElements()) {
                Appearance appearance = (Appearance)e.nextElement();
    	        buf.append(appearance.toXMLString());
	    }
    	}
	return buf.toString();
    }

    /**
     * Gets the omitted citation title for creating persistent id
     * @return String the omitted citation title
     */
    public String getOmittedCitationTitle()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getOmittedCitationTitle();

	String citationTitle = getCitationTitle();
	if(citationTitle == null || citationTitle.length() == 0)
	    return citationTitle;
	
	StringTokenizer st = new StringTokenizer(citationTitle, " \t,.");
	String s = "";
	String word = "";
	while (st.hasMoreTokens()) {
	    String token = st.nextToken();
	    boolean flag = false;
	    int indexOfDash = token.indexOf('\'');
	    for (int i = 0; i<articleAndPrepositionList.length; i++) {
		if (articleAndPrepositionList[i].equals(token)) {
		    flag = true;
		    break;
		}
		if (indexOfDash != -1 && articleAndPrepositionList[i].equals(token.substring(0, indexOfDash + 1))) {
		    token = token.substring(indexOfDash + 1, token.length());
		    break;
		}
	    }
	    if (!flag) {
		word = s;
		s += token.charAt(0);
	    }
	}
	if (s.length() == 1) {
	    return word;
	}
	return s.toLowerCase();
    }

    public String getViewName()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getViewName();

	StringBuffer buffer = new StringBuffer();
	getViewName(buffer);
	return buffer.toString();
    }

    public boolean getViewName(StringBuffer buffer)
    {
	if(entity != null)
	    return ((Publication)getEntity()).getViewName(buffer);

	boolean appended = getAuthorsViewName(buffer);

	/*
	String viewYear = null;
	if(year != null && year.length() != 0)
	    viewYear = year;
	else {
	    viewYear = getPersistentIDComponent(YEAR);
	}
	*/
	String viewYear = getYear();

	if(viewYear != null && viewYear.length() != 0) {
	    if(appended)
		buffer.append(' ');
	    buffer.append(viewYear);
	    appended = true;
	}

	return appended;
    }

    public String getAuthorsViewName()
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAuthorsViewName();
	StringBuffer buffer = new StringBuffer();
	getAuthorsViewName(buffer);
	return buffer.toString();
    }

    public boolean getAuthorsViewName(StringBuffer buffer)
    {
	if(entity != null)
	    return ((Publication)getEntity()).getAuthorsViewName(buffer);

	if(isNominal()) {
	    buffer.append(getPersistentIDComponent(AUTHOR));
	    return true;
	}

	boolean appended = false;

	if(authors != null &&
	   !authors.isEmpty()) {
	    switch(authors.size()) {
	    case 1:
		((Author)authors.elementAt(0)).getAuthorViewName(buffer);
		break;
	    case 2:
		((Author)authors.elementAt(0)).getAuthorViewName(buffer);
		buffer.append(" & ");
		((Author)authors.elementAt(1)).getAuthorViewName(buffer);
		break;
	    default:
		((Author)authors.elementAt(0)).getAuthorViewName(buffer);
		buffer.append(" et al.");
		break;
	    }
	    appended = true;
	}
	else if(authorNames != null && authorNames.length() != 0) {
	    buffer.append(authorNames);
	    appended = true;
	}

	return appended;
    }

    /**
     * Sets a <CODE>object</CODE> as the entity of the name
     *
     * @param object representing the entity
     */
    public void setEntity(Object object)
    {
	if(object == this || 
	   (object != null && !(object instanceof Publication)))
	    throw new IllegalArgumentException(object.getClass().getName() + " can't be an entity of " + getClass().getName());
	entity = (Publication)object;
	super.setEntity(object);
    }
}
