/*
 * Rank.java:  a Java implementation of Rank class
 * for TaxoNote, an user interface model for Nomencurator
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Rank.java,v 1.28 2002/10/01 08:00:21 nozomi Exp $
 * $Log: Rank.java,v $
 * Revision 1.28  2002/10/01 08:00:21  nozomi
 * add specimen pseudo rank
 *
 * Revision 1.27  2002/08/29 12:42:07  nozomi
 * tentative fix or notho- ranks handling
 *
 * Revision 1.26  2002/08/27 00:55:00  nozomi
 * fix insertion of rank to sorted Vector
 *
 * Revision 1.25  2002/08/24 06:42:43  nozomi
 * sorted Ranks names are available in an array
 *
 * Revision 1.24  2002/08/24 06:23:10  nozomi
 * sorted Ranks are available in final Vector or an array
 *
 * Revision 1.23  2002/08/19 21:40:45  nozomi
 * support partially-linked ranks
 *
 * Revision 1.22  2002/08/13 02:33:52  ryo
 * add userAddedRanks and its accessor
 *
 * Revision 1.21  2002/08/09 05:24:14  nozomi
 * fix getNonOptionalHigher/LowerRank bug
 *
 * Revision 1.20  2002/07/31 06:57:42  nozomi
 * accept orphan rank
 *
 * Revision 1.19  2002/07/02 08:21:50  nozomi
 * takes care of null rank name
 *
 * Revision 1.18  2002/06/06 09:06:46  nozomi
 * handle equivalent
 *
 * Revision 1.17  2002/06/06 03:56:55  nozomi
 * acccepts non-linked rank
 *
 * Revision 1.16  2002/05/29 02:57:42  nozomi
 * fix isOptional() determination in getNonOptionalHigher/LowerRank()
 *
 * Revision 1.15  2002/05/29 02:50:45  nozomi
 * add isOptional() to determine terminal rank in getNonOptionalHigher/LowerRank()
 *
 * Revision 1.14  2002/05/27 17:24:57  nozomi
 * simplify if clause
 *
 * Revision 1.13  2002/05/27 17:15:57  nozomi
 * fix rank hop
 *
 * Revision 1.12  2002/05/27 04:44:40  nozomi
 * fix infinty loop bug of optional rank handling
 *
 * Revision 1.11  2002/05/14 05:01:49  nozomi
 * improve optional ranks support
 *
 * Revision 1.10  2002/03/28 08:14:49  nozomi
 * RankCollator abandone
 *
 * Revision 1.9  2002/03/27 05:06:51  nozomi
 * use toLowerCase() to put/get by rank name
 *
 * Revision 1.8  2002/03/26 03:56:38  nozomi
 * use RankChoice to handle additional ranks
 *
 * Revision 1.7  2002/03/09 23:32:37  nozomi
 * extends Observable
 *
 * Revision 1.6  2002/03/03 23:46:33  nozomi
 * add empty Rank
 *
 * Revision 1.5  2002/02/27 23:45:35  nozomi
 * add equals()
 *
 * Revision 1.4  2002/02/23 21:51:04  nozomi
 * Add optional rank status
 *
 * Revision 1.3  2002/02/16 21:54:00  nozomi
 * Rank names capitalised, static ranks are taken from Hashtable
 *
 * Revision 1.2  2002/02/07 08:51:44  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

/**
 * Rank provides rank tracking without parsing name of rank
 *
 * @version 	01 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class Rank
    extends Observable
{
    /** Name of the rank */
    protected String name;
    
    /** Pointer to higher Rank which may be null */ 
    protected Rank higher;

    /** Pointer to lower Rank which may be null */
    protected Rank lower;

    /** Pointer to equivalent Rank provieds rank synonymy */
    protected Rank equivalent;

    /** boolean indicating whether it is optional rank */
    boolean optional;

    /**
     * Name of full ranks, i.e. Rank has non-null higher and 
     * lower value.
     */
    static String fullyLinkedNames[] = {
	"domain", "kingdom", "phylum", "class", "order", "family",
	"tribe", "genus", "section", "series", "species", 
	"variety", "form"
    };

    /**
     * sysnonymys of ranks
     */
    static String equivalents[][] = {
	{"division", "phylum"},
	{"nothogenus", "genus"},
	{"nothospecies", "species"}
    };

    /**
     * sysnonymys of ranks
     */
    static String partiallyLinkedHigherRankName[][] = {
	{"subclass", "class"},
	{"suborder", "order"},
	{"subfamily", "family"}
    };

    /**
     * Sysnonymys of ranks
     */
    static String partiallyLinkedLowerRankName[][] = {
	{"superclass", "class"},
    };

    /**
     * <code>Hashtable</code> to hold <code>Rank</code>s
     */
    static Hashtable ranks = new Hashtable();

    /**
     * <code>Vector</code> to hold <code>Rank</code>s
     */
    static Vector sortedRanks = new Vector();

    /**
     * <code>Vector</code> to hold the user added <code>Rank</code>s
     */
    static Vector userAddedRanks = new Vector();

    static RankObserver rankObserver = new RankObserver();

    static {
	/*
	 * Put deafult <code>Rank</code>s to the <code>Hashtable</code>
	 * with making links to upper and lower <code>Rank</code>s
	 */
	int rankIndex = 0;
	Rank currentRank = new Rank(fullyLinkedNames[rankIndex]);
	for(rankIndex++; rankIndex < fullyLinkedNames.length; rankIndex++) {
	    ranks.put(currentRank.getName(), currentRank);
	    sortedRanks.add(currentRank);
	    Rank nextRank = new Rank(fullyLinkedNames[rankIndex]);
	    currentRank.setLower(nextRank);
	    nextRank.setHigher(currentRank);
	    currentRank = nextRank;
	    nextRank = null;
	}
	ranks.put(currentRank.getName(), currentRank);
	sortedRanks.add(currentRank);

	/*
	 * Create and put partially linked <code>Rank</code>s to the
	 * <code>Hashtable</code>
	 */
	for(rankIndex = 0; rankIndex < partiallyLinkedHigherRankName.length; rankIndex++) {
	    currentRank = new Rank(partiallyLinkedHigherRankName[rankIndex][0], null, null);
	    Rank higherRank = get(partiallyLinkedHigherRankName[rankIndex][1]);
	    currentRank.setHigher(higherRank);
	    currentRank.setOptional(true);
	    /*
	    ranks.put(currentRank.getName(), currentRank);
	    sortedRanks.insertElementAt(currentRank, sortedRanks.indexOf(higherRank) + 1);
	    */
	    put(currentRank);
	}
	for(rankIndex = 0; rankIndex < partiallyLinkedLowerRankName.length; rankIndex++) {
	    currentRank = new Rank(partiallyLinkedLowerRankName[rankIndex][0], null, null);
	    Rank lowerRank = get(partiallyLinkedLowerRankName[rankIndex][1]);
	    currentRank.setLower(lowerRank);
	    currentRank.setOptional(true);
	    /*
	    ranks.put(currentRank.getName(), currentRank);
	    sortedRanks.insertElementAt(currentRank, sortedRanks.indexOf(lowerRank));
	    */
	    put(currentRank);
	}

	/*
	 * Create and put equivalent <code>Rank</code>s to the
	 * <code>Hashtable</code>
	 */
	for(rankIndex = 0; rankIndex < equivalents.length; rankIndex++) {
	    currentRank = new Rank(equivalents[rankIndex][0], 
				   get(equivalents[rankIndex][1]));
	    ranks.put(currentRank.getName(), currentRank);
	    sortedRanks.insertElementAt(currentRank, sortedRanks.indexOf(currentRank.getEquivalent()) + 1);
	    //    put(currentRank);
	}

	// pseudo rank for specimen
	currentRank = new Rank("specimen");
	ranks.put("specimen", currentRank);
	sortedRanks.addElement(currentRank);
	
	//anonymouse rank
	currentRank = new Rank("");
	ranks.put("", currentRank);
	
	get("tribe").setOptional(true);
	get("section").setOptional(true);
	get("series").setOptional(true);

	currentRank = null;

    }

    public static final Rank DOMAIN  = (Rank)ranks.get("domain");
    public static final Rank KINGDOM = (Rank)ranks.get("kingdom");
    public static final Rank PHYLUM  = (Rank)ranks.get("phylum");
    public static final Rank CLASS   = (Rank)ranks.get("class");
    public static final Rank ORDER   = (Rank)ranks.get("order");
    public static final Rank FAMILY  = (Rank)ranks.get("family");
    public static final Rank TRIBE   = (Rank)ranks.get("tribe"); 
    public static final Rank GENUS   = (Rank)ranks.get("genus");
    public static final Rank SECTION = (Rank)ranks.get("section");
    public static final Rank SERIES  = (Rank)ranks.get("series");
    public static final Rank SPECIES = (Rank)ranks.get("species"); 
    public static final Rank VARIETY = (Rank)ranks.get("variety"); 
    public static final Rank FORM    = (Rank)ranks.get("form");
    public static final Rank SPECIMEN = (Rank)ranks.get("specimen");


    /**
     * Constructs a species <code>Rank</code> as default.
     */
    public Rank()
    {
	this("species");
    }

    /**
     * Constructs a <code>Rank</code> with given <code>name</code>
     * pointing itself as both higher and lower <code>Rank</code>
     *
     * @param name name of the rank
     */
    public Rank(String name)
    {
	this(name, false);
    }

    /**
     * Constructs a <code>Rank</code> with given <code>name</code>
     * pointing itself as both higher and lower <code>Rank</code>
     *
     * @param name name of the rank
     */
    public Rank(String name, boolean autoLink)
    {
	this(name, null, null);

	Rank higher = null;
	Rank lower  = null;

	if(!autoLink) {
	    higher = this;
	    lower  = this;
	}
	else {
	    if(name.startsWith("super")) {
		lower = get(name.substring(5));
		if(lower != null)
		    put(this);
	    }
	    else if (name.startsWith("sub")) {
		higher = get(name.substring(3));
		if(higher != null)
		    put(this);
	    }
	    else if(name.startsWith("infra")) {
		String baseRankName = name.substring(5);
		higher = get(baseRankName);
		if(higher != null) {
		    String nextRankName = new String("sub" + baseRankName);
		    higher = get(nextRankName);
		    if(higher == null) {
			higher = get(baseRankName);
			Rank subRank = new Rank(nextRankName, null, null);
			subRank.setHigher(higher);
			put(subRank);
			higher = subRank;
		    }
		    put(this);
		}
	    }
	}
	setHigher(higher);
	setLower(lower);
    }

    /**
     * Construts a <code>Rank</code> with given <code>name</code>
     * pointing to an <code>equivalent</code> <code>Rank</code>
     *
     * @param name name of the rank
     * @param equivalent equivalent <code>Rank</code>
     */
    public Rank(String name, Rank equivalent)
    {
	this(name, null, null, equivalent);
    }

    /**
     * Construts a <code>Rank</code> with given <code>name</code>
     * pointing to <code>higher</code> and <code>lower</code>
     * <code>Rank</code>s
     *
     * @param name name of the rank
     * @param higher higher <code>Rank</code>
     * @param lower lower <code>Rank</code>
     */
    public Rank(String name, Rank higher, Rank lower)
    {
	this(name, higher, lower, null);
    }

    /**
     * Construts a <code>Rank</code> with given <code>name</code>
     * pointing to <code>higher</code>, <code>lower</code> and
     * <code>equivalent</code> <code>Rank</code>s
     *
     * @param name name of the rank
     * @param higher higher <code>Rank</code>
     * @param lower lower <code>Rank</code>
     * @param equivalent equivalent <code>Rank</code>
     */
    public Rank(String name, Rank higher, Rank lower, Rank equivalent)
    {
	setEquivalent(equivalent);
	setHigher(higher);
	setLower(lower);
	setName(name);
    }

    /**
     * Replaces a <code>Rank</code> in the <code>Hashtable</code>
     * with given <code>rank</code>
     *
     * @param rank to be replaced
     *
     */
    public static void put(Rank rank)
    {
	if(rank ==  null)
	    return;

	if(ranks.get(rank.getName()) != null)
	    return;

	ranks.put(rank.getName().toLowerCase(), rank);

	Rank next = rank.getEquivalent();
	if(next == null) {
	    next = rank.getHigher();
	    if(next != null) {
		sortedRanks.insertElementAt(rank, sortedRanks.indexOf(next) + 1);
	    }
	    else {
		next = rank.getLower();
		if(next != null) {
		    sortedRanks.insertElementAt(rank, sortedRanks.indexOf(next));
		}
		else {
		    next = rank.getHigher();
		    if(next != null) {
			sortedRanks.insertElementAt(rank, sortedRanks.indexOf(next) + 1);
		    }
		    else {
			sortedRanks.insertElementAt(rank, sortedRanks.size());
		    }
		}
	    }
	}
	else {
	    sortedRanks.insertElementAt(rank, sortedRanks.indexOf(next) + 1);
	}
	rank.addObserver(rankObserver);
	rank.setChanged();
	rank.notifyObservers(getRankNameArray());
	userAddedRanks.addElement(rank);
    }

    /**
     * Removes <code>rank</code> from the <code>Hashtable</code>
     *
     * @param rank to be removed
     *
     */
    public static void remove(Rank rank)
    {
	if(rank == null)
	    return;

	sortedRanks.removeElement(rank);

	ranks.remove(rank.getName());
    }

    /**
     * Removes <code>Rank</code> having <code>rankName</code> from the <code>Hashtable</code>
     *
     * @param rankName name of <code>Rank</code> to be removed
     *
     */
    public static void remove(String rankName)
    {
	remove(get(rankName));
    }

    /**
     * Returns a <code>Rank</code> of given <code>name</code>
     * in the <code>Hashtable</code>
     *
     * @return a <code>Rank</code> of given <code>name</code> or null
     * if no <code>Rank</code> in the <code>Hashtable</code> has the <code>name</code>.
     */
    public static Rank get(String name)
    {
	Rank rank = null;
	if(name != null) {
	    rank = (Rank)(ranks.get(name));
	    if(rank == null)
		rank = (Rank)(ranks.get(name.toLowerCase()));
	}
	return rank;
    }

    /**
	 * Get the user added <code>Rank</code>s
     */
    public static Vector getUserAddedRanks()
    {
    	return userAddedRanks;
    }

    /**
     * Sets name to be given <code>name</code>
     * and replace <code>Rank</code> in the <code>Hashtable</code> 
     * keyed by the name to this
     *
     * @param name to be set
     */
    public void setName(String name)
    {
	if(name != null && name.equals(this.name))
	    return;

	this.name = name;

	setChanged();
	notifyObservers(this.name);
    }

    /**
     * Returns the name of the <code>Rank</code>
     *
     * @return name <code>String</code> of the <code>Rank</code>
     */
    public String getName()
    {
	return name;
    }

    public boolean isOptional()
    {
	return optional;
    }

    public void setOptional(boolean state)
    {
	optional = state;
    }

    /**
     * Sets higher rank to be given <code>higher</code> <code>Rank</code>.
     *
     * @param higher <code>Rank</code>
     */
    public void setHigher(Rank higher)
    {
	this.higher = higher;
    }

    /**
     * Returns higher rank of this <code>Rank</code>.
     *
     * @return higher <code>Rank</code>, of null if no
     * <code>Rank</code> is pointed as higher <code>Rank</code>
     */
    public Rank getHigher()
    {
	return higher;
    }

    /**
     * Sets lower rank to be given <code>lower</code> <code>Rank</code>.
     *
     * @param lower <code>Rank</code>
     */
    public void setLower(Rank lower)
    {
	this.lower = lower;
    }

    /**
     * Returns lower rank of this <code>Rank</code>.
     *
     * @return lower <code>Rank</code>, of null if no
     * <code>Rank</code> is pointed as lower <code>Rank</code>
     */
    public Rank getLower()
    {
	return lower;
    }

    /**
     * Sets equivalent rank to be given <code>equivalent</code> <code>Rank</code>.
     *
     * @param equivalent <code>Rank</code>
     */
    public void setEquivalent(Rank equivalent)
    {
	this.equivalent = equivalent;
    }

    /**
     * Returns a <code>Rank</code> equivalent to this <code>Rank</code>.
     *
     * @return <code>Rank</code> equivalent to this, or null if no
     * <code>Rank</code> is equivalent to this
     */
    public Rank getEquivalent()
    {
	return equivalent;
    }

    /**
     * Returns whether this is fully linked to both higher and lower <code>Rank</code>s.
     * It also tries its equivalent <code>Rank</code> if given.
     *
     * @return true if both higher and lower <code>Rank</code> are non null,
     * or false if either is null
     */
    public boolean isFullyLinkedRank()
    {
	if(higher != null && lower != null)
	    return true;

	if(equivalent == null)
	    return false;

	return equivalent.isFullyLinkedRank();
    }

    /**
     * Returns nearest fully linked <code>Rank</code>
     *
     * @return nearest fully linked <code>Rank</code>
     */
    public Rank getFullyLinkedRank()
    {
	if(isFullyLinkedRank())
	    return this;

	if(equivalent != null)
	    return equivalent.getFullyLinkedRank();

	if(higher != null) 
	    return getHigher().getFullyLinkedRank();
	else if(lower != null)
	    return getLower().getFullyLinkedRank();
	else
	    return this; //or null ?
    }

    /**
     * Returns nearest fully linked <code>Rank</code>
     *
     * @return nearest fully linked <code>Rank</code>
     */
    public Rank getNonOptionalHigherRank()
    {
	if(isHighest() && !isOptional())
	    return this;
	Rank target = this;
	Rank higher = null;
	do {
	    higher = target.getHigher();
	    if(higher == null)
		higher =  target.getLower().getHigher();
	    if(higher.isHighest() && !higher.isOptional())
		return higher;
	    if(higher == null ||
	       (higher == target && target.isOptional()))
		higher = target.getLower().getFullyLinkedRank().getHigher();
	    target = higher.getFullyLinkedRank();
	} while(target.isOptional() && !target.isHighest());
	
	return target;
    }

    /**
     * Returns nearest fully linked <code>Rank</code>
     *
     * @return nearest fully linked <code>Rank</code>
     */
    public Rank getNonOptionalLowerRank()
    {
	if(isLowest() && !isOptional())
	    return this;
	Rank target = this;
	Rank lower = null;
	do {
	    lower = target.getLower();
	    if(lower == null)
		lower = target.getHigher().getLower();
	    if(lower.isLowest() && !lower.isOptional()) 
		return lower;
	    if(lower == null ||
	       (lower == target && target.isOptional())) {
		lower = target.getHigher().getFullyLinkedRank().getLower();
	    }
	    target = lower.getFullyLinkedRank();
	} while(target.isOptional() && !target.isLowest());
	
	return target;
    }

    /**
     * Returns whether this is the highest rank
     *
     * @return true if this is the highest rank
     */
    public boolean isHighest()
    {
	return (higher == this);
    }

    /**
     * Returns whether this is the lowest rank
     *
     * @return true if this is the lowest rank
     */
    public boolean isLowest()
    {
	return (lower == this);
    }

    /**
     * Returns whether this is higher than <code>rank</code>
     *
     * @param rank name of <code>Rank</code> to be compared
     *
     * @return true if this is higher than given <code>rank</code>
     */
    public boolean isHigher(String rank)
    {
	return isHigher(get(rank));
    }

    /**
     * Returns whether this is higher than given <code>rank</code>
     *
     * @param rank to be compared
     *
     * @return true if this is higher than given <code>rank</code>
     */
    public boolean isHigher(Rank rank)
    {
	if(rank == this)
	    return false;

	Rank[]  path1 = getRankPath(this.getFullyLinkedRank());
	Rank[]  path2 = getRankPath(rank.getFullyLinkedRank());

	if(path1.length < path2.length)
	    return true;
	else if (path1.length > path2.length)
	    return false;

	path1 = getRankPath(this);
	path2 = getRankPath(rank);

	if(path1.length < path2.length)
	    return true;
	else if (path1.length > path2.length)
	    return false;

	Rank r1 = getFullyLinkedRank();
	Rank r2 = rank.getFullyLinkedRank();
	if(r1 == r2.getHigher())
	    return true; 
	else if (r2 == r1.getHigher())
	    return false;

	return false; //is it right?
    }


    /**
     * Returns whether this is lower than given <code>rank</code>
     *
     * @param rank name to be compared
     *
     * @return true if this is lower than given <code>rank</code>
     */
    public boolean isLower(String rank)
    {
	return isLower(get(rank));
    }

    /**
     * Returns whether this is lower than given <code>rank</code>
     *
     * @param rank to be compared
     *
     * @return true if this is lower than given <code>rank</code>
     */
    public boolean isLower(Rank rank)
    {
	if(rank == this)
	    return false;

	Rank[]  path1 = getRankPath(this.getFullyLinkedRank());
	Rank[]  path2 = getRankPath(rank.getFullyLinkedRank());

	if(path1.length > path2.length)
	    return true;
	else if (path1.length < path2.length)
	    return false;

	path1 = getRankPath(this);
	path2 = getRankPath(rank);

	if(path1.length > path2.length)
	    return true;
	else if (path1.length < path2.length)
	    return false;

	Rank r1 = getFullyLinkedRank();
	Rank r2 = rank.getFullyLinkedRank();
	if(r1 == r2.getHigher())
	    return true; 
	else if (r2 == r1.getHigher())
	    return false;

	return false; //?
    }

    /**
     * Returns descending path from highest rank to
     * given <code>rank</code>
     *
     * @param rank to be tracked from top rank
     *
     * @return an Array of <code>Rank</code> indicating path from top to
     * given <code>rank</code>
     */
    public static Rank[] getRankPath(Rank rank)
    {
	if(rank == null)
	    return null;

	Vector ascendingPath = new Vector();

	if(!rank.isFullyLinkedRank()) {
	    if(rank.getHigher() == null) {
		ascendingPath.addElement(rank);
		rank = rank.getFullyLinkedRank().getHigher();
	    }
	    else {
		while(!rank.isFullyLinkedRank()) {
		    ascendingPath.addElement(rank);
		    rank = rank.getHigher();
		}
	    }
	}
	ascendingPath.addElement(rank);
	while(rank != null && !rank.isHighest()) {
	    rank = rank.getHigher();
	    ascendingPath.addElement(rank);
	}

	int i = ascendingPath.size();

	Rank[] rankArray = new Rank[i];

	int j = 0;
	while(i > 0) {
	    rankArray[j++] = (Rank)ascendingPath.elementAt(--i);
	}

	return rankArray;
    }

    /**
     * Returns true if <code>object</code> equals to this object
     *
     * @param object <code>Object</code> to be compared with this object
     *
     * @return true if <code>object</code> and this are the same object,
     * or <code>object</code> and this have the same name; false if
     * <code>objct</code> is not instance of <code>Rank</code>, or any conditions 
     * other than above enumerated
     */
    public boolean equals(Object object)
    {
	if(object == this)
	    return true;
	if(!(object instanceof Rank))
	    return false;
	Rank rank = (Rank)object;
	if(name.equals(rank.getName()))
	   return true;

	//here we need equivalent evaluation.....
	return false;

    }

    /**
     * Returnes an array of rank names appropriate to <code>TextListModel</code>.
     * 
     * @return Array of <code>String</code> representing rank names appropriate to <code>TextListModel</code>.
     */
    public static final String[][] getRankNameArray()
    {
	Enumeration nameEnum = sortedRanks.elements();
	String names[][] = new String[sortedRanks.size()][1];
	int index = 0;
	while(nameEnum.hasMoreElements()) {
	    names[index++] = new String[]{((Rank)(nameEnum.nextElement())).getName()};
	}
	return names;
    }

    /**
     * Returns <code>String</code> representing rank name
     *
     * @return String representing rank name
     */
    public String toString()
    {
	return getName();
    }

    /**
     * Adds <code>observer</code> to <code>Observer</code> list
     * 
     * @param observer <code>Observer</code> to be added to the list
     */
    public void addObserver(Observer observer)
    {
	rankObserver.addObserver(observer);
    }

    /**
     * Removes <code>observer</code> from <code>Observer</code> list
     * 
     * @param observer <code>Observer</code> to be removed from the list
     */
    public void deleteObserver(Observer observer)
    {
	rankObserver.deleteObserver(observer);
    }

    /**
     * Adds <code>observer</code> to <code>Observer</code> list
     * observing this
     * 
     * @param observer <code>Observer</code> to be added to the list
     */
    public void addObserver(RankObserver observer)
    {
	super.addObserver(observer);
    }

    /**
     * Removes <code>observer</code> from <code>Observer</code> list
     * observing this
     * 
     * @param observer <code>Observer</code> to be removed from the list
     */
    public void deleteObserver(RankObserver observer)
    {
	super.deleteObserver(observer);
    }

    /**
     * Returns highest <code>Rank</code>
     *
     * @return <code>Rank</code> at top of rank hierarchy
     */
    public static Rank getHighest()
    {
	return (Rank)sortedRanks.elementAt(0);
    }

    /**
     * Returns a Vector of sorted <CODE>Rank</CODE>s
     *
     * @return Vector of sorted <CODE>Rank</CODE>s
     */
    public static final Vector getSortedRanks()
    {
	return sortedRanks;
    }

    /**
     * Returns an array of sorted <CODE>Rank</CODE>s
     *
     * @return array of sorted <CODE>Rank</CODE>s
     */
    public static Rank[] getSortedArray()
    {
	int ranks = sortedRanks.size();
	Rank[] sorted = new Rank[ranks];
	System.arraycopy(sortedRanks.toArray(), 0, sorted, 0, ranks);
	return sorted;
    }

    /**
     * Returns an array of <CODE>String</CODE>s
     * representing sorted <CODE>Rank</CODE>s' name
     *
     * @return array of <CODE>Rank</CODE>s' name
     */
    public static String[] getRankNames()
    {
	String[] sortedRankNames = new String[sortedRanks.size()];
	int index = 0;
	Enumeration e = sortedRanks.elements();
	while(e.hasMoreElements()) {
	    sortedRankNames[index++] = 
		((Rank)(e.nextElement())).getName();
	}
	return sortedRankNames;
    }

}

class RankObserver
    extends Observable
    implements Observer
{
    public RankObserver()
    {
	super();
    }

    public void update(Observable observer, Object arg)
    {
	setChanged();
	notifyObservers(arg);
    }

}
