/*
 * DateEntry.java:  a Java implementation of a fielded date entry component
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 2000, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: DateEntry.java,v 1.2 2002/04/01 08:16:28 nozomi Exp $
 * $Log: DateEntry.java,v $
 * Revision 1.2  2002/04/01 08:16:28  nozomi
 * use java.util.Vector
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.awt;

import java.awt.Component;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.SimpleDateFormat;

import java.util.Date;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.Vector;

import jp.kyasu.awt.AWTResources;
import jp.kyasu.awt.Choice;
import jp.kyasu.awt.EventPoster;
import jp.kyasu.awt.TextArea;
import jp.kyasu.awt.TextField;

import org.nomencurator.awt.event.DateEvent;
import org.nomencurator.awt.event.DateListener;

public class DateEntry extends Object implements ItemListener, TextListener {

    private String [] _monthNames = null;
    private String separator;
    
    private int [] ymdIndex = null;
    
    private static int _lastY = 0;
    private static int _lastM = 0;
    private static int _lastD = 0;
    
    private int _year = 0;
    private int _month = 1;
    private int _day = 0;
    
    protected TextField yearField;
    protected Choice monthChoice;
    protected TextField dayField;

    protected boolean validDate;
    protected boolean validYear;
    protected boolean validDay;


    Component [] ymd = null;
    char[] format = null;

    transient Vector dateListeners;

    public boolean isValid()
    {
	return validDate;
    }
    
    public int getYear()
    {
	return _year;
    }

    public void setYear(int year)
    {
	yearField.setText(String.valueOf(_year = year));
	validYear = true;
	if(validDate = validDay)
	    notifyDateListeners();
    }

    public int getMonth()
    {
	return _month;
    }

    public void setMonth(int month)
    {
	if(month > 12 || month < 1)
	    return;

	monthChoice.select((_month = month) - 1);
	if(validDate = validYear & validDay)
	    notifyDateListeners();
    }
    
    public int getDay()
    {
	return _day;
    }

    public void setDay(int day)
    {
	dayField.setText(String.valueOf(_day = day));
	validDay = true;
	if(validDate = validYear)
	    notifyDateListeners();
    }

    public Date getDate()
    {
	return new Date(_year - 1900, _month, _day);
    }

    public void setDate(Date date)
    {
	setYear(date.getYear() + 1900);
	setMonth(date.getMonth());
	setDay(date.getDate());
	validDate = validYear = validDay = true;
	notifyDateListeners();
    }

    public Calendar getCalendar()
    {
	return new GregorianCalendar(_year, _month, _day);
    }

    public void getCalendar(Calendar calendar)
    {
	calendar.set(_year, _month, _day);
    }


    public void setCalendar(Calendar calendar)
    {
	setYear(calendar.get(Calendar.YEAR));
	setMonth(calendar.get(Calendar.MONTH));
	setDay(calendar.get(Calendar.DAY_OF_MONTH));
	validDate = validYear = validDay = true;
	notifyDateListeners();
    }

    public String getSeparator()
    {
	return separator;
    }

    public String[] getMonthName()
    {
	return _monthNames;
    }
    
    DateEntry(){
	this(false);
    }
    
    DateEntry(boolean useCurrentTime){
	super();
	dateListeners = null;
	if(useCurrentTime){
	    Calendar c = Calendar.getInstance();

	    if(_lastY == 0)
		_lastY = c.get(Calendar.YEAR);
	    if(_lastM == 0)
		_lastM = c.get(Calendar.MONTH); 
	    if(_lastD == 0)
		_lastD = c.get(Calendar.DAY_OF_MONTH);
	    _year =  _lastY;
	    _month = _lastM;
	    _day =  _lastD;
	}

	setLocaleInfo();
	
	yearField = _setNumber (4, _year);
	monthChoice = new Choice();
	dayField = _setNumber (2, _day);
	
	// -1 for the 13th month?
	int monthes = _monthNames.length - 1;
	for(int i = 0; i < monthes; i++)
	    monthChoice.add(_monthNames[i]);
	monthChoice.select(_month - 1);
	
	ymd = new Component[3];
	ymd[ymdIndex[0]] = yearField;
	ymd[ymdIndex[1]] = monthChoice;
	ymd[ymdIndex[2]] = dayField;
    }
    
    private TextField _setNumber (int width, int value){
	TextField tf = new TextField(width);
	tf.addTextListener(this);
	if(value != 0)
	    tf.setText(String.valueOf(value));
	return tf;
    }
    
    public void textValueChanged(TextEvent e) {
	Object eventSource = e.getSource();
	if(eventSource == yearField) {
	    String yearString = yearField.getText();
	    if(yearString.length() != 0){
		_year = Integer.decode(yearString).intValue();
		validYear = true;
		if(validDate = validDay)
		    notifyDateListeners();
	    }
	}
	else if(eventSource == dayField) {
	    String dayString = dayField.getText();
	    if(dayString.length() != 0){
		_day = Integer.decode(dayString).intValue();
		validDay = true;
		if(validDate = validYear)
		    notifyDateListeners();
	    }
	}
    }
    
    public void itemStateChanged(ItemEvent e) {
	Object eventSource = e.getSource();
	if(eventSource == monthChoice) {
	    _month = monthChoice.getSelectedIndex() + 1;
	    notifyDateListeners();
	}
    }

    private void notifyDateListeners() 
    {
	if(dateListeners == null)
	    return;
	DateEvent event = new DateEvent(this);
	for(Enumeration e = dateListeners.elements(); e.hasMoreElements();)
	    ((DateListener)e.nextElement()).dateValueChanged(event);
    }

    public synchronized void addDateListener(DateListener l) {
	if (l == null)
	    return;
	if(dateListeners == null)
	    dateListeners = new Vector();
	dateListeners.addElement(l);
    }

    public synchronized void removeDateListener(DateListener l) {
	if(dateListeners == null)
	    return;
	dateListeners.removeElement(l);
	if(dateListeners.size() == 0)
	    dateListeners = null;
    }

    private void setLocaleInfo() {
	DateFormatSymbols dfms = new DateFormatSymbols();
	_monthNames = dfms.getShortMonths();
	ymdIndex = new int[3];
	    
	//locale depending format
	format = new char[3];
	String dateFormat;
	try{
	    dateFormat = ((SimpleDateFormat)DateFormat.getDateInstance(DateFormat.LONG)).toPattern();
	}
	catch(Throwable e){
	    try{
		SimpleDateFormat sdf = new SimpleDateFormat();
		dateFormat = sdf.toPattern();
	    }
	    catch(Throwable ee){
		dateFormat = "yMd";
	    }
	}
	
	int cnt = 0;
	format[cnt] = dateFormat.charAt(0);
	char sep[] = new char[1];
	sep[0] = '/';
	for(int i = 1; i < dateFormat.length() && cnt < 3; i++){
	    if(dateFormat.charAt(i) != 'y' && dateFormat.charAt(i) != 'M' && dateFormat.charAt(i) != 'd'){
		sep[0] = dateFormat.charAt(i);
	    }
	    else if(format[cnt] != dateFormat.charAt(i))
		format[++cnt] = dateFormat.charAt(i);
	}
	for(int i = 0; i < 3; i++){
	    if(format[i] == 'y')
		ymdIndex[0] = i;
	    else if (format[i] == 'M')
		ymdIndex[1] = i;
	    else if(format[i] == 'd')
		ymdIndex[2] = i;
	}
	separator = new String(sep);
    }

}

