/*
 * Dialog.java:  a Dialog to display components
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 2000, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Dialog.java,v 1.2 2002/03/27 02:21:50 nozomi Exp $
 * $Log: Dialog.java,v $
 * Revision 1.2  2002/03/27 02:21:50  nozomi
 * add a utility method to get Frame from given Component
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.awt;

import java.awt.BorderLayout;

import java.awt.Component;

import jp.kyasu.awt.AWTResources;
import jp.kyasu.awt.Label;
import jp.kyasu.awt.Panel;

import jp.kyasu.graphics.VImage;

import org.nomencurator.awt.Button;

import org.nomencurator.awt.util.AWTtools;

/**
 * A <code>Dialog</code> class extending <code>jp.kyasu.awt.Dialog</code>
 * with <code>Buttons</code>
 *
 * @see 	jp.kyasu.awt.Dialog
 *
 * @version 	26 Mar 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class Dialog
    extends jp.kyasu.awt.Dialog
{
    /** type of Dialog */
    protected int type;

    protected VImage icon;

    /**
     * A utility method returns <code>java.awt.Frame</code> containing
     * <code>component</code>. 
     *
     * @param component <code>Component</code> required <code>Dialog</code>
     */
    public static java.awt.Frame getFrame(Component component)
    {
	while(!(component instanceof java.awt.Frame))
	    component = component.getParent();
	return (java.awt.Frame)component;
    }

    /**
     * Constructs an initially invisible Dialog with an empty title.
     *
     * @param component the requestor of the dialog.
     */
    public Dialog(Component component)
    {
	this(getFrame(component));
    }

    /**
     * Constructs an initially invisible Dialog with an empty title.
     *
     * @param parent the owner of the dialog.
     */
    public Dialog(java.awt.Frame parent)
    {
	this(parent, "", false);
    }

    /**
     * Constructs an initially invisible Dialog with an empty title.
     * A modal Dialog grabs all the input to the parent frame from the user.
     *
     * @param component the requestor of the dialog.
     * @param modal  if true, dialog blocks input to the parent window when
     *               shown.
     */
    public Dialog(Component component, boolean modal)
    {
	this(getFrame(component), modal);
    }

    /**
     * Constructs an initially invisible Dialog with an empty title.
     * A modal Dialog grabs all the input to the parent frame from the user.
     *
     * @param parent the owner of the dialog.
     * @param modal  if true, dialog blocks input to the parent window when
     *               shown.
     */
    public Dialog(java.awt.Frame parent, boolean modal)
    {
	this(parent, "", modal);
    }

    /**
     * Constructs an initially invisible Dialog with a title.
     *
     * @param component the requestor of the dialog.
     * @param title  the title of the dialog.
     */
    public Dialog(Component component, String title)
    {
	this(getFrame(component), title);
    }

    /**
     * Constructs an initially invisible Dialog with a title.
     *
     * @param parent the owner of the dialog.
     * @param title  the title of the dialog.
     */
    public Dialog(java.awt.Frame parent, String title)
    {
	this(parent, title, false);
    }

    /**
     * Constructs an initially invisible Dialog with a title.
     * A modal Dialog grabs all the input to the parent frame from the user.
     *
     * @param component the requestor of the dialog.
     * @param title  the title of the dialog.
     * @param modal  if true, dialog blocks input to the parent window when
     *               shown.
     */
    public Dialog(Component component, String title, boolean modal)
    {
	this(getFrame(component), title, modal);
    }

    /**
     * Constructs an initially invisible Dialog with a title.
     * A modal Dialog grabs all the input to the parent frame from the user.
     *
     * @param parent the owner of the dialog.
     * @param title  the title of the dialog.
     * @param modal  if true, dialog blocks input to the parent window when
     *               shown.
     */
    public Dialog(java.awt.Frame parent, String title, boolean modal)
    {
	this(parent, title, modal, MESSAGE);
    }

    /**
     * Constructs an initially invisible Dialog with a title.
     * A modal Dialog grabs all the input to the parent frame from the user.
     *
     * @param component the requestor of the dialog.
     * @param title  the title of the dialog.
     * @param modal  if true, dialog blocks input to the parent window when
     *               shown.
     */
    public Dialog(Component component, String title, boolean modal, int type)
    {
	this(getFrame(component), title, modal, type);
    }

    /**
     * Constructs an initially invisible Dialog with a title.
     * A modal Dialog grabs all the input to the parent frame from the user.
     *
     * @param parent the owner of the dialog.
     * @param title  the title of the dialog.
     * @param modal  if true, dialog blocks input to the parent window when
     *               shown.
     */
    public Dialog(java.awt.Frame parent, String title, boolean modal, int type)
    {
	super(parent, title, modal);
	this.type = type;
	if(title == null || title.length() == 0) {
	    switch(type) {
	    case INFORM:
		setTitle(AWTResources.getResourceString("kfc.dialog.informLabel", "Inform"));
		break;
	    case WARN:
		setTitle(AWTResources.getResourceString("kfc.dialog.informLabel", "Warn"));
		break;
	    case ERROR:
		setTitle(AWTResources.getResourceString("kfc.dialog.informLabel", "Error"));
		break;
	    case CONFIRM:
		setTitle(AWTResources.getResourceString("kfc.dialog.informLabel", "Confirm"));
		break;
	    default:
		break;
	    }
	} 
	setLayout();
	createComponents();
	createPanels();
	addComponents();
	addPanels();
    }

    protected Panel north;
    protected Panel east;
    protected Panel west;
    protected Panel south;
    protected Panel center;

    protected Label message;

    protected Button[] buttons;

    protected void setLayout()
    {
	setLayout(new BorderLayout());
    }

    protected void createComponents()
    {
	createButtons();
    }

    protected void createButtons()
    {
	buttons = new Button[1];
	buttons[0] = new OkButton();
    }

    protected void createPanels()
    {
	south = new Panel();

	if(type != MESSAGE) {
	    switch(type) {
	    case INFORM:
		icon = getInformIcon();
		break;
	    case WARN:
		icon = getWarnIcon();
		break;
	    case ERROR:
		icon = getErrorIcon();
		break;
	    case CONFIRM:
		icon = getQuestionIcon();
		break;
	    default:
		icon = null;
		break;
	    }
	    if(icon != null && west == null) {
		west = new Panel();
		west.add(new Label(icon));
	    }

	}
    }

    protected void addComponents()
    {
	addButtons();
    }

    protected void addButtons()
    {
	if(south != null && buttons != null)
	    for(int i = 0; i < buttons.length; i++)
		south.add(buttons[i]);
    }

    protected void addPanels()
    {
	if(north != null)
	    add(north, BorderLayout.NORTH);
	if(south != null)
	    add(south, BorderLayout.SOUTH);
	if(east != null)
	    add(east, BorderLayout.EAST);
	if(west != null)
	    add(west, BorderLayout.WEST);
	if(center != null)
	    add(center, BorderLayout.CENTER);

    }

}

class DialogButton extends Button
{

    public DialogButton(String label)
    {
	super(label);
    }

    public void run()
    {
	AWTtools.getFrame(this).dispose();
    }
}

class OkButton extends DialogButton {
    public OkButton() {
	super(AWTResources.getResourceString("kfc.dialog.okLabel", "OK"));
    }
}

class YesButton extends DialogButton {
    public YesButton() {
	super(AWTResources.getResourceString("kfc.dialog.yesLabel", "Yes"));
    }
}

class NoButton extends DialogButton {
    public NoButton() {
	super(AWTResources.getResourceString("kfc.dialog.noLabel", "No"));
    }
}

