/*
 *
 * SummaryEditor.java:  an editing object with a text conmponent 
 * summarizing something should be edited.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: SummaryEditor.java,v 1.1.1.1 2002/01/16 12:33:33 ryo Exp $
 *	$Log: SummaryEditor.java,v $
 *	Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 *	initial import into CVS
 *	
 *	
 *
 */

package org.nomencurator.awt;

import java.awt.AWTEventMulticaster;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.TextListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputMethodEvent;
import java.awt.event.InputMethodListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.beans.PropertyChangeListener;

import jp.kyasu.awt.KComponent;
import jp.kyasu.awt.KContainer;
import jp.kyasu.awt.Label;
import jp.kyasu.awt.Panel;

import org.nomencurator.awt.Button;
import org.nomencurator.awt.SummaryButton;

/**
 * The <code>SummaryEditor</code> class provides an editing object with
 * a summarizin component.
 * <p>
 * Editing actions can be bound to <code>Button</code> of <code>SummaryEditor</code> 
 * object by  * <code>ActionListener</code>.
 *
 * @see 	java.awt.Button
 * @see 	java.awt.ActionListener
 * @see 	jp.kyasu.awt.Kcomponent
 *
 * @version 	09 Jan 2000
 * @author 	Nozomi `James' Ytow
 */

//public abstract class SummaryEditor extends KComponent /*Object*/
public abstract class SummaryEditor extends Component
    implements ComponentListener, FocusListener,
	       //InputMethodListener, 
	       KeyListener, MouseListener, 
	       MouseMotionListener
	    //    implements ActionListener 
{

    protected Label label;
    protected Component component;
    protected SummaryButton[] buttons;
    //    protected ActionListener listeners;

    /*
    protected ComponentListener   componentListener;
    protected FocusListener       focusListener;
//    protected InputMethodListener inputMethodListener;
    protected KeyListener         keyListener;
    protected MouseListener       mouseListener;
    protected MouseMotionListener mouseMotionListener;
    */

    /**
     * Constructs a summary editor with specified label,
     * component and buttons.
     * @param label     A string label for the summary
     * @param component A component for the summary
     * @param buttons    A array of buttons to evoke editing action
     */
    public SummaryEditor(Label label,
			 Component component, 
			 SummaryButton[] buttons)
    {
	this(label, component, buttons, null);
    }

    /**
     * Constructs a summary editor with specified label,
     * component and buttons.
     * @param label     A string label for the summary
     * @param component A component for the summary
     * @param buttons    A array of buttons to evoke editing action
     * @param actionListener    An array of ActionListeners for the buttons
     */
    public SummaryEditor(Label label,
			 Component component, 
			 SummaryButton[] buttons, ActionListener[] actionListener)
    {
	super();

	//	listeners = null;
	/*
	componentListener = null;
	focusListener = null;
	inputMethodListener = null;
	keyListener = null;
	mouseListener = null;
	mouseMotionListener = null;
	*/

	this.label = label;
	
	this.component = component;
	if(this.component != null)
	    addListeners();
	
	if(buttons == null || buttons.length == 0)
	    createButtons();
	else 
	    this.buttons = buttons;
	if(this.buttons != null)
	    for(int i = 0; i < this.buttons.length; i++) {
		//		    this.buttons[i].addActionListener(this);
		if(this.buttons[i] != null)
		    this.buttons[i].setSummaryEditor(this);
	    }
	if(this.buttons != null &&
	   actionListener != null) {
		int a = 0;
		int b = 0;
		for(; 
		    a < actionListener.length && b < this.buttons.length;
		    a++, b++) {
		    this.buttons[b].addActionListener(actionListener[a]);
		}
	}
    }

    protected void addListeners()
    {
	component.addComponentListener(this);
	component.addFocusListener(this);
	//	component.addInputMethodListener(this);
	component.addKeyListener(this);
	component.addMouseListener(this);
	component.addMouseMotionListener(this);
    }


    //    protected abstract void createButtons();
    protected void createButtons()
    {
    }

    public Label getLabel()
    {
	return label;
    }

    public Component getComponent()
    {
	return component;
    }

    public Button[] getButtons()
    {
	if(buttons == null)
	    return (Button[])null;
	Button[] simpleButtons = new Button[buttons.length];
	for(int i = 0; i < buttons.length; i++) {
	    simpleButtons[i] = (Button)(buttons[i]);
	}
	return simpleButtons;
    }

    public Button getButton()
    {
	if(buttons != null)
	    return (Button)(buttons[0]);
	else
	    return (Button) null;
    }

    /**
     * Returns the preferred size of this summary editor.
     * @return the preferred dimensions for displaying this summary editor.
     */
    public Dimension getPreferredSize() {
	return getPreferredSize(component.getPreferredSize());
    }

    /**
     * Returns the preferred size of this summary editor.
     * It is used by subclasses only.
     * @param d Dimension of the component.
     */
    public Dimension getPreferredSize(Dimension d) {
	Dimension labelDimension = label.getPreferredSize();
	Dimension buttonDimension = new Dimension();
	for(int i = 0; i < buttons.length; i++) {
	    Dimension buttonDim = buttons[i].getPreferredSize();
	    buttonDimension.height += buttonDim.height;
	    if(buttonDimension.width < buttonDim.width)
		buttonDimension.width = buttonDim.width;
	}

	int width  = labelDimension.width + buttonDimension.width
	    + d.width;

	int height = labelDimension.height > buttonDimension.height?
	    labelDimension.height : buttonDimension.height;
	height = d.height > height? d.height : height;
	return new Dimension(width, height);
    }
    
    /**
     * Returns the minimum size of this summary editor.
     * @return the minimum dimension for displaying this summary editor.
     */
    public Dimension getMinimumSize() {
	return getMinimumSize(component.getMinimumSize());
    }

    /**
     * Returns the preferred size of this summary editor.
     * It is used by subclasses only.
     * @param d Dimension of the component.
     */
    public Dimension getMinimumSize(Dimension d) {
	Dimension labelDimension = label.getMinimumSize();
	Dimension buttonDimension = new Dimension();
	for(int i = 0; i < buttons.length; i++) {
	    Dimension buttonDim = buttons[i].getMinimumSize();
	    buttonDimension.height += buttonDim.height;
	    if(buttonDimension.width < buttonDim.width)
		buttonDimension.width = buttonDim.width;
	}

	int width  = labelDimension.width + buttonDimension.width
	    + d.width;
	int height = labelDimension.height > buttonDimension.height?
	    labelDimension.height : buttonDimension.height;
	height = d.height > height? d.height : height;
	return new Dimension(width, height);
    }

    /**
     * Add <code>SummaryEditor</code> object to a <code>Container</code>
     * with given <code>GridBagLayout</code> under given <code>GridBagConstraints</code>.
     * @param container        A container to which the <code>SummaryEditor</code> will be added
     * @param layout           A layout
     * @param constraints      A contraints to add the <code>SummaryEditor</code> object
     */
    public void addTo(Container container, GridBagLayout layout, GridBagConstraints constraints) {
	addTo(container, 1, layout, constraints);
    }
    
    /**
     * Add <code>SummaryEditor</code> object to a <code>Container</code>
     * with given <code>GridBagLayout</code> under given <code>GridBagConstraints</code>.
     * @param container        A container to which the <code>SummaryEditor</code> will be added
     * @param componentColumns Number of columns occupied by the component
     * @param layout           A layout
     * @param constraints      A contraints to add the <code>SummaryEditor</code> object
     */
    public void addTo(Container container, int componentColumns, GridBagLayout layout, GridBagConstraints constraints) {
	addTo(container, layout, constraints, componentColumns, 0.1, 0.8, 0.1);
    }
    
    /**
     * Add <code>SummaryEditor</code> object to a <code>Container</code>
     * with given <code>GridBagLayout</code> under given <code>GridBagConstraints</code>.
     * @param container        A container to which the <code>SummaryEditor</code> will be added
     * @param layout           A layout
     * @param constraints      A contraints to add the <code>SummaryEditor</code> object
     * @param componentColumns Number of columns occupied by the component
     * @param labelWeight      A weight of the label.  If negative, 0.1 assumed.
     * @param componentWeight  A weight of the component.  If negative, 1.0 - labelWeight - buttonWeight is assumed.
     * @param buttonWeight     A weight of the button.  If negative, 0.1 assumed.
     */
    public void addTo(Container container, GridBagLayout layout, GridBagConstraints constraints, 
		      int componentColumns, 
		      double labelWeight, double componentWeight, double buttonWeight) {

	//save parameters
	double weightx = constraints.weightx;
	int gridwidth = constraints.gridwidth;
	int gridx = constraints.gridx;
	int ipadx = constraints.ipadx;
	int anchor = constraints.anchor;
	int fill = constraints.fill;
	int insetsLeft = constraints.insets.left;
	int insetsRight = constraints.insets.right;

	if(labelWeight < 0.0)
	    labelWeight = 0.1d;
	if(buttonWeight < 0.0)
	    buttonWeight = 0.1d;
	if(componentWeight < 0.0)
	    componentWeight = 1.0d - labelWeight - buttonWeight;

	
	constraints.gridwidth = 1;
	if(gridx == GridBagConstraints.RELATIVE)
	    constraints.gridx = 0;
	
	constraints.insets.left = constraints.insets.right;
	constraints.insets.right /= 2;
	//	constraints.weightx = labelWeight;
	constraints.fill = GridBagConstraints.NONE;
	constraints.anchor = GridBagConstraints.EAST;
	//	layout.setConstraints(label, constraints);
	container.add(label, constraints);
	
	//	constraints.gridx = GridBagConstraints.RELATIVE;
	constraints.insets.left = insetsLeft;
	constraints.insets.right = insetsRight;
	
	//	constraints.weightx = componentWeight;
	constraints.fill = GridBagConstraints.HORIZONTAL;
	constraints.anchor = GridBagConstraints.CENTER;
	//	constraints.gridwidth = GridBagConstraints.REMAINDER;
	//	constraints.gridx = GridBagConstraints.RELATIVE;
	constraints.gridx++;
	constraints.gridwidth = componentColumns;
	constraints.gridheight = buttons.length;
	//	layout.setConstraints(component, constraints);
	container.add(component, constraints);
	constraints.gridx += componentColumns;

	constraints.gridwidth = 1;//GridBagConstraints.REMAINDER;
	constraints.fill = GridBagConstraints.HORIZONTAL;
	//constraints.fill = GridBagConstraints.BOTH;
	constraints.anchor = GridBagConstraints.WEST;
	//	constraints.ipadx = 6;
	//	constraints.gridx = GridBagConstraints.REMAINDER;
	constraints.gridheight = 1;
	if(buttons != null && buttons.length != 0) {
	    Insets insets = constraints.insets;
	    constraints.insets = (Insets)insets.clone();
	    constraints.insets.bottom = 0;
	    for(int i = 0; i < buttons.length; i++) {
		//		if(i == buttons.length - 1)
		//		    constraints.insets.bottom = insets.bottom;
		buttons[i].setSummaryEditor(this);
		//layout.setConstraints(buttons[i], constraints);
		container.add(buttons[i], constraints);
		constraints.gridy++;
		constraints.insets.top = 0;
	    }
	    constraints.insets = insets;
	    constraints.gridy--;
	}
	    
	constraints.weightx = weightx;
	
	//restore parameters
	constraints.fill = fill;
	constraints.anchor = anchor;
	constraints.ipadx = ipadx;
	constraints.gridx = gridx;
	constraints.gridwidth = gridwidth;
	constraints.weightx = weightx;
	
	/*
	  but a side effect...
	*/
	constraints.gridy++;
    }

    protected abstract void remove(int index);

    protected abstract void deselect(int index);
    
    protected abstract int[] getSelectedIndexes();

    public void setEnabled(boolean enabled)
    {
	if(label != null)
	    label.setEnabled(enabled);
	if(component != null)
	    component.setEnabled(enabled);
	setSelectionEnabled(enabled, 0);
    }

    public void setSelectionEnabled(boolean enabled)
    {
	setSelectionEnabled(enabled, 1);
    }

    protected void setSelectionEnabled(boolean enabled, int start)
    {
	if(buttons != null)
	    for(int i = start; i < buttons.length; i++)
		buttons[i].setEnabled(enabled);
    }


    public void actionPerformed(ActionEvent e)
    {
	Object eventSource = e.getSource();
	if(buttons != null) {
	    for(int i = 0; i < buttons.length; i++) {
		if(eventSource == buttons[i]) {
		    buttons[i].run();
		    return;
		}
	    }
	}
    }

    protected ComponentEvent eventConvert(ComponentEvent e)
    {
	return new ComponentEvent(this, e.getID());
    }

    public void componentResized(ComponentEvent e)
    {
	processComponentEvent(eventConvert(e));
    }

    public void componentMoved(ComponentEvent e)
    {
	processComponentEvent(eventConvert(e));
    }

    public void componentShown(ComponentEvent e)
    {
	processComponentEvent(eventConvert(e));
    }

    public void componentHidden(ComponentEvent e)
    {
	processComponentEvent(eventConvert(e));
    }

    public void focusGained(FocusEvent e)
    {
	processFocusEvent(eventConvert(e));
    }

    public void focusLost(FocusEvent e)
    {
	super.processFocusEvent(eventConvert(e));
    }

    protected FocusEvent eventConvert(FocusEvent e)
    {
	return new FocusEvent(this, e.getID(), e.isTemporary());
    }
    /*
    protected InputMethodEvent eventConvert(InputMethodEvent e)
    {
	return new InputMethodEvent(this, e.getID(), e.getText(),
				    e.getCommittedCharacterCount(), 
				    e.getCaret(), 
				    e.getVisiblePosition()
				    );
    }

    public void inputMethodTextChanged(InputMethodEvent e)
    {
	processInputMethodEvent(eventConvert(e));
    }

    public void caretPositionChanged(InputMethodEvent e)
    {
	processInputMethodEvent(eventConvert(e));
    }
    */
    protected MouseEvent eventConvert(MouseEvent e)
    {
	return new MouseEvent(this, e.getID(), 
			      e.getWhen(), e.getModifiers(),
			      e.getX(), e.getY(), e.getClickCount(),
			      e.isPopupTrigger());
    }


    public void mouseClicked(MouseEvent e) 
    {
	processMouseEvent(eventConvert(e));
    }

    public void mousePressed(MouseEvent e)
    {
	processMouseEvent(eventConvert(e));
    }

    public void mouseReleased(MouseEvent e)
    {
	processMouseEvent(eventConvert(e));
    }

    public void mouseEntered(MouseEvent e)
    {
	processMouseEvent(eventConvert(e));
    }

    public void mouseExited(MouseEvent e)
    {
	processMouseEvent(eventConvert(e));
    }

    public void mouseDragged(MouseEvent e)
    {
	processMouseEvent(eventConvert(e));
    }

    public void mouseMoved(MouseEvent e)
    {
	processMouseEvent(eventConvert(e));
    }

    protected KeyEvent eventConvert(KeyEvent e)
    {
	return new KeyEvent(this, e.getID(), 
			    e.getWhen(), e.getModifiers(),
			    e.getKeyCode(), e.getKeyChar());
    }

    public void keyTyped(KeyEvent e)
    {
	processKeyEvent(eventConvert(e));
    }

    public void keyPressed(KeyEvent e)
    {
	processKeyEvent(eventConvert(e));
    }

    public void keyReleased(KeyEvent e)
    {
	processKeyEvent(eventConvert(e));
    }

}
