/*
 * Tree.java:  Tree to replace javax.swing.JTree
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: Tree.java,v 1.9 2002/09/02 20:37:46 nozomi Exp $
 * $Log: Tree.java,v $
 * Revision 1.9  2002/09/02 20:37:46  nozomi
 * remove unused mouseListener code
 *
 * Revision 1.8  2002/08/15 00:52:25  nozomi
 * modified to use TreeMouseAdaptor in TreeView
 *
 * Revision 1.7  2002/08/01 04:19:46  nozomi
 * adapt to JDK 1.4
 *
 * Revision 1.6  2002/06/09 12:34:40  nozomi
 * change TreeModelEvent handling
 *
 * Revision 1.5  2002/05/25 18:09:37  nozomi
 * remove/addMouseListener(this) when setModel()
 *
 * Revision 1.4  2002/05/23 06:12:37  nozomi
 * tentative fix of re-draw issue
 *
 * Revision 1.3  2002/05/21 06:56:13  nozomi
 * change main() for test
 *
 * Revision 1.2  2002/03/06 03:35:23  nozomi
 * fix a mistake in comment
 *	
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.awt;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Rectangle;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.kyasu.awt.Frame;
import jp.kyasu.awt.List;
import jp.kyasu.awt.Panel;
import jp.kyasu.awt.ScrollPanel;
import jp.kyasu.awt.TextListModel;

import jp.kyasu.awt.text.TextListView;

import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.V3DBorder;
import jp.kyasu.graphics.VBorder;

import org.nomencurator.awt.tree.DefaultMutableTreeNode;
import org.nomencurator.awt.tree.JTree;
import org.nomencurator.awt.tree.TreeController;
import org.nomencurator.awt.tree.TextTreeModel;
import org.nomencurator.awt.tree.TreeNode;
import org.nomencurator.awt.tree.TreeSelectionModel;
import org.nomencurator.awt.tree.TreeView;

import org.nomencurator.awt.tree.event.ExpandVetoException;
import org.nomencurator.awt.tree.event.TreeExpansionEvent;
import org.nomencurator.awt.tree.event.TreeExpansionListener;
import org.nomencurator.awt.tree.event.TreeSelectionEvent;
import org.nomencurator.awt.tree.event.TreeSelectionListener;
import org.nomencurator.awt.tree.event.TreeWillExpandListener;

import org.nomencurator.util.tree.TreePath;
import org.nomencurator.util.tree.TreeModel;
import org.nomencurator.awt.tree.DefaultTreeModel;

/**
 * A GUI representation of hierarchical data.
 *
 * @version 	02 Sep 2002
 * @author 	Nozomi `James' Ytow
 */
public class Tree
    extends List
//    implements JTree
{
    /**
     * org.nomencurator.awt.TreeModel is a subcllass of TextListModel
     * and hence it can be handled by List.
     * Original specification in javax.swing.JTree is:
     * protected transient TreeModel treeModel;
     */

    /**
     *
     * TreeSelectionModel seems task of a controller,
     * hence it can be held by List as TextListController
     * implementing TreeSelectionModel interface.
     *
     */
    //     protected transient TreeSelectionModel selectionModel;


    /**
     *
     * True if root node visible, or false its child nodes are
     * highest visible node
     *
     */
    protected boolean rootVisible;

    /**
     * <code>TreeCellRenderer</code> used to render a node.
     * Default cellRenderer is used by UI if null.
     * Rendering must be handled by TreeView.
     */
    //protected transient TreeCellRenderer cellRenderer;

    /**
     *
     * Height of each row where zero indicates 
     * that the hight is determined by renderer.
     * It would belong to TreeView.
     * 
     */
    //    protected int rowHeight;

    /**
     *
     * Determines whether to display a handle for the root node of the tree.
     * <P>
     * A handle is a small icon displayed next to node enabling users to
     * expand or shrink the node by clicking.  Plus sign (+) indicates
     * expandable node or negative sign (-) indicates shrinkable node
     * in common interface.  Handles are always displayed for nodes
     * lower than top node
     * <P>
     * The root node is only the top level node if rootVisible specifies
     * to display root node.  If the root node is not displayed, all
     * child nodes of the root node at the top level of the tree.
     * <P>
     * Set ture this value ordinary if the root not is invisible, otherwise
     * tree will be displayed as if a list from which user can't recognise
     * these "list entries" are tree nodes in reality.
     *
    */
    protected boolean showsRootHandles;

    /**
     *
     * Construts a new event and redirect it to <code>SelectionListeners</code>
     *
     */
    //protected transient JTree.TreeSelectionRedirector selectionRedirector;


    /**
     *
     * Editor of entries of tree.  Default is null, i.e. tree is not editable.
     */
    //    protected transient TreeCellEditor cellEditor;


    /**
     *
     * Inidicates wheter tree is editable,  Dafault value is false
     */
    protected boolean editable;


    /**
     *
     * Suggests this tree is large model or not which sets code optimisation.
     * A large model can be used when hight of all cell are same.
     * Using large mode, UI cache very little information but transfer data
     * to model continuously.  Without large mode, UI chache most information
     * which reduces call of model methods.
     * <P>
     * It is only suggestion to UI which may ignore this value.
     * Default is false.
     */
    protected boolean largeModel;

    /**
     *
     * Number of rows visble at once which is used by <code>Scrollable</code> interface.
     * It determines appropriate size of display area.
     *
     * It is equivalent to List.rows.
     */
    //protected int visibleRowCount;


    /**
     *
     * If it is true, <code>stopCellEditing</code> is called to save modification when edition 
     * is interrupted by chenge of selection, data change in tree or other events.
     * If it is false, <code>cancellCellEditing</code> is called to discard modification
     * under such occasions.
     * Deault value is false
     *
     */
    protected boolean invokesStopCellEditing;


    /**
     *
     * If it is true, expansion of a node results in scrolling to show all descending 
     * nodes of the node.
     */
    protected boolean scrollsOnExpand;

    /**
     *
     * Number of mouse clicks necessary to expand/shrink a node
     *
     */
    protected int toggleClickCount;


    /**
     *
     * Vector of TreeModelListeners
     *
     * protected transient TreeModelListener treeModelListener;
     */
    protected transient Vector treeModelListener;

    /**
     *
     */
    protected boolean expandsSelectedPaths;

    //Constructors compatible with jp.kyasu.awt.List
    /**
     * Constructs a scrolling Tree without visible line.
     * 
     */
    public Tree()
    {
	this(0, false);
    }

    /**
     * Constructs a scrolling Tree initialized with the specified
     * number of visible lines.
     * 
     * @param rows the number of items to show.
     */
    public Tree(int rows) {
    	this(rows, false);
    }

    /**
     * Constructs a scrolling Tree initialized to display the specified
     * number of <code>rows</code>. If the value of <code>multipleMode</code> is
     * <code>true</code>, then the user can select multiple items from
     * the tree. If it is <code>false</code>, only one item at a time
     * can be selected.
     * @param rows         the number of items to show.
     * @param multipleMode if <code>true</code>, then multiple selections
     *                     are allowed; otherwise, only one item can be
     *                     selected at a time.
     */
    public Tree(int rows, boolean multipleMode)
    {
	this(rows, multipleMode, ScrollPanel.SCROLLBARS_BOTH);
    }

    /**
     * Constructs a scrolling Tree initialized to display the specified
     * number of rows, multipleMode, and scroll bar visibility..
     * @param rows         the number of items to show.
     * @param multipleMode if <code>true</code>, then multiple selections
     *                     are allowed; otherwise, only one item can be
     *                     selected at a time.
     * @param scrollbars   a constant that determines what scrollbars are
     *                     created to view the list.
     */
    public Tree(int rows, boolean multipleMode, int scrollbars)
    {
	this(RichTextStyle.DEFAULT_LIST_STYLE,
	     rows, multipleMode, scrollbars);
    }

    /**
     * Constructs a scrolling Tree with the specified style and number of rows.
     * @param richTextStyle the style of the text list model.
     * @param rows          the number of items to show.
     */
    public Tree(RichTextStyle richTextStyle, int rows)
    {
	this(richTextStyle, rows, false);
    }

    /**
     * Constructs a scrolling Tree with the specified style, number of rows,
     * and multipleMode.
     * @param richTextStyle the style of the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     */
    public Tree(RichTextStyle richTextStyle, int rows, boolean multipleMode)
    {
	this(richTextStyle, rows, multipleMode, ScrollPanel.SCROLLBARS_BOTH);
    }

    /**
     * Constructs a scrolling Tree with the specified style, number of rows,
     * multipleMode, and scroll bar visibility.
     * @param richTextStyle the style of the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     */
    public Tree(RichTextStyle richTextStyle, int rows, boolean multipleMode,
		int scrollbars)
    {
	this(new TextTreeModel(richTextStyle), rows, multipleMode,
	     scrollbars, new V3DBorder(false));
    }

    /**
     * Constructs a scrolling Tree with the specified model and number of rows.
     * @param treeModel the text list model.
     * @param rows          the number of items to show.
     */
    public Tree(TextTreeModel textTreeModel, int rows)
    {
	this(textTreeModel, rows, false);
    }

    /**
     * Constructs a scrolling Tree with the specified model, number of rows,
     * and multipleMode.
     * @param treeModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     */
    public Tree(TextTreeModel textTreeModel, int rows, boolean multipleMode) {
	this(textTreeModel, rows, multipleMode, ScrollPanel.SCROLLBARS_BOTH);
    }

    /**
     * Constructs a scrolling Tree with the specified model, number of rows,
     * multipleMode, and scroll bar visibility.
     * @param treeModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     */
    public Tree(TextTreeModel textTreeModel, int rows, boolean multipleMode,
		int scrollbars)
    {
	this(textTreeModel, rows, multipleMode, scrollbars,
	     new V3DBorder(false));
    }

    /**
     * Constructs a scrolling Tree with the specified model, number of rows,
     * multipleMode, scroll bar visibility, and border visual.
     * @param treeModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     * @param border        the border visual of the list.
     */
    public Tree(TextTreeModel textTreeModel, int rows, boolean multipleMode,
		int scrollbars, VBorder border)
    {
	super(textTreeModel, rows, multipleMode, scrollbars, border);
	//	super.addMouseListener(this);
    }

    /**
     * Constructs a scrolling Tree with the specified model.
     * This constructor is used by the subclasses only.
     */
    protected Tree(TextTreeModel textTreeModel)
    {
	this(textTreeModel, 0);
    }

    //Constructors compatible with javax.swing.JTree

    /**
     * Constructs a Tree widget having elemnts of <code>value</code> array
     * as child nodes of undisplayed new root node.  Leaf nodes in the tree are
     * any node witout child node as default.
     *
     * @param value an array of Objects
     */
    public Tree(Object[] value)
    {
	this();
	if(value == null || value.length == 0)
	    return;
	TreeNode root = (TreeNode)(getTextTreeModel()).getRoot();
	for(int i = 0; i < value.length; i++) {
	    root.insert(new DefaultMutableTreeNode(value[i]), i);
	}
	//Do we need to do something to tell the controller?
    }

    /**
     * Constructs a Tree widget having elemnts of <code>value</code> Enumeration
     * as child nodes of undisplayed new root node.  Leaf nodes in the tree are
     * any node witout child node as default.
     *
     * @param value a Enumeration
     */
    public Tree(Enumeration value)
    {
	this();
	if(value == null)
	    return;
	TreeNode root = (TreeNode)(getTextTreeModel()).getRoot();
	int i = 0;
	while(value.hasMoreElements()){
	    root.insert(new DefaultMutableTreeNode(value.nextElement()), i++);
	}
	//Do we need to do something to tell the controller?

    }

    /**
     * Constructs a Tree widget having elemnts of <code>value</code> Vector
     * as child nodes of undisplayed new root node.  Leaf nodes in the tree are
     * defined as any node witout child node by default.
     *
     * @param value a Vector
     */
    public Tree(Vector value)
    {
	this(value.elements());
    }

    /**
     * Constructs and returns a reference to Tree widget having elements which is value 
     * of key-value pair in <code>value</code> as child nodes of undisplayed new root 
     * node.  Leaf nodes in the tree are defined as any node witout child node by default.
     *
     * @param value a Hashtable
     * 
     */
    public Tree(Hashtable value)
    {
	this(value.elements());
    }


    /**
     * Returns refenrece to newly constructed Tree having specified <code>node</code> as
     * its root node.  Leaf nodes in the tree are defined as any node witout child node
     * by default.
     *
     * @param node TreeNode object to be the root of constructed Tree
     */
    public Tree(TreeNode node)
    {
	this(node, false);
    }


    /**
     * Returns reference to newly constructed Tree having specified <code>node</code> as
     * its root node.  Leaf nodes in the tree are determined depeding on given
     * <code>asksAllowsChildren</code>.
     *
     * @param node TreeNode object to be the root of constructed Tree
     * @param askedAllowsChildren false if leaf node is any node without child node, 
     * true if leaf node is a node which prohibits having child node.
     */
    public Tree(TreeNode node,
		boolean asksAllowsChildren)
    {
	this(new TextTreeModel(node, asksAllowsChildren));
    }

    /**
     * Creates a new TextListView.
     */
    protected TextListView createsListView(TextListModel listModel)
    {
	return new TreeView(listModel);
    }


    //methods of JTree
    /*
     * Creates and returns sample <code>TreeModel</code>
     *
     * @return default <code>TreeModel</code>
     */

    protected static TreeModel getDefaultTreeModel()
    {
	return new TextTreeModel();
    }


    /**
     * Returns a TreeModel wrapping object specified as <code>value</code>.  If <code>value</code> is
     * one of <UL>
     * <LI>array of Objects</LI>
     * <LI>Hashtable</LI>
     * <LI>Vector</LI>
     * </UL>,  the root of created TreeModel has objects as its child nodes.
     * Otherwise the given Object will be used as value of the root node.
     *
     * @param value The <code>Object</code> used as foundation of the <code>TreeModel</code>
     *
     * @return TreeModel wrapping the given object
     */
    protected static TextTreeModel createTextTreeModel(Object value)
    {
	return new TextTreeModel(new DefaultMutableTreeNode(value));
    }


    
    /**
     *
     * Returns look and feel object rendering this component.
     *
     * @return <code>TreeUI</code> object rendering this component
     */
    /*
    public TreeUI getUI()
    {
    }
    */


    /**
     *
     * Sets look and feel object rendering this component.
     *
     * @param ut Look and feel object of <code>TreeUI</code>
     *
     * @see UIDefaults#getUI(javax.swing.JComponent)
    */
    //    public void setUI(TreeUI ui);

    /**
     *
     * Notification from <code>UIManager</code> telling change of look and feel
     * which replaces current UI object with newer version from the 
     * <code>UIManager</code>.
     *
     * @see JComponent#updateUI
    */
    //    public void updateUI();


    /*
     *
     * Returns L&F class name rendering this component
     *
     * @return string "TreeUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI(javax.swing.JComponent)
     */
    //    public String getUIClassID();


    /**
     *
     * Returns current <code>TreeCellRenderer</code> rendering
     * each cell
     */
    //public TreeCellRenderer getCellRenderer();


    /**
     *
     * Sets <code>TreeCellRenderer</code> to render
     * each cell
     *
     * @param renderer <code>TreeCellRenderer</code> to render each cell 
     */
    //public void setCellRenderer(TreeCellRenderer renderer);


    /**
     *
     * Sets the tree is editable or not.  It triggers property changed event
     * if the new value differ from previous one.
     *
     * @param flag boolean to be true when the tree is editable
    */
    public void setEditable(boolean flag)
    {
	if(editable == flag)
	    return;

	editable = flag;
    }


    /**
     *
     * Returns true if the tree is editable
     *
     */
    public boolean isEditable()
    {
	return editable;
    }


    /**
     *
     * Sets a <code>TreeCellEditor</code> specified by <code>cellEditor</code>.
     * Null value of <code>cellEditor</code> implies the tree is not editable.
     * If this represents change in <code>cellEditor</code>, 
     * <code>propertyChange</code> method will be evoked fro all listerns
     *
     * @param cellEditor <code>TreeCellEditor</code> to be used
    */
    //public void setCellEditor(TreeCellEditor cellEditor);



    /**
     *
     * Returns a reference to <code>TreeCheelEditor</code> to be used
     * for edition of tree entries.
     *
     * @return <code>TreeCellEditor</code> in use, or null
     * if the tree is uneditable.
     */
    //public TreeCellEditor getCellEditor();


    /**
     *
     * Returns a reference to the <code>TextTreeModel</code>
     * providing data for the tree.
     *
     * @return <code>TextTreeModel</code> providing data
     */
    public TextTreeModel getTextTreeModel()
    {
	return (TextTreeModel)getModel();
    }

    /**
     *
     * Returns a reference to the <code>TreeModel</code>
     * providing data for the tree.
     *
     * @return <code>TreeModel</code> providing data
     */
    public TreeModel getTreeModel()
    {
	return getTextTreeModel().getTreeModel();
    }

    /**
     *
     * Sets a <code>TextTreeModel</code> to provide data
     *
     * @param model <code>TextTreeModel</code>  to provide data
     */
    /**
     * Sets <code>textListModel</code> as the model of this list.
     *
     * @param <code>textListModel</code> to be used as the model of this list
     */
    public synchronized void setModel(TextListModel textListModel)
    {
	if(listModel == textListModel)
	    return;

	//	super.removeMouseListener(this);
	super.setModel(textListModel);
	//	super.addMouseListener(this);
    }


    /**
     *
     * Returns a reference to the <code>TreeView</code>
     * providing data for the tree.
     *
     * @return <code>TreeView</code> providing data
     */
    public TreeView getTreeView()
    {
	return (TreeView)getView();
    }

    /**
     *
     * Sets a <code>TreeView</code> to provide data
     *
     * @param model <code>TreeView</code>  to provide data
     */
    public void setView(TreeView view)
    {
	if(listView == view)
	    return;

	listView = view;
    }

    /**
     *
     * Returns a reference to the <code>TreeController</code>
     * providing data for the tree.
     *
     * @return <code>TreeController</code> providing data
     */
    public TreeController getTreeController()
    {
	return (TreeController)getController();
    }

    /**
     *
     * Returns a reference to the <code>TreeModel</code>
     * providing data for the tree.
     *
     * @return <code>TreeModel</code> providing data
     */
    public TreeSelectionModel getTreeSelectionModel()
    {
	return (TreeSelectionModel)getController();
    }


    /**
     *
     * Sets a <code>TreeController</code> to provide data
     *
     * @param model <code>TreeController</code>  to provide data
     */
    public void setController(TreeController controller)
    {
	if(listController == controller)
	    return;

	listController = controller;
    }


    /**
     *
     * Returns true if the root node of the tree is displayed
     *
     * @return true if the root node is displayed
     */
    public boolean isRootVisible()
    {
	return rootVisible;
    }


    /**
     *
     * Determines whether root node of <code>TreeModel</code> is
     * visible.
     *
     * @param rootVisible true if the root node is displayed.
     */
    public void setRootVisible(boolean visibility)
    {
	if(rootVisible == visibility)
	    return;

	rootVisible = visibility;
    }


    /**
     * 
     * Determines whether handle of the root node is displayed.
     *
     * @param visible true if handle of the root node is displayed
     *
     */
    public void setShowsRootHandles(boolean visible)
    {
	if(showsRootHandles == visible)
	    return;

	showsRootHandles = visible; 
    }

    /**
     *
     * Returns true if handle of the root node is displayed.
     *
     * @return true if handle of the root node is displayed.
     */
    public boolean getShowsRootHandles()
    {
	return showsRootHandles;
    }


    /**
     *
     * Specifies hight of each cell in pixel.  If specified value is less than
     * or equal to zero, the current cell renderer is queried for row heidht.
     *
     * @param rowHeight hight of each cell in pixel
     */
    public void setRowHeight(int rowHeight)
    {
	//to be implemented
    }


    /**
     *
     * Returns hight of a row.  If returned value is less than or equal to zero,
     * the hight determined by the cell renderer.
     *
     * @return hight of a row in pixel.  Less than or equal to zero if the hight
     * is determined by cell renderer.
     *
     */
    public int getRowHeight()
    {
	return 0;
    }

    /**
     * 
     * Returns true if row hight has fixed value
     *
     * @return true if row hight has fixed value
     */
    public boolean isFixedRowHeight()
    {
	return true;
    }


    /**
     *
     * Sets wheter UI uses large model (its implementation isn't mandate for all UI).
     * It triggers propaty change of LARGE_MODEL_PROPERTY
     *
     * @param largeModel true to suggest larege model to UI
     */
    public void setLargeModel(boolean largeModel)
    {
	if(this.largeModel == largeModel)
	    return;

	this.largeModel = largeModel;
    }
    

    /**
     *
     * Returns true if the tree is set for larege model
     *
     * @return true if large model is suggested
     */
    public boolean isLargeModel()
    {
	return largeModel;
    }

    /**
     *
     * Sets what to do when edition is interrupted by selection of other nodes in the tree,
     * change of tree data etc.  If this propaty is true, modiciation is saved automatically
     * when edition is interrupted.
     * <p>
     * It invokes propaty change of INVOKES_STOP_CELL_EDITING_PROPERTY
     *
     * @param saveData If true, stopCellEditing is called to save data when edition inturrputed.
     * If false, cancelCellEditing is called and modification will be lost
     */
    public void setInvokesStopCellEditing(boolean saveData)
    {
	if(invokesStopCellEditing == saveData)
	    return;

	invokesStopCellEditing = saveData;
    }


    /**
     *
     * Returns indicator telling what should do when edition is inturrputed.
     *
     * @return indicator telling what should do when edition is inturrputed.
     */
    public boolean getInvokesStopCellEditing()
    {
	return invokesStopCellEditing;
    }


    /**
     *
     * Determines whether scroll out in the view port as much as lower nodes when a node
     * is extended.  True is the default value.
     *
     */
    public void setScrollsOnExpand(boolean scroll)
    {
	if(scrollsOnExpand == scroll)
	    return;

	scrollsOnExpand = scroll;
    }


    /**
     *
     * Returns true if the tree scrolls to show previously hidden nodes
     *
     * @return: true if the tree scrolls to show as many as descendent nodes 
     * when a node is extended
     */
    public boolean getScrollsOnExpand()
    {
	return scrollsOnExpand;
    }



    /**
     *
     * Sets number of mouse clicks necessary to extend/shrink a node.
     * Default is two.
     *
     *
     * @since 1.3
     */
    public void setToggleClickCount(int clickCount)
    {
	if(toggleClickCount == clickCount)
	    return;

	toggleClickCount = clickCount;
    }



    /**
     *
     * Returns nuber of mouse clicks necessary to extend/shrink a node
     *
     * @return nuber of mouse clicks necessary to extend/shrink a node 
     *
     * @since 1.3
     */
    public int getToggleClickCount()
    {
	return toggleClickCount;
    }


    /*
     *
     * Sets <code>expandsSelectedPaths</code> propaty.  If true, selection can be
     * modified anytime via <code>TreeSelectionModel</code> or cover method provided by
     * <code>JTree</code>, and parent of <code>TreePath</code> is extended to be visible,
     * where visible means extended parent path besides a visible rectangular area of JTree.
     * If false, selection of a node extends all parents but not make them visible.
     * It is convinient to extend all parents of a path but keep them invisible.
     * 
     * @param newValue new value of <code>expandsSelectedPaths</code>
     *
     * @since 1.3 
     */
    public void setExpandsSelectedPaths(boolean newValue)
    {
	if(expandsSelectedPaths == newValue)
	    return;
	expandsSelectedPaths = newValue;
    }


    /**
     *
     * Returns <code>expandsSelectedPaths</code> propaty.
     *
     * @return true if change of selection extends parent path
     *
     * @since 1.3
     */
    public boolean getExpandsSelectedPaths()
    {
	return expandsSelectedPaths;
    }


    /**
     *
     * Returns <code>isEditable</code>. 
     * It is invoked by UI before stating edition and make specified path editable.
     * It provides an entry point for subclasser to add filtered editing without
     * creating new editor.
     *
     * @return true if all parent nodes and the node itself are editable
     */
    public boolean isPathEditable(TreePath path)
    {
	return editable;
    }


    /**
     *
     * Overrides <code>getToolTipText</code> method of <code>JComponent</code> to make rendering hint usable.
     *
     * @override <code>getToolTipText</code> of <code>JComponent</code>
     *
     * @param event <code>MouseEvent</code> invoked dispaly of <code>ToolTip</code>
     *
     * @return <code>String</code> containing tool hing.  It is null if <code>event</code> is null.
     */
    //    public String getToolTipText(MouseEvent event);


    /**
     *
     * Not yet implemented
     *
     */
    /*
    public String convertValueToText(Object value,
				     boolean selected,
				     boolean expanded,
				     boolean leaf,
				     int row,
				     boolean hasFocus);
    */

    /**
     *
     * Returns number of rows displayed
     *
     * @return number of rows displayed
     */
    public int getRowCount()
    {
	return rows; //?
    }
    


    /**
     *
     * Selects a node identified given <code>path</code>.
     * The node becomes visible if any component of the path is hidden (under a shrunken node)
     * and if <code>getExpandsSelectedPaths</code> is true.
     *
     * @param path a <code>TreePath</code> specifying node to be selected
     */
    public void setSelectionPath(TreePath path)
    {
	getSelectionModel().setSelectionPath(path);
    }


    /**
     *
     * Select a set of node specified by each path in given array of <code>TreePath</code>.
     * Nodes become visible if any component of the path is hidden (under a shrunken node)
     * and if <code>getExpandsSelectedPaths</code> is true.
     *
     * @param paths an array of <code>TreePath</code> object specifying node to be selected
     */
    public void setSelectionPaths(TreePath[] paths)
    {
	getSelectionModel().setSelectionPaths(paths);
    }


    /**
     *
     * Sets path identified as a lead which is not selected itself.
     * Lead is not maintained by JTree but updated by UI
     *
     * @param path the new lead path
     *
     * @since 1.3
     */
    //    public void setLeadSelectionPath(TreePath newPath);


    /**
     *
     * Sets path identified as anchor which is not maintained by JTree but
     * updated by UI.
     *
     * @param path the new anchor path
     *
     * @since 1.3
     */
    //public void setAnchorSelectionPath(TreePath path);


    /**
     *
     * Selects a node at specified <code>row</code>
     *
     * @param row to be selected; zero indicates the first row
     */
    public void setSelectionRow(int row)
    {
	getSelectionModel().setSelectionRow(row);
    }

    /**
     *
     * Slects nodes corresponding to specified <code>rows</code>.
     * Elements of the <code>rows</code> of which value is < 0 
     * or >= getRowCount will be ignored as invalid element.
     * Selection will be cleard when all elements of <code>rows</code>
     * are invalid, which is equivalent to calling <code>clearSelection</code>
     *
     * @param rows an array of int specifying rows to be selected.  Its zero value
     * indicates the first row
     */
    public void setSelectionRows(int[] rows)
    {
	getSelectionModel().setSelectionRows(rows);
    }


    /**
     *
     * Appends a node specified by given <code>TreePath</code> to current selection.
     * If any component of the path is invisible and if getExpandsSelectedPaths is true,
     * the node will be visible.
     *
     * @param path <code>TreePath</code> specifying a node to be add
     */
    public void addSelectionPath(TreePath path)
    {
	getSelectionModel().addSelectionPath(path);
    }


    /**
     *
     * Append paths specified by given array of <code>TreePath</code> to current selection.
     * If any component of the path is invisible and if getExpandsSelectedPaths is true,
     * the node will be visible.
     *
     * @param paths an array of <code>TreePath</code> objects specifying a node to be append
     */
    public void addSelectionPaths(TreePath[] paths)
    {
	getSelectionModel().addSelectionPaths(paths);
    }


    /**
     * 
     * Append path at the specified <code>row</code> to current selection.
     *
     * @param row an int specifying a row of the node to be append,
     * where zero indicates the first row.
     */
    public void addSelectionRow(int row)
    {
	getSelectionModel().addSelectionRow(row);
    }



    /**
     *
     * Appends paths at specified rows to current selection
     *
     * @param rows an array of int specifying rows at which path to be append
     * to the selection, where zero implies the first row
     */
    public void addSelectionRows(int[] rows)
    {
	getSelectionModel().addSelectionRows(rows);
    }


    /**
     *
     * Returns last path component on the first node of currenct selection
     *
     * @return the last <code>Object</code> on the first selected <code>TreePath</code>,
     * or null if nothing is selected
     *
     * @see         TreePath.getLastPathComponent();
    */
    public Object getLastSelectedPathComponent()
    {
	return getSelectionModel().getLastSelectedPathComponent();
    }


    /**
     *
     * Returns a path identified as lead
     *
     * @return path identified as lead
     */
    public TreePath getLeadSelectionPath()
    {
	return getSelectionModel().getLeadSelectionPath();
    }


    /**
     *
     * Returns a path identified as anchor
     *
     * @return path identified as anchor
     *
     * @since 1.3
     */
    public TreePath getAnchorSelectionPath()
    {
	return getSelectionModel().getAnchorSelectionPath();
    }


    /**
     *
     * Returns path of the first selected node
     *
     * @return <code>TreePath</code> of the first selected node, or
     * null if nothing is selected
     */
    public TreePath getSelectionPath()
    {
	return getSelectionModel().getSelectionPath();
    }


    /**
     *
     * Returns paths of all selected nodes
     *
     * @return an array of <code>TreePath</code> indicating all selected nodes, or
     * null if nothing is selected
     */
    public TreePath[] getSelectionPaths()
    {
	return getSelectionModel().getSelectionPaths();
    }

    /**
     *
     * Returns all rows selected.
     * Call of this method is simply transfered to <code>TreeSelectionModel</code>.
     * Either null or an empty array will be returned if nothing selected, depending 
     * on implementation of <code>TreeSelectionModel</code>.
     *
     * @return an array of int indicating all selected rows where zero is the
     * first row
     */
    public int[] getSelectionRows()
    {
	return getSelectionModel().getSelectionRows();
    }


    /**
     *
     * Returns number of selected nodes
     *
     * @return number of selected nodes
     */
    public int getSelectionCount()
    {
	return getSelectionModel().getSelectionCount();
    }


    /**
     *
     * Returns the first selected row
     *
     * @return int specifying the first selected row where zero indicates the first row
     *
     */
    public int getMinSelectionRow()
    {
	return getSelectionModel().getMinSelectionRow();
    }


    /**
     *
     * Returns the last selected row
     *
     * @return int specifying the last selected row where zero indicates the first row
     *
     */
    public int getMaxSelectionRow()
    {
	return getSelectionModel().getMaxSelectionRow();
    }


    /**
     *
     * Returns row index correstponding to lead path
     *
     * @return int specifying row index correstponding to lead path
     * where zero indicates the first row, -1 indicates null lead path
     */
    public int getLeadSelectionRow()
    {
	return getSelectionModel().getLeadSelectionRow();
    }


    /**
     *
     * Returns true if node specified by <code>path</code> is selected
     *
     * @param path a <code>TreePath</code> identifying a node
     *
     * @returns true if specified node is selected
     */
    public boolean isPathSelected(TreePath path)
    {
	return getSelectionModel().isPathSelected(path);
    }



    /**
     *
     * Returnes true if node specified by <code>row</code> is selected
     *
     * @param row int value specifying display row where zero indicates
     * the first row
     *
     * @return ture if the node is selected
     */
    public boolean isRowSelected(int row)
    {
	return getSelectionModel().isRowSelected(row);
    }


    /**
     *
     * Returns an <code>Enumeration</code> consisted from descending nodes
     * of currently expanded path <code>parent</code>, or null if <code>parent</code>
     * is not expanded.  If nodes in the path was expanded or shrunk while iteration
     * with returnd <code>Enumeration</code>, this methods may not returns all expanded 
     * path or may return no longer expanded path.
     *
     * @param parent path to be examined
     *
     * @return <code>Enumeration</code> of descending nodes of <code>parent</code> or
     * null if <code>parent</code> is not expanded 
     */
    public Enumeration getExpandedDescendants(TreePath parent)
    {
	return getTextTreeModel().getExpandedDescendants(parent, getTreeView());
    }


    /**
     *
     * Returns true if the node specifyed by <code>path</code> has ever been expanded
     *
     * @return true if the path has been expanded
     */
    public boolean hasBeenExpanded(TreePath path)
    {
	return getTextTreeModel().hasBeenExpanded(path);
    }


    /**
     *
     * Returns true if a node specifyied by <code>path</code> is expanded currently.
     *
     * @param path a <code>TreePath</code> specifying node to be examine
     *
     * @return false if any node on the <code>path</code> is shrunken or true if all nodes on the 
     * <code>path</code> is exapanded
     */
    public boolean isExpanded(TreePath path)
    {
	return getTextTreeModel().isExpanded(path);
    }



    /**
     *
     * Returns true if the node at specified <code>row</code> is currently expanded
     *
     * @param row to be examined where zero indicates the first row
     *
     * @return ture if the node is currently expanded
     */
    public boolean isExpanded(int row)
    {
	return getTreeView().isExpanded(row);
    }


    /**
     *
     * Returns true if a node specified by <code>path</code> is shrunken.
     * It returns false if nodes on the <code>path</code> is not displayed
     *
     * @param path a <code>TreePath</code> to be examined
     *
     * @returns ture if nods on the <code>path</code> is shrunken, or false
     * if all nodes on the <code>path</code> is expanded
     */
    public boolean isCollapsed(TreePath path)
    {
	return getTextTreeModel().isCollapsed(path);
    }


    /**
     *
     * Returns true if the node at specified <code>row</code> is
     * shrunken
     *
     * @param row to be exmained where zero indicates the first row
     *
     * @return true if the node is currently expanded
     */
    public boolean isCollapsed(int row)
    {
	return getTreeView().isCollapsed(row);
    }


    /**
     *
     * Makes the node identified by <code>path</code> visible
     *
     * @param path a <code>TreePath</code> to be visible
     */
    public void makeVisible(TreePath path)
    {
	getTreeView().makeVisible(path);
    }


    /**
     *
     * Returns true if the node identified by <code>path</code> is visible, i.e. it is root or its all parents are
     * expanded.
     *
     * @returns true if the node is visible
     */
    public boolean isVisible(TreePath path)
    {
	return getTreeView().isVisible(path);
    }


    /**
     *
     * Returns a <code>Rectangle</code> where node specified by <code>path</code>
     * is rendererd.  It returns null if nodes on the <code>path</code> is hidden
     * (i.e. under hidden parent),
     * <p>
     * Note: it returns valid <code>Rectangle</code> even if the specified node
     * is not visible
     *
     * @param path a <code>TreePath</code> specifying a node
     *
     * @return <code>Rectangle</code> to where node is rendered or null
     */
    public Rectangle getPathBounds(TreePath path)
    {
	return getTreeView().getPathBounds(path);
    }


    /**
     *
     * Returns a <code>Rectangle</code> where the node specifyed by
     * given <code>row</code> is rendered
     *
     * @param row to be rendered where zero indicates the first row
     *
     * @return <code>Rectangle</code> where node to be rendered
     */
    public Rectangle getRowBounds(int row)
    {
	return getTreeView().getRowBounds(row);
    }

    /**
     *
     * Extends all nodes on given <code>path</code> except the last path component
     * and scroll that the node to be displayed.  It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param path a <code>TreePath</code> specifying a node to be displayed
     */
    public void scrollPathToVisible(TreePath path)
    {
	getTreeView().scrollPathToVisible(path);
    }

    /**
     *
     * Scrolles until the node specified by <code>row</code> to be displayed
     * with minimum amount of scroll to display.
     * It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param row int specifying row to scroll where zero indicates the first row
     */
    public void scrollRowToVisible(int row)
    {
	getTreeView().scrollRowToVisible(row);
    }


    /**
     *
     * Returns <code>TreePath</code> of the node at given <code>row</code>
     * or null if the <code>row</code> is invisible
     *
     * @param row in int
     *
     * @return <code>TreePath</code> of specified node, or null if
     * <code>row</code> < 0 or <code>row</code> > <code>getRowCount()</code>
    */
    public TreePath getPathForRow(int row)
    {
	return getTreeView().getPathForRow(row);
    }


    /**
     *
     * Returns row displaying node idnetified by <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     * @return int indicating displayed row where zero indicates the first
     * row, -1 if path componets are hidden under shrunken parent
     */
    public int getRowForPath(TreePath path)
    {
	return getTreeView().getRowForPath(path);
    }



    /**
     *
     * Expands and makes visible a node indetified by given <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     */
    public void expandPath(TreePath path)
    {
	getTextTreeModel().expandPath(path);
    }


    /**
     *
     * Expands and makes visible a node at specified <code>row</code>.
     * It does nothing if <code>row</code> < 0 or .<code>row</code> >=
     * <code>getRowCount</code>.
     *
     * @param row int specifying displayed row where zero indicates
     * the first row
     */
    public void expandRow(int row)
    {
	getTextTreeModel().expandRow(row);
    }


    /**
     *
     * Shrink and make visible a node identified by <code>path</code>
     *
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     */
    public void collapsePath(TreePath path)
    {
	getTextTreeModel().collapsePath(path);
    }

    /**
     *
     * Shrinnk a node at specified <code>row</code>
     * It does nothing if <code>row</code> < 0 or .<code>row</code> >=
     * <code>getRowCount</code>.
     *
     * @param row int specifying displayed row where zero indicates
     * the first row
     */
    public void collapseRow(int row)
    {
	getTextTreeModel().collapseRow(row);
    }


    /**
     *
     * Returns <code>TreePath</code> of a node at specified position
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return a <code>TreePath</code> of a node at specified position
     */
    public TreePath getPathForLocation(int x,
				       int y)
    {
	return getTreeView().getPathForLocation(x, y);
    }


    /**
     *
     * Returns row of a node at specified position
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return row of a node at specified position, where -1 indicates
     * the specified position is out of displayed area
     *
     * @see getClosestRowForLocation(int, int);
    */
    public int getRowForLocation(int x,
				 int y)

    {
	return getTreeView().getRowForLocation(x, y);
    }

    /**
     *
     * Returns <code>TreePath</code> of a node nearest to position specified by
     * <code>x</code>, <code>y</code>.
     * It returns valid <code>TreePath</code> always except when returns null if there is no visible node or no model.
     * To confirm the node is at <code>x</code>, <code>y</code> exactly, examine <code>x</code>, <code>y</code>
     * with margine of the node obtained using <code>getPathBounds(TreePath)</code>
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return <code>TreePath</code> of a node nearest to specified postion, or null
     * if there is no visible node or no model
     *
     * @see getPathForLocation(int, int)
     * @see getPathBounds(TreePath)
     */
    public TreePath getClosestPathForLocation(int x,
					      int y)
    {
	return getTreeView().getClosestPathForLocation(x, y);
    }

    /**
     *
     * Returns row of a node nearest to position specified by
     * <code>x</code>, <code>y</code>.
     * It returns valid row always except when returns -1 if there is no visible node or no model.
     * To confirm the node is at <code>x</code>, <code>y</code> exactly, examine <code>x</code>, <code>y</code>
     * with margine of the node obtained using <code>getRowBounds(int)</code>
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return row of a node nearest to specified postion, or -1
     * if there is no visible node or no model
     *
     * @see getRowForLocation(int, int)
     * @see getRowBounds(int)
     */
    public int getClosestRowForLocation(int x,
					int y)
    {
	return getTreeView().getClosestRowForLocation(x, y);
    }

    /**
     *
     * Returns true if the tree is under editing.
     * Item under edition can be obtaind using  getSelectionPath.
     *
     * @return true if user is editing nodes
     *
     * @see getSelectionPath();
    */
    public boolean isEditing()
    {
	return getTreeController().isEditing();
    }


    /**
     *
     * Terminates currend editin session.
     * DefaultTreeCellEditor object conservs all edtion on going on cells, which may
     * differ in other implementation.
     * It does nothing if the tree is not under edition.
     * <P>
     * Note:<BR>
     * Use <code>setInvokesStopCellEditing(boolean)</code> to save edition automatically 
     * when a user moved in the tree
     *
     * @return true if edition is on going and currently stopped, or false if
     * tree is not under edition
     */
    public boolean stopEditing()
    {
	return getTreeController().stopEditing();
    }


    /**
     *
     * Cancels current edit session.
     * It does nothing if tree is not under edition
     */
    public void cancelEditing()
    {
	getTreeController().cancelEditing();
    }


    /**
     *
     * Selects and starts edition of a node identified by <code>path</code>.
     * It fails if <code>CellEditor</code> does not allow edtion of specified
     * item.
     *
     * @param path a <code>TreePath</code> identifying a node
     */
    public void startEditingAtPath(TreePath path)
    {
	getTreeController().startEditingAtPath(path);
    }


    /**
     *
     * Returns path of the node under edition
     *
     * @return <code>TreePath</code> of the node under edition
     */
    public TreePath getEditingPath()
    {
	return getTreeController().getEditingPath();
    }


    /**
     *
     * Sets a <code>TreeSelectionModel</code> of the tree.
     * A null value indicates an empty selection model prohibiting selection
     *
     * @param selectionModel <code>TreeSelectionModel</code> to be used, or null
     * to disable selection
     *
     * @see <code>TreeSelectionModel</code>
    */
    public void setSelectionModel(TreeSelectionModel selectionModel)
    {
	listController = (TreeController)selectionModel;
    }

    /**
     *
     * Returns a <code>TreeSelectionModel</code> of the tree which is always non-null.
     * Set selection model to null to enforce using empty selection model if you want
     * to prohibit selection
     *
     * @return used <code>TreeSelectionModel</code>
     * 
     * @seesetSelectionModel(TreeSelectionModel)
    */
    public TreeSelectionModel getSelectionModel()
    {
	return (TreeSelectionModel)getController();
    }

    /**
     *
     * Returns a <code>TreePath</code> indicating a path from <code>index0</code>
     * to <code>index1</code>, or null if there is no tree.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     * @return an arraay of <code>TreePath</code> objects, of each for each node between
     * <code>index0</code> and <code>index1</code>, or null if there is no tree
     */
    protected TreePath[] getPathBetweenRows(int index0,
					    int index1)
    {
	return getTreeView().getPathBetweenRows(index0, index1);
    }

    /**
     *
     * Selects nodes at rows between <code>index0</code> and <code>index1</code>.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void setSelectionInterval(int index0,
				     int index1)
    {
	getSelectionModel().setSelectionInterval(index0, index1);
    }

    /**
     *
     * Appends paths at rows between <code>index0</code> and <code>index1</code>
     * to the selection
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void addSelectionInterval(int index0,
				     int index1)
    {
	getSelectionModel().addSelectionInterval(index0, index1);
    }

    /**
     *
     * Removes nodes at rows between <code>index0</code> and <code>index1</code>
     * from the selection
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void removeSelectionInterval(int index0,
					int index1)
    {
	getSelectionModel().removeSelectionInterval(index0, index1);
    }
    

    /**
     *
     * Removes the node identified by given <code>path</code>
     * from the selection
     *
     * @param path a <code>TreePath</code> identifying the node to be removed
     */
    public void removeSelectionPath(TreePath path)
    {
	getSelectionModel().removeSelectionPath(path);
    }


    /**
     *
     * Removes nodes identified by elements of given <code>paths</code>
     * from the selection
     *
     * @param paths an array of  <code>TreePath</code> of which element identifying each node to be removed
     */
    public void removeSelectionPaths(TreePath[] paths)
    {
	getSelectionModel().removeSelectionPaths(paths);
    }


    /**
     *
     * Removes path at <code>row</code> from current selection
     *
     * @param row specifying the node to be removed
     */
    public void removeSelectionRow(int row)
    {
	getSelectionModel().removeSelectionRow(row);
    }


    /**
     *
     * Removes paths at specified <code>rows</code> from the selection
     *
     * @param rows an array of int specifying paths to be removed where
     * zero value indicates the first row 
     */
    public void removeSelectionRows(int[] rows)
    {
	getSelectionModel().removeSelectionRows(rows);
    }


    /**
     *
     * Clear the selection
     */
    public void clearSelection()
    {
	getSelectionModel().clearSelection();
    }


    /**
     *
     * Returns true if selected range is empty currently
     *
     * @return true if selection is empty
     */
    public boolean isSelectionEmpty()
    {
	return getSelectionModel().isSelectionEmpty();
    }


    /**
     *
     * Adds a <code>TreeExpansion</code> event listener
     *
     * @param listener a <code>TreeExpansionListener</code> which is to be notified
     * when a tree node is expanded or shrunken (negative expansion)
     */
    public void addTreeExpansionListener(TreeExpansionListener listener)
    {
	getTreeController().addTreeExpansionListener(listener);
    }


    /**
     * 
     * Removes a <code>TreeExpansion</code> event listener
     *
     * @param listener a <code>TreeExpansionListener</code> to be removed
     */
    public void removeTreeExpansionListener(TreeExpansionListener listener)
    {
	getTreeController().removeTreeExpansionListener(listener);
    }


    /**
     *
     * Adds a <code>TreeWillExpand</code> event listener
     *
     * @param listener a <code>TreeWillExpand</code> which is to be notified
     * when a tree node is expanded or shrunken (negative expansion)
     */
    public void addTreeWillExpandListener(TreeWillExpandListener listener)
    {
	getTreeController().addTreeWillExpandListener(listener);
    }



    /**
     *
     * Removes a <code>TreeWillExpand</code> event listener
     *
     * @param listener a <code>TreeWillExpand</code> to be removed
     */
    public void removeTreeWillExpandListener(TreeWillExpandListener listener)
    {
	getTreeController().removeTreeWillExpandListener(listener);
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying expanded node
     */
    public void fireTreeExpanded(TreePath path)
    {
	//	getTextTreeModel().fireTreeExpanded(path);
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying shrunken node
     */
    public void fireTreeCollapsed(TreePath path)
    {
	//	getTextTreeModel().fireTreeCollapsed(path);
    }


    /**
     *
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to expand
     */
    public void fireTreeWillExpand(TreePath path)
	throws ExpandVetoException
    {
	//	getTextTreeModel().fireTreeWillExpand(path);
    }
	    
    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to shrink
     */
    public void fireTreeWillCollapse(TreePath path)
	throws ExpandVetoException
    {
	//	getTextTreeModel().fireTreeWillCollapse(path);
    }


    /**
     *
     * Adds a <code>TreeSelection</code> event listener
     *
     * @param listener a <code>TreeSelection</code> notified when a node
     * is selected or unselected(negative selection)
     */
    public void addTreeSelectionListener(TreeSelectionListener listener)
    {
	getTreeSelectionModel().addTreeSelectionListener(listener);
    }


    /**
     *
     * Removes a <code>TreeSelection</code> event listener
     *
     * @param listener a <code>TreeSelection</code> notified when a node
     * is selected or unselected(negative selection)
     */
    public void removeTreeSelectionListener(TreeSelectionListener listener)
    {
	getTreeSelectionModel().removeTreeSelectionListener(listener);
    }

    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param event <code>TreeSelectionEvent</code> constructed by 
     * <code>TreeSelectionModel</code> when a node is selected or deselected
     */
    protected void fireValueChanged(TreeSelectionEvent event)
    {
	//	getTextTreeModel().fireValueChanged(event);
    }

    /**
     *
     * Invoked when a modification in tree is sufficiently large to change
     * size of margine but not so huge to remove expanded node set (e.g.
     * when node is expanded, shrunken, or nodes are inserted into tree).
     * It is unnecessary to call this method; UI does when necessary
     */
    public void treeDidChange()
    {
	getTreeView().treeDidChange();
    }


    /**
     *
     * Sets number of displayed rows.
     * It simply calls <code>List.setRow(int)</code>
     *
     * @param rows number of rows to display
     */
    public void setVisibleRowCount(int rows)
    {
	setRows(rows);
    }


    /**
     *
     * Returns number of displayed rows in display area.
     * It simply returns <code>List.getRow(int)</code>
     *
     * @return number of displayd rows 
     */
    public int getVisibleRowCount()
    {
	return getRows();
    }


    /**
     *
     * Returns appropriate display size of JTree of which height is determined
     * by <code>getVisibleRowCount</code> and width is current appropriate width.
     *
     * @return <code>Dimension</code> object holding appropriate size
     */
    public Dimension getPreferredScrollableViewportSize()
    {
	return getPreferredSize();
	//	return getTreeView().getPreferredScrollableViewportSize();
    }


    /**
     *
     * Returns increment when scrolled which is determined by height of the first partially-displayed row,
     * or height of the next hidden line at scrolling direction.
     *
     * @param visibleRect visible area in the view port
     * @param orientation VERTICAL or HORIZONTAL
     * @param direction less than zero for upward or leftword scroll,
     * or larger than zero for downword or rightword scroll
     *
     * @return "unit" increment to scroll in specified direction
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect,
					  int orientation,
					  int direction)
    {
	return getTreeView().getScrollableUnitIncrement(visibleRect,
							orientation,
							direction);
    }

    /**
     *
     * Returns "block" increment which is determined by height or widht of <code>visibleRect</code>
     * depending on <code>orientation</code>.
     *
     * @param avisibleRect visbile vew area in view port
     * @param orientation <code>VERTICAL</code> or <code>HORIZONTAL</code>
     * @param direction less than zero for upward or leftword scroll,
     * or larger than zero for downword or rightword scroll
     *
     * @return "block" increment to scroll in specified direction
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect,
					   int orientation,
					   int direction)
    {
	return getTreeView().getScrollableBlockIncrement(visibleRect,
							 orientation,
							 direction);
    }
    
    /**
     *
     * Returns false to tell width of view port does not restrict width of table,
     * i.e. keeps tree from being smaller than view port, except when appropriate width 
     * for the tree is narrower than width of the view port
     *
     * @return false
     *
     */
    public boolean getScrollableTracksViewportWidth()
    {
	return getTreeView().getScrollableTracksViewportWidth();
    }

    /**
     *
     * Returns false to tell height of view port does not restrict height of table,
     * i.e. keeps tree from being smaller than view port, except when appropriate height
     * for the tree is shorter than height of the view port
     *
     * @return false
     *
     */
    public boolean getScrollableTracksViewportHeight()
    {
	return getTreeView().getScrollableTracksViewportHeight();
    }


    /**
     *
     * Sets expansion state of this JTree.  If <code>state</code> is ture,
     * <code>path</code> and its all parents are marked to epand.
     * If <code>state</code> is false, all parents of <code>path</code> is
     * displayed by <code>EXPANDED</code> but <code>path</code> itself is
     * shrunken.
     * <P>
     * It fails if denied by <code>TreeWillExpandListener</code>
     */
    protected void setExpandedState(TreePath path,
				    boolean state)
    {
	getTextTreeModel().setExpandedState(path, state);
    }


    /**
     *
     * Returns <code>Enumeration</code> of descendant nodes of <code>parent</code>
     */
    protected Enumeration getDescendantToggledPaths(TreePath parent)
    {
	return getTextTreeModel().getDescendantToggledPaths(parent);
    }

    /**
     *
     * Remove expanded descending nodes of <code>TreePath</code>s in
     * <code>toRemove</code>
     */
    protected void removeDescendantToggledPaths(Enumeration toRemove)
    {
	getTextTreeModel().removeDescendantToggledPaths(toRemove);
    }


    /**
     *
     * Clears cache of toggled <code>TreePath</code>s witout sending
     * <code>TreeExpansionListener</code> event
    */
    protected void clearToggledPaths()
    {
	getTextTreeModel().clearToggledPaths();
    }


    /**
     *
     * Constructs and returns an instance of <code>TreeModelHandler</code>
     * which updates expanded status when  <code>TreeModel</code> is modified
     *
     */
    //    protected TreeModelListener createTreeModelListener();


    /**
     *
     * Remove all selected discending nodes of <code>path</code>.
     * If <code>includePath</code> is true and <code>path</code> is selected,
     * it is removed from selection.
     *
     * @return true if discending nodes are removed
     *
     * @since 1.3
     */
    protected boolean removeDescendantSelectedPaths(TreePath path,
						    boolean includePath)
    {
	return getSelectionModel().removeDescendantSelectedPaths(path, includePath);
    }


    //not yet impelmented
    /**
     *
     * Returns <code>String</code> expression of this <code>JTree</code>.
     * It exists only for debugging and contents and format of returned <code>String</code>
     * may depend on implementation.  Returned <code>String</code> may be empty but never
     * be null.
     *
     * @return <code>String</code> expression of <code>JTree</code>
     */
    protected String paramString()
    {
	return new String();
    }


    //not yet impelmented
    /**
     * Returns <code>AccessibleContext</code> relating to this <code>JTree</code>
     * which will be an instance of <code>AccessibleJTree</code> newly constructed
     * depending on necessity.
     *.
     * @return <code>AccessibleJTree</code> works as <code>AccessibleContext</code>
     * of this <code>JTree</code>
     */
    /*
    public AccessibleContext getAccessibleContext()
    {
	return new AccessibleContext();
    }
    */


    public static void main(String[] args)
    {
	DefaultMutableTreeNode  colours  = new DefaultMutableTreeNode("colors");
	
	colours.add(new DefaultMutableTreeNode("blue"));
	colours.add(new DefaultMutableTreeNode("violet"));
	colours.add(new DefaultMutableTreeNode("red"));
	colours.add(new DefaultMutableTreeNode("yellow"));

	DefaultMutableTreeNode sports = new DefaultMutableTreeNode("sports");
	sports.add(new DefaultMutableTreeNode("basketball"));
	sports.add(new DefaultMutableTreeNode("soccer"));
	sports.add(new DefaultMutableTreeNode("football"));
	sports.add(new DefaultMutableTreeNode("hockey"));

	TextTreeModel colourTextTreeModel = new TextTreeModel(colours);
	TextTreeModel sportsTextTreeModel = new TextTreeModel(sports);
	DefaultTreeModel colourTreeModel = (DefaultTreeModel)(colourTextTreeModel.getTreeModel());
	DefaultTreeModel sportsTreeModel = (DefaultTreeModel)(sportsTextTreeModel.getTreeModel());

	
	final Tree colourTree = new Tree(colourTextTreeModel, 10);
	final Tree sportsTree = new Tree(sportsTextTreeModel, 10);
	final Frame f = new Frame();
	f.setLayout(new BorderLayout());
	f.add(colourTree, BorderLayout.WEST);
	f.add(sportsTree, BorderLayout.EAST);
	final jp.kyasu.awt.Button button = new jp.kyasu.awt.Button("xchg");
	button.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    TextTreeModel tmp = (TextTreeModel)sportsTree.getModel();
		    sportsTree.setModel((TextTreeModel)colourTree.getModel());
		    colourTree.setModel(tmp);
		}
	    });
	f.add(button, BorderLayout.CENTER);
	f.pack();
	f.show();
    }

}
