/*
 * DefaultMutableTreeNode.java:  DefaultMutableTreeNode 
 * to replace javax.swing.tree.DefaultMutableTreeNode
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: DefaultMutableTreeNode.java,v 1.4 2002/06/06 21:37:07 nozomi Exp $
 * $Log: DefaultMutableTreeNode.java,v $
 * Revision 1.4  2002/06/06 21:37:07  nozomi
 * evoke root node's fireing methods
 *
 * Revision 1.3  2002/06/06 19:12:50  nozomi
 * fire TreeExpansionEvent
 *
 * Revision 1.2  2002/03/10 06:55:53  nozomi
 * modify log format
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.awt.tree;

import java.util.Vector;

import org.nomencurator.awt.tree.TreeNode;

import org.nomencurator.awt.tree.event.ExpandVetoException;
import org.nomencurator.awt.tree.event.TreeExpansionEvent;
import org.nomencurator.awt.tree.event.TreeExpansionListener;
import org.nomencurator.awt.tree.event.TreeWillExpandListener;

import org.nomencurator.util.tree.TreePath;

/**
 * An implementation of <code>MutalbeTreeNode<code>
 * supporting expand/collapse state
 *
 * @version	06 June 2002
 * @author 	Nozomi `James' Ytow
 */
public class DefaultMutableTreeNode
    extends org.nomencurator.util.tree.DefaultMutableTreeNode
    implements TreeNode
{

    //true if the node is expanded
    protected boolean expanded;

    /** Vector to hold <CODE>TextExpansionListener</CODE>s */
    transient protected Vector treeExpansionListeners;

    /** Vector to hold <CODE>TextWillExpandListener</CODE>s */
    transient protected Vector treeWillExpandListeners;

    //TreeNode interfaces
    /**
     * Sets expanded status to <code>show</code>.  If status is same to <code>show</code>
     * it does nothing, else it may call non-trivial functions specific to the 
     * implementation.
     *
     * @param show status to be set
     */
    public void setExpanded(boolean show)
    {
	if(show == expanded)
	    return;

	TreePath path = new TreePath(getPath());
	try {
	    if(show) {
		((DefaultMutableTreeNode)getRoot()).fireTreeWillExpand(path);
	    }
	    else {
		((DefaultMutableTreeNode)getRoot()).fireTreeWillCollapse(path);
	    }
	}
	catch (ExpandVetoException e) {
	    return;
	}
	
	expanded = show;

	if(show) {
	    ((DefaultMutableTreeNode)getRoot()).fireTreeExpanded(path);
	}
	else {
	    ((DefaultMutableTreeNode)getRoot()).fireTreeCollapsed(path);
	}
	

    }

    /**
     * Returns current expanded status.
     *
     * @return current expanded status in boolean
     */
    public boolean isExpanded()
    {
	return expanded;
    } 


    //Constructors
    /**
     * Create a tree node without parent nor child
     */
    public DefaultMutableTreeNode() 
    {
	this(null);
    }

    /**
     * Create a tree node without parent nor child
     * but initialise it with the given object
     */
    public DefaultMutableTreeNode(Object userObject)
    {
	this(userObject, true);
    }

    /**
     * Create a tree node without parent nor child
     * but initialise it with the given object and
     * given allowance
     * 
     */
    public DefaultMutableTreeNode(Object userObject, boolean allowsChildren)
    {
	super(userObject, allowsChildren);
	expanded = true;
    }

    /*
     * Adds a TreeExpansionListener to the lister list
     *
     * @param listener a listener of TreeExpansionListener to be add
     */
    public void addTreeExpansionListener(TreeExpansionListener listener)
    {
	if(treeExpansionListeners == null)
	    treeExpansionListeners = new Vector();

	treeExpansionListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeExpansionlLister from the listeners list
     *
     * @param listener a listener of TreeExpansionListener to be removed
     */
    public void removeTreeExpansionListener(TreeExpansionListener listener)
    {
	treeExpansionListeners.remove(listener);
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying expanded node
     */
    public void fireTreeExpanded(TreePath path)
    {
	if(treeExpansionListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeExpansionListeners) {
	    listeners = treeExpansionListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeExpansionListener)listeners[--i]).treeExpanded(e);
	}
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying shrunken node
     */
    public void fireTreeCollapsed(TreePath path)
    {
	if(treeExpansionListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeExpansionListeners) {
	    listeners = treeExpansionListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeExpansionListener)listeners[--i]).treeCollapsed(e);
	}
    }


    /*
     * Adds a TreeWillExpandListener to the lister list
     *
     * @param listener a listener of TreeWillExpandListener to be add
     */
    public void addTreeWillExpandListener(TreeWillExpandListener listener)
    {
	if(treeWillExpandListeners == null)
	    treeWillExpandListeners = new Vector();

	treeWillExpandListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeWillExpandListener from the listeners list
     *
     * @param listener a listener of TreeWillExpandListener to be removed
     */
    public void removeTreeWillExpandListener(TreeWillExpandListener listener)
    {
	treeWillExpandListeners.remove(listener);
    }

    /**
     *
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to expand
     */
    public void fireTreeWillExpand(TreePath path)
	throws ExpandVetoException
    {
	if(treeWillExpandListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeWillExpandListeners) {
	    listeners = treeWillExpandListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeWillExpandListener)listeners[--i]).treeWillExpand(e);
	}
    }
	    
    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to shrink
     */
    public void fireTreeWillCollapse(TreePath path)
	throws ExpandVetoException
    {
	if(treeWillExpandListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeWillExpandListeners) {
	    listeners = treeWillExpandListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeWillExpandListener)listeners[--i]).treeWillCollapse(e);
	}
    }


}
