/*
 * DefaultTreeModel.java:  DefaultTreeModel to replace it in javax.swing.tree.
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: DefaultTreeModel.java,v 1.2 2002/06/09 12:38:26 nozomi Exp $
 * $Log: DefaultTreeModel.java,v $
 * Revision 1.2  2002/06/09 12:38:26  nozomi
 * synchronize add/removeTreeExpansionListener
 *
 * Revision 1.1  2002/06/06 21:33:07  nozomi
 * initial import into CVS repository
 */

package org.nomencurator.awt.tree;

import java.util.Enumeration;
import java.util.Vector;
import java.util.Stack;

import org.nomencurator.util.tree.TreeModel;
import org.nomencurator.util.tree.TreeNode;
import org.nomencurator.util.tree.TreePath;

import org.nomencurator.util.tree.event.TreeModelEvent;
import org.nomencurator.util.tree.event.TreeModelListener;

import org.nomencurator.awt.tree.event.ExpandVetoException;
import org.nomencurator.awt.tree.event.TreeExpansionEvent;
import org.nomencurator.awt.tree.event.TreeExpansionListener;
import org.nomencurator.awt.tree.event.TreeWillExpandListener;

/**
 * An implementaiont of <code>TreeModel<code>
 *
 * @version	06 June 2002
 * @author 	Nozomi `James' Ytow
 */
public class DefaultTreeModel
    extends org.nomencurator.util.tree.DefaultTreeModel
    implements TreeExpansionListener,
	       TreeWillExpandListener
{
    /** Vector to hold <CODE>TextExpansionListener</CODE>s */
    transient protected Vector treeExpansionListeners;

    /** Vector to hold <CODE>TextWillExpandListener</CODE>s */
    transient protected Vector treeWillExpandListeners;

    //Constructors
    /**
     * Constructs a tree of which any node may have child node.
     *
     * @param root TreeNode object to be root of the tree.
     *
     * @see DefaultTreeModel(TreeNode, boolean)
     */
    public DefaultTreeModel(TreeNode node)
    {
	this(node, false);
    }

    /**
     * Constructs a single column text list model with the specified rich text
     * style.
     *
     * @param root TreeNode object to be root of the tree.
     * @param allowance false if any child node may have 
     * child node, or true if only specified node may have children.
     */
    public DefaultTreeModel(TreeNode node, 
			    boolean allowance)
    {
	super(node, allowance);
    }
    
    /*
     * Adds a TreeExpansionListener to the lister list
     *
     * @param listener a listener of TreeExpansionListener to be add
     */
    public synchronized void addTreeExpansionListener(TreeExpansionListener listener)
    {
	if(treeExpansionListeners == null)
	    treeExpansionListeners = new Vector();

	treeExpansionListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeExpansionlLister from the listeners list
     *
     * @param listener a listener of TreeExpansionListener to be removed
     */
    public synchronized void removeTreeExpansionListener(TreeExpansionListener listener)
    {
	treeExpansionListeners.remove(listener);
    }

    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying expanded node
     */
    public void fireTreeExpanded(TreePath path)
    {
	if(treeExpansionListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeExpansionListeners) {
	    listeners = treeExpansionListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeExpansionListener)listeners[--i]).treeExpanded(e);
	}
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying shrunken node
     */
    public void fireTreeCollapsed(TreePath path)
    {
	if(treeExpansionListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeExpansionListeners) {
	    listeners = treeExpansionListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeExpansionListener)listeners[--i]).treeCollapsed(e);
	}
    }


    /*
     * Adds a TreeWillExpandListener to the lister list
     *
     * @param listener a listener of TreeWillExpandListener to be add
     */
    public void addTreeWillExpandListener(TreeWillExpandListener listener)
    {
	if(treeWillExpandListeners == null)
	    treeWillExpandListeners = new Vector();

	treeWillExpandListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeWillExpandListener from the listeners list
     *
     * @param listener a listener of TreeWillExpandListener to be removed
     */
    public void removeTreeWillExpandListener(TreeWillExpandListener listener)
    {
	treeWillExpandListeners.remove(listener);
    }

    /**
     *
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to expand
     */
    public void fireTreeWillExpand(TreePath path)
	throws ExpandVetoException
    {
	if(treeWillExpandListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeWillExpandListeners) {
	    listeners = treeWillExpandListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeWillExpandListener)listeners[--i]).treeWillExpand(e);
	}
    }
	    
    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to shrink
     */
    public void fireTreeWillCollapse(TreePath path)
	throws ExpandVetoException
    {
	if(treeWillExpandListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized (treeWillExpandListeners) {
	    listeners = treeWillExpandListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeWillExpandListener)listeners[--i]).treeWillCollapse(e);
	}
    }

    /**
     * Called by expansion of a node in a tree
     */
    public void treeExpanded(TreeExpansionEvent event)
    {
	fireTreeExpanded(event.getPath());
    }


    /**
     * Called by shrink of a node in a tree
     */
    public void treeCollapsed(TreeExpansionEvent event)
    {
	fireTreeCollapsed(event.getPath());
    }

    /**
     * Invoked when a tree node will be expanded
     */
    public void treeWillExpand(TreeExpansionEvent event)
	throws ExpandVetoException
    {
	fireTreeWillExpand(event.getPath());
    }


    /**
     * Invoked when a tree node will be collapsed
     */
    public void treeWillCollapse(TreeExpansionEvent event)
	throws ExpandVetoException
    {
	fireTreeWillCollapse(event.getPath());
    }
}
