/*
 * JTreel.java:  interface compatible with javax.swing.JTree
 *
 * Copyright (c) 2001 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: JTree.java,v 1.1.1.1 2002/01/16 12:33:33 ryo Exp $
 * $Log: JTree.java,v $
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.awt.tree;

import java.awt.Dimension;
import java.awt.Rectangle;

import java.util.Enumeration;

import org.nomencurator.awt.tree.event.ExpandVetoException;
import org.nomencurator.awt.tree.event.TreeExpansionEvent;
import org.nomencurator.awt.tree.event.TreeExpansionListener;
import org.nomencurator.awt.tree.event.TreeSelectionEvent;
import org.nomencurator.awt.tree.event.TreeSelectionListener;
import org.nomencurator.awt.tree.event.TreeWillExpandListener;

import org.nomencurator.util.tree.TreePath;
import org.nomencurator.util.tree.TreeModel;

import org.nomencurator.util.tree.event.TreeModelListener;


public interface JTree
{

    //Bound property name
    /** Bound property name for <code>cellRenderer</code> */
    public static final String CELL_RENDERER_PROPERTY = "cellRenderer";

    /** Bound property name for <code>treeModel</code> */
    public static final String TREE_MODEL_PROPERTY = "treeModel";

    /** Bound property name for <code>rootVisible</code> */
    public static final String ROOT_VISIBLE_PROPERTY ="rootVisible";

    /** Bound property name for <code>showsRootHandles</code> */
    public static final String SHOWS_ROOT_HANDLES_PROPERTY = "showsRootHandles";

    /** Bound property name for <code>rowHeight</code> */
    public static final String ROW_HEIGHT_PROPERTY = "rowHeight";

    /** Bound property name for <code>cellEditor</code> */
    public static final String CELL_EDITOR_PROPERTY = "cellEditor";

    /** Bound property name for <code>editable</code> */
    public static final String EDITABLE_PROPERTY = "editable";

    /** Bound property name for <code>largeModel</code> */
    public static final String LARGE_MODEL_PROPERTY = "largeModel";

    /** Bound property name for <code>selectionModel</code> */
    public static final String SELECTION_MODEL_PROPERTY = "selectionModel";

    /** Bound property name for <code>visibleRowCount</code> */
    public static final String VISIBLE_ROW_COUNT_PROPERTY = "visibleRowCount";

    /** Bound property name for <code>messagesStopCellEditing</code> */
    public static final String INVOKES_STOP_CELL_EDITING_PROPERTY = "messagesStopCellEditing";

    /** Bound property name for <code>scrollsOnExpand</code> */
    public static final String SCROLLS_ON_EXPAND_PROPERTY = "scrollsOnExpand";

    /** Bound property name for <code>toggleClickCount</code> */
    public static final String TOGGLE_CLICK_COUNT_PROPERTY = "toggleClickCount";

    /** Bound property name for <code>leadSelectionPath</code> 
     * @since 1.3 */
    public static final String LEAD_SELECTION_PATH_PROPERTY = "leadSelectionPath";

    /** Bound property name for <code>anchorSelectionPath</code> 
     * @since 1.3 */
    public static final String ANCHOR_SELECTION_PATH_PROPERTY = "anchorSelectionPath";

    /** Bound property name for <code>expandsSelectedPaths</code>
     * @since 1.3 */
    public static final String EXPANDS_SELECTED_PATHS_PROPERTY = "expandsSelectedPaths";



    /**
     * Creates and returns a default TreeModel
     *
     */
    public TreeModel getDefaultTreeModel();

    /**
     * Returns a TreeModel wrapping object specified as <code>value</code>.  If <code>value</code> is
     * one of <UL>
     * <LI>array of Objects</LI>
     * <LI>Hashtable</LI>
     * <LI>Vector</LI>
     * </UL>,  the root of created TreeModel has objects as its child nodes.
     * Otherwise the given Object will be used as value of the root node.
     *
     * @param value The <code>Object</code> used as foundation of the <code>TreeModel</code>
     *
     * @return TreeModel wrapping the given object
     */
    public TreeModel createTreeModel(Object value);

    /**
     *
     * Returns look and feel object rendering this component.
     *
     * @return <code>TreeUI</code> object rendering this component
     */
    //public TreeUI getUI();


    /**
     *
     * Sets look and feel object rendering this component.
     *
     * @param ut Look and feel object of <code>TreeUI</code>
     *
     * @see UIDefaults#getUI(javax.swing.JComponent)
    */
    //    public void setUI(TreeUI ui);

    /**
     *
     * Notification from <code>UIManager</code> telling change of look and feel
     * which replaces current UI object with newer version from the 
     * <code>UIManager</code>.
     *
     * @see JComponent#updateUI
    */
    //    public void updateUI();


    /*
     *
     * Returns L&F class name rendering this component
     *
     * @return string "TreeUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI(javax.swing.JComponent)
     */
    //    public String getUIClassID();


    /**
     *
     * Returns current <code>TreeCellRenderer</code> rendering
     * each cell
     */
    //public TreeCellRenderer getCellRenderer();


    /**
     *
     * Sets <code>TreeCellRenderer</code> to render
     * each cell
     *
     * @param renderer <code>TreeCellRenderer</code> to render each cell 
     */
    //public void setCellRenderer(TreeCellRenderer renderer);


    /**
     *
     * Sets the tree is editable or not.  It triggers property changed event
     * if the new value differ from previous one.
     *
     * @param flag boolean to be true when the tree is editable
    */
    public void setEditable(boolean flag);


    /**
     *
     * Returns true if the tree is editable
     *
     */
    public boolean isEditable();


    /**
     *
     * Sets a <code>TreeCellEditor</code> specified by <code>cellEditor</code>.
     * Null value of <code>cellEditor</code> implies the tree is not editable.
     * If this represents change in <code>cellEditor</code>, 
     * <code>propertyChange</code> method will be evoked fro all listerns
     *
     * @param cellEditor <code>TreeCellEditor</code> to be used
    */
    //public void setCellEditor(TreeCellEditor cellEditor);



    /**
     *
     * Returns a reference to <code>TreeCheelEditor</code> to be used
     * for edition of tree entries.
     *
     * @return <code>TreeCellEditor</code> in use, or null
     * if the tree is uneditable.
     */
    //public TreeCellEditor getCellEditor();


    /**
     *
     * Returns a reference to the <code>TreeModel</code>
     * providing data for the tree.
     *
     * @return <code>TreeModel</code> providing data
     */
    //    public TreeModel getModel();
    public TreeModel getTreeModel();

    /**
     *
     * Sets a <code>TreeModel</code> to provide data
     *
     * @param model <code>TreeModel</code>  to provide data
     */
    public void setModel(TreeModel model);



    /**
     *
     * Returns true if the root node of the tree is displayed
     *
     * @return true if the root node is displayed
     */
    public boolean isRootVisible();


    /**
     *
     * Determines whether root node of <code>TreeModel</code> is
     * visible.
     *
     * @param rootVisible true if the root node is displayed.
     */
    public void setRootVisible(boolean rootVisible);


    /**
     * 
     * Determines whether handle of the root node is displayed.
     *
     * @param visible true if handle of the root node is displayed
     *
     */
    public void setShowsRootHandles(boolean visible);

    /**
     *
     * Returns true if handle of the root node is displayed.
     *
     * @return true if handle of the root node is displayed.
     */
    public boolean getShowsRootHandles();


    /**
     *
     * Specifies hight of each cell in pixel.  If specified value is less than
     * or equal to zero, the current cell renderer is queried for row heidht.
     *
     * @param rowHeight hight of each cell in pixel
     */
    public void setRowHeight(int rowHeight);


    /**
     *
     * Returns hight of a row.  If returned value is less than or equal to zero,
     * the hight determined by the cell renderer.
     *
     * @return hight of a row in pixel.  Less than or equal to zero if the hight
     * is determined by cell renderer.
     *
     */
    public int getRowHeight();

    /**
     * 
     * Returns true if row hight has fixed value
     *
     * @return true if row hight has fixed value
     */
    public boolean isFixedRowHeight();


    /**
     *
     * Sets wheter UI uses large model (its implementation isn't mandate for all UI).
     * It triggers propaty change of LARGE_MODEL_PROPERTY
     *
     * @param largeModel true to suggest larege model to UI
     */
    public void setLargeModel(boolean largeModel);


    /**
     *
     * Returns true if the tree is set for larege model
     *
     * @return true if large model is suggested
     */
    public boolean isLargeModel();

    /**
     *
     * Sets what to do when edition is interrupted by selection of other nodes in the tree,
     * change of tree data etc.  If this propaty is true, modiciation is saved automatically
     * when edition is interrupted.
     * <p>
     * It ivokes propaty change of INVOKES_STOP_CELL_EDITING_PROPERTY.
     *
     * @param saveData If true, stopCellEditing is called to save data when edition inturrputed.
     * If false, cancelCellEditing is called and modification will be lost
     */
    public void setInvokesStopCellEditing(boolean saveData);


    /**
     *
     * Returns indicator telling what should do when edition is inturrputed.
     *
     * @return indicator telling what should do when edition is inturrputed.
     */
    public boolean getInvokesStopCellEditing();


    /**
     *
     * Determines whether scroll out in the view port as much as lower nodes when a node
     * is extended.  True is the default value.
     *
     */
    public void setScrollsOnExpand(boolean scroll);


    /**
     *
     * Returns true if the tree scrolls to show previously hidden nodes
     *
     * @return: true if the tree scrolls to show as many as descendent nodes 
     * when a node is extended
     */
    public boolean getScrollsOnExpand();


    /**
     *
     * Sets number of mouse clicks necessary to extend/shrink a node.
     * Default is two.
     *
     *
     * @since 1.3
     */
    public void setToggleClickCount(int clickCount);



    /**
     *
     * Returns nuber of mouse clicks necessary to extend/shrink a node
     *
     * @return nuber of mouse clicks necessary to extend/shrink a node 
     *
     * @since 1.3
     */
    public int getToggleClickCount();


    /*
     *
     * Sets <code>expandsSelectedPaths</code> propaty.  If true, selection can be
     * modified anytime via <code>TreeSelectionModel</code> or cover method provided by
     * <code>JTree</code>, and parent of <code>TreePath</code> is extended to be visible,
     * where visible means extended parent path besides a visible rectangular area of JTree.
     * If false, selection of a node extends all parents but not make them visible.
     * It is convinient to extend all parents of a path but keep them invisible.
     * 
     * @param newValue new value of <code>expandsSelectedPaths</code>
     *
     * @since 1.3 
     */
    public void setExpandsSelectedPaths(boolean newValue);


    /**
     *
     * Returns <code>expandsSelectedPaths</code> propaty.
     *
     * @return true if change of selection extends parent path
     *
     * @since 1.3
     */
    public boolean getExpandsSelectedPaths();


    /**
     *
     * Returns <code>isEditable</code>. 
     * It is ivoked by UI before stating edition and make specified path editable.
     * It provides an entry point for subclasser to add filtered editing without
     * creating new editor.
     *
     * @return true if all parent nodes and the node itself are editable
     */
    public boolean isPathEditable(TreePath path);


    /**
     *
     * Overrides <code>getToolTipText</code> method of <code>JComponent</code> to make rendering hint usable.
     *
     * @override <code>getToolTipText</code> of <code>JComponent</code>
     *
     * @param event <code>MouseEvent</code> ivoked dispaly of <code>ToolTip</code>
     *
     * @return <code>String</code> containing tool hing.  It is null if <code>event</code> is null.
     */
    //    public String getToolTipText(MouseEvent event);


    /**
     *
     * Not yet implemented
     *
     */
    /*
    public String convertValueToText(Object value,
				     boolean selected,
				     boolean expanded,
				     boolean leaf,
				     int row,
				     boolean hasFocus);
    */

    /**
     *
     * Returns number of rows displayed
     *
     * @return number of rows displayed
     */
    public int getRowCount();
    


    /**
     *
     * Selects a node identified given <code>path</code>.
     * The node becomes visible if any component of the path is hidden (under a shrunken node)
     * and if <code>getExpandsSelectedPaths</code> is true.
     *
     * @param path a <code>TreePath</code> specifying node to be selected
     */
    public void setSelectionPath(TreePath path);


    /**
     *
     * Select a set of node specified by each path in given array of <code>TreePath</code>.
     * Nodes become visible if any component of the path is hidden (under a shrunken node)
     * and if <code>getExpandsSelectedPaths</code> is true.
     *
     * @param paths an array of <code>TreePath</code> object specifying node to be selected
     */
    public void setSelectionPaths(TreePath[] paths);


    /**
     *
     * Sets path identified as a lead which is not selected itself.
     * Lead is not maintained by JTree but updated by UI
     *
     * @param path the new lead path
     *
     * @since 1.3
     */
    //    public void setLeadSelectionPath(TreePath newPath);


    /**
     *
     * Sets path identified as anchor which is not maintained by JTree but
     * updated by UI.
     *
     * @param path the new anchor path
     *
     * @since 1.3
     */
    //public void setAnchorSelectionPath(TreePath path);


    /**
     *
     * Selects a node at specified <code>row</code>
     *
     * @param row to be selected; zero indicates the first row
     */

    public void setSelectionRow(int row);

    /**
     *
     * Slects nodes corresponding to specified <code>rows</code>.
     * Elements of the <code>rows</code> of which value is < 0 
     * or >= getRowCount will be ignored as invalid element.
     * Selection will be cleard when all elements of <code>rows</code>
     * are invalid, which is equivalent to calling <code>clearSelection</code>
     *
     * @param rows an array of int specifying rows to be selected.  Its zero value
     * indicates the first row
     */
    public void setSelectionRows(int[] rows);


    /**
     *
     * Appends a node specified by given <code>TreePath</code> to current selection.
     * If any component of the path is invisible and if getExpandsSelectedPaths is true,
     * the node will be visible.
     *
     * @param path <code>TreePath</code> specifying a node to be add
     */
    public void addSelectionPath(TreePath path);


    /**
     *
     * Append paths specified by given array of <code>TreePath</code> to current selection.
     * If any component of the path is invisible and if getExpandsSelectedPaths is true,
     * the node will be visible.
     *
     * @param paths an array of <code>TreePath</code> objects specifying a node to be append
     */
    public void addSelectionPaths(TreePath[] paths);


    /**
     * 
     * Append path at the specified <code>row</code> to current selection.
     *
     * @param row an int specifying a row of the node to be append,
     * where zero indicates the first row.
     */
    public void addSelectionRow(int row);


    /**
     *
     * Appends paths at specified rows to current selection
     *
     * @param rows an array of int specifying rows at which path to be append
     * to the selection, where zero implies the first row
     */
    public void addSelectionRows(int[] rows);


    /**
     *
     * Returns last path component on the first node of currenct selection
     *
     * @return the last <code>Object</code> on the first selected <code>TreePath</code>,
     * or null if nothing is selected
     *
     * @see         TreePath.getLastPathComponent();
    */
    public Object getLastSelectedPathComponent();

    /**
     *
     * Returns a path identified as lead
     *
     * @return path identified as lead
     */
    public TreePath getLeadSelectionPath();


    /**
     *
     * Returns a path identified as anchor
     *
     * @return path identified as anchor
     *
     * @since 1.3
     */
    public TreePath getAnchorSelectionPath();


    /**
     *
     * Returns path of the first selected node
     *
     * @return <code>TreePath</code> of the first selected node, or
     * null if nothing is selected
     */
    public TreePath getSelectionPath();


    /**
     *
     * Returns paths of all selected nodes
     *
     * @return an array of <code>TreePath</code> indicating all selected nodes, or
     * null if nothing is selected
     */
    public TreePath[] getSelectionPaths();

    /**
     *
     * Returns all rows selected.
     * Call of this method is simply transfered to <code>TreeSelectionModel</code>.
     * Either null or an empty array will be returned if nothing selected, depending 
     * on implementation of <code>TreeSelectionModel</code>.
     *
     * @return an array of int indicating all selected rows where zero is the
     * first row
     */
    public int[] getSelectionRows();


    /**
     *
     * Returns number of selected nodes
     *
     * @return number of selected nodes
     */
    public int getSelectionCount();


    /**
     *
     * Returns the first selected row
     *
     * @return int specifying the first selected row where zero indicates the first row
     *
     */
    public int getMinSelectionRow();

    /**
     *
     * Returns the last selected row
     *
     * @return int specifying the last selected row where zero indicates the first row
     *
     */
    public int getMaxSelectionRow();

    /**
     *
     * Returns row index correstponding to lead path
     *
     * @return int specifying row index correstponding to lead path
     * where zero indicates the first row, -1 indicates null lead path
     */
    public int getLeadSelectionRow();


    /**
     *
     * Returns true if node specified by <code>path</code> is selected
     *
     * @param path a <code>TreePath</code> identifying a node
     *
     * @returns true if specified node is selected
     */
    public boolean isPathSelected(TreePath path);

    /**
     *
     * Returnes true if node specified by <code>row</code> is selected
     *
     * @param row int value specifying display row where zero indicates
     * the first row
     *
     * @return ture if the node is selected
     */
    public boolean isRowSelected(int row);

    /**
     *
     * Returns an <code>Enumeration</code> consisted from descending nodes
     * of currently expanded path <code>parent</code>, or null if <code>parent</code>
     * is not expanded.  If nodes in the path was expanded or shrunk while iteration
     * with returnd <code>Enumeration</code>, this methods may not returns all expanded 
     * path or may return no longer expanded path.
     *
     * @param parent path to be examined
     *
     * @return <code>Enumeration</code> of descending nodes of <code>parent</code> or
     * null if <code>parent</code> is not expanded 
     */
    public Enumeration getExpandedDescendants(TreePath parent);


    /**
     *
     * Returns true if the node specifyed by <code>path</code> has ever been expanded
     *
     * @return true if the path has been expanded
     */
    public boolean hasBeenExpanded(TreePath path);


    /**
     *
     * Returns true if a node specifyied by <code>path</code> is expanded currently.
     *
     * @param path a <code>TreePath</code> specifying node to be examine
     *
     * @return false if any node on the <code>path</code> is shrunken or true if all nodes on the 
     * <code>path</code> is exapanded
     */
    public boolean isExpanded(TreePath path);


    /**
     *
     * Returns true if the node at specified <code>row</code> is currently expanded
     *
     * @param row to be examined where zero indicates the first row
     *
     * @return ture if the node is currently expanded
     */
    public boolean isExpanded(int row);


    /**
     *
     * Returns true if a node specified by <code>path</code> is shrunken.
     * It returns false if nodes on the <code>path</code> is not displayed
     *
     * @param path a <code>TreePath</code> to be examined
     *
     * @returns ture if nods on the <code>path</code> is shrunken, or false
     * if all nodes on the <code>path</code> is expanded
     */
    public boolean isCollapsed(TreePath path);


    /**
     *
     * Returns true if the node at specified <code>row</code> is
     * shrunken
     *
     * @param row to be exmained where zero indicates the first row
     *
     * @return true if the node is currently expanded
     */
    public boolean isCollapsed(int row);


    /**
     *
     * Makes the node identified by <code>path</code> visible
     *
     * @param path a <code>TreePath</code> to be visible
     */
    public void makeVisible(TreePath path);


    /**
     *
     * Returns true if the node identified by <code>path</code> is visible, i.e. it is root or its all parents are
     * expanded.
     *
     * @returns true if the node is visible
     */
    public boolean isVisible(TreePath path);


    /**
     *
     * Returns a <code>Rectangle</code> where node specified by <code>path</code>
     * is rendererd.  It returns null if nodes on the <code>path</code> is hidden
     * (i.e. under hidden parent),
     * <p>
     * Note: it returns valid <code>Rectangle</code> even if the specified node
     * is not visible
     *
     * @param path a <code>TreePath</code> specifying a node
     *
     * @return <code>Rectangle</code> to where node is rendered or null
     */
    //    public Rectangle getPathBounds(TreePath path);


    /**
     *
     * Returns a <code>Rectangle</code> where the node specifyed by
     * given <code>row</code> is rendered
     *
     * @param row to be rendered where zero indicates the first row
     *
     * @return <code>Rectangle</code> where node to be rendered
     */
    //    public Rectangle getRowBounds(int row);

    /**
     *
     * Extends all nodes on given <code>path</code> except the last path component
     * and scroll that the node to be displayed.  It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param path a <code>TreePath</code> specifying a node to be displayed
     */
    public void scrollPathToVisible(TreePath path);

    /**
     *
     * Scrolles until the node specified by <code>row</code> to be displayed
     * with minimum amount of scroll to display.
     * It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param row int specifying row to scroll where zero indicates the first row
     */
    public void scrollRowToVisible(int row);


    /**
     *
     * Returns <code>TreePath</code> of the node at given <code>row</code>
     * or null if the <code>row</code> is invisible
     *
     * @param row in int
     *
     * @return <code>TreePath</code> of specified node, or null if
     * <code>row</code> < 0 or <code>row</code> > <code>getRowCount()</code>
    */
    public TreePath getPathForRow(int row);


    /**
     *
     * Returns row displaying node idnetified by <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     * @return int indicating displayed row where zero indicates the first
     * row, -1 if path componets are hidden under shrunken parent
     */
    public int getRowForPath(TreePath path);


    /**
     *
     * Expands and makes visible a node indetified by given <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     */
    public void expandPath(TreePath path);


    /**
     *
     * Expands and makes visible a node at specified <code>row</code>.
     * It does nothing if <code>row</code> < 0 or .<code>row</code> >=
     * <code>getRowCount</code>.
     *
     * @param row int specifying displayed row where zero indicates
     * the first row
     */
    public void expandRow(int row);


    /**
     *
     * Shrink and make visible a node identified by <code>path</code>
     *
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     */
    public void collapsePath(TreePath path);

    /**
     *
     * Shrinnk a node at specified <code>row</code>
     * It does nothing if <code>row</code> < 0 or .<code>row</code> >=
     * <code>getRowCount</code>.
     *
     * @param row int specifying displayed row where zero indicates
     * the first row
     */
    public void collapseRow(int row);


    /**
     *
     * Returns <code>TreePath</code> of a node at specified position
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return a <code>TreePath</code> of a node at specified position
     */
    public TreePath getPathForLocation(int x,
				       int y);


    /**
     *
     * Returns row of a node at specified position
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return row of a node at specified position, where -1 indicates
     * the specified position is out of displayed area
     *
     * @see getClosestRowForLocation(int, int);
    */
    public int getRowForLocation(int x,
				 int y);


    /**
     *
     * Returns <code>TreePath</code> of a node nearest to position specified by
     * <code>x</code>, <code>y</code>.
     * It returns valid <code>TreePath</code> always except when returns null if there is no visible node or no model.
     * To confirm the node is at <code>x</code>, <code>y</code> exactly, examine <code>x</code>, <code>y</code>
     * with margine of the node obtained using <code>getPathBounds(TreePath)</code>
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return <code>TreePath</code> of a node nearest to specified postion, or null
     * if there is no visible node or no model
     *
     * @see getPathForLocation(int, int)
     * @see getPathBounds(TreePath)
     */
    public TreePath getClosestPathForLocation(int x,
					      int y);

    /**
     *
     * Returns row of a node nearest to position specified by
     * <code>x</code>, <code>y</code>.
     * It returns valid row always except when returns -1 if there is no visible node or no model.
     * To confirm the node is at <code>x</code>, <code>y</code> exactly, examine <code>x</code>, <code>y</code>
     * with margine of the node obtained using <code>getRowBounds(int)</code>
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return row of a node nearest to specified postion, or -1
     * if there is no visible node or no model
     *
     * @see getRowForLocation(int, int)
     * @see getRowBounds(int)
     */
    public int getClosestRowForLocation(int x,
					int y);

    /**
     *
     * Returns true if the tree is under editing.
     * Item under edition can be obtaind using  getSelectionPath.
     *
     * @return true if user is editing nodes
     *
     * @see getSelectionPath();
    */
    public boolean isEditing();


    /**
     *
     * Terminates currend editin session.
     * DefaultTreeCellEditor object conservs all edtion on going on cells, which may
     * differ in other implementation.
     * It does nothing if the tree is not under edition.
     * <P>
     * Note:<BR>
     * Use <code>setInvokesStopCellEditing(boolean)</code> to save edition automatically 
     * when a user moved in the tree
     *
     * @return true if edition is on going and currently stopped, or false if
     * tree is not under edition
     */
    public boolean stopEditing();


    /**
     *
     * Cancels current edit session.
     * It does nothing if tree is not under edition
     */
    public void cancelEditing();


    /**
     *
     * Selects and starts edition of a node identified by <code>path</code>.
     * It fails if <code>CellEditor</code> does not allow edtion of specified
     * item.
     *
     * @param path a <code>TreePath</code> identifying a node
     */
    public void startEditingAtPath(TreePath path);


    /**
     *
     * Returns path of the node under edition
     *
     * @return <code>TreePath</code> of the node under edition
     */
    public TreePath getEditingPath();


    /**
     *
     * Sets a <code>TreeSelectionModel</code> of the tree.
     * A null value indicates an empty selection model prohibiting selection
     *
     * @param selectionModel <code>TreeSelectionModel</code> to be used, or null
     * to disable selection
     *
     * @see <code>TreeSelectionModel</code>
    */
    //    public void setSelectionModel(TreeSelectionModel selectionModel);

    /**
     *
     * Returns a <code>TreeSelectionModel</code> of the tree which is always non-null.
     * Set selection model to null to enforce using empty selection model if you want
     * to prohibit selection
     *
     * @return used <code>TreeSelectionModel</code>
     * 
     * @seesetSelectionModel(TreeSelectionModel)
    */
    public TreeSelectionModel getSelectionModel();

    /**
     *
     * Returns a <code>TreePath</code> indicating a path from <code>index0</code>
     * to <code>index1</code>, or null if there is no tree.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     * @return an arraay of <code>TreePath</code> objects, of each for each node between
     * <code>index0</code> and <code>index1</code>, or null if there is no tree
     */
    public TreePath[] getPathBetweenRows(int index0,
					    int index1);

    /**
     *
     * Selects nodes at rows between <code>index0</code> and <code>index1</code>.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void setSelectionInterval(int index0,
				     int index1);

    /**
     *
     * Appends paths at rows between <code>index0</code> and <code>index1</code>
     * to the selection
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void addSelectionInterval(int index0,
				     int index1);


    /**
     *
     * Removes nodes at rows between <code>index0</code> and <code>index1</code>
     * from the selection
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void removeSelectionInterval(int index0,
					int index1);
    

    /**
     *
     * Removes the node identified by given <code>path</code>
     * from the selection
     *
     * @param path a <code>TreePath</code> identifying the node to be removed
     */
    public void removeSelectionPath(TreePath path);


    /**
     *
     * Removes nodes identified by elements of given <code>paths</code>
     * from the selection
     *
     * @param paths an array of  <code>TreePath</code> of which element identifying each node to be removed
     */
    public void removeSelectionPaths(TreePath[] paths);


    /**
     *
     * Removes path at <code>row</code> from current selection
     *
     * @param row specifying the node to be removed
     */
    public void removeSelectionRow(int row);


    /**
     *
     * Removes paths at specified <code>rows</code> from the selection
     *
     * @param rows an array of int specifying paths to be removed where
     * zero value indicates the first row 
     */
    public void removeSelectionRows(int[] rows);


    /**
     *
     * Clear the selection
     */
    public void clearSelection();


    /**
     *
     * Returns true if selected range is empty currently
     *
     * @return true if selection is empty
     */
    public boolean isSelectionEmpty();


    /**
     *
     * Adds a <code>TreeExpansion</code> event listener
     *
     * @param listener a <code>TreeExpansionListener</code> which is to be notified
     * when a tree node is expanded or shrunken (negative expansion)
     */
    public void addTreeExpansionListener(TreeExpansionListener listener);


    /**
     * 
     * Removes a <code>TreeExpansion</code> event listener
     *
     * @param listener a <code>TreeExpansionListener</code> to be removed
     */
    public void removeTreeExpansionListener(TreeExpansionListener listener);


    /**
     *
     * Adds a <code>TreeWillExpand</code> event listener
     *
     * @param listener a <code>TreeWillExpand</code> which is to be notified
     * when a tree node is expanded or shrunken (negative expansion)
     */
    public void addTreeWillExpandListener(TreeWillExpandListener listener);


    /**
     *
     * Removes a <code>TreeWillExpand</code> event listener
     *
     * @param listener a <code>TreeWillExpand</code> to be removed
     */
    public void removeTreeWillExpandListener(TreeWillExpandListener tel);


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying expanded node
     */
    public void fireTreeExpanded(TreePath path);


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying shrunken node
     */
    public void fireTreeCollapsed(TreePath path);


    /**
     *
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to expand
     */
    public void fireTreeWillExpand(TreePath path)
	throws ExpandVetoException;
	    
    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to shrink
     */
    public void fireTreeWillCollapse(TreePath path)
	throws ExpandVetoException;


    /**
     *
     * Adds a <code>TreeSelection</code> event listener
     *
     * @param listener a <code>TreeSelection</code> notified when a node
     * is selected or unselected(negative selection)
     */
    public void addTreeSelectionListener(TreeSelectionListener listener);


    /**
     *
     * Removes a <code>TreeSelection</code> event listener
     *
     * @param listener a <code>TreeSelection</code> notified when a node
     * is selected or unselected(negative selection)
     */
    public void removeTreeSelectionListener(TreeSelectionListener tsl);

    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param event <code>TreeSelectionEvent</code> constructed by 
     * <code>TreeSelectionModel</code> when a node is selected or deselected
     */
    public void fireValueChanged(TreeSelectionEvent event);

    /**
     *
     * Ivoked when a modification in tree is sufficiently large to change
     * size of margine but not so huge to remove expanded node set (e.g.
     * when node is expanded, shrunken, or nodes are inserted into tree).
     * It is unnecessary to call this method; UI does when necessary
     */
    public void treeDidChange();

    /**
     *
     * Sets number of displayed rows.
     * This methods works only if the receiver is contained in JScrollPane
     * to adjust size of the scroll pane
     *
     * @param rows number of rows to display
     */
    public void setVisibleRowCount(int rows);


    /**
     *
     * Returns number of displayed rows in display area
     *
     * @return number of displayd rows 
     */
    public int getVisibleRowCount();


    /**
     *
     * Returns appropriate display size of JTree of which height is determined
     * by <code>getVisibleRowCount</code> and width is current appropriate width.
     *
     * @return <code>Dimension</code> object holding appropriate size
     */
    public Dimension getPreferredScrollableViewportSize();


    /**
     *
     * Returns increment when scrolled which is determined by height of the first partially-displayed row,
     * or height of the next hidden line at scrolling direction.
     *
     * @param visibleRect visible area in the view port
     * @param orientation VERTICAL or HORIZONTAL
     * @param direction less than zero for upward or leftword scroll,
     * or larger than zero for downword or rightword scroll
     *
     * @return "unit" increment to scroll in specified direction
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect,
					  int orientation,
					  int direction);

    /**
     *
     * Returns "block" increment which is determined by height or widht of <code>visibleRect</code>
     * depending on <code>orientation</code>.
     *
     * @param avisibleRect visbile vew area in view port
     * @param orientation <code>VERTICAL</code> or <code>HORIZONTAL</code>
     * @param direction less than zero for upward or leftword scroll,
     * or larger than zero for downword or rightword scroll
     *
     * @return "block" increment to scroll in specified direction
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect,
					   int orientation,
					   int direction);
    
    /**
     *
     * Returns false to tell width of view port does not restrict width of table,
     * i.e. keeps tree from being smaller than view port, except when appropriate width 
     * for the tree is narrower than width of the view port
     *
     * @return false
     *
     */
    public boolean getScrollableTracksViewportWidth();


    /**
     *
     * Returns false to tell height of view port does not restrict height of table,
     * i.e. keeps tree from being smaller than view port, except when appropriate height
     * for the tree is shorter than height of the view port
     *
     * @return false
     *
     */
    public boolean getScrollableTracksViewportHeight();


    /**
     *
     * Sets expansion state of this JTree.  If <code>state</code> is ture,
     * <code>path</code> and its all parents are marked to epand.
     * If <code>state</code> is false, all parents of <code>path</code> is
     * displayed by <code>EXPANDED</code> but <code>path</code> itself is
     * shrunken.
     * <P>
     * It fails if denied by <code>TreeWillExpandListener</code>
     */
    public void setExpandedState(TreePath path,
				    boolean state);


    /**
     *
     * Returns <code>Enumeration</code> of descendant nodes of <code>parent</code>
     */
    public Enumeration getDescendantToggledPaths(TreePath parent);


    /**
     *
     * Remove expanded descending nodes of <code>TreePath</code>s in
     * <code>toRemove</code>
     */
    public void removeDescendantToggledPaths(Enumeration toRemove);


    /**
     *
     * Clears cache of toggled <code>TreePath</code>s witout sending
     * <code>TreeExpansionListener</code> event
    */
    public void clearToggledPaths();


    /**
     *
     * Constructs and returns an instance of <code>TreeModelHandler</code>
     * which updates expanded status when  <code>TreeModel</code> is modified
     *
     */
    public TreeModelListener createTreeModelListener();


    /**
     *
     * Remove all selected discending nodes of <code>path</code>.
     * If <code>includePath</code> is true and <code>path</code> is selected,
     * it is removed from selection.
     *
     * @return true if discending nodes are removed
     *
     * @since 1.3
     */
    public boolean removeDescendantSelectedPaths(TreePath path,
						    boolean includePath);


    /**
     *
     * Returns <code>String</code> expression of this <code>JTree</code>.
     * It exists only for debugging and contents and format of returned <code>String</code>
     * may depend on implementation.  Returned <code>String</code> may be empty but never
     * be null.
     *
     * @return <code>String</code> expression of <code>JTree</code>
     */
    public String paramString();


    /**
     * Returns <code>AccessibleContext</code> relating to this <code>JTree</code>
     * which will be an instance of <code>AccessibleJTree</code> newly constructed
     * depending on necessity.
     *.
     * @return <code>AccessibleJTree</code> works as <code>AccessibleContext</code>
     * of this <code>JTree</code>
     */
    //    public AccessibleContext getAccessibleContext();

}
