/*
 * TextTreeModel.java:  a Text expression of a TreeModel
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: TextTreeModel.java,v 1.15 2002/10/08 16:29:22 nozomi Exp $
 * $Log: TextTreeModel.java,v $
 * Revision 1.15  2002/10/08 16:29:22  nozomi
 * add getRowForObject()
 *
 * Revision 1.14  2002/08/22 22:59:03  nozomi
 * remove unused lines; introduce getLastDisplayedDiscendentNode()
 *
 * Revision 1.13  2002/08/22 02:32:11  nozomi
 * fix child multiplication bug in treeNodesChanged()
 *
 * Revision 1.12  2002/08/09 05:25:15  nozomi
 * fix insertion bug
 *
 * Revision 1.11  2002/08/06 06:16:32  nozomi
 * another, incomplete code to fix addition-to-hidden-node bug
 *
 * Revision 1.10  2002/08/06 05:35:01  nozomi
 * fix addition-to-hidden-node bug
 *
 * Revision 1.9  2002/06/25 10:35:39  t.okada
 * CastException in treeNodesRemoved() method is corrected.
 *
 * Revision 1.8  2002/06/09 12:39:19  nozomi
 * change TreeModelEvent handling
 *
 * Revision 1.7  2002/06/03 04:34:57  nozomi
 * fix treeStructureChanged()
 *
 * Revision 1.6  2002/05/27 00:02:26  nozomi
 * implements treeNodesRemoved()
 *
 * Revision 1.5  2002/03/10 08:53:31  nozomi
 * fix multiple insertion bug
 *
 * Revision 1.4  2002/03/10 06:56:21  nozomi
 * insertion of higher node support
 *
 * Revision 1.3  2002/03/09 23:33:04  nozomi
 * add event handling
 *
 * Revision 1.2  2002/03/08 23:09:29  nozomi
 * modification in even handling
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.awt.tree;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import java.awt.event.MouseEvent;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;

import jp.kyasu.awt.DefaultTextListModel;

import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextBuffer;

import org.nomencurator.awt.tree.DefaultMutableTreeNode;
import org.nomencurator.awt.tree.DefaultTreeModel;
import org.nomencurator.awt.tree.TreeNode;

import org.nomencurator.awt.tree.event.ExpandVetoException;
import org.nomencurator.awt.tree.event.TreeExpansionEvent;
import org.nomencurator.awt.tree.event.TreeExpansionListener;
import org.nomencurator.awt.tree.event.TreeSelectionEvent;
import org.nomencurator.awt.tree.event.TreeSelectionListener;
import org.nomencurator.awt.tree.event.TreeWillExpandListener;

import org.nomencurator.graphics.RichTextable;
import org.nomencurator.graphics.Textable;

import org.nomencurator.util.tree.MutableTreeNode;
import org.nomencurator.util.tree.TreeModel;
import org.nomencurator.util.tree.TreePath;

import org.nomencurator.util.tree.event.TreeModelEvent;
import org.nomencurator.util.tree.event.TreeModelListener;

/**
 * An graphical expression of tree using <code>List</code>.
 * <p>
 * Since a tree can be represented by a indented list graphically,
 * a TreeModel can be implemented as a subclass ListModel which
 * can be used as a model in Tree widget class extending List class
 * of KFC.
 *
 * @version 	08 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class TextTreeModel
    extends DefaultTextListModel
    implements TreeModel,
	       TreeModelListener,
	       TreeExpansionListener,
	       TreeSelectionListener,
	       TreeWillExpandListener
{

    protected DefaultTreeModel treeModel;

    protected TreeNode root;

    transient protected Hashtable expandedStatus;

    transient protected Vector treeExpansionListeners;
    transient protected Vector treeWillExpandListeners;

    transient protected Vector treeSelectionListeners;

    private static int[] singleColumn;

    transient Vector visiblePaths;
    transient Hashtable rowOfPath;
    transient Hashtable rowOfNode;
    transient Hashtable rowOfObject;

    static {
	singleColumn = new int[1];
	singleColumn[0] = 1;
    }

    //Constructors
    /**
     * Constructs a tree 
     *
     */
    public TextTreeModel()
    {
	this(new DefaultMutableTreeNode());
    }

    /**
     * Constructs a tree of which any node may have child node.
     *
     * @param root TreeNode object to be root of the tree.
     *
     * @see DefaultTreeModel(TreeNode, boolean)
     */
    
    public TextTreeModel(TreeNode root)
    {
	this(root, false);
    }

    /**
     * Constructs a tree under the root <code>node</code>
     * specifying whether any node may have children nodes
     * or only specified node may have children nodes. 
     * 
     * @param node root node of the tree
     * @param allowance false if any child node may have 
     * child node, or true if only specified node may have children.
     *
     * @see #asksAllowsChildren
     */

    public TextTreeModel(TreeNode node,
		     boolean allowance)
    {
	this(new DefaultTreeModel(node, allowance));
    }
    

    /**
     * Constructs a tree under the root <code>node</code>
     * specifying whether any node may have children nodes
     * or only specified node may have children nodes. 
     * 
     * @param node root node of the tree
     * @param allowance false if any child node may have 
     * child node, or true if only specified node may have children.
     *
     * @see #asksAllowsChildren
     */

    public TextTreeModel(TreeModel model)
    {
	this(model, RichTextStyle.DEFAULT_LIST_STYLE);
    }
    
    /**
     * Constructs a single column tree model with the specified rich text
     * style.
     *
     * @param richTextStyle the rich text style.
     */
    public TextTreeModel(RichTextStyle ritchTextStyle)
    {
	this(new DefaultTreeModel(new DefaultMutableTreeNode()),
	     singleColumn, ritchTextStyle);
    }
    
    /**
     * Constructs a single column text list model with the specified rich text
     * style.
     *
     * @param root TreeNode object to be root of the tree.
     * @param richTextStyle the rich text style.
     */
    public TextTreeModel(TreeModel model,
			 RichTextStyle ritchTextStyle)
    {
	super(ritchTextStyle);
	initialize(model);
    }
    
    /**
     * Constructs a text list model with the specified column widths.
     *
     * @param colWidths the column widths.
     */
    public TextTreeModel(int columnWidths[])
    {
	this(new DefaultTreeModel(new DefaultMutableTreeNode()), columnWidths);
    }

    /**
     * Constructs a text list model with the specified column widths.
     *
     * @param colWidths the column widths.
     */
    public TextTreeModel(TreeModel model,
			 int columnWidths[])
    {
	this(model, columnWidths.length, columnWidths);
    }

    /**
     * Constructs a text list model with the specified column widths.
     *
     * @param colWidths the column widths.
     */
    public TextTreeModel(TreeModel model,
			 int columnWidths[],
			 RichTextStyle richTextStyle)
    {
	this(model, columnWidths.length, columnWidths, richTextStyle);
    }

    /**
     * Constructs a text list model with the specified number of columns
     * and column widths.
     *
     * @param columns   the number of columns.
     * @param colWidths the column widths.
     */
    public TextTreeModel(TreeModel model,
			 int columns,
			 int colWidths[])
    {
	this(model, columns, colWidths, RichTextStyle.DEFAULT_LIST_STYLE);
    }

    /**
     * Constructs a text list model with the specified number of columns,
     * column widths, and rich text style.
     *
     * @param columns       the number of columns.
     * @param colWidths     the column widths.
     * @param richTextStyle the rich text style.
     */
    public TextTreeModel(TreeModel model,
			 int columns, 
			 int colWidths[],
			 RichTextStyle richTextStyle)
    {
	super(columns, colWidths, richTextStyle);
	initialize(model);
    }

    protected void initialize(TreeModel model)
    {
	visiblePaths   = new Vector();
	rowOfPath      = new Hashtable();
	rowOfNode      = new Hashtable();
	rowOfObject    = new Hashtable();
	expandedStatus = new Hashtable();
	treeExpansionListeners = new Vector();
	treeWillExpandListeners = new Vector();
	treeSelectionListeners = new Vector();

	treeModel = (DefaultTreeModel)model;

	if(treeModel == null)
	    return;

	root = (TreeNode)(treeModel.getRoot());

	int index = getItemCount();
	replaceItems(index, index, (TreeNode)treeModel.getRoot());

	treeModel.addTreeModelListener(this);
	treeModel.addTreeExpansionListener(this);
	treeModel.addTreeWillExpandListener(this);
	//	treeModel.addTreeSelectionLitener(this);
    }

    public void replaceItems(int begin, int end, TreeNode node)
    {
	if(!((DefaultMutableTreeNode)node).isRoot() && 
	   !((TreeNode)node.getParent()).isExpanded())
	    return;

	TreePath path = null;
	for(int i = begin; i < end; i++) {
	  path = (TreePath)visiblePaths.elementAt(begin);
	  rowOfPath.remove(path);
	  TreeNode lastNode = (TreeNode)path.getLastPathComponent();
	  rowOfNode.remove(lastNode);
	  rowOfObject.remove(lastNode.getUserObject());
	  visiblePaths.removeElementAt(begin);
	}

	Text[] texts = toTexts(node, begin);
	replaceItems(begin, end, texts);

	Integer row;
	for(int i = begin; i < visiblePaths.size(); i++) {
	  path = (TreePath)visiblePaths.elementAt(i);
	  row = new Integer(i);
	  rowOfPath.put(path, row);
	  TreeNode lastNode = (TreeNode)path.getLastPathComponent();
	  rowOfNode.put(lastNode, row);
	  rowOfObject.put(lastNode.getUserObject(), row);
	}
    }

    public void removeItem(int begin)
    {
	removeItems(begin, begin + 1);
    }

    public void removeItems(int begin, int end)
    {
	replaceItems(begin, end,
		     new String[0][getColumnCount()]);
	TreePath path;
	for(int i = end; i > begin; ) {
	    path = (TreePath)visiblePaths.elementAt(--i);
	    visiblePaths.removeElementAt(i);
	    rowOfPath.remove(path);
	    rowOfNode.remove(path.getLastPathComponent());
	    rowOfObject.remove(((TreeNode)path.getLastPathComponent()).getUserObject());
	}
	Integer row;
	for(int i = begin; i < visiblePaths.size(); i++) {
	  path = (TreePath)visiblePaths.elementAt(i);
	  row = new Integer(i);
	  rowOfPath.put(path, row);
	  rowOfNode.put(path.getLastPathComponent(), row);
	  rowOfObject.put(((TreeNode)path.getLastPathComponent()).getUserObject(), row);
	}
    }


    public void initializeContents()
    {
    }
    
    protected Text toText(TreeNode node)
    {
	TextBuffer buffer = new TextBuffer();
	toTextBuffer(buffer, node);
	return buffer.toText();
    }

    protected Text[] toTexts(TreeNode node)
    {
	return toTexts(node, -1);
    }

    protected Text[] toTexts(TreeNode node, int startLine)
    {
	Vector textVector = new Vector();
	toTextVector(textVector, node, startLine);
	Text[] texts = new Text[textVector.size()];
	Enumeration textEnum = textVector.elements();
	int i = 0;
	while(textEnum.hasMoreElements())
	    texts[i++] = (Text)(textEnum.nextElement());
	return texts;
    }

    protected int toTextVector(Vector texts, TreeNode node)
    {
	return toTextVector(texts, node, -1);
    }

    protected int toTextVector(Vector texts, TreeNode node, int startLine)
    {
	TextBuffer buffer = new TextBuffer();
	boolean expanded = toTextBuffer(buffer, node, startLine);
	texts.addElement(buffer.toText());
	if(startLine > -1)
	    startLine++;
	buffer = null;
	if(!expanded)
	    return startLine;
	Enumeration children = node.children();
	while(children.hasMoreElements()) {
	    TreeNode tn = (TreeNode)(children.nextElement());
	    startLine = toTextVector(texts, tn, startLine);
	}
	return startLine;
    }

    protected boolean toTextBuffer(TextBuffer buffer, TreeNode node)
    {
	return toTextBuffer(buffer, node, -1);
    }

    protected boolean toTextBuffer(TextBuffer buffer, TreeNode node, int startLine)
    {
	TreePath path = ((DefaultTreeModel)getTreeModel()).getTreePath(node); 
	//	int depth = node.getDepth();
	int level = ((DefaultMutableTreeNode)node).getLevel();
	for(int i = 0; i < level; i++)
	    buffer.append("\t");

	boolean expanded = false;
	Object expandedState =
	    expandedStatus.get(path);
	boolean leaf = node.isLeaf();
	if(expandedState == null && ! leaf) {
	    //if(expandedState == null) {
	    expandedStatus.put(path, Boolean.TRUE);
	    expandedState
		= expandedStatus.get(path);
	}

	if(leaf) 
	    buffer.append(' ');
	else {
	    if(((Boolean)expandedState).booleanValue()) {
		expanded = true;
		buffer.append('-');
	    }
	    else {
		buffer.append('+');
	    }
	}

	Object value = node;
	if(value instanceof DefaultMutableTreeNode) {
	    value = ((DefaultMutableTreeNode)value).getUserObject();
	    if(value == null)
		value = node;
	}

	Text expression = null;
	if(value instanceof Text)
	    expression = (Text)value;
	else if (value instanceof String)
	    expression = new Text((String)value);
	else if (value instanceof Textable)
	    expression = ((Textable)value).toText();
	else if (value.toString() != null)
	    expression = new Text(value.toString());
	else
	    expression = new Text("");

	buffer.append(expression);	

	if(startLine > -1) {
	    visiblePaths.add(startLine, path);
	    Integer row = new Integer(startLine);
	    rowOfNode.put(node, row);
	    rowOfPath.put(path, row);
	    rowOfObject.put(node.getUserObject(), row);
	    startLine++;
	}

	return expanded;
    }

    public TreeModel getTreeModel()
    {
	return treeModel;
    }

    public void setTreeModel(TreeModel model)
    {
	if(treeModel == model)
	    return;

	if(treeModel != null)
	    treeModel.removeTreeModelListener(this);
	treeModel = (DefaultTreeModel)model;
	treeModel.addTreeModelListener(this);
    }

    //TreeModelListener interfaces
    /**
     * Invoked when a node or a set of siblings was modified.
     * Neither location of the node in the tree nor array of its child nodes 
     * was not modified but other attributes were changed which may arrect
     * tree's representation.  For eample when file name was changed but 
     * its position in file system tree was not altered.
     * <p>
     * Vale of childIndices and children will be null to tell change of root.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes.
     */
    public void treeNodesChanged(TreeModelEvent event)
    {
	TreeNode rootNode = (TreeNode)((TreeModel)(event.getSource())).getRoot();
	TreePath path = event.getTreePath();
	TreeNode lastNode = (TreeNode)path.getLastPathComponent();
	int index = getRowForPath(path);
	/*
	if(index < 0) {
	    TreeNode parent = (TreeNode)lastNode.getParent();
	    ((org.nomencurator.awt.tree.TreeNode)lastNode.getParent()).setExpanded(true);
	    lastNode.setExpanded(true);
	    index = getRowForPath(path);
	}
	*/
	if(index < 0)
	    return; //?
	
	int end = getRowForNode(getLastDisplayedDiscendentNode(lastNode)) + 1;

	if(lastNode == rootNode)
	    end = getItemCount();

	replaceItems(index, end, lastNode);
	return;
    }

    /**
     * Returns <CODE>TreeNode</CODE> representing the displayed last node
     * in the subtree under <CODE>node</CODE>
     *
     * @param node <CODE>TreeNode</CODE> representing the root of subtree
     * to be examined
     *
     * @return TreeNode representing the displayed last node
     */
    protected TreeNode getLastDisplayedDiscendentNode(TreeNode node)
    {
	while(node.isExpanded() && !node.isLeaf())
	    node = (TreeNode)(node.getChildAt(node.getChildCount() - 1));
	return node;
    }

    /**
     * Invoked after insertion of a node.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */

    public void treeNodesInserted(TreeModelEvent event)
    {
	TreePath parent = event.getTreePath();
	int[]    indices = event.getChildIndices();
	Object[] children = event.getChildren();

	if(indices == null || children == null) {
	    replaceItems(0, 0, (DefaultMutableTreeNode)parent.getPathComponent(0));
	    return;
	}

	Object state = expandedStatus.get(parent);

	if(state == null && ! ((TreeNode)parent.getLastPathComponent()).isLeaf()) {
	    expandedStatus.put(parent, Boolean.TRUE);
	    state = expandedStatus.get(parent);
	}

	if(state == null || !((Boolean)state).booleanValue())
	    return;


	int parentRow;
	int index;
	for(int i = 0; i < indices.length; i++ ) {
	    parentRow = getRowForPath(parent);
	    if(parentRow == -1)
		parentRow = getRowForNode((TreeNode)(parent.getLastPathComponent()));
	    index = indices[i];

	    //the first child
	    if(index == 0)
		parentRow++;

	    //second or later children
	    else {
		TreeNode node = getNodeForPath(parent);

		int childrens = node.getChildCount();

		//insert somwhere between first and last children
		if(childrens > index + 1) {
		    parentRow = getRowForNode(node.getChildAt(index + 1));
		}

		//the last child
		else {
		    TreePath path = parent;
		    //		    parent = parent.getParentPath();

		    //root node
		    if(parent.getParentPath() == null) {
			parentRow = getItemCount();
		    }
		    //not root
		    else {
			node = (TreeNode)(node.getChildAt(index - 1));
			parentRow = getRowForNode(getLastDisplayedDiscendentNode(node)) + 1;
		    }
		}
	    }


	    index = parentRow;

	    replaceItems(index, index, (DefaultMutableTreeNode)children[i]);
	}
    }

    /**
     * Invoked after removal of a node from a tree.
     * When a sbutree removed a tree, this method is ivoked only once for
     * the root node of the removed subtree but never each sibling nodes removed.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */
    public void treeNodesRemoved(TreeModelEvent event)
    {
	DefaultTreeModel source = (DefaultTreeModel)event.getSource();
	TreePath parent = event.getTreePath();
	int[]    indices = event.getChildIndices();
	Object[] children = event.getChildren();

	if(indices == null || children == null) {
	    replaceItems(0, 0, (DefaultMutableTreeNode)parent.getPathComponent(0));
	    return;
	}

	Object state = expandedStatus.get(parent);

	if(state == null && ! ((TreeNode)parent.getLastPathComponent()).isLeaf()) {
	    expandedStatus.put(parent, Boolean.TRUE);
	    state = expandedStatus.get(parent);
	}

	if(state == null || !((Boolean)state).booleanValue())
	    return;


	int parentRow;
	int index;
	for(int i = indices.length; i > 0; ) {
	    --i;
	    parentRow = getRowForPath(parent);
	    if(parentRow == -1)
		parentRow = getRowForNode((TreeNode)(parent.getLastPathComponent()));
	    index = indices[i];

	    //the first child
	    if(index == 0)
		parentRow++;

	    //second or later children
	    else {
		TreeNode node = getNodeForPath(parent);

		int childrens = node.getChildCount();

		//insert somwhere between first and last children
		if(childrens > index + 1) {
		    parentRow = getRowForNode(node.getChildAt(index + 1));
		}

		//the last child
		else {
		    TreePath path = parent;
		    parent = parent.getParentPath();

		    //root node
		    if(parent == null) {
			parentRow = getItemCount();
		    }
		    //not root
		    else {
			parentRow = getRowForPath(parent);
			if(parentRow < 1) {
			    parentRow = getItemCount();
			}
			else{
			    parentRow--;
			}
		    }
		}
	    }


	    index = parentRow;

	    replaceItems(index, index + 1,
			 new String[0][getColumnCount()]);
	}
    }

    /**
     * Invoked when tree structure under a node modified drastically.
     * When the path length given by <code>TreeModelEvent#getPath()</code> is one
     * and the first element is not the current root node, the first element is 
     * the root of the new tree.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * null.
     */
    public void treeStructureChanged(TreeModelEvent event)
    {
	Object[] path   = event.getPath();

	if(path.length == 1 && /*root */getRoot() != path[0]) {
	    setRoot((TreeNode)path[0]);
	}
    }


    // DefaultTreeModel wrapper including 
    // org.nomencurator.util.tree.TreeModel interface
    /*
     * Sets whether or not leafness of TreeNodes by passing getAllowsChildren()
     * or isLeaf().
     * Sets true to send getAllowsChildren(), or false to send isLeaf().
    */

    public void setAsksAllowsChildren(boolean allowance)
    {
	((DefaultTreeModel)treeModel).setAsksAllowsChildren(allowance);
    }


    /**
     * Tells leafness determination method
     *
     * @return true if a leaf node is only node prohibited to have child note,
     * or false if any node without child node (even if it allowed to have children) 
     * are leaf nodes.
     *
     * @see #asksAllowsChildren
     */
    public boolean asksAllowsChildren()
    {
	return ((DefaultTreeModel)treeModel).asksAllowsChildren();
    }


    /**
     * Sets <code>node</code> as root node  of the tree.
     *
     * @exception IllegalArgumentException if <code>node</code> is null
     */

    public void setRoot(TreeNode node)
    {
	if(node == root/*getRoot()*/)
	    return;
	root = node;
	//treeModel.setRoot(node);
	//	try{throw new Error();}catch(Error e){e.printStackTrace();}
	//	((DefaultTreeModel)treeModel).setRoot(node);
	replaceItems(0, getItemCount(), node);
    }

    /**
     * Gets root node of the tree.
     * It returns null only if there is no node in the tree (how it can happen?)
     *
     * @return root of the tree
     *
     */
    public Object getRoot()
    {
	return ((DefaultTreeModel)treeModel).getRoot();
    }

    /**
     * Returns index of <code>child</code> node of <code>parent</code> node
     *
     */

    public int getIndexOfChild(Object parent,
			       Object child)
    {
	return ((DefaultTreeModel)treeModel).getIndexOfChild(parent, child);
    }


    /**
     * Returns <code>index</code>th child node of <code>parent</code>.
     * The <code>parent</code> must be node obtained from this data source.
     * If <code>index</code> is out of valid range (i.e. index < 0 || 
     * index >= getChildCount(parent)) it does not return null.
     *
     * @param parent a node in this tree obtained from this data source
     *
     * @return indexth child of parent
     */

    public Object getChild(Object parent,
			   int index)
    {
	return ((DefaultTreeModel)treeModel).getChild(parent, index);
    }


    /**
     * Returns number of children of <code>parent</code> which must be obtaind
     * from this data source previously.  If <code>parent</code> is a leaf or
     * doesn't have a children node, returns 0.
     *
     * @param parent a node in this tree previously obtained from this data source
     * 
     * @return number of <code>parent</code>'s child nodes
     */

    public int getChildCount(Object parent)
    {
	return ((DefaultTreeModel)treeModel).getChildCount(parent);
    }

    /**
     * Returns wheter specifed <code>node</code> is a leaf node
     * by using method depending on value of asksAllowsChildren. 
     *
     * @param node to be determined
     *
     * @return true if the node is a leaf node
     *
     * @see #asksAllowsChildren
     * @see TreeModel#isLeaf(java.lang.Object)
    */
    public boolean isLeaf(Object node)
    {
	return ((DefaultTreeModel)treeModel).isLeaf(node);
    }


    /**
     * Call this method when you modified TreeNode on which this model
     * depends.  The model will notify modification to its all listeners.
     */

    public void reload()
    {
	((DefaultTreeModel)treeModel).reload();
    }


    /**
     * Sets <code>object</code> to the TreeNode identified by <code>path</code>
     * and post node chnaged event.
     *
     * @param path path to the node
     * @param object to be set as user object
     */
    public void valueForPathChanged(TreePath path,
				    Object object)
    {
	((DefaultTreeModel)treeModel).valueForPathChanged(path, object);
    }

    /**
     * Inserts <code>child</code> node as <code>index</code>th child of
     * <code>parent</code> node, and ivoke nodesWereInserted.
     *
     * @param child node to be inserted
     * @param parent node to have inderted child node
     * @parem index place where the node to be inserted
     *
     */
    public void insertNodeInto(MutableTreeNode child,
			       MutableTreeNode parent,
			       int index)
    {
	((DefaultTreeModel)treeModel).insertNodeInto(child, parent, index);
    }

    /**
     * Remove <code>node</code> from its parent and send
     * <code>nodesWereRemoved</code> message to crate appropriate event.
     *
     * @param node a node to be removed from the tree
     *
     * @exception IllegalArgumentException if the node without parent node
     */
    public void removeNodeFromParent(MutableTreeNode node)
    {
	((DefaultTreeModel)treeModel).removeNodeFromParent(node);
    }

    /**
     * Invoke after changing tree representaion
     * 
     * @param  node
     */
    public void nodeChanged(TreeNode node)
    {
	((DefaultTreeModel)treeModel).nodeChanged(node);
    }

    /**
     * Call this method when you modified a TreeNode on which this TreeModel
     * depends.  The model tells all listeners that a tree under <code>node</code>
     * has been modified.
     *
     */
    public void reload(TreeNode node)
    {
	((DefaultTreeModel)treeModel).reload(node);
    }


    /**
     * Call this method when you inserted TreeNodes as children of <code>node</node>
     * at positions specified by <code>indices</code> which must be sorted in 
     * ascending order.
     *
     * @param node parent node of inserted nodes
     * @param indices where nodes were insertde
     */
     public void nodesWereInserted(TreeNode node,
				   int[] indices)
    {
	((DefaultTreeModel)treeModel).nodesWereInserted(node, indices);
    }


    /**
     * Invoke this method after removal of nodes from <code>node</code>s child list
     * where <code>indices</code> is an array of removed nodes' indices in ascending order,
     * and <code>removed</code> is an array of removed nodes.
     * 
     *
     */
    public void nodesWereRemoved(TreeNode node,
				 int[] indices,
				 Object[] removed)
    {
	((DefaultTreeModel)treeModel).nodesWereRemoved(node, indices, removed);
    }

    /**
     * Invoke this method after changing tree's reprenstaion under nodes
     * specified by <code>indices</code> of a parent <code>node</code>
     */
     public void nodesChanged(TreeNode node,
			      int[] indices)
    {
	((DefaultTreeModel)treeModel).nodesChanged(node, indices);
    }


    /**
     *
     * Invoke when tree structure changed recursively
     *
     */

    public void nodeStructureChanged(TreeNode node)
    {
	((DefaultTreeModel)treeModel).nodeStructureChanged(node);
    }


    /**
     * Builds a path from the parent to the root node where the original <code>node</code>
     * is the last element of returned array.  Length of the array indicates depth of the
     * <code>node</code> in the tree.
     *
     * @param node TreeNode to get its path from the root
     * @return an array of TreeNodes inidicating path from root to the given node
     */
    public TreeNode[] getPathToRoot(org.nomencurator.util.tree.TreeNode node)
    {
	org.nomencurator.util.tree.TreeNode[] nodes = 
	    ((DefaultTreeModel)treeModel).getPathToRoot(node);

	TreeNode[] treeNodes = new TreeNode[nodes.length];
	for(int i = 0; i < nodes.length; i++)
	    treeNodes[i] = (TreeNode)nodes[i];

	nodes = null;

	return treeNodes;
    }


    /**
     * Adds a TreeModelListener
     *
     * @param listener a listener of TreeModelListener to be add
     */
    public void addTreeModelListener(TreeModelListener listener)
    {
	((DefaultTreeModel)treeModel).addTreeModelListener(listener);
    }


    /**
     * Revmoves a TreeModelListener from the listeners list
     *
     * @param listener a listener of TreeModelListener to be removed
     */
    public void removeTreeModelListener(TreeModelListener listener)
    {
	((DefaultTreeModel)treeModel).removeTreeModelListener(listener);
    }

    /**
     * Returns EventListers of specified <code>listenerType</code>
     *
     * @returns all objects recieving specified <code>listenerType</code> event
     *
     * @since 1.3
     */

    /* not yet implemented....is EventListener good?
    public EventListener[] getListeners(Class listenerType)
    {
	return ((DefaultTreeModel)treeModel).getListeners(listenerType);
    }
    */

    // Serialization support.  TBI
    /*
    private void writeObject(ObjectOutputStream s) throws IOException {
    }

    private void readObject(ObjectInputStream s) 
        throws IOException, ClassNotFoundException {
    }
    */


    //Methods from JTree
    /**
     *
     * Returns true if the node specifyed by <code>path</code> has ever been expanded
     *
     * @return true if the path has been expanded
     */
    public boolean hasBeenExpanded(TreePath path)
    {
	return (path != null && expandedStatus.get(path) != null);
    }

    /**
     *
     * Returns an <code>Enumeration</code> consisted from descending nodes
     * of currently expanded path <code>parent</code>, or null if <code>parent</code>
     * is not expanded.  If nodes in the path was expanded or shrunk while iteration
     * with returnd <code>Enumeration</code>, this methods may not returns all expanded 
     * path or may return no longer expanded path.
     *
     * @param parent path to be examined
     *
     * @return <code>Enumeration</code> of descending nodes of <code>parent</code> or
     * null if <code>parent</code> is not expanded 
     */
    public Enumeration getExpandedDescendants(TreePath parent, TreeView treeView)
    {
	//returns null if parent is null or parent is not expanded
	if(!isExpanded(parent)) 
	    return null;

	//get nodes in status table
	Enumeration keys = expandedStatus.keys();
	Vector elements  = null;

	if(keys != null) {
	    while(keys.hasMoreElements()) {
		TreePath path = (TreePath)(keys.nextElement());
		if(path != parent) {
		    Boolean  status = (Boolean)(expandedStatus.get(path));
		    
		    if(status != null &&
		       status.booleanValue() &&
		       parent.isDescendant(path) &&
		       treeView.isVisible(path)
		       ) {
			if(elements == null)
			    elements = new Vector();
			elements.addElement(path);
		    }
		}
	    }
	}

	if(elements == null)
	    return DefaultMutableTreeNode.EMPTY_ENUMERATION;

	return elements.elements();
    }

    /**
     *
     * Returns true if a node specifyied by <code>path</code> is expanded currently.
     *
     * @param path a <code>TreePath</code> specifying node to be examine
     *
     * @return false if any node on the <code>path</code> is shrunken or true if all nodes on the 
     * <code>path</code> is exapanded
     */
    public boolean isExpanded(TreePath path)
    {
	if(path == null)
	    return false;

	Boolean status = (Boolean)expandedStatus.get(path);

	//false if the path is not contained in the tree 
	//or the spechified node is not expanded
	if(status == null || ! status.booleanValue())
	    return false;

	//Confirm its parents also epxanded...

	TreePath parent = path.getParentPath();

	//The node is the root
	if(parent == null)
	    return true;

	//Check expanded status recursively
	return isExpanded(parent);
    }

    /**
     *
     * Returns true if a node specified by <code>path</code> is shrunken.
     * It returns false if nodes on the <code>path</code> is not displayed
     *
     * @param path a <code>TreePath</code> to be examined
     *
     * @returns ture if nods on the <code>path</code> is shrunken, or false
     * if all nodes on the <code>path</code> is expanded
     */
    public boolean isCollapsed(TreePath path)
    {
	return !isExpanded(path);
    }

    /**
     *
     * Sets expansion state of this JTree.  If <code>state</code> is ture,
     * <code>path</code> and its all parents are marked to epand.
     * If <code>state</code> is false, all parents of <code>path</code> is
     * displayed by <code>EXPANDED</code> but <code>path</code> itself is
     * shrunken.
     * <P>
     * It fails if denied by <code>TreeWillExpandListener</code>
     */
    public void setExpandedState(TreePath path,
				    boolean state)
    {
	//Return if given path is null
	if(path == null)
	    return;

	Object currentState = expandedStatus.get(path);
	//Do nothing if given state is already set
	if(currentState != null &&
	   state == ((Boolean)currentState).booleanValue())
	    return;

	//Expanding a node
	if(state) {
	    Stack nodes = new Stack();
	    TreePath parent = path;

	    while(parent != null) {
		if(isExpanded(parent)) {
		    //if already expanded, unnecessary to climb up to the root node
		    parent = null;
		}
		else {
		    nodes.push(parent);
		    parent = parent.getParentPath();
		}
	    }

	    while(!nodes.isEmpty()) {
		parent = (TreePath)nodes.pop();
		if(!isExpanded(parent)) {
		    try {
			fireTreeWillExpand(parent);
		    }
		    catch (ExpandVetoException e) {
			return;
		    }
		    TreeNode toBeExpanded
			= (TreeNode)(parent.getLastPathComponent());
		    int begin = getRowForNode(toBeExpanded);
		    expandedStatus.put(parent, Boolean.TRUE);

		    if(!toBeExpanded.isLeaf()) {
			begin++;
			int end = getItemCount();
			if(begin  < end)
			    end = begin;
			begin--;
			//			removeItem(begin);
			replaceItems(begin, end, toBeExpanded);
			//			removeItem(begin + toBeExpanded.getChildCount());

		    }
		    fireTreeExpanded(parent);
		}
	    }

	    
	}
	//...or shrinking a node
	else {
	    try {
		fireTreeWillCollapse(path);
	    }
	    catch (ExpandVetoException e) {
		return;
	    }

	    expandedStatus.put(path, Boolean.FALSE);


	    TreeNode toBeClosed
		= (TreeNode)(path.getLastPathComponent());

	    org.nomencurator.util.tree.TreeNode node = toBeClosed;
	    int closingRow = getRowForNode(toBeClosed);

	    if(!node.isLeaf()) {
		int start = getRowForNode(node.getChildAt(0)) - 1;
		if(start < 0)
		    start = 0;
		org.nomencurator.util.tree.TreeNode lastChild = node.getChildAt(node.getChildCount() - 1);
		while(!lastChild.isLeaf()) {
		    //while(!node.isLeaf()) {
		    node = lastChild;
		    lastChild = node.getChildAt(node.getChildCount() - 1);
		}

		int end = getRowForNode(lastChild);
		if(end == -1)
		    end = getRowForNode(node);
		end++;

		replaceItems(start, end, toBeClosed);
	    }

	    fireTreeCollapsed(path);

	    if (removeDescendantSelectedPaths(path, false) &&
		!isPathSelected(path)) {
		addSelectionPath(path);
	    }
	}

    }

    /**
     *
     * Returns <code>Enumeration</code> of descendant nodes of <code>parent</code>
     *
     */
    public Enumeration getDescendantToggledPaths(TreePath parent)
    {
	if(parent == null)
	    return null;

	Vector descendants = new Vector();
	Enumeration nodes  = expandedStatus.keys();
	TreePath path;

	while(nodes.hasMoreElements()) {

	    path = (TreePath)nodes.nextElement();

	    if(parent.isDescendant(path))
		descendants.addElement(path);
	}

	return descendants.elements();
    }

    /**
     *
     * Remove expanded descending nodes of <code>TreePath</code>s in
     * <code>toRemove</code>
     */
    public void removeDescendantToggledPaths(Enumeration toRemove)
    {
	if(toRemove == null)
	    return;

	while(toRemove.hasMoreElements()) {
	    Enumeration descendants = 
		getDescendantToggledPaths((TreePath)toRemove.nextElement());
	    if(descendants != null) {
		while(descendants.hasMoreElements()) {
		    expandedStatus.remove(descendants.nextElement());
		}
	    }
	}
    }


    /**
     *
     * Clears cache of toggled <code>TreePath</code>s witout sending
     * <code>TreeExpansionListener</code> event
    */
    public void clearToggledPaths()
    {
	expandedStatus.clear();
    }

    /**
     *
     * Extends all nodes on given <code>path</code> except the last path component
     * and scroll that the node to be displayed.  It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param path a <code>TreePath</code> specifying a node to be displayed
     */
    public void scrollPathToVisible(TreePath path)
    {
	;
    }

    /**
     *
     * Scrolles until the node specified by <code>row</code> to be displayed
     * with minimum amount of scroll to display.
     * It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param row int specifying row to scroll where zero indicates the first row
     */
    public void scrollRowToVisible(int row)
    {
	scrollPathToVisible(getPathForRow(row));
    }


    /**
     *
     * Returns <code>TreePath</code> of the node at given <code>row</code>
     * or null if the <code>row</code> is invisible
     *
     * @param row in int
     *
     * @return <code>TreePath</code> of specified node, or null if
     * <code>row</code> < 0 or <code>row</code> > <code>getRowCount()</code>
    */
    public TreePath getPathForRow(int row)
    {
	return (TreePath)visiblePaths.elementAt(limitRange(row));
    }

    /**
     *
     * Returns <code>TreePath</code> of the node at given <code>row</code>
     * or null if the <code>row</code> is invisible
     *
     * @param row in int
     *
     * @return <code>TreePath</code> of specified node, or null if
     * <code>row</code> < 0 or <code>row</code> > <code>getRowCount()</code>
    */
    public TreeNode getNodeForRow(int row)
    {
	return getNodeForPath(getPathForRow(limitRange(row)));
    }

    protected int limitRange(int row)
    {
	int upperLimit = visiblePaths.size() - 1;
	return row > upperLimit? upperLimit:row;
    }

    /**
     *
     * Returns <code>TreePath</code> of the node at given <code>row</code>
     * or null if the <code>row</code> is invisible
     *
     * @param row in int
     *
     * @return <code>TreePath</code> of specified node, or null if
     * <code>row</code> < 0 or <code>row</code> > <code>getRowCount()</code>
    */
    public TreeNode getNodeForPath(TreePath path)
    {
	return (TreeNode)(path.getLastPathComponent());
    }


    /**
     *
     * Returns row displaying node idnetified by <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     * @return int indicating displayed row where zero indicates the first
     * row, -1 if path componets are hidden under shrunken parent
     */
    public int getRowForPath(TreePath path)
    {
	return getRowForKey(rowOfPath, path);
    }

    public int getRowForNode(org.nomencurator.util.tree.TreeNode node)
    {
	return getRowForKey(rowOfNode, node);
    }

    public int getRowForObject(Object object)
    {
	return getRowForKey(rowOfObject, object);
    }

    public int getRowForKey(Hashtable hashtable, Object key)
    {
	Object row = hashtable.get(key);
	if(row == null)
	    return -1;

	return ((Integer)row).intValue();
    }
    

    /**
     *
     * Expands and makes visible a node indetified by given <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     */
    public void expandPath(TreePath path)
    {
	if(path == null)
	    return;

	if(!isLeaf(path.getLastPathComponent())) {
	    setExpandedState(path, true);
	}
    }


    /**
     *
     * Expands and makes visible a node at specified <code>row</code>.
     * It does nothing if <code>row</code> < 0 or .<code>row</code> >=
     * <code>getRowCount</code>.
     *
     * @param row int specifying displayed row where zero indicates
     * the first row
     */
    public void expandRow(int row)
    {
	expandPath(getPathForRow(row));
    }


    /**
     *
     * Shrink and make visible a node identified by <code>path</code>
     *
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     */
    public void collapsePath(TreePath path)
    {
	setExpandedState(path, false);
    }

    /**
     *
     * Shrinnk a node at specified <code>row</code>
     * It does nothing if <code>row</code> < 0 or .<code>row</code> >=
     * <code>getRowCount</code>.
     *
     * @param row int specifying displayed row where zero indicates
     * the first row
     */
    public void collapseRow(int row)
    {
	collapsePath(getPathForRow(row));
    }

    /**
     *
     * Returns a <code>TreePath</code> indicating a path from <code>index0</code>
     * to <code>index1</code>, or null if there is no tree.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     * @return an arraay of <code>TreePath</code> objects, of each for each node between
     * <code>index0</code> and <code>index1</code>, or null if there is no tree
     */
    public TreePath[] getPathBetweenRows(int index0,
					    int index1)
    {
	if(index0 > index1) {
	    int tmp = index0;
	    index0 = index1;
	    index1 = tmp;
	}

	if(index0 < 0 || index1 > visiblePaths.size())
	    return null;
	TreePath[] paths = new TreePath[index1 - index0];
	for(int i = index0; i < index1; i++)
	    paths[i] = (TreePath)visiblePaths.elementAt(i);

	return paths;
    }


    /*
     * Adds a TreeExpansionListener to the lister list
     *
     * @param listener a listener of TreeExpansionListener to be add
     */
    public void addTreeExpansionListener(TreeExpansionListener listener)
    {
	treeExpansionListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeExpansionlLister from the listeners list
     *
     * @param listener a listener of TreeExpansionListener to be removed
     */
    public void removeTreeExpansionListener(TreeExpansionListener listener)
    {
	treeExpansionListeners.remove(listener);
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying expanded node
     */
    public void fireTreeExpanded(TreePath path)
    {
	if(treeExpansionListeners == null ||
	   treeExpansionListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized(treeExpansionListeners) {
	    listeners = treeExpansionListeners.toArray();
	}

	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeExpansionListener)listeners[--i]).treeExpanded(e);
	}
    }


    /*
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying shrunken node
     */
    public void fireTreeCollapsed(TreePath path)
    {
	if(treeExpansionListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized(treeExpansionListeners) {
	    listeners = treeExpansionListeners.toArray();
	}

	TreeExpansionEvent e = null;
	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeExpansionListener)listeners[--i]).treeCollapsed(e);
	}
    }


    /*
     * Adds a TreeWillExpandListener to the lister list
     *
     * @param listener a listener of TreeWillExpandListener to be add
     */
    public void addTreeWillExpandListener(TreeWillExpandListener listener)
    {
	treeWillExpandListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeWillExpandListener from the listeners list
     *
     * @param listener a listener of TreeWillExpandListener to be removed
     */
    public void removeTreeWillExpandListener(TreeWillExpandListener listener)
    {
	treeWillExpandListeners.remove(listener);
    }


    /**
     *
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to expand
     */
    public void fireTreeWillExpand(TreePath path)
	throws ExpandVetoException
    {
	if(treeWillExpandListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized(treeWillExpandListeners) {
	    listeners = treeWillExpandListeners.toArray();
	}
	TreeExpansionEvent e = null;

	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeWillExpandListener)listeners[--i]).treeWillExpand(e);
	}
    }
	    
    /**
     * Notifys an event to all <code>EventListeners</code> registered 
     * as listeners of this event type.
     * The instance of event is constructed using parameters passed to
     * triggering method.
     *
     * @param path <code>TreePath</code> identifying node to shrink
     */
    public void fireTreeWillCollapse(TreePath path)
	throws ExpandVetoException
    {
	if(treeWillExpandListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized(treeWillExpandListeners) {
	    listeners = treeWillExpandListeners.toArray();
	}

	TreeExpansionEvent e = null;
	for(int i = listeners.length; i > 0; ) {
	    if(e == null)
		e = new TreeExpansionEvent(this, path);
	    ((TreeWillExpandListener)listeners[--i]).treeWillCollapse(e);
	}
    }

    /*
     * Adds a TreeSelectionListener to the lister list
     *
     * @param listener a listener of TreeSelectionListener to be add
     */
    public void addTreeSelectionListener(TreeSelectionListener listener)
    {
	treeSelectionListeners.addElement(listener);
    }

    /**
     * Revmoves a TreeSelectionListener from the listeners list
     *
     * @param listener a listener of TreeSelectionListener to be removed
     */
    public void removeTreeSelectionListener(TreeSelectionListener listener)
    {
	treeSelectionListeners.remove(listener);
    }

    /**
     *
     * Appends a node specified by given <code>TreePath</code> to current selection.
     * If any component of the path is invisible and if getExpandsSelectedPaths is true,
     * the node will be visible.
     *
     * @param path <code>TreePath</code> specifying a node to be add
     */
    public void addSelectionPath(TreePath path)
    {
	;
    }


    /**
     *
     * Returns true if node specified by <code>path</code> is selected
     *
     * @param path a <code>TreePath</code> identifying a node
     *
     * @returns true if specified node is selected
     */
    public boolean isPathSelected(TreePath path)
    {
	return false;
    }

    /**
     *
     * Remove all selected discending nodes of <code>path</code>.
     * If <code>includePath</code> is true and <code>path</code> is selected,
     * it is removed from selection.
     *
     * @return true if discending nodes are removed
     *
     * @since 1.3
     */
    public boolean removeDescendantSelectedPaths(TreePath path,
						 boolean includePath)
    {
	return true;
    }

    /**
     * Invoked when mouse clicked
     *
     */
    public void doubleCliked(TreePath path)
    {
	if(isLeaf(path.getLastPathComponent()))
	    return;

	Object expanded = expandedStatus.get(path);

	if(expanded == null)
	    return;

	if(((Boolean)expanded).booleanValue())
	    collapsePath(path);
	else
	    expandPath(path);
    } 

    /**
     *
     * Called by expansion of a node in a tree
     *
     */
    public void treeExpanded(TreeExpansionEvent event)
    {
    }


    /**
     *
     * Called by shrink of a node in a tree
     *
     */
    public void treeCollapsed(TreeExpansionEvent event)
    {
    }


    /**
     *
     * Invoked when a tree node will be expanded
     *
     */
    public void treeWillExpand(TreeExpansionEvent event)
	throws ExpandVetoException
    {
    }


    /**
     *
     * Invoked when a tree node will be shrunken
     *
     */
    public void treeWillCollapse(TreeExpansionEvent event)
	throws ExpandVetoException
    {
    }

    /**
     * Invoked when value in current selection is changed
     *
     * @param event <code>TreeSelectionEvent</code> describing
     * change in value
     */
    public void valueChanged(TreeSelectionEvent event)
    {
    }
}
