/*
 *
 * TreeSelectionModel.java:  Selection model interface for Tree
 *
 * Copyright (c) 2001 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: TreeSelectionModel.java,v 1.1.1.1 2002/01/16 12:33:34 ryo Exp $
 *	$Log: TreeSelectionModel.java,v $
 *	Revision 1.1.1.1  2002/01/16 12:33:34  ryo
 *	initial import into CVS
 *	
 *	
 *
 */

package org.nomencurator.awt.tree;

import java.awt.Dimension;

import java.beans.PropertyChangeListener;

import org.nomencurator.util.tree.TreePath;

import org.nomencurator.util.tree.event.TreeModelListener;

import org.nomencurator.awt.tree.event.TreeSelectionListener;

public interface TreeSelectionModel
{
    //Fields
    /**
     *
     * Selection may contain only one <code>TreePath</code>
     *
     */
    public static final int SINGLE_TREE_SELECTION = 1;

    /**
     *
     * Selection of multiple <code>TreePath</code>s is allowed
     * only if they are contiguous.  It is applicable only when
     * a <code>RowMapper</code> instance is provided.  If no
     * <code>RowMapper</code> is available, it is equivalent with
     * DISCONTIGUOUS_TREE_SELECTION
     */
    public static final int CONTIGUOUS_TREE_SELECTION = 2;


    /**
     *
     * Multiple selection is allowed without contiguity constraints
     *
     */

    public static final int DISCONTIGUOUS_TREE_SELECTION = 4;


    //Methods compatible with javax.swing.tree.TreeSelectionModel
    /**
     *
     * Selects selection <code>mode</code> which must be one of SINGLE_TREE_SELECTION,
     * CONTIGUOUS_TREE_SELECTION or DISCONTIGUOUS_TREE_SELECTION.
     * <P>
     * Selected area can be changed if current selection is invalid under specified 
     * new <code>mode</code>.  When three <code>TreePath</code>s are selected and then
     * mode is switched to SINGLE_TREE_SELECTION, for example, only one <code>TreePath</code>
     * will be chosen.  Choice of <code>TreePath</code> depends on implementation.
     */
    public void setSelectionMode(int mode);


    /**
     * 
     * Returns current selection mode, i.e. one of SINGLE_TREE_SELECTION
     * CONTIGUOUS_TREE_SELECTION or DISCONTIGUOUS_TREE_SELECTION
     */
    public int getSelectionMode();


    /**
     *
     * Selects specified <code>path</code>.  
     * <code>TreeSelectionListeners</code> will be notified if it implies
     * a modification.  It is equivalent with calling <code>clearSelection</code>
     * when <code>path</code> is null.
     *
     * @param path <code>TreePath</code> to be selected
     */
    public void setSelectionPath(TreePath path);


    /**
     *
     * Selects <code>paths</code>.
     * 
     * <code>TreeSelectionListeners</code> will be notified if it implies
     * a modification.  It is equivalent with calling <code>clearSelection</code>
     * when <code>paths</code> is null.
     *
     * @param paths an array of <code>TreePath</code> to be selected
     */
    public void setSelectionPaths(TreePath[] paths);


    /**
     *
     * Append <code>path</code> to current selected range.
     * If the <code>path</code> is out of the current selection rage,
     * <code>TreeSelectionListeners</code> will be notified.
     * Nothing happens if <code>path</code> is null.
     *
     * @param path <code>TreePath</code> to be added to current selection
     */

    public void addSelectionPath(TreePath path);


    /**
     *
     * Append elements of <code>paths</code> to current selected range.
     * If the <code>path</code> is out of the current selection rage,
     * <code>TreeSelectionListeners</code> will be notified.
     * Nothing happens if <code>paths</code> is null.
     *
     * @param paths an array of  <code>TreePath</code> to be added to current selection
     */
    public void addSelectionPaths(TreePath[] paths);


    /**
     *
     * Remove <code>path</code> from selected range.
     * If the <code>path</code> is in the selected range,
     * <code>TreeSelectionListeners</code> will be notified.
     * Nothing happens if <code>path</code> is null.
     *
     * @param path <code>TreePath</code> to be added to current selection
     */

    public void removeSelectionPath(TreePath path);

    /**
     *
     * Remove each element of <code>paths</code> from selected range.
     * If a path is in the selected range,
     * <code>TreeSelectionListeners</code> will be notified.
     * Nothing happens if <code>paths</code> is null.
     *
     * @param paths an array of  <code>TreePath</code> to be added to current selection
     */

    public void removeSelectionPaths(TreePath[] paths);


    /**
     *
     * Returns first path of selected range.
     * Implementor may define the "first path" which is unnecessary
     * to be the <code>TreePath</code> with minimum row number determined
     * by <code>RowMapper</code>
     */
    public TreePath getSelectionPath();


    /**
     *
     * Returns paths in selected range.
     * If nothing is selected, it returns null or
     * empty array.
     *
     * /
     public TreePath[] getSelectionPaths();


     /**
      *
      * Returns number of selected paths 
      *
      */
    public int getSelectionCount();


    /**
     *
     * Returns true if given <code>path</code> is in 
     * the current selection range
     */
    public boolean isPathSelected(TreePath path);


    /**
     *
     * Returns true if selection is emtpy.
     *
     */
    public boolean isSelectionEmpty();


    /**
     *
     * Clear current selection.
     * If it changes current selection (i.e. non-empty selection range
     * has been cleared), it notifies to selection listeners
     */
    public void clearSelection();


    /**
     *
     * Sets an instance of <code>RowMapper</code> which enables
     * to determine which row corresponds to which <code>TreePath</code>
     */
    public void setRowMapper(RowMapper mapper);


    /**
     *
     * Returns an instance of <code>RowMapper</code> which enables
     * mapping between <code>TreePath</code> and row.
     *
     */
    public RowMapper getRowMapper();


    /**
     *
     * Returns array of selected rows.
     * If node <code>TreePath</code> is selected or no <code>RowMapper</code> is set,
     * it returns null or empty array.
     */
    public int[] getSelectionRows();


    /**
     *
     * Returns minimum row number obtained from <code>RowMapper</code> on 
     * current selected <code>TreePath</code>.  It returns -1 if nothins selected
     * or no <code>RowMapper</code> is assigned.
     */
    public int getMinSelectionRow();


    /**
     *
     * Returns maximum row number obtained from <code>RowMapper</code> on 
     * current selected <code>TreePath</code>.  It returns -1 if nothins selected
     * or no <code>RowMapper</code> is assigned.
     */
    public int getMaxSelectionRow();


    /**
     *
     * It returns true if <code>row</code> is selected
     */
    public boolean isRowSelected(int row);


    /**
     *
     * Updates mapping from TreePath of this Object to row,
     * ivoked when a node is expanded, for example.
     * <P>
     * It is unnecessary for users to ivoke this method ordinarily
     * but JTree or its relating listers ivoke it instead.
     * If you implemnts your own view class, you need to call
     * it explicitly, however.
     * 
     */
    public void resetRowSelection();


    /**
     *
     * Returns lead index in selected range, which is last-added index
     * 
     */
    public int getLeadSelectionRow();


    /**
     *
     * Returns path added to the tree in last which may differ from 
     * leadSelectionPath propaty of JTree.
     *
     */
    public TreePath getLeadSelectionPath();


    /**
     *  Adds a <code>PropertyChangeListener</code> to listener list which 
     * is registered in all proparites.
     * <P>
     * PropertyChangeEvent is triggerd when selection mode is changed
     *
     * @param listener <code>PropertyChangeListener</code> to be add
     */
    public void addPropertyChangeListener(PropertyChangeListener listener);


    /**
     *
     * Removes a <code>PropertyChangeListener</code> from listener list for 
     * all propaties.
     *
     * @param listener <code>PropertyChangeListener</code> to be removed
     *
     */
    public void removePropertyChangeListener(PropertyChangeListener listener);


    /**
     *
     * Adds a <code>listener</code> to the listener list which is notified
     * whenever set of selected <code>TreePath</code> is changed
     *
     * @param listener <code>TreeSelectionListener</code> to be add
     */
    public void addTreeSelectionListener(TreeSelectionListener lister);

    /**
     *
     * Removes a <code>listener</code> from the listener list which is notified
     * whenever set of selected <code>TreePath</code> is changed
     *
     * @param listener <code>TreeSelectionListener</code> to be removed
     */
    public void removeTreeSelectionListener(TreeSelectionListener listener);

    //Methods necessary to support JTree
    /**
     *
     * Selects a node at specified <code>row</code>
     *
     * @param row to be selected; zero indicates the first row
     */
    public void setSelectionRow(int row);

    /**
     *
     * Slects nodes corresponding to specified <code>rows</code>.
     * Elements of the <code>rows</code> of which value is < 0 
     * or >= getRowCount will be ignored as invalid element.
     * Selection will be cleard when all elements of <code>rows</code>
     * are invalid, which is equivalent to calling <code>clearSelection</code>
     *
     * @param rows an array of int specifying rows to be selected.  Its zero value
     * indicates the first row
     */
    public void setSelectionRows(int[] rows);

    /**
     * 
     * Append path at the specified <code>row</code> to current selection.
     *
     * @param row an int specifying a row of the node to be append,
     * where zero indicates the first row.
     */
    public void addSelectionRow(int row);

    /**
     *
     * Appends paths at specified rows to current selection
     *
     * @param rows an array of int specifying rows at which path to be append
     * to the selection, where zero implies the first row
     */
    public void addSelectionRows(int[] rows);

    /**
     *
     * Returns last path component on the first node of currenct selection
     *
     * @return the last <code>Object</code> on the first selected <code>TreePath</code>,
     * or null if nothing is selected
     *
     * @see         TreePath.getLastPathComponent();
    */
    public Object getLastSelectedPathComponent();

    /**
     *
     * Returns a path identified as anchor
     *
     * @return path identified as anchor
     *
     * @since 1.3
     */
    public TreePath getAnchorSelectionPath();

    /**
     *
     * Returns paths of all selected nodes
     *
     * @return an array of <code>TreePath</code> indicating all selected nodes, or
     * null if nothing is selected
     */
    public TreePath[] getSelectionPaths();

    /**
     *
     * Selects nodes at rows between <code>index0</code> and <code>index1</code>.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void setSelectionInterval(int index0,
				     int index1);

    /**
     *
     * Appends paths at rows between <code>index0</code> and <code>index1</code>
     * to the selection
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void addSelectionInterval(int index0,
				     int index1);

    /**
     *
     * Removes nodes at rows between <code>index0</code> and <code>index1</code>
     * from the selection
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     */
    public void removeSelectionInterval(int index0,
					int index1);

    /**
     *
     * Removes path at <code>row</code> from current selection
     *
     * @param row specifying the node to be removed
     */
    public void removeSelectionRow(int row);

    /**
     *
     * Removes paths at specified <code>rows</code> from the selection
     *
     * @param rows an array of int specifying paths to be removed where
     * zero value indicates the first row 
     */
    public void removeSelectionRows(int[] rows);

    /**
     *
     * Remove all selected discending nodes of <code>path</code>.
     * If <code>includePath</code> is true and <code>path</code> is selected,
     * it is removed from selection.
     *
     * @return true if discending nodes are removed
     *
     * @since 1.3
     */
    public boolean removeDescendantSelectedPaths(TreePath path,
						    boolean includePath);

}
