/*
 * TreeView.java:  a TreeView for Tree user interface
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: TreeView.java,v 1.4 2002/08/15 00:50:53 nozomi Exp $
 * $Log: TreeView.java,v $
 * Revision 1.4  2002/08/15 00:50:53  nozomi
 * modified to use TreeMouseAdaptor
 *
 * Revision 1.3  2002/06/09 12:40:44  nozomi
 * change TreeModelEvent handling
 *
 * Revision 1.2  2002/03/10 06:56:33  nozomi
 * remove empty lines
 *
 * Revision 1.1.1.1  2002/01/16 12:33:34  ryo
 * initial import into CVS
 */

package org.nomencurator.awt.tree;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.kyasu.awt.TextListModel;

import jp.kyasu.awt.text.TextListController;
import jp.kyasu.awt.text.TextListView;

import jp.kyasu.graphics.RichTextStyle;

import org.nomencurator.awt.tree.DefaultMutableTreeNode;
import org.nomencurator.awt.tree.TextTreeModel;
import org.nomencurator.awt.tree.TreeNode;

import org.nomencurator.util.tree.DefaultTreeModel;
import org.nomencurator.util.tree.MutableTreeNode;
import org.nomencurator.util.tree.TreeModel;
import org.nomencurator.util.tree.TreePath;

import org.nomencurator.util.tree.event.TreeModelEvent;
import org.nomencurator.util.tree.event.TreeModelListener;

import org.nomencurator.awt.tree.event.ExpandVetoException;
import org.nomencurator.awt.tree.event.TreeExpansionEvent;
import org.nomencurator.awt.tree.event.TreeExpansionListener;
import org.nomencurator.awt.tree.event.TreeSelectionEvent;
import org.nomencurator.awt.tree.event.TreeSelectionListener;
import org.nomencurator.awt.tree.event.TreeWillExpandListener;

/**
 * Since a tree can be represented by a indented list graphically,
 * a <CODE>TreeView</CODE> can be implemented as a subclass of
 * <CODE>TextListView</CODE> used as a view in 
 * <CODE>Tree</CODE> widget class extending <CDOE>List</CODE> class of KFC.
 *
 * @version 	15 Aug 2002
 * @author 	Nozomi `James' Ytow
 */
public class TreeView
    extends TextListView
    implements TreeModelListener,
	       TreeExpansionListener,
	       TreeSelectionListener,
	       TreeWillExpandListener
{
    public TreeView(TextListModel listModel)
    {
	this((TextTreeModel)listModel);
    }

    public TreeView(TreeModel treeModel)
    {
	this(new TextTreeModel(treeModel));
    }

    public TreeView(TextTreeModel textTreeModel)
    {
	super(textTreeModel);

	//treeModel handling....
	Object node = textTreeModel.getTreeModel().getRoot();
	
	textTreeModel.addTreeModelListener(this);
	textTreeModel.addTreeExpansionListener(this);
	textTreeModel.addTreeWillExpandListener(this);
	textTreeModel.addTreeSelectionListener(this);

	new TreeMouseAdaptor(this);
    }

    protected TextListController createsController()
    {
	return new TreeController(this);
    }


    /**
     * Draw a node in the tree.
     *
     * @param TreeNode to be drawn
     */
    protected void drawNode(TreeNode node)
    {
	if(node.isLeaf() || node.isExpanded()) {
	    Enumeration children = node.children();
	    while(children.hasMoreElements()) {
		drawNode((TreeNode)(children.nextElement()));
	    }
	}
    }


    public TextTreeModel getTextTreeModel()
    {
	return (TextTreeModel)getModel();
    }

    // TreeModelListener's methods
    /**
     * Invoked when a node or a set of siblings was modified.
     * Neither location of the node in the tree nor array of its child nodes 
     * was not modified but other attributes were changed which may arrect
     * tree's representation.  For eample when file name was changed but 
     * its position in file system tree was not altered.
     * <p>
     * Vale of childIndices and children will be null to tell change of root.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes.
     */
    public void treeNodesChanged(TreeModelEvent event)
    {

    }

    /**
     * Invoked after insertion of a node.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */

    public void treeNodesInserted(TreeModelEvent event)
    {

    }

    /**
     * Invoked after removal of a node from a tree.
     * When a sbutree removed a tree, this method is invoked only once for
     * the root node of the removed subtree but never each sibling nodes removed.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */
    public void treeNodesRemoved(TreeModelEvent event)
    {

    }

    /**
     * Invoked when tree structure under a node modified drastically.
     * When the path length given by <code>TreeModelEvent#getPath()</code> is one
     * and the first element is not the current root node, the first element is 
     * the root of the new tree.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * null.
     */
    public void treeStructureChanged(TreeModelEvent event)
    {
    }

    //Methods for JTree support
    /**
     *
     * Returns true if the node at specified <code>row</code> is currently expanded
     *
     * @param row to be examined where zero indicates the first row
     *
     * @return ture if the node is currently expanded
     */
    public boolean isExpanded(int row)
    {
	return getTextTreeModel().isExpanded(getPathForRow(row));
    }

    /**
     *
     * Returns true if the node at specified <code>row</code> is
     * shrunken
     *
     * @param row to be exmained where zero indicates the first row
     *
     * @return true if the node is currently expanded
     */
    public boolean isCollapsed(int row)
    {
	return getTextTreeModel().isCollapsed(getPathForRow(row));
    }

    /**
     *
     * Makes the node identified by <code>path</code> visible
     *
     * @param path a <code>TreePath</code> to be visible
     */
    public void makeVisible(TreePath path)
    {
	if(path == null)
	    return;

	path = path.getParentPath();

	TextTreeModel model = getTextTreeModel();
	if(model != null)
	    getTextTreeModel().expandPath(path);
    }

    /**
     *
     * Returns true if the node identified by <code>path</code> is visible, i.e. it is root or its all parents are
     * expanded.
     *
     * @returns true if the node is visible
     */
    public boolean isVisible(TreePath path)
    {
	if(path == null)
	    return false;

	path = path.getParentPath();

	if(path == null)
	    return true;

	return getTextTreeModel().isExpanded(path);
    }

    /**
     *
     * Returns <code>TreePath</code> of the node at given <code>row</code>
     * or null if the <code>row</code> is invisible
     *
     * @param row in int
     *
     * @return <code>TreePath</code> of specified node, or null if
     * <code>row</code> < 0 or <code>row</code> > <code>getRowCount()</code>
    */
    public TreePath getPathForRow(int row)
    {
	return getTextTreeModel().getPathForRow(row + getVisibleBegin().lineIndex);
	//	return getTextTreeModel().getPathForRow(row);
    }

    /**
     *
     * Returns row displaying node idnetified by <code>path</code>
     *
     * @param path a <code>TreePath</code> indentifying a node
     *
     * @return int indicating displayed row where zero indicates the first
     * row, -1 if path componets are hidden under shrunken parent
     */
    public int getRowForPath(TreePath path)
    {
	return getTextTreeModel().getRowForPath(path) - getVisibleBegin().lineIndex;
    }


    /**
     *
     * Returns a <code>Rectangle</code> where node specified by <code>path</code>
     * is rendererd.  It returns null if nodes on the <code>path</code> is hidden
     * (i.e. under hidden parent),
     * <p>
     * Note: it returns valid <code>Rectangle</code> even if the specified node
     * is not visible
     *
     * @param path a <code>TreePath</code> specifying a node
     *
     * @return <code>Rectangle</code> to where node is rendered or null
     */
    public Rectangle getPathBounds(TreePath path)
    {
	return null;
    }


    /**
     *
     * Returns a <code>Rectangle</code> where the node specifyed by
     * given <code>row</code> is rendered
     *
     * @param row to be rendered where zero indicates the first row
     *
     * @return <code>Rectangle</code> where node to be rendered
     */
    public Rectangle getRowBounds(int row)
    {
	return getPathBounds(getPathForRow(row));
    }

    /**
     *
     * Extends all nodes on given <code>path</code> except the last path component
     * and scroll that the node to be displayed.  It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param path a <code>TreePath</code> specifying a node to be displayed
     */
    public void scrollPathToVisible(TreePath path)
    {
	;
    }

    /**
     *
     * Scrolles until the node specified by <code>row</code> to be displayed
     * with minimum amount of scroll to display.
     * It works only if this <code>JTree</code>
     * is contained in <code>JScrollPane</code>
     *
     * @param row int specifying row to scroll where zero indicates the first row
     */
    public void scrollRowToVisible(int row)
    {
	;
    }

    /**
     *
     * Returns <code>TreePath</code> of a node at specified position
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return a <code>TreePath</code> of a node at specified position
     */
    public TreePath getPathForLocation(int x,
				       int y)
    {
	TreePath path = getClosestPathForLocation(x, y);

	if(path != null) {
	    Rectangle bounds = getPathBounds(path);
	    if(bounds == null)
		return path;
            if(x >= bounds.x && x < (bounds.x + bounds.width) &&
               y >= bounds.y && y < (bounds.y + bounds.height))
                return path;
	}
	return null;
    }


    /**
     *
     * Returns row of a node at specified position
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return row of a node at specified position, where -1 indicates
     * the specified position is out of displayed area
     *
     * @see getClosestRowForLocation(int, int);
    */
    public int getRowForLocation(int x,
				 int y)

    {
	return getRowForPath(getPathForLocation(x, y));
    }

    /**
     *
     * Returns <code>TreePath</code> of a node nearest to position specified by
     * <code>x</code>, <code>y</code>.
     * It returns valid <code>TreePath</code> always except when returns null if there is no visible node or no model.
     * To confirm the node is at <code>x</code>, <code>y</code> exactly, examine <code>x</code>, <code>y</code>
     * with margine of the node obtained using <code>getPathBounds(TreePath)</code>
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return <code>TreePath</code> of a node nearest to specified postion, or null
     * if there is no visible node or no model
     *
     * @see getPathForLocation(int, int)
     * @see getPathBounds(TreePath)
     */
    public TreePath getClosestPathForLocation(int x,
					      int y)
    {
	return getPathForRow(getClosestRowForLocation(x, y));
    }

    /**
     *
     * Returns row of a node nearest to position specified by
     * <code>x</code>, <code>y</code>.
     * It returns valid row always except when returns -1 if there is no visible node or no model.
     * To confirm the node is at <code>x</code>, <code>y</code> exactly, examine <code>x</code>, <code>y</code>
     * with margine of the node obtained using <code>getRowBounds(int)</code>
     *
     * @param x int value indicating horizontal distance in pixel from
     * left edge of display area (excluding the left margin)
     * @param y int value indicating vertical distance in pixel from
     * top edge of display area (excluding the top margin)
     *
     * @return row of a node nearest to specified postion, or -1
     * if there is no visible node or no model
     *
     * @see getRowForLocation(int, int)
     * @see getRowBounds(int)
     */
    public int getClosestRowForLocation(int x,
					int y)
    {
	return getLineIndexNearby(getVisiblePosition(), new Point(x, y));
    }

    /**
     *
     * Returns appropriate display size of JTree of which height is determined
     * by <code>getVisibleRowCount</code> and width is current appropriate width.
     *
     * @return <code>Dimension</code> object holding appropriate size
     */
    public Dimension getPreferredScrollableViewportSize()
    {
	return getPreferredSize();
    }


    /**
     *
     * Returns a <code>TreePath</code> indicating a path from <code>index0</code>
     * to <code>index1</code>, or null if there is no tree.
     *
     * @param index0 int specifying starting row where zero indicates the first row
     * @param index1 int specifying last row
     *
     * @return an arraay of <code>TreePath</code> objects, of each for each node between
     * <code>index0</code> and <code>index1</code>, or null if there is no tree
     */
    public TreePath[] getPathBetweenRows(int index0,
					    int index1)
    {
	return getTextTreeModel().getPathBetweenRows(index0, index1);
    }

    /**
     *
     * Returns increment when scrolled which is determined by height of the first partially-displayed row,
     * or height of the next hidden line at scrolling direction.
     *
     * @param visibleRect visible area in the view port
     * @param orientation VERTICAL or HORIZONTAL
     * @param direction less than zero for upward or leftword scroll,
     * or larger than zero for downword or rightword scroll
     *
     * @return "unit" increment to scroll in specified direction
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect,
					  int orientation,
					  int direction)
    {
	//	getTextLineAt(lineIndex);
	return 0;
    }

    /**
     *
     * Returns "block" increment which is determined by height or widht of <code>visibleRect</code>
     * depending on <code>orientation</code>.
     *
     * @param avisibleRect visbile vew area in view port
     * @param orientation <code>VERTICAL</code> or <code>HORIZONTAL</code>
     * @param direction less than zero for upward or leftword scroll,
     * or larger than zero for downword or rightword scroll
     *
     * @return "block" increment to scroll in specified direction
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect,
					   int orientation,
					   int direction)
    {
	return 0;
    }
    
    /**
     *
     * Returns false to tell width of view port does not restrict width of table,
     * i.e. keeps tree from being smaller than view port, except when appropriate width 
     * for the tree is narrower than width of the view port
     *
     * @return false
     *
     */
    public boolean getScrollableTracksViewportWidth()
    {
	return false;
    }

    /**
     *
     * Returns false to tell height of view port does not restrict height of table,
     * i.e. keeps tree from being smaller than view port, except when appropriate height
     * for the tree is shorter than height of the view port
     *
     * @return false
     *
     */
    public boolean getScrollableTracksViewportHeight()
    {
	return false;
    }


    /**
     *
     * Invoked when a modification in tree is sufficiently large to change
     * size of margine but not so huge to remove expanded node set (e.g.
     * when node is expanded, shrunken, or nodes are inserted into tree).
     * It is unnecessary to call this method; UI does when necessary
     */
    public void treeDidChange()
    {
    }

    /**
     *
     * Called by expansion of a node in a tree
     *
     */
    public void treeExpanded(TreeExpansionEvent event)
    {
    }


    /**
     *
     * Called by shrink of a node in a tree
     *
     */
    public void treeCollapsed(TreeExpansionEvent event)
    {
    }

    /**
     * 
     * Invoked when value in current selection is changed
     *
     * @param event <code>TreeSelectionEvent</code> describing
     * change in value
     */
    public void valueChanged(TreeSelectionEvent event)
    {
    }

    /**
     *
     * Invoked when a tree node will be expanded
     *
     */
    public void treeWillExpand(TreeExpansionEvent event)
	throws ExpandVetoException
    {
    }


    /**
     *
     * Invoked when a tree node will be shrunken
     *
     */
    public void treeWillCollapse(TreeExpansionEvent event)
	throws ExpandVetoException
    {
    }
}
