/*
 *
 * TreeSelectionEvent.java:  Selection event for Tree
 *
 * Copyright (c) 2001 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: TreeSelectionEvent.java,v 1.1.1.1 2002/01/16 12:33:34 ryo Exp $
 *	$Log: TreeSelectionEvent.java,v $
 *	Revision 1.1.1.1  2002/01/16 12:33:34  ryo
 *	initial import into CVS
 *	
 *	
 *
 */

package org.nomencurator.awt.tree.event;

import java.util.EventObject;

import org.nomencurator.util.tree.TreePath;

public class TreeSelectionEvent
    extends EventObject
{
    /**
     *
     * An array of <code>TreePath</code> to which this event relates
     *
     */
    protected TreePath[] paths;


    /**
     *
     * An array of boolean indicating each path is new
     */
    protected boolean[] areNew;


    /**
     *
     * leadSelectionPath before the path modification may be null
     *
     */
    protected TreePath oldLeadSelectionPath;


    /**
     *
     * leadSelectionPath after the path modification may be null
     *
     */
    protected TreePath newLeadSelectionPath;


    //Constructors
    /**
     *
     * Indicates modiciation in <code>TreeSelectionModel</code>.
     * <code>paths</code> indicates paths added to or removed from
     * selected range
     *
     * @param source event source
     * @param paths modified path in selection range
     *
     */
    public TreeSelectionEvent(Object source,
			      TreePath[] paths,
			      boolean[] areNew,
			      TreePath oldLeadSelectionPath,
			      TreePath newLeadSelectionPath)
    {
	super(source);
	this.paths = paths;
	this.areNew = areNew;
	this.oldLeadSelectionPath =
	    oldLeadSelectionPath;
	this.newLeadSelectionPath =
	    newLeadSelectionPath;
    }

    
    /**
     *
     * Indicates modiciation in <code>TreeSelectionModel</code>.
     * <code>path</code> indicates path added to or removed from
     * selected range
     *
     * @param source event source
     * @param path modified path in selection range
     * @param isNew whether path is new to the selected range;
     * false if the path has been removed from the selection 
     *
     */
    public TreeSelectionEvent(Object source,
			      TreePath path,
			      boolean isNew,
			      TreePath oldLeadSelectionPath,
			      TreePath newLeadSelectionPath)
    {
	super(source);
	paths = new TreePath[1];
	paths[0] = path;
	areNew = new boolean[1];
	areNew[0] = isNew;
	this.oldLeadSelectionPath =
	    oldLeadSelectionPath;
	this.newLeadSelectionPath =
	    newLeadSelectionPath;
    }


    //Methods
    /**
     *
     * Returns paths added to or removed from the selection
     *
     */
    public TreePath[] getPaths()
    {
	return paths;
    }


    /**
     *
     * Returns first path which added to or removed from the selection
     *
     */
    public TreePath getPath()
    {
	return paths[0];
    }

    
    /**
     *
     * Returns true if the first path is added to selection,
     * or false if removed.
     *
     */
    public boolean isAddedPath()
    {
	return areNew[0];
    }


    /**
     *
     * Returns true if specified <code>path</code> is added to the selection,
     * or false if removed.
     *
     */
    public boolean isAddedPath(TreePath path)
    {
	if(path != null) {
	    for (int i = 0; i < paths.length; i++) {
		if(paths[i].equals(path))
		    return areNew[i];
	    }
	}
       
	throw new IllegalArgumentException();
    }



    /**
     *
     * Returns true if <code>TreePath</code> specified by <code>index</code> is
     * added to the selection, or false if the path was in the selection but
     * out currently.
     *
     * @since 1.3
     */
    public boolean isAddedPath(int index)
    {
	if(paths == null || index < 0 || index > paths.length)
	    throw new IllegalArgumentException();

	return areNew[index];
    }


    /**
     *
     * Returns old lead selection path
     */
    public TreePath getOldLeadSelectionPath()
    {
	return oldLeadSelectionPath;
    }


    /**
     *
     * Returns newLeadSelectionPath
     *
     */

    public TreePath getNewLeadSelectionPath()
    {
	return newLeadSelectionPath;
    }


    /**
     *
     * Returns copy of receiver with newSource
     *
     */

    public Object cloneWithSource(Object newSource)
    {
	return new TreeSelectionEvent(newSource,
				      paths,
				      areNew,
				      oldLeadSelectionPath,
				      newLeadSelectionPath);

    }

}
