/*
 * JdbcConnection.java: 
 *
 * Copyright (c) 2002 Ryo FUJIMOTO
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: JdbcConnection.java,v 1.30 2002/12/06 09:37:39 t.okada Exp $
 * $Log: JdbcConnection.java,v $
 * Revision 1.30  2002/12/06 09:37:39  t.okada
 * modify SQL Nomencurator MkIII
 *
 * Revision 1.29  2002/10/11 05:34:04  ryo
 * fix SQLException bug of auto commit
 *
 * Revision 1.28  2002/10/10 06:22:32  nozomi
 * adapt to method name modification in NameUsage in another call
 *
 * Revision 1.27  2002/10/10 06:21:02  nozomi
 * adapt to method name modification in NameUsage
 *
 * Revision 1.26  2002/09/17 05:01:41  nozomi
 * fix method name
 *
 * Revision 1.25  2002/09/02 08:12:33  t.okada
 * remove multibyte character
 *
 * Revision 1.24  2002/08/27 08:12:27  ryo
 * add the processing for adding type
 *
 * Revision 1.23  2002/08/26 09:24:40  t.okada
 * add publisher and place to publication
 *
 * Revision 1.22  2002/08/23 01:59:36  t.okada
 * removal sun.java2d.pipe.NullPixelPipe
 *
 * Revision 1.21  2002/08/22 13:28:36  t.okada
 * It corrects for a locale colum addition.
 *
 * Revision 1.20  2002/08/20 13:15:20  t.okada
 * addition notes column and lines column
 *
 * Revision 1.19  2002/08/15 05:16:55  ryo
 * change type of year into String
 *
 * Revision 1.18  2002/08/06 13:47:29  ryo
 * create authority data temporarily
 *
 * Revision 1.17  2002/08/02 11:20:17  ryo
 * fix annotation linkage processing bug
 *
 * Revision 1.16  2002/08/01 13:08:46  ryo
 * fix Annotation linkage bug
 *
 * Revision 1.15  2002/07/30 08:46:03  ryo
 * fix search via servlet bug
 *
 * Revision 1.14  2002/07/29 04:59:57  ryo
 * change column name of NameUsage table from 'note' to 'notes'
 *
 * Revision 1.13  2002/07/18 14:40:19  t.okada
 * note item is added to nameusage
 *
 * Revision 1.12  2002/07/12 16:53:29  ryo
 * move member and accessor from *Connection to AbstractConnection
 *
 * Revision 1.11  2002/07/08 07:43:14  ryo
 * modify SQL statement because unused table removed
 *
 * Revision 1.10  2002/07/03 08:32:28  t.okada
 * Encoding at the time of connection is set as Unicode.
 *
 * Revision 1.9  2002/06/27 13:14:12  ryo
 * fix NullPointerException bug
 *
 * Revision 1.8  2002/06/26 11:15:20  ryo
 * CRLF conversion
 *
 * Revision 1.7  2002/06/25 14:45:07  ryo
 * remove the processing which generates NamedObject in the surroundings of NamedObject
 *
 * Revision 1.6  2002/06/25 10:08:53  ryo
 * change database column name from 'number' to 'issue'
 *
 * Revision 1.5  2002/06/24 09:23:38  ryo
 * change the timing of a commitment and add the processing which adds data to a link table
 *
 * Revision 1.4  2002/06/21 14:09:42  ryo
 * add copyright
 *
 * Revision 1.2  2002/06/21 13:02:56  ryo
 * add copyright
 *
 */

package org.nomencurator.broker;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.ResultSet;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Properties;
import java.util.Locale;
import java.util.Hashtable;

import org.nomencurator.NamedObject;
import org.nomencurator.NameUsage;
import org.nomencurator.Appearance;
import org.nomencurator.Annotation;
import org.nomencurator.Publication;
import org.nomencurator.Author;
import org.nomencurator.editor.NameUsageListRow;

import org.nomencurator.controller.linktable.NameUsageHigher;
import org.nomencurator.controller.linktable.NameUsageAuthority;
import org.nomencurator.controller.linktable.NameUsageRecorder;
import org.nomencurator.controller.linktable.AppearanceNameUsage;
import org.nomencurator.controller.linktable.PublicationAppearance;
import org.nomencurator.controller.linktable.PublicationAuthor;
import org.nomencurator.controller.linktable.AppearanceAnnotation;
import org.nomencurator.controller.linktable.AnnotationFrom;
import org.nomencurator.controller.linktable.AnnotationTo;
//import org.nomencurator.controller.linktable.LinkRecord;

public class JdbcConnection extends AbstractConnection {

	/**
	 * connection used for access to a database.
	 */
	protected Connection connection = null;

	/**
	 * Constructor.
	 */
	public JdbcConnection() {
	}

	/**
	 * Constructor.
	 */
	public JdbcConnection(
		String driverName,
		String url,
		String username,
		String password) {
		this.driverName = driverName;
		this.url = url;
		this.username = username;
		this.password = password;
	}

	/**
	 * Get connection type.
	 * @return String connection type
	 */
	public String getType() {
		return "JDBC";
	}

	/**
	 * Connect to a database.
	 * @return Connection connection to a database.
	 * @see AbstractConnection#connect()
	 */
	public Connection connect() {
		if (connection == null) {
			try {
				Class.forName(driverName);
				
				Properties props = new Properties();
				  props.put("user", username);
				  props.put("password", password);
				  props.put("charSet","UTF8"); //set encode name
				
				connection = DriverManager.getConnection(url,props);
				
				//connection = DriverManager.getConnection(url, username, password);
			} catch (ClassNotFoundException e) {
				e.printStackTrace();
				connection = null;
			} catch (SQLException e) {
				e.printStackTrace();
				connection = null;
			}
		}
		return connection;
	}

	/**
	 * Disconnect from a database.
	 * @see AbstractConnection#disconnect()
	 */
	public void disconnect() {
		try {
			if (connection != null)
				connection.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Search NameUsage objects.
	 * @param name name string of NameUsage object.
	 * @param author author string of Author object.
	 * @param year year of Publication object.
	 * @return Vector result record set.
	 * @see AbstractConnection#searchNameUsage(String, String, String)
	 */
	public Vector searchNameUsage(String name, String author, String year) {
		// create sql statement
		StringBuffer sql = new StringBuffer();
//		sql.append(
//			"select distinct a.id, a.name, c.authors, c.year "
//				+ "from nameusage a, author b, publication c, appearance d, "
//				+ "lt_publication_authorlist e, "
//				+ "lt_publication_appearances f, "
//				+ "lt_appearance_nameusages g "
//				+ "where b.id = e.linkto "
//				+ "and e.linkfrom = c.id "
//				+ "and c.id = f.linkfrom "
//				+ "and f.linkto = d.id "
//				+ "and d.id = g.linkfrom "
//				+ "and g.linkto = a.id "
//				+ "and a.name like '%"
//				+ name
//				+ "%' ");
//		if (!author.equals("")) {
//			sql.append("and (b.surname like '%" + author + "%' ");
//			sql.append("or b.firstname like '%" + author + "%' ");
//			sql.append("or b.middlename like '%" + author + "%') ");
//		}
//		if (!year.equals("")) {
//			sql.append("and c.year like '%" + year + "%'");
//		}
		sql.append(
			" select distinct n.id, n.ascribedname, p.authors, p.year "
				+ " from "
				+ " nameusage n, author a, publication p, appearance ap, "
				+ " authorPublication a_p "
				+ " where  "
				+ " p.id = a_p.publication "
				+ " and a_p.author = a.id "
				+ " and p.id = ap.publication "
				+ " and ap.id = n.appearance "
				+ " and n.ascribedname like '%" + name + "%' ");
		if (!author.equals("")) {
			sql.append("and (a.surname like '%" + author + "%' ");
			sql.append("or a.firstname like '%" + author + "%' ");
			sql.append("or a.middlename like '%" + author + "%') ");
		}
		if (!year.equals("")) {
			sql.append("and p.year like '%" + year + "%'");
		}

		// execute sql statement and get result
		Vector v = new Vector();
		try {
			Statement s = connection.createStatement();
System.out.println(sql.toString());
			ResultSet r = s.executeQuery(sql.toString());
			while (r.next()) {
				NameUsageListRow row = new NameUsageListRow();
				row.setObjectId(new String(r.getString(1)));
				row.setName(new String(r.getString(2)));
				row.setAuthor(new String(r.getString(3)));
				row.setYear(new String(r.getString(4)));
				v.add(row);
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return v;
	}

	/**
	 * Get a NamedObject.
	 * @param oid object id of NamedObject.
	 * @return NamedObject NamedObject.
	 * @see AbstractConnection#getNamedObject(String)
	 */
	public NamedObject getNamedObject(String oid) {
		NamedObject namedObject = null;
		if (oid == null || oid.length() <= 0)
			return null;
		try {
			if (oid.startsWith("NameUsage")) {
				namedObject = getNameUsage(oid);
			} else if (oid.startsWith("Appearance")) {
				namedObject = getAppearance(oid);
			} else if (oid.startsWith("Publication")) {
				namedObject = getPublication(oid);
			} else if (oid.startsWith("Author")) {
				namedObject = getAuthor(oid);
			} else if (oid.startsWith("Annotation")) {
				namedObject = getAnnotation(oid);
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return namedObject;
	}

	/**
	 * Get a NameUsage.
	 * @param oid object id of NameUsage.
	 * @return NameUsage NameUsage.
	 */
	private NameUsage getNameUsage(String oid) throws SQLException {
		NamedObjectBroker broker = NamedObjectBroker.getInstance();
		boolean toBeResolved = false;
		NameUsage nameUsage = null;

		// create sql statement
		Statement s = connection.createStatement();
		ResultSet r;
		StringBuffer sql = new StringBuffer();
//		sql.append(
//			"select id, rank, name, authority, "
//				+ "appearance, highertaxon, "
//				+ "type, year, incertaeseds, notes, locale "
//				+ "from nameusage "
//				+ "where id = '"
//				+ oid
//				+ "'");
		sql.append(
			"select nu.id, nu.rank, nu.ascribedname, nu.authority, "
				+ "nu.appearance, nu.highertaxon, "
				+ "nu.type, nu.year, nu.incertaesedis, nob.notes, nu.locale "
				+ "from nameusage nu, namedobject nob "
				+ "where nu.id = '"
				+ oid
				+ "' "
				+ "and nu.id = nob.id ");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		String authorityOid = null;
		String appearanceOid = null;
		String highertaxonOid = null;
		boolean type = false;
		boolean incertaeseds = false;
		while (r.next()) {
			nameUsage = new NameUsage();
			String str;
			str = r.getString(2);//nu.rank");
			if (r.wasNull())
				str = null;
			nameUsage.setRank(str);
			str = r.getString(3);//"nu.ascribedname");
			if (r.wasNull())
				str = null;
			nameUsage.setAscribedName(str);

			str = r.getString(10);//"nob.notes");
			if (r.wasNull())
				str = null;
			nameUsage.setNote(str);
			
			str = r.getString(11);//"nu.locale");
			if (r.wasNull()) {
				str = null;
				Locale locale = new Locale(" ", "");
				nameUsage.setLocale(locale);
			} else {
				String lang, coun;
				if(!str.equals("") && str.length() > 2) {
					lang = str.substring(0, str.indexOf("_"));
					coun = str.substring(str.indexOf("_") + 1) ;
				} else {
					lang = " ";
					coun = "";
				}
				Locale locale = new Locale(lang, coun);
				nameUsage.setLocale(locale);
			}
			
			authorityOid = r.getString(4);//"nu.authority");
			if (r.wasNull())
				authorityOid = null;
			appearanceOid = r.getString(5);//"nu.appearance");
			if (r.wasNull())
				appearanceOid = null;
			highertaxonOid = r.getString(6);//"nu.highertaxon");
			if (r.wasNull())
				highertaxonOid = null;
			type = r.getBoolean(7);//"nu.type");
			nameUsage.setType(type);
			incertaeseds = r.getBoolean(9);//"nu.incertaesedis");
			nameUsage.setIncertaeSedis(incertaeseds);
		}
		if (authorityOid != null) {
//			NameUsageAuthority.getInstance().addLinkRecord(
//				new LinkRecord(oid, authorityOid));
//			NameUsage authority = new NameUsage();
//			authority.setPersistentID(authorityOid);
//			nameUsage.setAuthority(authority);
			
			
			String pID = authorityOid;
			NameUsage authority =
				(NameUsage) broker.getNamedObjectInPool(pID);
			if (authority == null) {
				authority = new NameUsage();
				authority.setName(pID);
				broker.putUnresolved(authority);
			}
			
			
			
		}
		if (appearanceOid != null) {
//			NameUsageRecorder.getInstance().addLinkRecord(
//				new LinkRecord(oid, appearanceOid));
//			Appearance app = new Appearance();	// for servlet connection.
//			app.setPersistentID(appearanceOid);
//			nameUsage.setAppearance(app);

			String pID = appearanceOid;
			Appearance appearance = (Appearance) broker.getNamedObjectInPool(pID);
			if (appearance != null) {
				toBeResolved = true;
			} else {
				appearance = new Appearance();
				appearance.setName(pID);
				broker.putUnresolved(appearance);
				appearance.addNameUsage(nameUsage);
			}

		}
		if (highertaxonOid != null) {
//			NameUsageHigher.getInstance().addLinkRecord(
//				new LinkRecord(oid, highertaxonOid));
			String pID = highertaxonOid;
		    NameUsage higherTaxon = (NameUsage)broker.getNamedObjectInPool(pID);
		    if(higherTaxon != null) {
				toBeResolved = true;
		    }
		    else {
				higherTaxon = new NameUsage();
				higherTaxon.setName(pID);
				broker.putUnresolved(higherTaxon);
				if(higherTaxon.getLowerTaxa() == null)
					higherTaxon.setLowerTaxa(new Vector());
				higherTaxon.getLowerTaxa().addElement(nameUsage);
			}
		}
		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkfrom "
//				+ "from lt_annotation_annotators "
//				+ "where linkto = '"
//				+ oid
//				+ "'");
		sql.append(
			"select annotation "
				+ "from annotators "
				+ "where annotator = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String linkfrom = r.getString("annotation");
			if (r.wasNull())
				continue;
//			AnnotationFrom.getInstance().addLinkRecord(new LinkRecord(linkfrom, oid));
//			Annotation ann = new Annotation();
//			ann.setPersistentID(linkfrom);
//			nameUsage.addAnnotation(ann);

		    String pID = linkfrom;;
		    Annotation a =
			(Annotation)broker.getNamedObjectInPool(pID);
		    if(a != null) {
				if(nameUsage.getAnnotations() == null)
				   nameUsage.setAnnotations(new Vector());
				nameUsage.getAnnotations().addElement(a);
				toBeResolved = true;
		    }
		    else {
				a = new Annotation();
				a.setName(pID);
				broker.putUnresolved(a);
				nameUsage.addAnnotation(a);
		    }

		}

		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkto "
//				+ "from lt_nameusage_lowertaxa "
//				+ "where linkfrom = '"
//				+ oid
//				+ "'");
		sql.append(
			"select id "
				+ "from nameusage "
				+ "where highertaxon = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String linkto = r.getString("id");
			if (r.wasNull())
				continue;
//			NameUsageHigher.getInstance().addLinkRecord(new LinkRecord(linkto, oid));
//			NameUsage lower = new NameUsage();	// for servlet connection
//			lower.setPersistentID(linkto);
//			nameUsage.addLowerTaxon(lower);


		    String pID = linkto;
		    NameUsage n = (NameUsage)broker.getNamedObjectInPool(pID);
		    if(n != null) {
				if(nameUsage.getLowerTaxa() == null)
				    nameUsage.setLowerTaxa(new Vector());
				nameUsage.getLowerTaxa().addElement(nameUsage);
				toBeResolved = true;
			}
		    else {
				n = new NameUsage();
				n.setName(pID);
				broker.putUnresolved(n);
				nameUsage.addLowerTaxon(n);
		    }


		}

		nameUsage.setPersistentID(oid);

		if(toBeResolved)
		    broker.resolve(nameUsage);

		return nameUsage;
	}

	/**
	 * Get an Appearance.
	 * @param oid object id of Appearance.
	 * @return Appearance Appearance.
	 */
	private Appearance getAppearance(String oid) throws SQLException {
		NamedObjectBroker broker = NamedObjectBroker.getInstance();
		boolean toBeResolved = false;
		Appearance appearance = null;

		// create sql statement
		Statement s = connection.createStatement();
		ResultSet r;
		StringBuffer sql = new StringBuffer();
		sql.append(
			"select id, page, lines, appearance, publication "
				+ "from appearance "
				+ "where id = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			appearance = new Appearance();
			String str;
			str = r.getString("page");
			if (r.wasNull())
				str = null;
			appearance.setPage(str);
			
			str = r.getString("lines");
			if(r.wasNull())
				str = null;
			appearance.setLines(str);
			
			str = r.getString("appearance");
			if (r.wasNull())
				str = null;
			appearance.setAppearance(str);
			str = r.getString("publication");
			if (!r.wasNull()) {
//				PublicationAppearance.getInstance().addLinkRecord(new LinkRecord(str, oid));
//				Publication pub = new Publication();	// for servlet connection.
//				pub.setPersistentID(str);
//				appearance.setPublication(pub);
			    String pID = str;
			    Publication p =
					(Publication)broker.getNamedObjectInPool(pID);
			    if(p != null) {
					appearance.setPublication(p);
					toBeResolved = true;
			    }
			    else {
					p = new Publication();
					p.setName(pID);
					broker.putUnresolved(p);
					p.addAppearance(appearance);
			    }

			}
		}

		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkto from lt_appearance_nameusages "
//				+ "where linkfrom = '"
//				+ oid
//				+ "'");
		sql.append(
			"select id from nameusage "
				+ "where appearance = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String str = r.getString("id");
			if (!r.wasNull()) {
				//AppearanceNameUsage.getInstance().addLinkRecord(new LinkRecord(oid, str));
				
			    String pID = str;
			    NameUsage n =
				(NameUsage)broker.getNamedObjectInPool(pID);
			    if(n != null) {
					appearance.getNameUsages().addElement(n);
					toBeResolved = true;
			    }
			    else {
					n = new NameUsage();
					n.setName(pID);
					broker.putUnresolved(n);
					appearance.addNameUsage(n);
			    }

			}
		}

		//
		sql = new StringBuffer();
		sql.append(
			"select id from annotation "
				+ "where appearance = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String str = r.getString("id");
			if (!r.wasNull()) {
//				AppearanceAnnotation.getInstance().addLinkRecord(new LinkRecord(oid, str));
//				Annotation ann = new Annotation();
//				ann.setPersistentID(str);
//				appearance.addAnnotation(ann);

			    String pID = str;
			    Annotation a =
				(Annotation)broker.getNamedObjectInPool(pID);
			    if(a != null) {
					appearance.getAnnotations().addElement(a);
					toBeResolved = true;
			    }
			    else {
					a = new Annotation();
					a.setName(pID);
					broker.putUnresolved(a);
					appearance.addAnnotation(a);
			    }

			}
		}

		return appearance;
	}

	/**
	 * Get an Author object.
	 * @param oid object id of Author.
	 * @return Author Author.
	 */
	private Author getAuthor(String oid) throws SQLException {
		Author author = null;
		// create sql statement
		Statement s = connection.createStatement();
		ResultSet r;
		StringBuffer sql = new StringBuffer();
		sql.append(
			"select id, surname, firstname, middlename, title, "
				+ "epithet, surnamePrefix "
				+ "from author "
				+ "where id = '"
				+ oid
				+ "'");
//System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			author = new Author();
			String str;
			str = r.getString("surname");
			if (r.wasNull())
				str = "";
			author.setSurname(str);
			str = r.getString("firstname");
			if (r.wasNull())
				str = "";
			author.setFirstName(str);
			str = r.getString("middlename");
			if (r.wasNull())
				str = "";
			author.setMiddleName(str);
			str = r.getString("title");
			if (r.wasNull())
				str = null;
			author.setTitle(str);
			str = r.getString("epithet");
			if (r.wasNull())
				str = null;
			author.setEpithet(str);
			str = r.getString("surnamePrefix");
			if (r.wasNull())
				str = null;
			author.setFeudality(str);
			author.setPersistentID(oid);
		}
		return author;
	}

	/**
	 * Get a Publication object.
	 * @param oid object id of Publication.
	 * @return Publication Publication.
	 */
	private Publication getPublication(String oid) throws SQLException {
		NamedObjectBroker broker = NamedObjectBroker.getInstance();
		boolean toBeResolved = false;
		Hashtable hash = new Hashtable();
		int authorCount = 0;

		Publication publication = null;
		// create sql statement
		Statement s = connection.createStatement();
		ResultSet r;
		StringBuffer sql = new StringBuffer();
		sql.append(
			"select p.id, p.authors, p.affiliations, "
				+ "p.citationtitle, p.contentstitle, p.isxn, p.year, p.volume, "
				+ "p.issue, p.firstpage, p.lastpage, n.notes, p.partof, "
				+ "p.received, p.revised, p.publisher, p.place, p.accepted "
				+ "from publication p, namedobject n "
				+ "where p.id = '"
				+ oid
				+ "' and p.id = n.id ");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			publication = new Publication();
			String str;
			//			publication.getAuthorListSummary();
			str = r.getString(3);//"p.affiliations");
			if (r.wasNull())
				str = null;
			publication.setAffiliationsString(str);
			str = r.getString(2);//"p.authors");
			if (r.wasNull())
				str = null;
			publication.setAuthorNames(str);
			str = r.getString(4);//"p.citationtitle");
			if (r.wasNull())
				str = null;
			publication.setCitationTitle(str);
			str = r.getString(5);//"p.contentstitle");
			if (r.wasNull())
				str = null;
			publication.setContentsTitle(str);
			str = r.getString(6);//"p.isxn");
			if (r.wasNull())
				str = null;
			publication.setISXN(str);
			str = r.getString(7);//"p.year");
			if (r.wasNull())
				str = null;
			publication.setYear(str);
			str = r.getString(8);//"p.volume");
			if (r.wasNull())
				str = null;
			publication.setVolume(str);
			str = r.getString(9);//"p.issue");
			if (r.wasNull())
				str = null;
			publication.setNumber(str);
			str = r.getString(10);//"p.firstpage");
			if (r.wasNull())
				str = null;
			publication.setFirstPage(str);
			str = r.getString(11);//"p.lastpage");
			if (r.wasNull())
				str = null;
			publication.setLastPage(str);
			str = r.getString(16);//"p.publisher");
			if (r.wasNull())
				str = null;
			publication.setPublisher(str);
			str = r.getString(17);//"p.place");
			if (r.wasNull())
				str = null;
			publication.setPlace(str);
			str = r.getString(12);//"n.notes");
			if (r.wasNull())
				str = null;
			publication.setNotes(str);
			//			publication.setReceived(r.getDate("received"));
			//			publication.setRevised(r.getDate("revised"));
			//			publication.setAccepted(r.getDate("accepted"));
			//			r.getString("partof")
			publication.setPersistentID(oid);
		}

		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkto, seq "
//				+ "from lt_publication_authorlist "
//				+ "where linkfrom = '"
//				+ oid
//				+ "' "
//				+ "order by seq ");
		sql.append(
			"select author, sequence "
				+ "from authorpublication "
				+ "where publication = '"
				+ oid
				+ "' "
				+ "order by sequence ");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String linkto = r.getString("author");
			if (r.wasNull())
				continue;
			String seq = r.getString("sequence");
			if (r.wasNull())
				continue;
//			PublicationAuthor.getInstance().addLinkRecord(new LinkRecord(oid, linkto, seq));
//			Author author = new Author();	// for servlet connection
//			author.setPersistentID(linkto);
//			publication.addAuthor(author);
			
		    String pID = linkto;
		    hash.put(seq, pID);
		    authorCount++;
		}

		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkto "
//				+ "from lt_publication_appearances "
//				+ "where linkfrom = '"
//				+ oid
//				+ "' ");
		sql.append(
			"select id "
				+ "from appearance "
				+ "where publication = '"
				+ oid
				+ "' ");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String linkto = r.getString("id");
			if (r.wasNull())
				continue;
			//PublicationAppearance.getInstance().addLinkRecord(new LinkRecord(oid, linkto));
		    String pID = linkto;
		    Appearance ap = (Appearance)broker.getNamedObjectInPool(pID);
		    if(ap != null) {
				publication.getAppearances().addElement(ap);
				toBeResolved = true;
		    }
		    else {
				ap = new Appearance();
				ap.setName(pID);
				broker.putUnresolved(ap);
				publication.addAppearance(ap);
		    }
		}


		if(authorCount > 0) {
//		    publication.setAuthors(new Vector(authorCount));
		    String pID = null;
		    for(int i = 0; i < authorCount; i++) {
				pID = (String)hash.get(Integer.toString(i));
				Author au =
				    (Author)broker.getNamedObjectInPool(pID);
				if(au != null) {
//				    publication.getAuthors().addElement(au);
				    publication.addAuthor(au);
				    toBeResolved = true;
				}
				else {
				    au = new Author();
				    au.setName(pID);
				    broker.putUnresolved(au);
				    publication.addAuthor(au);
				}
		    }
		}

		return publication;
	}

	/**
	 * Get an Annotation object.
	 * @param oid object id of Annotation.
	 * @return Annotation Annotation.
	 */
	private Annotation getAnnotation(String oid) throws SQLException {
		NamedObjectBroker broker = NamedObjectBroker.getInstance();
		boolean toBeResolved = false;
		Annotation annotation = null;
		// create sql statement
		Statement s = connection.createStatement();
		ResultSet r;
		StringBuffer sql = new StringBuffer();
		sql.append(
			"select id, linktype, appearance "
				+ "from annotation "
				+ "where id = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			annotation = new Annotation();
			String str;
			str = r.getString("linktype");
			if (r.wasNull())
				str = null;
			annotation.setLinkType(str);
			str = r.getString("appearance");
			if (r.wasNull())
				continue;
//			AppearanceAnnotation.getInstance().addLinkRecord(new LinkRecord(str, oid));
//			Appearance app = new Appearance();
//			app.setPersistentID(str);
//			annotation.setAppearance(app);
		    String pID = str;
		    Appearance ap =
			(Appearance)broker.getNamedObjectInPool(pID);
		    if(ap != null) {
				annotation.setAppearance(ap);
				toBeResolved = true;
		    }
		    else {
				ap = new Appearance();
				ap.setName(pID);
				broker.putUnresolved(ap);
				ap.addAnnotation(annotation);
		    }

		}

		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkto "
//				+ "from lt_annotation_annotatants "
//				+ "where linkfrom = '"
//				+ oid
//				+ "'");
		sql.append(
			"select annotatant "
				+ "from annotatants "
				+ "where annotation = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String linkto = r.getString("annotatant");
			if (r.wasNull())
				continue;
//			AnnotationTo.getInstance().addLinkRecord(new LinkRecord(oid, linkto));
//			NameUsage nu = new NameUsage();
//			nu.setPersistentID(linkto);
//			annotation.addAnnotatant(nu);

		    String pID = linkto;
		    NameUsage n =
			(NameUsage)broker.getNamedObjectInPool(pID);
		    if(n != null) {
				if(annotation.getAnnotatants() == null)
				    annotation.setAnnotatants(new Vector());
				annotation.getAnnotatants().addElement(n);
				toBeResolved = true;
		    }
		    else {
				n = new NameUsage();
				n.setName(pID);
				broker.putUnresolved(n);
				annotation.addAnnotatant(n);
		    }


		}

		//
		sql = new StringBuffer();
//		sql.append(
//			"select linkto "
//				+ "from lt_annotation_annotators "
//				+ "where linkfrom = '"
//				+ oid
//				+ "'");
		sql.append(
			"select annotator "
				+ "from annotators "
				+ "where annotation = '"
				+ oid
				+ "'");
System.out.println(sql.toString());
		r = s.executeQuery(sql.toString());
		while (r.next()) {
			String linkto = r.getString("annotator");
			if (r.wasNull())
				continue;
//			AnnotationFrom.getInstance().addLinkRecord(new LinkRecord(oid, linkto));
//			NameUsage nu = new NameUsage();
//			nu.setPersistentID(linkto);
//			annotation.addAnnotator(nu);

		    String pID = linkto;
		    NameUsage n =
			(NameUsage)broker.getNamedObjectInPool(pID);
		    if(n != null) {
				if(annotation.getAnnotators() == null)
				    annotation.setAnnotators(new Vector());
				annotation.getAnnotators().addElement(n);
				toBeResolved = true;
		    }
		    else {
				n = new NameUsage();
				n.setName(pID);
				broker.putUnresolved(n);
				annotation.addAnnotator(n);
		    }
		}
		annotation.setPersistentID(oid);

		return annotation;
	}

	/**
	 * Save NamedObjects.
	 * @param Vector<NamedObject>
	 * @return int inserted row count
	 * @see AbstractConnection#saveNamedObject(Vector)
	 */
	public int saveNamedObject(Vector v) {
		int rows = 0;
		boolean autoCommit = true;
		try {
			autoCommit = connection.getAutoCommit();
			connection.setAutoCommit(false);
		} catch (SQLException e) {
			e.printStackTrace();
		}
		for (Enumeration e = v.elements(); e.hasMoreElements();) {
			Object o = e.nextElement();
			if (o == null)
				continue;
			try {
				if (o instanceof NameUsage) {
					rows += saveNameUsage((NameUsage) o);
				} else if (o instanceof Appearance) {
					rows += saveAppearance((Appearance) o);
				} else if (o instanceof Publication) {
					rows += savePublication((Publication) o);
				} else if (o instanceof Author) {
					rows += saveAuthor((Author) o);
				} else if (o instanceof Annotation) {
					rows += saveAnnotation((Annotation) o);
				}
			} catch (SQLException ex) {
				ex.printStackTrace();
			}
		}
		try {
			if (rows > 0)
				connection.commit();
			connection.setAutoCommit(autoCommit);
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return rows;
	}

	/**
	 * Save NameUsage.
	 * @param o NameUsage.
	 * @return int inserted row count.
	 */
	private int saveNameUsage(NameUsage o) throws SQLException {
		// create sql statement
		Statement s = connection.createStatement();
		StringBuffer sql = new StringBuffer();
//		sql.append(
//			"insert into nameusage (id, rank, name, authority, "
//				+ "appearance, highertaxon, "
//				+ "type, year, incertaeseds, notes, locale) values (");
		sql.append(
			"insert into nameusage (id, rank, ascribedname, authority, "
				+ "appearance, highertaxon, "
				+ "type, year, incertaesedis, locale) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getRank() + "',");
		sql.append("'" + o.getAscribedName() + "',");
		if (o.getAuthority() != null)
			sql.append("'" + o.getAuthority().getPersistentID() + "',");
		else
			sql.append("null,");
		if (o.getAppearance() != null)
			sql.append("'" + o.getAppearance().getPersistentID() + "',");
		else
			sql.append("null,");
		if (o.getHigherTaxon() != null)
			sql.append("'" + o.getHigherTaxon().getPersistentID() + "',");
		else
			sql.append("null,");
//		sql.append("'" + o.isType() + "',");
		sql.append(o.isType() + ",");
		sql.append("null,");
//		sql.append("'" + o.isIncertaeSedis() + "',");
		sql.append(o.isIncertaeSedis() + ",");
		if(o.getLocale() != null)
			sql.append("'" + o.getLocale().toString() + "'");
		else
			sql.append("null");
		sql.append(")");
System.out.println(sql.toString());
		int r = s.executeUpdate(sql.toString());

		sql = new StringBuffer();
		sql.append("insert into namedobject (id, persistentID, notes) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getPersistentID() + "',");
		if (o.getNote() != null)
			sql.append("'" + o.getNote() + "'");
		else
			sql.append("null");
		sql.append(")");
System.out.println(sql.toString());
		r = s.executeUpdate(sql.toString());


//		if (o.getAppearance() != null) {
//			sql = new StringBuffer();
//			sql.append("insert into lt_appearance_nameusages (linkfrom, linkto) ");
//			sql.append("values (");
//			sql.append("insert into lt_appearance_nameusages (linkfrom, linkto) ");
//			sql.append("values (");
//			sql.append("'" + o.getAppearance().getPersistentID() + "',");
//			sql.append("'" + o.getPersistentID() + "')");
////System.out.println(sql.toString());
//			r = s.executeUpdate(sql.toString());
//		}

		Vector v;
//		Vector v = o.getAnnotations();
//		if (v != null) {
//			for (Enumeration e = v.elements(); e.hasMoreElements();) {
//				Annotation annotation = (Annotation) e.nextElement();
//				sql = new StringBuffer();
//				sql.append("insert into lt_annotation_annotators (linkfrom, linkto) ");
//				sql.append("values (");
//				sql.append("'" + annotation.getPersistentID() + "',");
//				sql.append("'" + o.getPersistentID() + "')");
////System.out.println(sql.toString());
//				r = s.executeUpdate(sql.toString());
//			}
//		}

//		NameUsage nameUsage = o.getHigherTaxon();
//		if (nameUsage != null) {
//			sql = new StringBuffer();
//			sql.append("insert into lt_nameusage_lowertaxa (linkfrom, linkto) ");
//			sql.append("values (");
//			sql.append("'" + nameUsage.getPersistentID() + "',");
//			sql.append("'" + o.getPersistentID() + "')");
////System.out.println(sql.toString());
//			r = s.executeUpdate(sql.toString());
//		}

//		v = o.getLowerTaxa();
//		if (v != null) {
//			for (Enumeration e = v.elements(); e.hasMoreElements();) {
//				nameUsage = (NameUsage) e.nextElement();
//				sql = new StringBuffer();
//				sql.append("insert into lt_nameusage_lowertaxa (linkfrom, linkto) ");
//				sql.append("values (");
//				sql.append("'" + o.getPersistentID() + "',");
//				sql.append("'" + nameUsage.getPersistentID() + "')");
////System.out.println(sql.toString());
//				r = s.executeUpdate(sql.toString());
//			}
//		}
		return 1;
	}

	/**
	 * Save Appearance.
	 */
	private int saveAppearance(Appearance o) throws SQLException {
		// create sql statement
		Statement s = connection.createStatement();
		StringBuffer sql = new StringBuffer();
		sql.append("insert into appearance (id, page, lines, appearance, publication) ");
		sql.append("values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getPage() + "',");
		sql.append("'" + o.getLines() + "',");
		sql.append("'" + o.getAppearance() + "',");
		if (o.getPublication() != null)
			sql.append("'" + o.getPublication().getPersistentID() + "')");
		else
			sql.append("null)");
System.out.println(sql.toString());
		int r = s.executeUpdate(sql.toString());

//		if (o.getPublication() != null) {
//			sql = new StringBuffer();
//			sql.append("insert into lt_publication_appearances");
//			sql.append("(linkfrom, linkto) values (");
//			sql.append("'" + o.getPublication().getPersistentID() + "',");
//			sql.append("'" + o.getPersistentID() + "')");
////System.out.println(sql.toString());
//			r = s.executeUpdate(sql.toString());
//		}
		sql = new StringBuffer();
		sql.append("insert into namedobject (id, persistentID, notes) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("null");
		sql.append(")");
System.out.println(sql.toString());
		r = s.executeUpdate(sql.toString());

		return 1;
	}

	/**
	 * Save Publication.
	 */
	private int savePublication(Publication o) throws SQLException {
		// create sql statement
		Statement s = connection.createStatement();
		StringBuffer sql = new StringBuffer();
		sql.append("insert into publication (id, authors, affiliations, ");
		sql.append("citationtitle, contentstitle, isxn, year, volume, ");
		sql.append("issue, firstpage, lastpage, partof, ");
		sql.append("received, revised, publisher, place, accepted )\r\n");
		sql.append(" values (");
		sql.append("'" + o.getPersistentID() + "',");
		if (o.getAuthorNames() != null && o.getAuthorNames().length() != 0)
			sql.append("'" + o.getAuthorNames() + "',");
		else
			sql.append("null,");
		if (o.getAffiliations() != null && o.getAffiliations().length() != 0)
			sql.append("'" + o.getAffiliations() + "',");
		else
			sql.append("null,");
		if (o.getCitationTitle() != null && o.getCitationTitle().length() != 0)
			sql.append("'" + o.getCitationTitle() + "',");
		else
			sql.append("null,");
		if (o.getContentsTitle() != null && o.getContentsTitle().length() != 0)
			sql.append("'" + o.getContentsTitle() + "',");
		else
			sql.append("null,");
		if (o.getISXN() != null && o.getISXN().length() != 0)
			sql.append("'" + o.getISXN() + "',");
		else
			sql.append("null,");
		if (o.getYear() != null && o.getYear().length() != 0)
			sql.append("'" + o.getYear() + "',");
		else
			sql.append("null,");
		if (o.getVolume() != null && o.getVolume().length() != 0)
			sql.append("'" + o.getVolume() + "',");
		else
			sql.append("null,");
		if (o.getNumber() != null && o.getNumber().length() != 0)
			sql.append("'" + o.getNumber() + "',");
		else
			sql.append("null,");
		if (o.getFirstPage() != null && o.getFirstPage().length() != 0)
			sql.append("'" + o.getFirstPage() + "',");
		else
			sql.append("null,");
		if (o.getLastPage() != null && o.getLastPage().length() != 0)
			sql.append("'" + o.getLastPage() + "',");
		else
			sql.append("null,");
		if (o.getContainer() != null && o.getContainer().length() != 0)
			sql.append("'" + o.getContainer().getPersistentID() + "',");
		else
			sql.append("null,");
		if (o.getReceived() != null)
			sql.append("'" + o.getReceived() + "',");
		else
			sql.append("null,");
		if (o.getRevised() != null)
			sql.append("'" + o.getRevised() + "',");
		else
			sql.append("null,");
		if(o.getPublisher() != null && o.getPublisher().length() != 0)
			sql.append("'" + o.getPublisher() + "',");
		else
			sql.append("null,");
		if(o.getPlace() != null && o.getPlace().length() != 0)
			sql.append("'" + o.getPlace() + "',");
		else
			sql.append("null,");
		if (o.getAccepted() != null)
			sql.append("'" + o.getAccepted() + "')");
		else
			sql.append("null)");
System.out.println(sql.toString());
		int r = s.executeUpdate(sql.toString());

		sql = new StringBuffer();
		sql.append("insert into namedobject (id, persistentID, notes) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getPersistentID() + "',");
		if (o.getNotes() != null && o.getNotes().length() != 0)
			sql.append("'" + o.getNotes() + "'");
		else
			sql.append("null");
		sql.append(")");
System.out.println(sql.toString());
		r = s.executeUpdate(sql.toString());


		Vector v = o.getAuthors();
		if (v != null) {
			for (int i = 0; i < v.size(); i++) {
				Author author = (Author) v.elementAt(i);
				if (author == null)
					continue;
				sql = new StringBuffer();
//				sql.append("insert into lt_publication_authorlist");
//				sql.append("(linkfrom, linkto, seq) values (");
				sql.append("insert into authorpublication");
				sql.append("(publication, author, sequence) values (");
				sql.append("'" + o.getPersistentID() + "',");
				sql.append("'" + author.getPersistentID() + "',");
				sql.append(i + ")");
System.out.println(sql.toString());
				r = s.executeUpdate(sql.toString());
			}
		}
		return 1;
	}

	/**
	 * Save Author.
	 */
	private int saveAuthor(Author o) throws SQLException {
		// create sql statement
		Statement s = connection.createStatement();
		StringBuffer sql = new StringBuffer();
		sql.append("insert into author (id, surname, firstname, middlename, ");
		sql.append("title, epithet, surnamePrefix) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getSurname() + "',");
		sql.append("'" + o.getFirstName() + "',");
		sql.append("'" + o.getMiddleName() + "',");
		sql.append("'" + o.getTitle() + "',");
		sql.append("'" + o.getEpithet() + "',");
		sql.append("'" + o.getFeudality() + "')");
//System.out.println(sql.toString());
		int r = s.executeUpdate(sql.toString());

		sql = new StringBuffer();
		sql.append("insert into namedobject (id, persistentID, notes) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("null");
		sql.append(")");
//System.out.println(sql.toString());
		r = s.executeUpdate(sql.toString());

		return 1;
	}

	/**
	 * Save Annotation
	 */
	private int saveAnnotation(Annotation o) throws SQLException {
		// create sql statement
		Statement s = connection.createStatement();
		StringBuffer sql = new StringBuffer();
		sql.append("insert into annotation (id, linktype, appearance) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getLinkType() + "',");
		if (o.getAppearance() != null)
			sql.append("'" + o.getAppearance().getPersistentID() + "')");
		else
			sql.append("null)");
System.out.println(sql.toString());
		int r = s.executeUpdate(sql.toString());

		sql = new StringBuffer();
		sql.append("insert into namedobject (id, persistentID, notes) values (");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("'" + o.getPersistentID() + "',");
		sql.append("null");
		sql.append(")");
System.out.println(sql.toString());
		r = s.executeUpdate(sql.toString());

		int seq = 0;
		Vector v = o.getAnnotatants();
		for (Enumeration e = v.elements(); e.hasMoreElements();) {
			NameUsage nameUsage = (NameUsage) e.nextElement();
			sql = new StringBuffer();
//			sql.append("insert into lt_annotation_annotatants (linkfrom, linkto) values (");
			sql.append("insert into annotatants (annotation, annotatant, sequence) values (");
			sql.append("'" + o.getPersistentID() + "',");
			sql.append("'" + nameUsage.getPersistentID() + "',");
			sql.append(" " + Integer.toString(seq++) +  ")");
System.out.println(sql.toString());
			r = s.executeUpdate(sql.toString());
		}

		seq = 0;
		v = o.getAnnotators();
		for (Enumeration e = v.elements(); e.hasMoreElements();) {
			NameUsage nameUsage = (NameUsage) e.nextElement();
			sql = new StringBuffer();
//			sql.append("insert into lt_annotation_annotators (linkfrom, linkto) values (");
			sql.append("insert into annotators (annotation, annotator, sequence) values (");
			sql.append("'" + o.getPersistentID() + "',");
			sql.append("'" + nameUsage.getPersistentID() + "',");
			sql.append(" " + Integer.toString(seq++) +  ")");
System.out.println(sql.toString());
			r = s.executeUpdate(sql.toString());
		}
		return 1;
	}

}
