/*
 * NamedObjectBroker.java:  a broker of NamedObjects
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2002 Nozomi `James' Ytow, Ryo Fujimoto, T. Okada
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: NamedObjectBroker.java,v 1.46 2002/10/16 17:27:33 nozomi Exp $
 * $Log: NamedObjectBroker.java,v $
 * Revision 1.46  2002/10/16 17:27:33  nozomi
 * nomanal Apperance handling added
 *
 * Revision 1.45  2002/10/15 17:11:54  nozomi
 * tentative fix of null Publication
 *
 * Revision 1.44  2002/10/10 16:24:24  nozomi
 * a Publication may contain no Appearance
 *
 * Revision 1.43  2002/10/02 02:52:12  nozomi
 * modify NamedObject hanlding
 *
 * Revision 1.42  2002/09/17 05:03:42  nozomi
 * use indirect pointer
 *
 * Revision 1.41  2002/09/10 07:13:57  nozomi
 * support NameResolver
 *
 * Revision 1.40  2002/09/08 14:16:29  nozomi
 * support Author search
 *
 * Revision 1.39  2002/09/07 17:12:39  nozomi
 * support putting objects in a vector
 *
 * Revision 1.38  2002/09/07 01:40:08  nozomi
 * support search of unsaved objects
 *
 * Revision 1.37  2002/09/06 14:32:58  nozomi
 * improved connection handling
 *
 * Revision 1.36  2002/09/06 14:24:53  ryo
 * fix NullPointerException bug
 *
 * Revision 1.35  2002/09/06 03:36:18  nozomi
 * looks for unsaved objects at first
 *
 * Revision 1.34  2002/09/06 03:33:09  nozomi
 * support unsaved objects handling including save and search
 *
 * Revision 1.33  2002/09/01 10:56:08  nozomi
 * utilise ObjectPool for models, nodes and trees
 *
 * Revision 1.32  2002/08/30 03:31:57  nozomi
 * add putNamedObject(NamedObject)
 *
 * Revision 1.31  2002/08/30 03:10:07  nozomi
 * fix bug in CVS log string
 *
 * Revision 1.30  2002/08/30 02:22:21  nozomi
 * add put* and get* methods
 *
 * Revision 1.29  2002/08/19 03:05:54  ryo
 * fix ClassCastException bug
 *
 * Revision 1.28  2002/08/15 12:57:54  ryo
 * fix NumberFormatException bug
 *
 * Revision 1.27  2002/08/14 13:09:45  ryo
 * modify getAuthority(), return null if authority is not found
 *
 * Revision 1.26  2002/08/07 05:15:28  ryo
 * add disconnect and connect processing before and after setting connection interfaces
 *
 * Revision 1.25  2002/08/06 13:48:04  ryo
 * add getAuthority() function
 *
 * Revision 1.24  2002/08/02 11:20:34  ryo
 * modify AnnotationEditModel linkage processing
 *
 * Revision 1.23  2002/08/01 21:29:20  nozomi
 * JDK version dependency handling
 *
 * Revision 1.22  2002/08/01 07:20:00  ryo
 * comments out JDK 1.4 API
 *
 * Revision 1.21  2002/08/01 04:53:37  nozomi
 * adapt to JDK 1.4
 *
 * Revision 1.20  2002/07/29 05:00:13  ryo
 * add check a validity of connection
 *
 * Revision 1.19  2002/07/16 07:59:57  ryo
 * fix NullPointerException bug when searching and saving data
 *
 * Revision 1.18  2002/07/08 07:43:30  ryo
 * fix NullPointerException bug
 *
 * Revision 1.17  2002/07/05 06:15:47  t.okada
 * bugfix NameUsage Tree unlink problem
 *
 * Revision 1.16  2002/07/02 08:23:05  nozomi
 * null connectionInterface handling
 *
 * Revision 1.15  2002/07/02 02:26:15  ryo
 * remove unused import line
 *
 * Revision 1.14  2002/07/01 11:38:40  ryo
 * change the sequence of tree creating
 *
 * Revision 1.13  2002/07/01 04:51:12  t.okada
 * ObjectPool processing addition
 *
 * Revision 1.12  2002/06/25 14:47:51  ryo
 * fix multi appearance bug
 *
 * Revision 1.11  2002/06/21 11:34:00  ryo
 * add the processing of multiple connection
 *
 * Revision 1.10  2002/06/17 10:20:15  ryo
 * comment out JDBC connection
 *
 * Revision 1.9  2002/06/17 09:57:34  ryo
 * add the function which creates a tree
 *
 * Revision 1.8  2002/06/17 08:37:48  t.okada
 * add objectPool member
 *
 * Revision 1.7  2002/06/10 12:35:15  ryo
 * add getNamedObjectEditModelTree()
 *
 * Revision 1.6  2002/06/05 12:31:39  ryo
 * add setLocaldbTopDirectory()
 *
 * Revision 1.5  2002/06/04 13:45:29  ryo
 * add register method and remove unused function and class
 *
 * Revision 1.4  2002/06/03 12:14:42  t.okada
 * class below localdb implement.
 *
 * Revision 1.3  2002/05/31 08:00:45  t.okada
 * remove errors
 *
 * Revision 1.2  2002/05/29 03:17:01  nozomi
 * translation of comments done
 *
 * Revision 1.1  2002/05/28 06:01:17  nozomi
 * initial commition to CVS repository
 *
 */	

package org.nomencurator.broker;

import java.util.Map;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
/*if[JDK1.4]*/
import java.sql.Savepoint;
/*end[JDK1.4]*/
import java.sql.Statement;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.SQLWarning;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;

import org.nomencurator.Name;
import org.nomencurator.NamedObject;
import org.nomencurator.NameUsage;
import org.nomencurator.Nomencurator;
import org.nomencurator.Appearance;
import org.nomencurator.Publication;
import org.nomencurator.Author;
import org.nomencurator.Annotation;

import org.nomencurator.util.ObjectPool;

import org.nomencurator.editor.NameUsageListRow;

import org.nomencurator.editor.model.NamedObjectEditModel;
import org.nomencurator.editor.model.NamedObjectNode;
import org.nomencurator.editor.model.NameTreeModel;
import org.nomencurator.editor.model.NameUsageEditModel;
import org.nomencurator.editor.model.AppearanceEditModel;
import org.nomencurator.editor.model.PublicationEditModel;
import org.nomencurator.editor.model.AuthorEditModel;
import org.nomencurator.editor.model.AnnotationEditModel;

import org.nomencurator.controller.linktable.NameUsageHigher;
import org.nomencurator.controller.linktable.NameUsageAuthority;
import org.nomencurator.controller.linktable.NameUsageRecorder;
import org.nomencurator.controller.linktable.AppearanceNameUsage;
import org.nomencurator.controller.linktable.PublicationAppearance;
import org.nomencurator.controller.linktable.PublicationAuthor;
import org.nomencurator.controller.linktable.AnnotationFrom;
import org.nomencurator.controller.linktable.AnnotationTo;
import org.nomencurator.controller.linktable.LinkRecord;

/**
 * A broker of <code>NamedObject</code>s
 *
 * @version 	17 Oct 2002
 * @author 	Nozomi `James' Ytow
 * @author      T. Okada
 * @author 	Ryo Fujimoto
 */
public class NamedObjectBroker
    implements Connection
{
    /** <code>Vector</code> representing a list of <code>Connection</code>s */
    protected Vector connections;

    protected Vector connectionInterfaces;
    
    private static NamedObjectBroker _instance = null;
    
    //private ObjectPool _objectPool = new ObjectPool();

    private Nomencurator curator = Nomencurator.getInstance();

    protected ObjectPool nodePool = new ObjectPool();

    protected ObjectPool modelPool = new ObjectPool();

    protected ObjectPool treePool = new ObjectPool();

    protected ObjectPool viewPool = new ObjectPool();

    protected Hashtable nextSuffix;
    protected Hashtable publications;

    protected ObjectPool byNamePool = new ObjectPool();
    
    protected ObjectPool byAuthorPool = new ObjectPool();
    
    protected ObjectPool byYearPool = new ObjectPool();
    
    protected ObjectPool byRankPool = new ObjectPool();

    protected ObjectPool authorByNamePool = new ObjectPool();

    protected Hashtable unsavedPool;

    private NamedObjectBroker()
    {
	this(null);
    }

    /**
     * Constructs a <code>NamedObjectBroker</code> holding
     * <code>connection</code>
     */
    private NamedObjectBroker(Connection connection)
    {
	add(connection);
    }
    
    public static NamedObjectBroker getInstance()
    {
	if(_instance == null){
	    _instance = new NamedObjectBroker();
	}
	return _instance;
    }
	
    /**
     * get NameUsage list from the name, author, year
     * @param NameUsage
     * @param String name
     * @param String author
     * @param String year
     * @return Vector<NameUsageListRow> name record list
     */
    public Vector getNameUsageList(String name, String author, String year)
    {
	Vector list = searchNameUsage(name, author, year);
	if(list == null)
	    list = new Vector();
	if(connectionInterfaces != null) {
	    for (Enumeration e = connectionInterfaces.elements(); e.hasMoreElements(); ) {
		AbstractConnection co = (AbstractConnection)e.nextElement();
		if (co == null || !co.getValid())
			continue;
		Vector v = co.searchNameUsage(name, author, year);
		if (v != null && v.size() > 0)
		    list = getSum(list, v);
	    }
	}
    	return list;
    }

    /**
     * get NameUsage list from the name, author, year
     * @param NameUsage
     * @param String name
     * @param String author
     * @param String year
     * @return Vector<NameUsageListRow> name record list
     */
    public Vector searchNameUsage(String name, String author, String year)
    {
	if((byNamePool == null || byNamePool.size() == 0 ) &&
	   (byAuthorPool == null || byAuthorPool.size() == 0 ) &&
	   (byYearPool == null || byYearPool.size() == 0 ))
	    return null;
	Vector byName = null;
	Vector byAuthor = null;
	Vector byYear = null;
	Vector v = null;

	int keys = 0;

	if(name != null && name.length() > 0) {
	    byName = getByName(name);
	    keys |= 1;
	}
	if(author != null && author.length() > 0) {
	    byAuthor = getByAuthor(author);
	    keys |= 2;
	}
	if(year != null && year.length() > 0) {
	    byYear = getByYear(year);
	    keys |= 4;
	}

	switch(keys) {
	case 0: //no key was given
	    v = new Vector();
	    //	    Enumeration e = byYearPool.elements();
	    Enumeration e = byYearPool.keys();
	    while(e.hasMoreElements()) {
		v = NamedObjectBroker.getSum(v, getByYear((String)e.nextElement()));
	    }
	    break;
	case 1: //look up only by name
	    v = byName;
	    break;
	case 2: //look up only by author
	    v = byAuthor;
	    break;
	case 4: //look up only by year
	    v = byYear;
	    break;
	case 7: //look up by all keys
	    v = NamedObjectBroker.getProduct(NamedObjectBroker.getProduct(byName, byAuthor), byYear);
	    break;
	case 3: //look up by name and author
	    v = NamedObjectBroker.getProduct(byName, byAuthor);
	    break;
	case 5: //look up by name and year
	    v = NamedObjectBroker.getProduct(byName, byYear);
	    break;
	case 6: //look up by author and year
	    v = NamedObjectBroker.getProduct(byAuthor, byYear);
	    break;
	}

	if(v == null || v.isEmpty())
	    return null;

	Vector rows = new Vector();
	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    NameUsage n = (NameUsage)e.nextElement();
	    rows.addElement(new NameUsageListRow(n.getName(), 
						 n.getUsedName(), 
						 n.getAppearance().getPublication().getAuthorListSummary(),
						 n.getYear()));
	}

	v = null;
	e = null;
	return rows;
    }

	
    /**
     * get NamedObjects list for the tree and all EditPanel
     * @param NamedObject 
     * @param int higher count
     * @param int lower count
     * @return NameUsage
     */
    public NameUsage getNameUsageTree(NameUsage nameUsage, int higher, int lower)
    {
	return getNameUsageTree(nameUsage.persistentID(), higher, lower);
    }

    /**
     * get NamedObjects list for the tree and all EditPanel
     * @param persistentID of a <CODE>NamedObject</CODE>
     * @param int higher count
     * @param int lower count
     * @return NameUsage
     */
    public NameUsage getNameUsageTree(String persistentID, int higher, int lower)
    {
	// get NameUsage
	NameUsage nu = (NameUsage)getNamedObject(persistentID);

	// get Appearance
	String oid = null;
	Appearance appearance = nu.getAppearance();

	//if (appearance == null) return nu;	// error.
	if (appearance == null) {

	}
	if (appearance.isNominal()) {
	    appearance = (Appearance)getNamedObjectEntity(appearance.getName());
	    
	    //	    resolve(appearance);
	    
	}
	
	// linkage process of NameUsage's hierarchy
	getHigherNameUsages(nu, higher); // get higher
	getLowerNameUsages(nu, lower); // get lower
	
	// get Publication
	Publication publication = appearance.getPublication();

	if (publication == null) return nu;	// error.
	if (publication.isNominal()) {
	    publication = (Publication)getNamedObjectEntity(publication.getName());
	    //	    resolve(publication);
	}
	
	// get Appearance & NameUsage without nameUsage
	Vector as = publication.getAppearances();
	Enumeration e = null;
	if(as != null) {
	    e = as.elements();
	    while(e.hasMoreElements()) {
		Appearance ap = (Appearance)e.nextElement();
		if(ap.isNominal()) {
		    ap = (Appearance)getNamedObjectInBackends(ap.getName());
		    Vector v = ap.getNameUsages();
		    if(v != null) {
			Enumeration en = v.elements();
			while(en.hasMoreElements()) {
			    NameUsage n = (NameUsage)en.nextElement();
			    if(n.isNominal()) {
				n = (NameUsage)getNamedObjectInBackends(n.getName());
				//			resolve(n);
				getHigherNameUsages(n, higher);
				getHigherNameUsages(n, lower);
			    }
			}
		    }
		}
	    }
	}
	
	// get Authors
	as = publication.getAuthors();
	if(as != null) {
	    e = as.elements();
	    while(e.hasMoreElements()) {
		Author a = (Author)e.nextElement();
		if(a.isNominal()) {
		    a = (Author)getNamedObjectInBackends(a.getName());
		    //		resolve(a);
		}
	    }
	}
	
	
	// get Authority and Annotation
	as = appearance.getNameUsages();
	if(as != null) {
	    e = as.elements();
	    while(e.hasMoreElements()) {
		NameUsage n = (NameUsage)e.nextElement();
		
		if(n.isNominal()) {
		    n = (NameUsage)getNamedObjectInBackends(n.getName());
		    //		resolve(n);
		}
		
		Vector ans = n.getAnnotations();
		if (ans != null) {
		    Enumeration ee = ans.elements();
		    while (ee.hasMoreElements()) {
			Annotation an = (Annotation)ee.nextElement();
			if(an.isNominal()) {
			    //resolve(getNamedObjectInBackends(an.getName()));
			    getNamedObjectInBackends(an.getName());
			}
		    }
		}
		
		NameUsage auth = n.getAuthority();
		if(auth != null && auth != n && auth.isNominal()) {
		    //resolve(getNamedObjectInBackends(n.getName()));
		    getNamedObjectInBackends(n.getName());
		}
	    }
	}

	return nu;
    }

    /**
     * Get higher NameUsage.
     * @param nameUsage
     * @param higher
     * @return nameUsage's higherTaxon.
     */
    private NameUsage getHigherNameUsages(NameUsage nameUsage)
    {
	NameUsage higher = (NameUsage)nameUsage.getEntity();
	nameUsage = higher;
	while(higher != null) {
	    nameUsage = higher;
	    higher = getHigherNameUsages(higher, 1);
	}
	
	return nameUsage;
    }


    /**
     * Get higher NameUsage.
     * @param nameUsage
     * @param higher
     * @return nameUsage's higherTaxon.
     */
    private NameUsage getHigherNameUsages(NameUsage nameUsage, int higher)
    {
	if(higher == -1)
	    return getHigherNameUsages(nameUsage);
	higher--;
	if ((higher < 0) || (nameUsage == null))
	    return null;
	/*
	String higherOid = NameUsageHigher.getInstance().getTo(nameUsage.getPersistentID());
	NameUsage higherNameUsage = (NameUsage)getNamedObject(higherOid);
	*/
	NameUsage higherNameUsage = nameUsage.getHigherTaxon();
	if (higherNameUsage == null)
	    return null;
	if (higherNameUsage.isNominal()) {
	    higherNameUsage = (NameUsage)getNamedObjectInBackends(higherNameUsage.getName());
	    //resolve(higherNameUsage);
	}
	higherNameUsage = 
	    (NameUsage)higherNameUsage.getEntity();

	/*
	if (higherNameUsage != null) {
	    nameUsage.setHigherTaxon(higherNameUsage);
	    higherNameUsage.addLowerTaxon(nameUsage);
	    higherNameUsage.setAppearance(nameUsage.getAppearance());
	    nameUsage.getAppearance().addNameUsage(higherNameUsage);
	    //higherNameUsage.setHigherTaxon(getHigherNameUsages(higherNameUsage, higher));
	    getHigherNameUsages(higherNameUsage, higher);
	}
	*/
	return higherNameUsage;
    }
    
    /**
     * Get lower NameUsage.
     * @param nameUsage
     * @param lower
     * @return nameUsage's lowerTaxon.
     */
    private void getLowerNameUsages(NameUsage nameUsage, int lower)
    {
	lower--;

	if ((lower < 0) || (nameUsage == null))
	    return;

	Vector lowerTaxa = nameUsage.getLowerTaxa();
	if(lowerTaxa == null || lowerTaxa.isEmpty())
	    return;

	Enumeration e = lowerTaxa.elements();
	while (e.hasMoreElements()) {
	    NameUsage lowerTaxon = (NameUsage)e.nextElement();
	    if(lowerTaxon.isNominal()) {
		lowerTaxon = (NameUsage)getNamedObjectInBackends(lowerTaxon.getName());
		//		resolve(lowerTaxon);
	    }
	    lowerTaxon = 
		(NameUsage)lowerTaxon.getEntity();
	    getLowerNameUsages(lowerTaxon, lower);
	}
	
	/*
	NameUsageHigher nuh = NameUsageHigher.getInstance();
	Vector lowerOids = NameUsageHigher.getInstance().getFromList(nameUsage.getPersistentID());
	if (lowerOids != null) {
	    for (Enumeration e = lowerOids.elements(); e.hasMoreElements();) {
		String lowerOid = (String)e.nextElement();
		NameUsage lowerNameUsage = (NameUsage)getNamedObject(lowerOid);
		lowerNameUsage.setAppearance(nameUsage.getAppearance());
		nameUsage.getAppearance().addNameUsage(lowerNameUsage);
		lowerNameUsage.setHigherTaxon(nameUsage);
		//lowerNameUsage.setLowerTaxa(getLowerNameUsages(lowerNameUsage, lower));
		getLowerNameUsages(lowerNameUsage, lower);
		nameUsage.addLowerTaxon(lowerNameUsage);
	    }
	}
	*/
	return;
    }

    /**
     * get Authority
     * @param NameUsage 
     * @return NameUsage
     */
    public NameUsage getAuthority(NameUsage authority) {
	if (authority == null)
	    return null;
	// get NameUsage
	String oid = authority.getPersistentID();
	authority = (NameUsage)getNamedObject(oid);
	if (authority == null)
	    return null;
	// get Appearance
	/*
	oid = NameUsageRecorder.getInstance().getTo(authority.getPersistentID());
	Appearance appearance = (Appearance)getNamedObject(oid);
	*/
	Appearance appearance = authority.getAppearance();
	if (appearance == null)
	    return authority;
	if (appearance.isNominal())
	    appearance = (Appearance)getNamedObjectEntity(appearance.getName());
	appearance.addNameUsage(authority);
	authority.setAppearance(appearance);
	// get Publication
	/*
	oid = PublicationAppearance.getInstance().getFrom(appearance.getPersistentID());
	Publication publication = (Publication)getNamedObject(oid);
	*/
	Publication publication = appearance.getPublication();
	if (publication == null)
	    return authority;
	if (publication.isNominal())
	    publication = (Publication)getNamedObjectEntity(publication.getName());
	publication.addAppearance(appearance);
	appearance.setPublication(publication);
	
	// get Authors
	/*
	Vector v = PublicationAuthor.getInstance().getToLinkRecords(publication.getPersistentID());

	publication.setAuthors(new Vector()); // clear dummy author for servlet connection.
	Vector authors = new Vector();
	authors.setSize(v.size());
	if (v != null) {
	    for (Enumeration e = v.elements(); e.hasMoreElements();) {
		LinkRecord record = (LinkRecord)e.nextElement();
		Author author = (Author)this.getNamedObject(record.getTo());
		if (author != null) {
		    String idstring = record.getID();
		    int id = 0;
		    if (idstring == null || idstring.length() == 0) {
			id = authors.indexOf(null);
		    } else {
			try {
			    id = new Integer(idstring).intValue();
			} catch (NumberFormatException ex) {
			    id = 0;
			}
		    }
		    authors.set(id, author);
		}
	    }
	}
	for (Enumeration e = authors.elements(); e.hasMoreElements();) {
	    Author author = (Author)e.nextElement();
	    if (author == null)
		continue;
	    author.addPublication(publication);
	    publication.addAuthor(author);
	}
	*/
	return authority;
    }

    /**
     * get Authority
     * @param String 
     * @return NameUsage
     */
    public NameUsage getAuthority(String oid) {
	/*
	NameUsage authority = new NameUsage();
	authority.setPersistentID(oid);
	return getAuthority(authority);
	*/
	NameUsage authority = (NameUsage)getNamedObjectInPool(oid);
	if(authority == null || authority.isNominal())
	    authority = (NameUsage)getNamedObjectInBackends(oid);
	/*
	if(authority != null)
	    resolve(authority);
	*/
	return authority;

    }

    /**
     * get NamedObjectEditModels tree
     * @param NamedObject 
     * @param int higher count
     * @param int lower count
     * @return NameUsage
     */
    public NameUsageEditModel getNamedObjectEditModelTree(NameUsage nameUsage, int higher, int lower)
    {
	return getNamedObjectEditModelTree(getNameUsageTree(nameUsage, higher, lower));
    }

    /**
     * get NamedObjectEditModels tree
     * @param NamedObject 
     * @param int higher count
     * @param int lower count
     * @return NameUsage
     */
    public NameUsageEditModel getNamedObjectEditModelTree(NameUsage nu)
    {
	//nu = (NameUsage)getNameUsageTree(nu, higher, lower);
	if (nu == null)
	    return null;

	//if objects are nominal, get entity
	if(nu.isNominal())
	    getNamedObjectEntity(nu.getName());

	Appearance appearance = nu.getAppearance();
	if(appearance.isNominal())
	    getNamedObjectEntity(appearance.getName());

	Publication publication = appearance.getPublication();
	if(publication == null) {
	    publication = new Publication();
	    appearance.setPublication(publication);
	}
	if(publication.isNominal())
	    getNamedObjectEntity(publication.getName());

	// create PublicationEditModel, AppearanceEditModel and AuthorEditModel
	PublicationEditModel publicationEditModel = 
	    (PublicationEditModel)modelPool.get(publication.getName());
	if(publicationEditModel == null) { 
	    publicationEditModel = new PublicationEditModel(publication);
	    modelPool.put(publicationEditModel);
	}

	Hashtable editModels = new Hashtable();

	int appNum = publicationEditModel.getAppearanceEditModels().size();

	for(int i = 0; i < appNum; i++) {
	    AppearanceEditModel appearanceEditModel = 
		(AppearanceEditModel)publicationEditModel.getAppearanceEditModels().elementAt(i);
	    appearance = (Appearance)appearanceEditModel.getObject();
	    if(appearance.isNominal()) {
		appearance = (Appearance)getNamedObjectEntity(appearance.getName());
	    }
;
	    editModels.put(appearance.getPersistentID(), appearanceEditModel); // be used when linking annotation
	    if(modelPool.get(appearance.getName()) == null)
		modelPool.put(appearanceEditModel);
	    
	    // create NameUsageEditModel
	    if(appearance.getNameUsages() != null) {
	    int nameNum = appearance.getNameUsages().size();
	    for(int j = 0; j < nameNum; j++) {
		NameUsage nameUsageObject = (NameUsage)appearance.getNameUsages().elementAt(j);
		if(nameUsageObject.isNominal())
		    nameUsageObject = (NameUsage)getNamedObjectInBackends(nameUsageObject.getName());

		NameUsage nameUsageHigher = nameUsageObject.getHigherTaxon();
		if(nameUsageHigher != null) {
		    if(nameUsageHigher.isNominal())
			nameUsageHigher = (NameUsage)getNamedObjectInBackends(nameUsageHigher.getName());
		}
		NameUsageEditModel nameUsageEditModel = 
		    (NameUsageEditModel)modelPool.get(nameUsageObject.getName());
		if(nameUsageEditModel == null) { 
		    nameUsageEditModel = 
			new NameUsageEditModel(nameUsageObject,
					       true, null, null, null, appearanceEditModel, null, false);
		    modelPool.put(nameUsageEditModel);
		}

		/*
		nameUsageObject.setHigherTaxon(nameUsageHigher);
		*/
		editModels.put(nameUsageObject.persistentID(), nameUsageEditModel);
				// create Authrity EditModel

		NameUsage authority = nameUsageObject.getAuthority();
		if(authority != null) {
		    if(authority.isNominal())
			authority = (NameUsage)getNamedObjectInBackends(authority.getName());
		    NameUsageEditModel authorityEditModel =
			(NameUsageEditModel)modelPool.get(authority.getName());

		    if(authorityEditModel == null) {
			authorityEditModel =
			    new NameUsageEditModel(authority,
						   true, null, null, null, null, null, false);
			modelPool.put(authorityEditModel);
		    }
		    nameUsageEditModel.setAuthorityEditModel(authorityEditModel);
		}
	    }
	    }

	    // create AnnotationEditModel
	    // create only, no linkage
	    int annoNum = 0;
	    if (appearance.getAnnotations() != null)
	    	annoNum = appearance.getAnnotations().size();
	    for(int j = 0; j < annoNum; j++) {
		Annotation annotation = (Annotation)appearance.getAnnotations().elementAt(j);
		AnnotationEditModel annotationEditModel =
		    (AnnotationEditModel)modelPool.get(annotation.getName());
		if(annotationEditModel == null) {
		    annotationEditModel =
			new AnnotationEditModel(annotation,
						true, appearanceEditModel, null, null);
		    modelPool.put(annotationEditModel);
		}
		editModels.put(annotation.getPersistentID(), annotationEditModel); // be used when linking annotation
		//				appearanceEditModel.addAnnotationEditModel(annotationEditModel);
	    }
	}
	

	// create NameUsageEditModels Tree
	for(int i = 0; i < appNum; i++) {
	    appearance = (Appearance)publication.getAppearances().elementAt(i);
	    if(appearance.getNameUsages() != null) {
	    int nameNum = appearance.getNameUsages().size();
	    for(int j = 0; j < nameNum; j++) {
		NameUsage nameUsageObject = (NameUsage)appearance.getNameUsages().elementAt(j);
		if(nameUsageObject.isNominal())
		    nameUsageObject = (NameUsage)getNamedObjectInBackends(nameUsageObject.getName());
		NameUsageEditModel nameUsageEditModel = (NameUsageEditModel)editModels.get(nameUsageObject.persistentID());
		NameUsage higher = nameUsageObject.getHigherTaxon();
		if(higher != null) {
		    if(higher.isNominal())
			higher = (NameUsage)getNamedObjectInBackends(higher.getName());
		    NameUsageEditModel higherNameUsageEditModel = 
			(NameUsageEditModel)modelPool.get(nameUsageObject.getHigherTaxon().getName());
			//(NameUsageEditModel)editModels.get(nameUsageObject.getHigherTaxon().persistentID());
		    if(higherNameUsageEditModel != null) {
			/*
			nameUsageEditModel.setHigherEditModel(higherNameUsageEditModel);
			if(higherNameUsageEditModel.getLowerEditModels() == null) {
			    higherNameUsageEditModel.setLowerEditModels(new Vector());
			}
			higherNameUsageEditModel.getLowerEditModels().add(nameUsageEditModel);
			*/
			higherNameUsageEditModel.addLowerEditModel(nameUsageEditModel);
		    }
		}
	    }
	    }
	}
	
	// create Annotation relaytion
	Vector v = publication.getAppearances();
	for (Enumeration e = publication.getAppearances().elements(); e.hasMoreElements(); ) {
	    Appearance ap = (Appearance)e.nextElement();
	    if (ap.getAnnotations() == null)
		continue;
	    for (Enumeration ee = ap.getAnnotations().elements(); ee.hasMoreElements(); ) {
		Annotation ann = (Annotation)ee.nextElement();
		AnnotationEditModel annEM = (AnnotationEditModel)editModels.get(ann.getPersistentID());
				// set annotatants.
		if (ann.getAnnotatants() != null) {
		    for (Enumeration eee = ann.getAnnotatants().elements(); eee.hasMoreElements(); ) {
			NameUsage nameUsageObject = (NameUsage)eee.nextElement();
			NameUsageEditModel nameUsageEditModel = 
			    (NameUsageEditModel)modelPool.get(nameUsageObject.getName());
			if(nameUsageEditModel == null) {
			    nameUsageEditModel = 			
				new NameUsageEditModel(nameUsageObject,
						       true, null, null, null, null, null, false);
			    modelPool.put(nameUsageEditModel);
			}
			if(nameUsageEditModel != null) {
			    if(annEM != null && annEM.getAnnotants() != null)
				annEM.getAnnotants().add(nameUsageEditModel);
			}
		    }
		}
				// set annotator.
		if (ann.getAnnotators() != null) {
		    for (Enumeration eee = ann.getAnnotators().elements(); eee.hasMoreElements(); ) {
			NameUsage nameUsageObject = (NameUsage)eee.nextElement();
			//			NameUsageEditModel nameUsageEditModel = (NameUsageEditModel)editModels.get(nameUsageObject.getPersistentID());
			NameUsageEditModel nameUsageEditModel =
			    (NameUsageEditModel)modelPool.get(nameUsageObject.getName());
			if(nameUsageEditModel != null) {
			    annEM.getAnnotators().add(nameUsageEditModel);
			    nameUsageEditModel.addAnnotation(annEM);
			}
		    }
		}
				// set appearance.
		if (ann.getAppearance() != null) {
		    //		    AppearanceEditModel appEM = (AppearanceEditModel)editModels.get(ann.getAppearance().getPersistentID());
		    AppearanceEditModel appEM = 
			(AppearanceEditModel)modelPool.get(ann.getAppearance().getName());

		    if(appEM != null)
		    appEM.addAnnotationEditModel(annEM);
		    if(annEM != null)
		    annEM.setAppearanceEditModel(appEM);
		}
	    }
	}

	// output TextList
	/*
	int authNum = publicationEditModel.getAuthorEditModels().size();
	for(int i = 0; i < authNum; i++) {
	    publicationEditModel.addAuthor((AuthorEditModel)publicationEditModel.getAuthorEditModels().elementAt(i));	
	}
	AuthorEditModel authorEditModel = null;
	if(authNum > 0)
	    authorEditModel 
		= (AuthorEditModel)publicationEditModel.getAuthorEditModels().elementAt(0);
	else
	    authorEditModel = new AuthorEditModel();
	authorEditModel.add(publicationEditModel);
	*/
	publicationEditModel.getAuthorList().setObjects(publication.getAuthors());
	AuthorEditModel authorEditModel = null;
	if(publicationEditModel.getAuthorEditModels().size() > 0)
	    authorEditModel 
		= (AuthorEditModel)publicationEditModel.getAuthorEditModels().elementAt(0);
	else {
	    authorEditModel = new AuthorEditModel();
	    authorEditModel.add(publicationEditModel);
	}

	for(int i = 0; i < appNum;  i++) {
	    publicationEditModel.addAppearance((AppearanceEditModel)publicationEditModel.getAppearanceEditModels().elementAt(i));
	}

	//NameUsageEditModel nameUsageEditModel = (NameUsageEditModel)editModels.get(nu.persistentID());
	NameUsageEditModel nameUsageEditModel = 
	    (NameUsageEditModel)modelPool.get(nu.getName());
	if(nameUsageEditModel == null) {
	    nameUsageEditModel = new NameUsageEditModel(nu);
	    modelPool.put(nameUsageEditModel);
	}
	if(nameUsageEditModel.getLowerEditModels() != null) {
	    int nameNum = nameUsageEditModel.getLowerEditModels().size();
	    for(int i = 0; i < nameNum; i++) {
		nameUsageEditModel.getLower().update((NameUsageEditModel)nameUsageEditModel.getLowerEditModels().elementAt(i));
	    }
	}

	return nameUsageEditModel;
	//return new NameUsageEditModel(nu);
	
    }
    
    /**
     * save a NamedObject
     * @param NamedObject
     * @return int inserted row count
     */
    public int saveNamedObject(NamedObject o) {
	if(curator.get(o.persistentID()) == null) {
	    curator.put(o.persistentID(), o);
	}
	
	Vector v = new Vector();
	v.add(o);
	return saveNamedObject(v);
    }
    
    /**
     * save NamedObjects
     * @param Vector<NamedObject>
     * @return int inserted row count
     */
    public int saveNamedObject(Vector v) {
	if(v != null && !v.isEmpty()) {
	    new Error(((NamedObject)v.elementAt(0)).getName()).printStackTrace();
	}
	int rows = 0;
	Vector writables = new Vector();
	for (Enumeration e = connectionInterfaces.elements(); e.hasMoreElements(); ) {
	    AbstractConnection co = (AbstractConnection)e.nextElement();
	    if (co != null && co.getValid() && co.getWritable())
		writables.addElement(co);
	}

	for (Enumeration e = writables.elements(); e.hasMoreElements(); ) {
	    AbstractConnection co = (AbstractConnection)e.nextElement();
	    rows += co.saveNamedObject(v);
	}
	return rows;
    }
    
    /**
     * Adds <code>connection</code> to the list of <code>Connection</code>s
     * if it is not yet there.  
     *
     * @param connection <code>Connection</code> to be added to the list
     */
    public void add(Connection connection)
    {
	if(connection == null)
	    return;

	if(connections == null)
	    connections = new Vector();

	if(connections.contains(connection))
	    return;

	connections.addElement(connection);
    }

    /**
     * Removes <code>connection</code> from the list of <code>Connection</code>s
     * if it is not yet there.  
     *
     * @param connection <code>Connection</code> to be removed from the list
     */
    public void remove(Connection connection)
    {
	if(connections == null ||	   
	   !connections.contains(connection))
	    return;

	connections.remove(connection);
    }

	/**
	 * Adds <code>connection</code> to the list of <code>Connection</code>s
	 * if it is not yet there.  
	 *
	 * @param connection <code>Connection</code> to be added to the list
	 */
	public void addConnectionInterface(AbstractConnection c) {
		if (c == null)
			return;

		if (connectionInterfaces == null)
			connectionInterfaces = new Vector();

		if (connectionInterfaces.contains(c))
			return;

		connectionInterfaces.addElement(c);
	}

	/**
	 * Removes <code>connection</code> from the list of <code>Connection</code>s
	 * if it is not yet there.  
	 *
	 * @param connection <code>Connection</code> to be removed from the list
	 */
	public void removeConnectionInterface(AbstractConnection c) {
		if (connectionInterfaces == null || !connectionInterfaces.contains(c))
			return;

		connectionInterfaces.remove(c);
	}

    /**
     * Returns <code>Vector</code> of <code>Connection</code>s
     *
     * @return Vector of <code>Connection</code>s
     */
    public synchronized Vector getConnections()
    {
	return (Vector)connections.clone();
    }

    /**
     * Returns a <code>Statement</code> used to send DBMS a SQL query.
     * SQL query without parameter is executed using a <code>Statement</code>
     * ordinarily.  <code>PreparedStatement</code> object is more efficient
     * when the same SQL query executed multiple times.
     * <P>
     * <code>ResultSet</code>created by susing returned <code>Statement<code>
     * is forward only type which can be used for read-only parallel processing
     * by default.
     *
     * @return Statement object newly craeted
     *
     * @exception SQLException when DBMS access error happend
     *
     */
    public Statement createStatement()
	throws SQLException
    {
	if(connections == null)
	    return null;

	Enumeration e = getConnections().elements();
	while(e.hasMoreElements()) {
	    //((Connection)e.nextElement()).
	}
	return null;
    }

    /**
     * Returns <code>PreparedStatement</code> object to send SQL query with parameters to a
     * DBMS.  SQL query with or without IN parameter can be precompiled and stored into
     * <code>PreparedStatement</code> object.  Therefore, this object can be used
     * for efficient excuttion of the query in multiple times.
     * <P>
     * Note: This method is optimized to execute SQL query with paramters where precomile has
     * some efferct.  The <code>prepareStatement</code> method will send the query to DBMS
     * for precompile if the driver supports precompilation.  In this case the query will not
     * be send to DBMS until PreparedStatement is executed.
     * It doesn't affect users directly,
     * but affects to that which method throw <code>SQLException</code>.  
     * The <code>ResultSet</code> created by using returned <code>PreparedStatement</code>
     * is forward only type allowing read-only parallel processing by default.
     *
     * @param sql SQL phrase which may contain one or more '?' IN parameter placeholder.
     *
     * @return PreparedStatement contains precompiled SQL query
     *
     * @exception SQLException when DBMS access error happend
     */
    public PreparedStatement prepareStatement(String sql)
	throws SQLException
    {
	return null;
    }

    /**
     * Returns a newly created <code>CallableStatement</code> object to call 
     * stored procedure of database.  <code>CallableStatement</code> provides
     * methods to set its IN and OUT parameters and to call stored procedure.
     * <P>
     * Note: This method is optimized to call stored procedure of dtabase.
     * Some driver send call phrase to database when <code>prepareCall</code>
     * method is executed, and others wait for execution of <code>CallableStatement</code> object
     * It doesn't affect users directly,
     * but affects to that which method throw <code>SQLException</code>.  
     * The <code>ResultSet</code> created by using returned <code>CallableStatement</code>
     * is forward only type allowing read-only parallel processing by default.
     *
     * @param sql SQL phrase which may contain one or more '?' parameter placeholder.
     * This phrase is escapse sequence of JDBC function call.
     *
     * @return newly created <code>CallableStatement</code> containing precompiled SQL phrase
     *
     * @exception SQLException when DBMS access error happend
     */
    public CallableStatement prepareCall(String sql)
	throws SQLException
    {
	return null;
    }
    
    /**
     * Returns native SQL <code>String</code> of specified <code>sql</code>.
     * JDBC driver can convert JDBC SQL to its native SQL.
     * This method returns the native SQL expression which the
     * driver send to DBMS.
     *
     * @param sql <code>String</code> representing a SQL phrase which may contain
     * one or more '?" paramater place holder.
     *
     * @return String representing native SQL expression of <code>sql</code>
     *
     * @exception SQLException when DBMS access error happend
     */
    public String nativeSQL(String sql)
	throws SQLException
    {
	return null;
    }


    /**
     * Sets automatic commit mode of this <code>Conenction</code>.
     * If a <code>Conenction</code>.is at automatic commit mode,
     * its all SQL phrases are executed and commited as independent
     * transactions.  Else SQL phrases are grouped into a transaction
     * terminated by calling <code>commit</code> method or 
     * <code>rollback</code> method.
     * A new <code>Connection</code> is automatic commit mode by
     * default.  Commitment is evoked by earlier one of termination
     * of SQL query or evokement of the next execution.
     * If the query returns <code>ResultSet</code>, the query terminates
     * when last line of <code>ResultSet</code> is taken or it is closed.
     * In advanced cases, single query may returns multiple results as well as
     * output parameters.  In these cases, commit occurs when all resulsts and
     * output parameters have been retrieved.
     *
     * @param autoCommit true if auto commit is enabled, or false if disabled.
     *
     * @exception SQLException when DBMS access error occured
     */
    public void setAutoCommit(boolean autoCommit)
	throws SQLException
    {
    }


    /**
     * Returns current auto commit status
     *
     * @return current auto commit status
     *
     * @exception SQLException when DBMS access error occured
     *
     * @see setAutoCommit(boolean)
     *
     * @exception SQLException when DBMS access error occured
     */
    public boolean getAutoCommit()
	throws SQLException
    {
	return true;
    }

    /**
     * Makes all modification valid which were made after last commit/rollback
     * and release all database lock which <code>Connection</code> holds.
     * This method is used only when automatic commit mode is disabled.
     *
     * @exception SQLException when DBMS access error occured
     *
     */
    public void commit()
	throws SQLException
    {
    }

    /**
     * Makes all modification invalid which were made after last commit/rollback
     * and release all database lock which <code>Connection</code> holds.
     * This method is used only when automatic commit mode is disabled.
     *
     * @exception SQLException when DBMS access error occured
     */
    public void rollback()
	throws SQLException
    {
    }

    /**
     * Immediately releases the database of this <code>Connection</code> and JDBC resource
     * without waiting for automatic release.
     * <P>
     * Note: <code>Connection</code> is closed automatically when gabage collected.
     * Some leathal error closes <code>Connection</code> as its result.
     *
     * @exception SQLException when DBMS access error occured
     */
    public void close()
	throws SQLException
    {
    }

    /**
     * Returns whether <code>Connection</code> is closed or not
     *
     * @return ture if <code>Connection</code> is closed, or false if open.
     *
     * @exception SQLException when DBMS access error occured
     */
    public boolean isClosed()
                 throws SQLException
    {
	return true;
    }

    /**
     * Returns metadata on the database of this <code>Connection</code>.
     * The database of this <code>Connection</code> provides description on
     * table, supported SQL grammer, stored procedures and ability of
     * this <code>Connection</code>.  These data are retreived from
     * <code>DatabaseMetaData</code> object.
     *
     * @return DatabaseMetaData object for this <code>Connection</code>
     *
     * @exception SQLException when DBMS access error occured
     */
    public DatabaseMetaData getMetaData()
	throws SQLException
    {
	return null;
    }

    /**
     * Sets this <code>Connection</code> to be read only mode
     * as hint for database optimization.
     *
     * Note: this method can't be called while in transaction
     *
     * @param readOnly true to enable read only mode, or
     * false to disable
     *
     * @exception SQLException when DBMS access error occured
     */
    public void setReadOnly(boolean readOnly)
	throws SQLException
    {
    }

    /**
     * Returns whether <code>Connection</code> is read only mode.
     *
     * @return true if <code>Connection</code> is read only mode,
     * or false if not.
     *
     * @exception SQLException when DBMS access error occured
     */
    public boolean isReadOnly()
	throws SQLException
    {
	return true;
    }

    /**
     * Sets catalog name to select working subspace in the database of
     * this <code>Connection</code>.  It will be ignored if the
     * driver does not support catalog.
     *
     * @param catlog <code>String</code> representing catalog name to be selected
     *
     * @exception SQLException when DBMS access error occured
     */
    public void setCatalog(String catalog)
	throws SQLException
    {
    }

    /**
     * Retruns current catlog name of this <code>Connection</code>
     *
     * @return String representing current catlog name or null
     *
     * @exception SQLException when DBMS access error occured
     */
    public String getCatalog()
	throws SQLException
    {
	return null;
    }

    /**
     * Try to change the transaction isolation level to specified <code>level</code>.
     * Specifiable constants of transaction isolation are defined in 
     * <code>Connection</code> interface.
     * <P>
     * Note: this method can't be called in a transaction
     *
     * @param level one of TRANSACTION_* isolation values except TRANSACTION_NONE.
     * Other values may be unsupported by some databases.
     *
     * @exception SQLException when DBMS access error occured
     *
     * @see DatabaseMetaData.supportsTransactionIsolationLevel(int)
    */
    public void setTransactionIsolation(int level)
	throws SQLException
    {
    }

    /**
     * Gets current transaction isolation level of this
     * <CODE>Conenction</CODE>
     *
     * @return int representing current transaction isolation,
     * one of TRANSACTION_* mode value
     *
     * @exception SQLException when DBMS access error occured
     */
    public int getTransactionIsolation()
	throws SQLException
    {
	return 0;
    }

    /**
     * Returns the first <code>SQLWarning</code>s notified by call relating
     * this <code>Connection</code>
     * <P>
     * Note: following warnings will be chained to this <code>SQLWarning</code>
     *
     * @return the first <code>SQLWarning</code> or null
     *
     * @exception SQLException when DBMS access error occured
     */
    public SQLWarning getWarnings()
	throws SQLException
    {
	return null;
    }

    /**
     * Clears all warnings has been notified relating to this <code>Connection</code>.
     * The <code>getWarnings</code> afther call of this method returns null until
     * new warning on the <code>Connection</code> is notified.
     *
     * @exception SQLException when DBMS access error occured
     */
    public void clearWarnings()
	throws SQLException
    {
    }

    /**
     * Creates <code>Statement</code> object which creates <code>ResultSet</code>
     * of specified <code>reslutSetType</code> and <code>reslutSetConcurrency</code>.
     * This method is same with the above <code>createStatement</code> method but
     * default type and concurrency of <code>ResultSet</code> can be overridden.
     *
     * @param resultSetType <code>int</code> representing <code>ResultSet<code> type.
     * See ResultSet.TYPE_XXX.
     * @param resultSetConcurrency <code>int</code> representing <code>ResultSet<code> 
     * concurrency type.  See ResultSet.CONCUR_XXX.
     *
     * @return Statement newly created
     *
     * @exception SQLException when DBMS access error occured
     *
     * @since 1.2
     *
     * @see What Is in the JDBC 2.0 API
     */
    public Statement createStatement(int resultSetType,
				     int resultSetConcurrency)
	throws SQLException
    {
	return null;
    }

    /**
     * Creates <code>PreparedStatement</code> object which creates <code>ResultSet</code>
     * of specified <code>reslutSetType</code> and <code>reslutSetConcurrency</code>.
     * This method is same with the above <code>prepareStatement</code> method but
     * default type and concurrency of <code>ResultSet</code> can be overridden.
     *
     * @param resultSetType <code>int</code> representing <code>ResultSet<code> type.
     * See ResultSet.TYPE_XXX.
     * @param resultSetConcurrency <code>int</code> representing <code>ResultSet<code> 
     * concurrency type.  See ResultSet.CONCUR_XXX.
     *
     * @return Statement newly created
     *
     * @exception SQLException when DBMS access error occured
     *
     * @since 1.2
     *
     * @see What Is in the JDBC 2.0 API
     */
    public PreparedStatement prepareStatement(String sql,
					      int resultSetType,
					      int resultSetConcurrency)
	throws SQLException
    {
	return null;
    }

    /**
     * Creates <code>CallableStatement</code> object which creates <code>ResultSet</code>
     * of specified <code>reslutSetType</code> and <code>reslutSetConcurrency</code>.
     * This method is same with the above <code>prepareCall</code> method but
     * default type and concurrency of <code>ResultSet</code> can be overridden.
     *
     * @param resultSetType <code>int</code> representing <code>ResultSet<code> type.
     * See ResultSet.TYPE_XXX.
     * @param resultSetConcurrency <code>int</code> representing <code>ResultSet<code> 
     * concurrency type.  See ResultSet.CONCUR_XXX.
     *
     * @return Statement newly created
     *
     * @exception SQLException when DBMS access error occured
     *
     * @since 1.2
     *
     * @see What Is in the JDBC 2.0 API
     */
    public CallableStatement prepareCall(String sql,
					 int resultSetType,
					 int resultSetConcurrency)
	throws SQLException
    {
	return null;
    }

    /**
     * Gets type <code>Map</code> object relating to this <code>Connection</code>.
     * A null <code>Map</code> will be returened except when an application add
     * an entry to the type <code>Map</code>
     *
     * @return java.util.Map object relating to this <code>Connection</code>
     *
     * @since 1.2
     *
     * @see What Is in the JDBC 2.0 API
     */
    public Map getTypeMap()
	throws SQLException
    {
	return null;
    }

    /**
     * Sets specified type <code>map</code> as type <code>Map</code> of
     * this <code>Connection</code>.  A type <code>Map</code> is used
     * in custom mapping between SQUL strucutre type and individual type.
     *
     * @param map <code>java.util.Map</code> to be set for default type mapping
     * of this <code>Connection</code>
     *
     * @since 1.2
     *
     * @see What Is in the JDBC 2.0 API
     */
    public void setTypeMap(Map map)
	throws SQLException
    {
    }
    
    /**
     * Get NamedObject having <CODE>persistentID</CODE>
     * from backend sources inclding  local file and DBMS. 
     *
     * @param persistentID
     * @return NamedObject
     */
    public NamedObject getNamedObjectInBackends(String persistentID)
    {
	return getNamedObjectEntity(persistentID);
    }

    /**
     * Get NamedObject having <CODE>persistentID</CODE>
     * from backend sources inclding  local file and DBMS. 
     *
     * @param persistentID
     * @return NamedObject
     */
    public NamedObject getNamedObjectEntity(String persistentID)
    {
	// check ObjectPool at first
	NamedObject namedObject = (NamedObject)curator.get(persistentID);

	if(namedObject == null) 
	    namedObject = getUnsavedObject(persistentID);

	if(namedObject != null && !namedObject.isNominal())
	    return namedObject;

	namedObject = null;

	Enumeration e = connectionInterfaces.elements();
	while(namedObject == null &&  e.hasMoreElements() ) {
	    AbstractConnection co = (AbstractConnection)e.nextElement();
	    if (co == null || !co.getValid())
		continue;
	    namedObject = co.getNamedObject(persistentID);
	}

	if(namedObject != null) {
	    resolve(namedObject);
	    curator.put(persistentID, namedObject);
	}

	return namedObject;
    }
    
    /**
     * Get NamedObject from somewhere by NamedObject. (local, DB, etc.)
     * @param NamedObject
     * @return NamedObject
     */
    public NamedObject getNamedObject(NamedObject queryTemplate) {
	return null;
    }
    
	
    /**
	 * Save NamedObject to somewhere. (local, DB, etc.)
	 * @param Connection target
	 * @param NamedObject namedObject
	 * @return NamedObject
	 */
	public void saveNamedObject(Connection target, NamedObject namedObject) {
	
	}

	/**
	 * Set the top directory of local database.
	 * @param rootPath the top directory for local database.
	 */
	public void setLocaldbTopDirectory(String rootPath) {
		for (Enumeration e = connectionInterfaces.elements(); e.hasMoreElements(); ) {
			Object o = e.nextElement();
			if (o instanceof LocaldbConnection) {
				((LocaldbConnection)o).setPath(rootPath);
				break;
			}
		}
	}
	
	/**
	 * Get the top directory of local database.
	 * @return the top directory string of local database.
	 */
	public String getLocaldbTopDirectory() {
	    /*if(connectionInterfaces != null)*/ {
		for (Enumeration e = connectionInterfaces.elements(); e.hasMoreElements(); ) {
		    Object o = e.nextElement();
		    if (o instanceof LocaldbConnection) {
			return ((LocaldbConnection)o).getPath();
		    }
		}
	    }
	    return null;
	}

    /**
     * Set connection interfaces
     * @param v connection interfaces.
     */
    public void setConnectionInterfaces(Vector v) {
	disconnectionAll();
	connectionInterfaces = v;
	connectionAll();
    }
    
    /**
     * Get connection interfaces.
     * @return Vector connection interfaces.
     */
    public Vector getConnectionInterfaces() {
	return connectionInterfaces;
    }
    
    /**
     * Connect all connection interface.
     */
    public void connectionAll() {
	if (connectionInterfaces == null || connectionInterfaces.size() <= 0)
	    return;
	for (Enumeration e = connectionInterfaces.elements(); e.hasMoreElements(); ) {
	    AbstractConnection co = (AbstractConnection)e.nextElement();
	    if (co.getValid())
		co.connect();
	}
    }
    
    /**
	 * Disconnect all connection interface.
	 */
	public void disconnectionAll() {
		if (connectionInterfaces == null || connectionInterfaces.size() <= 0)
			return;
		for (Enumeration e = connectionInterfaces.elements(); e.hasMoreElements(); ) {
			AbstractConnection co = (AbstractConnection)e.nextElement();
			if (co.getValid())
				co.disconnect();
		}
	}

    /**
     * Changes holdability of the ResultSet object 
     * created  by using this Connection object
     * to specified <code>holdability</code>
     *
     * @param holdability holdability constant of <CODE>ResultSet</CODE>, either
     * <CODE>ResultSet.HOLD_CURSORS_OVER_COMMIT</CODE> or <CODE>ResultSet.CLOSE_CURSORS_AT_COMMIT</CODE>
     *
     * @exception SQLException when DBMS access error occured, specified paramter is not <CODE>ResultSet</CODE>'s
     * constant representing holdability, or specified <CODE>holdablity</CODE> is not supported
     *
     * @since 1.4
     *
     * @see getHoldability()
     * @see ResultSet
     */
    public void setHoldability(int holdability)
                    throws SQLException
    {
    }


    /**
     * Gets holdability of the ResultSet object 
     * created  by using this Connection object
     *
     * @return int representing holdability of the ResultSet object, eithr
     * <CODE>ResultSet.HOLD_CURSORS_OVER_COMMIT</CODE> or <CODE>ResultSet.CLOSE_CURSORS_AT_COMMIT</CODE>
     *
     * @exception SQLException when DBMS access error occured
     *
     * @since 1.4
     *
     * @see setHoldability(int)
     * @see ResultSet
     */
    public int getHoldability()
                   throws SQLException
    {
	return 0; //!
    }

    /**
     * Creates and returns a new <CODE>Savepoint</CODE> object representing
     * a nameless save point in current transaction
     *
     * @return Savepoint crated
     *
     * @exception SQLException when DBMS access error occured, or
     * this <CODE>Connection</CODE> object is autmatic commit mode
     *
     * @since 1.4
     *
     * @see Savepoint
     */
    /*if[JDK1.4]*/
    public Savepoint setSavepoint()
                       throws SQLException
    {
	return null;
    }
    /*end[JDK1.4]*/


    /**
     * Creates and returns a new <CODE>Savepoint</CODE> object representing
     * a save point with specified <CODE>name</CODE> in current transaction
     *
     * @param name <CODE>String</CODE> representing name of the save point
     *
     * @return Savepoint created
     *
     * @exception SQLException when DBMS access error occured, or
     * this <CODE>Connection</CODE> object is autmatic commit mode
     *
     * @since 1.4
     *
     * @see Savepoint
     */
    /*if[JDK1.4]*/
    public Savepoint setSavepoint(String name)
	throws SQLException
    {
	return null;
    }
    /*end[JDK1.4]*/

    /**
     * Rollbacks all modifications made after set of specified <CODE>Savepoint</CODE>
     * object.  Use this method only if automatic commit is disabled.
     *
     * @param savepoint <CODE>Savepoint</CODE> object to rollback to
     *
     * @exception SQLException when DBMS access error occured, or
     * this <CODE>Connection</CODE> object is autmatic commit mode
     * @since 1.4
     *
     * @see Savepoint
     * @see rollback()
     */
    /*if[JDK1.4]*/
    public void rollback(Savepoint savepoint)
	throws SQLException
    {
	;
    }
    /*end[JDK1.4]*/

    /**
     * Removed specified <CODE>Savepoint</CODE> object from
     * current transaction.  Reference to the <CODE>Savepoint</CODE> object
     * after removal throws <CODE>SQLException</CODE>
     *
     * @param savepoint <CODE>Savepoint</CODE> object to be removed
     *
     * @exception SQLException when DBMS access error occured, or
     * specified <CODE>Savepoint</CODE> object is invalid in
     * current transaction
     *
     * @since 1.4
     *
     * @see Savepoint
     * @see rollback()
     */
    /*if[JDK1.4]*/
    public void releaseSavepoint(Savepoint savepoint)
	throws SQLException
    {
	;
    }
    /*end[JDK1.4]*/

    /**
     * Creates a <CODE>Statement</CODE> generating a <CODE>ResultSet</CODE> object
     * with specified type, concurrency and holdability.
     * This method is same as createSetatement method above, but allows to
     * override default type, concurrency and holdability.
     *
     * @param resultSetType <CODE>ResultSet</CODE> constant, either 
     * <CODE>ResultSet.TYPE_FORWARD_ONLY</CODE>,
     * <CODE>ResultSet.TYPE_SCROLL_INSENSITIVE</CODE> or
     * <CODE>ResultSet.TYPE_SCROLL_SENSITIVE</CODE>
     * @param resultSetConcurrency <CODE>ResultSet</CODe> constant, either
     * <CODE>ResultSet.CONCUR_READ_ONLY</CODE> or
     * <CODE>ResultSet.CONCUR_UPDATABLE</CODE>
     * @param resultSetHoldability <CODE>ResultSet</CODE> constant, either
     * <CODE>ResultSet.HOLD_CURSORS_OVER_COMMIT</CODE> or
     * <CODE>ResultSet.CLOSE_CURSORS_AT_COMMIT</CODE>
     *
     * @return Statement object newly created which creates a <CODE>ResultSet</CODE>
     * object with specified type, concurrency and holdability
     *
     * @exception SQLException when DBMS access error occured, or
     * a specified parameter is not <CODE>ResultSet</CODE> constants
     * representing type, concurrency or holdablity.
     *
     * @since 1.2
     *
     * @see ResultSet
     * @see rollback()
     */
    public Statement createStatement(int resultSetType,
				     int resultSetConcurrency,
				     int resultSetHoldability)
	throws SQLException
    {
	return null;
    }

    /**
     * Craeates a <CODE>PreparedStatement</CODE> which creates a <CODE>ResultSet</CODE>
     * with specified type, concurrency and holdability
     * <P>
     * This method is same above <CODE>prepareStatement</CODE> but allows to override
     * default type, concurrency and holdability of <CODE>ResultSet</CODE>
     *
     * @param sql <CODE>String</CODE> representing SQL statement to be send to DBMS.
     * It may contain one or more ? IN parameter.
     * @param resultSetType <CODE>ResultSet</CODE> constant, either of
     * <CODE>ResultSet.TYPE_FORWARD_ONLY</CODE>,
     * <CODE>ResultSet.TYPE_SCROLL_INSENSITIVE</CODE> or
     * <CODE>ResultSet.TYPE_SCROLL_SENSITIVE</CODE>
     * @param resultSetConcurrency <CODE>ResultSet</CODE> constant, either
     * <CODE>ResultSet.CONCUR_READ_ONLY</CODE> or
     * <CODE>ResultSet.CONCUR_UPDATABLE</CODE>
     * @param resultSetHoldability <CODE>ResultSet</CODE> constant, either
     * <CODE>ResultSet.HOLD_CURSORS_OVER_COMMIT</CODE> or
     * <CODE>ResultSet.CLOSE_CURSORS_AT_COMMIT</CODE>
     *
     * @return PreparedStatement object containing precompiled SQL statement which
     * crates <CODE>ResultSet</CODE> object with specified type, concurrency
     * and holdability.
     *
     * @exception SQLException when DBMS access error occured, or
     * a specified parameter is not <CODE>ResultSet</CODE> constants
     * representing type, concurrency or holdablity.
     *
     * @since 1.4
     *
     * @see ResultSet
     */
    public PreparedStatement prepareStatement(String sql,
					      int resultSetType,
					      int resultSetConcurrency,
					      int resultSetHoldability)
	throws SQLException
    {
	return null;
    }

    /**
     * Craeates a <CODE>CallableStatement</CODE> object which creates a <CODE>ResultSet</CODE>
     * with specified type, concurrency and holdability
     * <P>
     * This method is same above <CODE>prepareCall</CODE> but allows to override
     * default type, concurrency and holdability of <CODE>ResultSet</CODE>
     *
     * @param sql <CODE>String</CODE> representing SQL statement to be send to DBMS.
     * It may contain one or more ? IN parameter.
     * @param resultSetType <CODE>ResultSet</CODE> constant, either of
     * <CODE>ResultSet.TYPE_FORWARD_ONLY</CODE>,
     * <CODE>ResultSet.TYPE_SCROLL_INSENSITIVE</CODE> or
     * <CODE>ResultSet.TYPE_SCROLL_SENSITIVE</CODE>
     * @param resultSetConcurrency <CODE>ResultSet</CODE> constant, either
     * <CODE>ResultSet.CONCUR_READ_ONLY</CODE> or
     * <CODE>ResultSet.CONCUR_UPDATABLE</CODE>
     * @param resultSetHoldability <CODE>ResultSet</CODE> constant, either
     * <CODE>ResultSet.HOLD_CURSORS_OVER_COMMIT</CODE> or
     * <CODE>ResultSet.CLOSE_CURSORS_AT_COMMIT</CODE>
     *
     * @return PreparedStatement object containing precompiled SQL statement which
     * crates <CODE>ResultSet</CODE> object with specified type, concurrency
     * and holdability.
     *
     * @exception SQLException when DBMS access error occured, or
     * a specified parameter is not <CODE>ResultSet</CODE> constants
     * representing type, concurrency or holdablity.
     *
     * @since 1.4
     *
     * @see ResultSet
     */
    public CallableStatement prepareCall(String sql,
					 int resultSetType,
					 int resultSetConcurrency,
					 int resultSetHoldability)
	throws SQLException
    { return null; }

    /**
     * Creates a default PreparedStatement object with a function to get automatic
     * generation key.  Specified parameter determines whether driver is allowed
     * to get automatic generation key.  This parameter will be ignored
     * if SQL statement is not INSERT statement.
     * <P>
     * Note: this method is optimized to process parameterized SQL statements where
     * precompilation is effective.  The prepareStatement method will send the
     * statement to DBMS if the driver supports precompilation.  Some drivers
     * do not support precompilation.  In this case statements will not be sent
     * to DBMS until <CODE>PrepareStatement</CODE> execution.  It does not affect
     * to users directly, but which method trows SQLException.
     * <P>
     * <CODE>ResultSet</CODE> created by returned <CODE>PreparedStatement</CODE>
     * has TYPE_FORWARD_ONLY as its type and CONCUR_READ_ONLY as its concurrency level
     * by default.
     *
     * @param sql <CODE>String</CODE> representing a SQL statement which may contain
     * one or more '?' IN parameter placeholders.
     * @param autoGeneratedKeys a flag indicating whether automatic generated key,
     * one of <CODE>Statement</CODE> contatns
     *
     * @return PreparedStatement object containing precompiled SQL statement
     *
     * @exception SQLException when DBMS access error occured, or
     * <CODE>authoGeneratedKeys</CODE> is not a <CODE>Statement</CODE> constants
     * representing automatic generation key flag
     *
     * @since 1.4
     */
    public PreparedStatement prepareStatement(String sql,
					      int autoGeneratedKeys)
	throws SQLException
    {	return  null;}

    
    /**
     * Creates a default <CODE>PreparedStatement</CODE> object which returns
     * automatic generated key specified by <CODE>columnIndexes</CODE>.
     * This array contains target table column indices of automatic generated keys.
     * This array is ignored if SQL statement is not INSERT statement.
     * <P>
     * A SQL statement with or without IN paramter can be precompiled and
     * stored into a <CODE>PreparedStatement</CODE> object.
     * Therefore, this object can be used the SQL statement multiple times
     * effectively.
     * <P>
     * Note: this method is optimized to process parameterized SQL statements where
     * precompilation is effective.  The prepareStatement method will send the
     * statement to DBMS if the driver supports precompilation.  Some drivers
     * do not support precompilation.  In this case statements will not be sent
     * to DBMS until <CODE>PrepareStatement</CODE> execution.  It does not affect
     * to users directly, but which method trows SQLException.
     * <P>
     * <CODE>ResultSet</CODE> created by returned <CODE>PreparedStatement</CODE>
     * has TYPE_FORWARD_ONLY as its type and CONCUR_READ_ONLY as its concurrency level
     * by default.
     *
     * @param sql <CODE>String</CODE> representing a SQL statement which may contain
     * one or more '?' IN parameter placeholders.
     * @param autoGeneratedKeys a flag indicating whether automatic generated key,
     * one of <CODE>Statement</CODE> contatns
     * @param columnINdexes an array representing columns should be returned from
     * inserted row.
     *
     * @return PreparedStatement object containing precompiled SQL statement with
     * function to return automatic generated key according to specified 
     * <CODE>columnIndexes</CODE>
     *
     * @exception SQLException when DBMS access error occured
     *
     * @since 1.4
     */
    public PreparedStatement prepareStatement(String sql,
					      int[] columnIndexes)
	throws SQLException
    { return null;}
    
    /**
     * Creates a default <CODE>PreparedStatement</CODE> object which returns
     * automatic generated key specified by <CODE>columnNames</CODE>.
     * This array contains target table column names of automatic generated keys.
     * This array will be ignored if SQL statement is not INSERT statement.
     * <P>
     * A SQL statement with or without IN paramter can be precompiled and
     * stored into a <CODE>PreparedStatement</CODE> object.
     * Therefore, this object can be used the SQL statement multiple times
     * effectively.
     * <P>
     * Note: this method is optimized to process parameterized SQL statements where
     * precompilation is effective.  The prepareStatement method will send the
     * statement to DBMS if the driver supports precompilation.  Some drivers
     * do not support precompilation.  In this case statements will not be sent
     * to DBMS until <CODE>PrepareStatement</CODE> execution.  It does not affect
     * to users directly, but which method trows SQLException.
     * <P>
     * <CODE>ResultSet</CODE> created by returned <CODE>PreparedStatement</CODE>
     * has TYPE_FORWARD_ONLY as its type and CONCUR_READ_ONLY as its concurrency level
     * by default.
     *
     * @param sql <CODE>String</CODE> representing a SQL statement which may contain
     * one or more '?' IN parameter placeholders.
     * @param autoGeneratedKeys a flag indicating whether automatic generated key,
     * one of <CODE>Statement</CODE> contatns
     * @param columnINdexes an array representing columns should be returned from
     * inserted row.
     *
     * @return PreparedStatement object containing precompiled SQL statement with
     * function to return automatic generated key according to specified 
     * <CODE>columnNames</CODE>
     *
     * @exception SQLException when DBMS access error occured
     *
     * @since 1.4
     */
    public PreparedStatement prepareStatement(String sql,
					      String[] columnNames)
	throws SQLException
    {return null;}

    public void putNamedObjects(Vector v)
    {
	if(v == null || v.isEmpty())
	    return;
	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    putNamedObject((NamedObject)e.nextElement());
	}
    }

    public void putNamedObject(NamedObject object)
    {
	curator.put(object);
    }

    public void putNode(NamedObjectNode node)
    {
	nodePool.put(node);
    }

    public void putModels(Vector v)
    {
	if(v == null || v.isEmpty())
	    return;
	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    putModel((NamedObjectEditModel)e.nextElement());
	}
    }

     public void putModel(NamedObjectEditModel model)
    {
	modelPool.put(model.getName(), model);
    }

    public void putTree(NameTreeModel tree)
    {
	treePool.put(tree);
    }

    public NamedObjectNode getNode(String name)
    {
	return (NamedObjectNode)nodePool.get(name);
    }

    public NamedObjectEditModel getModel(String name)
    {
	Object o = modelPool.get(name);
	if(o != null){
	    System.err.println(o.getClass());
	    System.err.println(o);
	}
	return (NamedObjectEditModel)o;
	    //	return (NamedObjectEditModel)modelPool.get(name);
    }

    public NameTreeModel getTree(String name)
    {
	return (NameTreeModel)treePool.get(name);
    }

    public void putByName(NameUsage usage)
    {
	byNamePool.put(usage);
    }

    public void putByAuthor(NameUsage usage)
    {
	byAuthorPool.put(usage);
    }

    public void putByYear(NameUsage usage)
    {
	byYearPool.put(usage);
    }

    public void putUnsavedObjects(Vector v)
    {
	if(v == null || v.isEmpty())
	    return;
	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    putUnsavedObject((NamedObject)e.nextElement());
	}
    }

    public void putUnsavedObject(Name object)
    {
	if(object == null || object.getName() == null)
	    return;

	String name = object.getName();

	if(name == null)
	    return;

	if(unsavedPool == null) {
	    unsavedPool = new Hashtable();
	}

	unsavedPool.put(name, object);
	curator.putToResolver(name, object);

	if(object instanceof NameUsage) {
	    NameUsage n = (NameUsage)object;
	    putByName(n);
	    putByAuthor(n);
	    putByYear(n);
	}

	if(object instanceof Author) {
	    putAuthorByName((Author)object);
	}
    }

    public void putAuthorByName(Author author)
    {
	String name = author.getFullname();
	
	if(name == null)
	    return;

	for (int i = name.length(); i > 0; i--) {
	    if(Character.isWhitespace(name.charAt(i - 1)))
		continue;
	    String partName = name.substring(0, i);
	    Vector v = getAuthorByName(partName);
	    if(v == null) {
		v = new Vector();
		authorByNamePool.put(partName, v);
	    }
	    if(!v.contains(author))
		v.addElement(author);
	}
    }



    public Vector getByName(String name)
    {
	return (Vector)byNamePool.get(name);
    }

    public Vector getByAuthor(String name)
    {
	return (Vector)byAuthorPool.get(name);
    }
    public Vector getByYear(String year)
    {
	return (Vector)byYearPool.get(year);
    }

    public Vector getAuthorByName(String name)
    {
	return (Vector)authorByNamePool.get(name);
    }

    public void saveAllUnsavedObject()
    {
	Vector v = getUnsavedObjects();

	if(v == null || v.isEmpty())
	    return;

	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    NamedObject object = (NamedObject)e.nextElement();
	    if(saveNamedObject(object) > 0)
		removeUnsavedObject(object);
	}
	clearUnsavedObject();
    }

    public void saveUnsavedObject(NamedObject object)
    {
	object = getUnsavedObject(object.getName());
	if(object == null)
	    return;

	if(saveNamedObject(object) > 0)
	    removeUnsavedObject(object);
    }

    public static Vector getProduct(Vector v1, Vector v2) 
    {
	if(v1 == null || v1.isEmpty() ||
	   v2 == null || v2.isEmpty())
	    return null;

	Vector v = v1;
	if(v2.size() < v1.size()) {
	    v1 = v2;
	    v2 = v;
	}
	v = new Vector();

	Enumeration e = v1.elements();
	while(e.hasMoreElements()) {
	    Object o = e.nextElement();
	    if(v2.contains(o))
		v.addElement(o);
	}
	return v;
    }

    public static Vector getSum(Vector v1, Vector v2) 
    {
	if(v1 == null || v1.isEmpty()) {
	    if(v2 == null || v2.isEmpty())
	       return null;
	    return v2;
	}

	if(v2 == null || v2.isEmpty())
	    return v1;
   
	Vector v = v1;
	if(v2.size() < v1.size()) {
	    v1 = v2;
	    v2 = v;
	}
	v = (Vector)v2.clone();

	Enumeration e = v1.elements();
	while(e.hasMoreElements()) {
	    Object o = e.nextElement();
	    if(!v2.contains(o))
		v.addElement(o);
	}
	return v;
    }

    public NamedObject resolve(Object key, NamedObject name)
    {
	return curator.resolve(key, name);
    }

    public NamedObject resolve(NamedObject name)
    {
	return (NamedObject)curator.resolve(name);
    }

    public NamedObject putUnresolved(NamedObject name)
    {
	return curator.putUnresolved(name);
    }

    public NamedObject getUnresolved(Object key)
    {
	return curator.getUnresolved(key);
    }

    /**
     * Get NamedObject having <CODE>persistentID</CODE>
     * from eithr <CODE>ObjectPool</CODE> or
     * other backend sources inclding  local file and DBMS, 
     * with preference of <CODE>ObjectPool</CODE>.
     *
     * @param persistentID
     * @return NamedObject
     */
    public NamedObject getNamedObject(String persistentID) {
	//	NamedObject namedObject = getNamedObjectInPool(persistentID);
	NamedObject namedObject = getUnsavedObject(persistentID);
	if(namedObject == null)
	    namedObject = getNamedObjectEntity(persistentID);
	return namedObject;
    }

    /**
     * Get NamedObject with <CODE>persistentID</CODE> from
     * <CODE>ObjctPool</CODE>
     *
     * @param persistentID
     * @return NamedObject
     */
    public NamedObject getNamedObjectInPool(String persistentID)
    {
	/*
	NamedObject namedObject = getUnsavedObject(persistentID);
	if(namedObject == null)
	    namedObject = getUnresolved(persistentID);
	return namedObject;
	*/
	return getUnsavedObject(persistentID);
    }

    public NamedObject getUnsavedObject(String name)
    {
	if(unsavedPool == null ||
	   name == null || name.length() == 0)
	    return null;
	return (NamedObject)unsavedPool.get(name);
    }

    public NamedObject removeUnsavedObject(NamedObject object)
    {
	if(unsavedPool == null || object == null)
	    return null;
	return (NamedObject)unsavedPool.remove(object);
    }

    public NamedObject removeUnsavedObject(String name)
    {
	if(unsavedPool == null ||
	   name == null || name.length() == 0)
	    return null;
	return removeUnsavedObject(getUnsavedObject(name));
    }

    public void clearUnsavedObject()
    {
	clearUnsavedObject(false);
    }

    public void clearUnsavedObject(boolean force)
    {
	if(unsavedPool == null)
	    return;
	synchronized(unsavedPool) {
	    if(force) {
		if(unsavedPool != null)
		    unsavedPool.clear();
	    }
	    if(unsavedPool.isEmpty())
		unsavedPool = null;
	}
    }

    public Vector getUnsavedObjects()
    {
	if(unsavedPool == null)
	    return null;
	Vector v = new Vector();
	synchronized(unsavedPool) {
	    Enumeration e = unsavedPool.elements();
	    while(e.hasMoreElements()) {
	    	Object o = e.nextElement();
	    	if (!v.contains(o))
				v.addElement(o);
	    }
	}
	return v;
    }

    public String getViewName(NameTreeModel tree)
    {
	if(nextSuffix == null) {
	    nextSuffix = new Hashtable();
	    publications = new Hashtable();
	}
	String viewName = tree.getViewName(false);
	StringBuffer buffer = new StringBuffer(viewName);
	String suffix = (String)nextSuffix.get(viewName);
	String publicationName = tree.getPublicationName();

	Hashtable publicationNames = 
	    (Hashtable)publications.get(viewName);
	if(suffix != null) {
	    if(publicationNames != null) {
		viewName = (String)publicationNames.get(tree.getPublicationName());
		if(viewName != null) {
		    tree.setViewName(viewName);
		    return viewName;
		}
		viewName = buffer.toString();
	    }
	    if(publicationNames == null) {
		publicationNames = new Hashtable();
		publications.put(viewName, publicationNames);
	    }
	    nextSuffix.put(viewName, new String(new char[]{(char)(suffix.charAt(suffix.length() - 1) + 1)}));
	    buffer.append(suffix);
	}
	else {
	    NameTreeModel model = getTree(viewName);
	    if(model != null && model != tree) {
		String modelPublicationName = model.getPublicationName();
		if(modelPublicationName.equals(publicationName))
		   return viewName;

		publicationNames = new Hashtable();
		publications.put(viewName, publicationNames);
		buffer.append('a');
		model.setViewName(buffer.toString());
		publicationNames.put(modelPublicationName, model.getViewName());
		buffer = new StringBuffer(viewName);
		buffer.append('b');
		nextSuffix.put(viewName, "c");
	    }
	}
	viewName = buffer.toString();
	tree.setViewName(viewName);
	if(publicationNames != null)
	    publicationNames.put(publicationName, viewName);
	return viewName;
    }
}
