/*
 * ObjectPool.java: 
 *
 * Copyright (c) 2002 Takehisa Okada,  Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ObjectPool.java,v 1.24 2002/11/12 03:14:07 nozomi Exp $
 * $Log: ObjectPool.java,v $
 * Revision 1.24  2002/11/12 03:14:07  nozomi
 * use Name if possible, instead of NamedObject
 *
 * Revision 1.23  2002/10/10 09:03:33  nozomi
 * fix cast bug in searchNameUsage()
 *
 * Revision 1.22  2002/09/26 05:13:20  ryo
 * add contains check
 *
 * Revision 1.21  2002/09/20 09:14:00  nozomi
 * fix null object bug in putUnsavedObject()
 *
 * Revision 1.20  2002/09/17 05:05:09  nozomi
 * add resolve(Object, NamedObject)
 *
 * Revision 1.19  2002/09/10 07:13:57  nozomi
 * support NameResolver
 *
 * Revision 1.18  2002/09/08 14:16:29  nozomi
 * support Author search
 *
 * Revision 1.17  2002/09/07 17:12:03  nozomi
 * avoid duplicated put into Vector for search
 *
 * Revision 1.16  2002/09/07 03:22:13  nozomi
 * remove debug info
 *
 * Revision 1.15  2002/09/07 01:40:08  nozomi
 * support search of unsaved objects
 *
 * Revision 1.14  2002/09/06 08:51:56  nozomi
 * return from clearUnsavedObject(boolean) if the pool is empty
 *
 * Revision 1.13  2002/09/06 06:05:44  nozomi
 * automatic put to by-Pools if unsaved object is a NameUsage
 *
 * Revision 1.12  2002/09/06 03:09:27  nozomi
 * fix method names; removeUnusedObject(NamedObject)
 *
 * Revision 1.11  2002/09/06 02:58:36  nozomi
 * fix a mistype in putUnusedObject()
 *
 * Revision 1.10  2002/09/06 02:56:52  nozomi
 * modify putUnusedObject() to save NamedObject
 *
 * Revision 1.9  2002/09/06 02:48:49  nozomi
 * add unusedPool and pools to search by name, author or year
 *
 * Revision 1.8  2002/09/02 15:39:41  nozomi
 * use node name, instead of view name, as a key to tree
 *
 * Revision 1.7  2002/09/01 10:47:55  nozomi
 * remove unused println
 *
 * Revision 1.6  2002/09/01 01:47:20  nozomi
 * fix bugs in getModel() and getViewName()
 *
 * Revision 1.5  2002/08/30 04:44:22  nozomi
 * get() returns null if Hashtable is null
 *
 * Revision 1.4  2002/08/30 03:14:13  nozomi
 * change definition of get/setName and get/setViewName
 *
 * Revision 1.3  2002/08/30 03:10:07  nozomi
 * fix bug in CVS log string
 *
 * Revision 1.2  2002/08/30 02:22:21  nozomi
 * add put* and get* methods
 *
 * Revision 1.1  2002/07/01 04:49:46  t.okada
 * add ObjectPool
 */

package org.nomencurator.broker;

import java.util.Vector;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Enumeration;
import java.util.Iterator;

import org.nomencurator.Author;
import org.nomencurator.Name;
import org.nomencurator.Nameable;
import org.nomencurator.NamedObject;
import org.nomencurator.NameResolver;
import org.nomencurator.NameUsage;

import org.nomencurator.editor.NameUsageListRow;
import org.nomencurator.editor.model.NamedObjectNode;
import org.nomencurator.editor.model.NameTreeModel;
import org.nomencurator.editor.model.NamedObjectEditModel;

/**
 * <CODE>ObjectPool</CODE> provides a pool of <CODE>Object</CODE>
 *
 * @version 12 Nov 2002
 * @author Takehisa Okada
 * @author 	Nozomi `James' Ytow
 */
public class ObjectPool
{

    /** Object pool size. */
    private int _poolMaxSize;
    
    private Hashtable _objectPool;
    private ObjectList _objectList;
    
    protected Hashtable nodePool;
    protected ObjectList nodeList;

    protected Hashtable modelPool;
    protected ObjectList modelList;

    protected Hashtable treePool;
    protected ObjectList treeList;

    protected Hashtable viewPool;
    protected ObjectList viewList;

    protected Hashtable nextSuffix;
    protected Hashtable publications;

    protected Hashtable byNamePool;
    protected ObjectList byNameList;
    
    protected Hashtable byAuthorPool;
    protected ObjectList byAuthorList;
    
    protected Hashtable byYearPool;
    protected ObjectList byYearList;
    
    protected Hashtable byRankPool;
    protected ObjectList byRankList;

    protected Hashtable authorByNamePool;
    protected ObjectList authorByNameList;

    protected Hashtable unsavedPool;

    protected NameResolver resolver;

    
    
    /** Constructor. */
    public ObjectPool()
    {
	this(65536);//Provisional value
    }
    
    /**
     * Constructor.
     */
    public ObjectPool(int poolMaxSize)
    {
	_poolMaxSize = poolMaxSize;
	_objectPool = new Hashtable(_poolMaxSize);
	_objectList = new ObjectList();
	resolver = new NameResolver();
    }
	
    public void setPoolMaxSize(int size)
    {
	_poolMaxSize = size;
    }
    
    public int getPoolMaxSize()
    {
	return _poolMaxSize;
    }
    
    /** Add a <Code>NamedObject</CODE> */
    //    public void put(NamedObject obj)
    public void put(Name obj)
    {
	if(obj == null)
	    return;
	//	addObject(obj.persistentID(), obj);
	addObject(obj.getName(), obj);
    }

    /** Add a <Code>NamedObject</CODE> */
    //    public void addObject(String persistentID, NamedObject obj)
    public void addObject(String persistentID, Name obj)
    {
	put(_objectPool, _objectList, persistentID, obj);
	//putToResolver(persistentID, obj);
	resolver.resolve(obj);
    }

    public void putNode(NamedObjectNode node)
    {
	if(nodePool == null) {
	    nodePool = new Hashtable(_poolMaxSize);
	    nodeList = new ObjectList();
	}

	put(nodePool, nodeList, node.getName(), node);
    }

    public void putModel(NamedObjectEditModel model)
    {
	if(modelPool == null) {
	    modelPool = new Hashtable(_poolMaxSize);
	    modelList = new ObjectList();
	}

	put(modelPool, modelList, model.getName(), model);
    }

    public void putTree(NameTreeModel tree)
    {
	if(treePool == null) {
	    treePool = new Hashtable(_poolMaxSize);
	    treeList = new ObjectList();
	}

	//	put(treePool, treeList, getViewName(tree), tree);
	//	put(treePool, treeList, ((Nameable)tree.getRoot()).getName(), tree);
	put(treePool, treeList, tree.getName(), tree);
    }

    public void putView(NameTreeModel tree)
    {
	if(viewPool == null) {
	    viewPool = new Hashtable(_poolMaxSize);
	    viewList = new ObjectList();
	}
	
	String viewName =  getViewName(tree);

	NameTreeModel view = getView(viewName);
	if(view != null) {
	    //marge two trees
	}

	put(viewPool, viewList, getViewName(tree), view);
    }


    public void putByName(NameUsage usage)
    {
	if(byNamePool == null) {
	    byNamePool = new Hashtable(_poolMaxSize);
	    byNameList = new ObjectList();
	}

	String name = usage.getUsedName();
	
	if(name == null)
	    return;

	for (int i = name.length(); i > 0; i--) {
	    if(Character.isWhitespace(name.charAt(i - 1)))
		continue;
	    String partName = name.substring(0, i);
	    Vector v = getByName(partName);
	    if(v == null) {
		v = new Vector();
		put(byNamePool, byNameList, partName, v);
	    }
	    if(!v.contains(usage))
		v.addElement(usage);
	}
    }

    public void putByAuthor(NameUsage usage)
    {
	Vector authors = usage.getAuthors();
	if(authors == null ||
	   authors.isEmpty())
	    return;

	if(byAuthorPool == null) {
	    byAuthorPool = new Hashtable(_poolMaxSize);
	    byAuthorList = new ObjectList();
	}

	Enumeration a = authors.elements();
	while(a.hasMoreElements()) {
	    String name = 
		((Author)a.nextElement()).getFullSurname();
	    if(name == null || name.length() == 0)
		continue;
	    
	    for (int i = name.length(); i > 0; i--) {
		if(Character.isWhitespace(name.charAt(i - 1)))
		    continue;
		String partName = name.substring(0, i);
		Vector v = getByAuthor(partName);
		if(v == null) {
		    v = new Vector();
		    put(byAuthorPool, byAuthorList, partName, v);
		}
		
		if(!v.contains(usage))
		    v.addElement(usage);
	    }
	}
    }

    public void putByYear(NameUsage usage)
    {
	if(byYearPool == null) {
	    byYearPool = new Hashtable(_poolMaxSize);
	    byYearList = new ObjectList();
	}

	String year = usage.getYear();
	if(year == null)
	    return;
	for (int i = year.length(); i > 0; i--) {
	    if(Character.isWhitespace(year.charAt(i - 1)))
		continue;
	    String partYear = year.substring(0, i);
	    Vector v = getByYear(partYear);
	    if(v == null) {
		v = new Vector();
		put(byYearPool, byYearList, partYear, v);
	    }
	    if(!v.contains(usage))
		v.addElement(usage);
	}
    }

    public void putAuthorByName(Author author)
    {
	if(authorByNamePool == null) {
	    authorByNamePool = new Hashtable(_poolMaxSize);
	    authorByNameList = new ObjectList();
	}

	String name = author.getFullname();
	
	if(name == null)
	    return;

	for (int i = name.length(); i > 0; i--) {
	    if(Character.isWhitespace(name.charAt(i - 1)))
		continue;
	    String partName = name.substring(0, i);
	    Vector v = getAuthorByName(partName);
	    if(v == null) {
		v = new Vector();
		put(authorByNamePool, authorByNameList, partName, v);
	    }
	    if(!v.contains(author))
		v.addElement(author);
	}
    }

    //    public void putUnsavedObject(NamedObject object)
    public void putUnsavedObject(Name object)
    {
	if(object == null || object.getName() == null)
	    return;

	String name = object.getName();

	if(name == null)
	    return;

	if(unsavedPool == null) {
	    unsavedPool = new Hashtable(_poolMaxSize);
	}

	unsavedPool.put(name, object);
	putToResolver(name, object);

	if(object instanceof NameUsage) {
	    NameUsage n = (NameUsage)object;
	    putByName(n);
	    putByAuthor(n);
	    putByYear(n);
	}

	if(object instanceof Author) {
	    putAuthorByName((Author)object);
	}
    }


    public String getViewName(NameTreeModel tree)
    {
	if(nextSuffix == null) {
	    nextSuffix = new Hashtable();
	    publications = new Hashtable();
	}
	String viewName = tree.getViewName(false);
	StringBuffer buffer = new StringBuffer(viewName);
	String suffix = (String)nextSuffix.get(viewName);
	String publicationName = tree.getPublicationName();

	Hashtable publicationNames = 
	    (Hashtable)publications.get(viewName);
	if(suffix != null) {
	    if(publicationNames != null) {
		viewName = (String)publicationNames.get(tree.getPublicationName());
		if(viewName != null) {
		    tree.setViewName(viewName);
		    return viewName;
		}
		viewName = buffer.toString();
	    }
	    if(publicationNames == null) {
		publicationNames = new Hashtable();
		publications.put(viewName, publicationNames);
	    }
	    nextSuffix.put(viewName, new String(new char[]{(char)(suffix.charAt(suffix.length() - 1) + 1)}));
	    buffer.append(suffix);
	}
	else {
	    NameTreeModel model = getTree(viewName);
	    if(model != null && model != tree) {
		String modelPublicationName = model.getPublicationName();
		if(modelPublicationName.equals(publicationName))
		   return viewName;

		publicationNames = new Hashtable();
		publications.put(viewName, publicationNames);
		buffer.append('a');
		model.setViewName(buffer.toString());
		publicationNames.put(modelPublicationName, model.getViewName());
		buffer = new StringBuffer(viewName);
		buffer.append('b');
		nextSuffix.put(viewName, "c");
	    }
	}
	viewName = buffer.toString();
	tree.setViewName(viewName);
	if(publicationNames != null)
	    publicationNames.put(publicationName, viewName);
	return viewName;
    }

    public void put(Hashtable pool, ObjectList list, String name, Object obj)
    {
	if(_poolMaxSize == 0)
	    return;
	if(name == null || name.length() == 0)
	    return;

	Object previous = pool.get(name);
	if(previous != null &&
	   obj == ((ObjectNode)previous).getObject())
	    return;
	previous = null;

	pool.put(name, list.addFirst(obj));
	
	if(_objectList.size() >= _poolMaxSize) {
	    name = (String)list.deleteLastNode();
	    if(name != null)
		pool.remove(name);
	}
    }

    
	
    /**
     * Get Object
     */
    public Object getObject(String persistentID)
    {
	return get(persistentID, _objectPool, _objectList);
    }

    public NamedObjectNode getNode(NamedObject object)
    {
	return getNode(object.getName());
    }

    public NamedObjectNode getNode(String name)
    {
	return (NamedObjectNode)get(name, nodePool, nodeList);
    }

    public NamedObjectEditModel getModel(NamedObject object)
    {
	if(object == null)
	    return null;
	return getModel(object.getName());
    }

    public NamedObjectEditModel getModel(String name)
    {
	return (NamedObjectEditModel)get(name, modelPool, modelList);
    }

    public NameTreeModel getTree(NamedObject object)
    {
	return getTree(object.getName());
    }

    public NameTreeModel getTree(String name)
    {
	return (NameTreeModel)get(name, treePool, treeList);
    }

    public NameTreeModel getView(String name)
    {
	return (NameTreeModel)get(name, viewPool, viewList);
    }

    public Vector getByName(String name)
    {
	return (Vector)get(name, byNamePool, byNameList);
    }

    public Vector getByAuthor(String name)
    {
	return (Vector)get(name, byAuthorPool, byAuthorList);
    }

    public Vector getByYear(String year)
    {
	return (Vector)get(year, byYearPool, byYearList);
    }

    public Vector getAuthorByName(String name)
    {
	return (Vector)get(name, authorByNamePool, authorByNameList);
    }

    /**
     * Gets an <CODE>Object</CODE> having the <CODE>name</CODE>,
     * or null if no <CODE>Object</CODE> is named as <CODE>name</CODE>,
     *
     * @param name <CODE>String</CODE> specifying name of an <CODE>Object</CODE>
     */
    public Object get(String name, Hashtable pool, ObjectList list)
    {
	if(_poolMaxSize == 0 ||
	   pool == null ||
	   name == null ||
	   name.length() == 0)
	    return null;
	
	Object object = pool.get(name);
	if(object != null) {
	    ObjectNode objectNode = (ObjectNode)object;
	    object = objectNode.getObject();
	    
	    if(object != null) {
		list.deleteNode(objectNode);
		pool.put(name, list.addFirst(object));
	    }
	}
	return object;
    }
    
    private void putObjectNode(String persistentID, ObjectNode objectNode) {
	_objectPool.put(persistentID, objectNode);
	_objectList.addNode(objectNode);
    }
    
    private ObjectNode getObjectNode(String persistentID) {
	return (ObjectNode)_objectPool.get(persistentID);
    }

    public NamedObject getUnsavedObject(String name)
    {
	if(unsavedPool == null ||
	   name == null || name.length() == 0)
	    return null;
	return (NamedObject)unsavedPool.get(name);
    }

    public NamedObject removeUnsavedObject(NamedObject object)
    {
	if(unsavedPool == null || object == null)
	    return null;
	return (NamedObject)unsavedPool.remove(object);
    }

    public NamedObject removeUnsavedObject(String name)
    {
	if(unsavedPool == null ||
	   name == null || name.length() == 0)
	    return null;
	return removeUnsavedObject(getUnsavedObject(name));
    }

    public void clearUnsavedObject()
    {
	clearUnsavedObject(false);
    }

    public void clearUnsavedObject(boolean force)
    {
	if(unsavedPool == null)
	    return;
	synchronized(unsavedPool) {
	    if(force) {
		if(unsavedPool != null)
		    unsavedPool.clear();
	    }
	    if(unsavedPool.isEmpty())
		unsavedPool = null;
	}
    }

    public Vector getUnsavedObjects()
    {
	if(unsavedPool == null)
	    return null;
	Vector v = new Vector();
	synchronized(unsavedPool) {
	    Enumeration e = unsavedPool.elements();
	    while(e.hasMoreElements()) {
	    	Object o = e.nextElement();
	    	if (!v.contains(o))
				v.addElement(o);
	    }
	}
	return v;
    }

    /**
     * get NameUsage list from the name, author, year
     * @param NameUsage
     * @param String name
     * @param String author
     * @param String year
     * @return Vector<NameUsageListRow> name record list
     */
    public Vector searchNameUsage(String name, String author, String year)
    {
	if((byNamePool == null || byNamePool.size() == 0 ) &&
	   (byAuthorPool == null || byAuthorPool.size() == 0 ) &&
	   (byYearPool == null || byYearPool.size() == 0 ))
	    return null;
	Vector byName = null;
	Vector byAuthor = null;
	Vector byYear = null;
	Vector v = null;

	int keys = 0;

	if(name != null && name.length() > 0) {
	    byName = getByName(name);
	    keys |= 1;
	}
	if(author != null && author.length() > 0) {
	    byAuthor = getByAuthor(author);
	    keys |= 2;
	}
	if(year != null && year.length() > 0) {
	    byYear = getByYear(year);
	    keys |= 4;
	}

	switch(keys) {
	case 0: //no key was given
	    v = new Vector();
	    //	    Enumeration e = byYearPool.elements();
	    Enumeration e = byYearPool.keys();
	    while(e.hasMoreElements()) {
		v = NamedObjectBroker.getSum(v, getByYear((String)e.nextElement()));
	    }
	    break;
	case 1: //look up only by name
	    v = byName;
	    break;
	case 2: //look up only by author
	    v = byAuthor;
	    break;
	case 4: //look up only by year
	    v = byYear;
	    break;
	case 7: //look up by all keys
	    v = NamedObjectBroker.getProduct(NamedObjectBroker.getProduct(byName, byAuthor), byYear);
	    break;
	case 3: //look up by name and author
	    v = NamedObjectBroker.getProduct(byName, byAuthor);
	    break;
	case 5: //look up by name and year
	    v = NamedObjectBroker.getProduct(byName, byYear);
	    break;
	case 6: //look up by author and year
	    v = NamedObjectBroker.getProduct(byAuthor, byYear);
	    break;
	}

	if(v == null || v.isEmpty())
	    return null;

	Vector rows = new Vector();
	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    NameUsage n = (NameUsage)e.nextElement();
	    rows.addElement(new NameUsageListRow(n.getName(), 
						 n.getUsedName(), 
						 n.getAppearance().getPublication().getAuthorListSummary(),
						 n.getYear()));
	}

	v = null;
	e = null;
	return rows;
    }

    //public NamedObject putToResolver(Object key, NamedObject obj)
    public NamedObject putToResolver(Object key, Name obj)
    {
	if(!obj.isNominal() || resolver.get(key) != null)
	    return (NamedObject)resolver.resolve(obj);
	return (NamedObject)resolver.put(key, obj);
    }

    public NamedObject resolve(Object key, NamedObject name)
    {
	return (NamedObject)resolver.resolve(key, name);
    }

    public NamedObject resolve(NamedObject name)
    {
	return (NamedObject)resolver.resolve(name);
    }

    public NamedObject putUnresolved(NamedObject name)
    {
	return (NamedObject)resolver.put(name.getName(), name);
    }

    public NamedObject getUnresolved(Object key)
    {
	return (NamedObject)resolver.get(key);
    }

    /*	
	public static void main(String [] args) {
	
	ObjectPool op = new ObjectPool();
	
	NameUsage nu = new NameUsage();
	Long t1 = new Long(System.currentTimeMillis());
	System.out.println(t1.toString() + "  START!!");
	
	for(int i = 0; i < 100000; i++) {
	op.addObject(Integer.toString(i), nu);
	}
	
	Long t15 = new Long(System.currentTimeMillis());
	System.out.println(t15.toString() + "  FLAG!!");
	
	for(int i = 0; i < 10000; i++) {
	NameUsage nu2 = (NameUsage)op.getObject(Integer.toString(i));
	}
	
	Long t2 = new Long(System.currentTimeMillis());
	System.out.println(t2.toString() + "  END!!");
	
	}
    */	
}

class ObjectList
{
    private ObjectNode _top = null;
    private ObjectNode _last = null;
    private int _size = 0;
    
    public ObjectList() { }
    
    public ObjectNode add(Object object)
    {
	ObjectNode objectNode = new ObjectNode(object);
	addNode(objectNode);
	return objectNode;
    }
    
    public ObjectNode addFirst(Object object)
    {
	ObjectNode objectNode = new ObjectNode(object);
	addNodeFirst(objectNode);
	return objectNode;
    }
    
    public void addNode(ObjectNode objectNode)
    {
	if(_last == null) {
	    objectNode.setPrev(null);
	    objectNode.setNext(null);
	    _top = objectNode;
	    _last = objectNode;
	}
	else {
	    _last.setNext(objectNode);
	    objectNode.setPrev(_last);
	    _last = objectNode;
	}
	_size++;
    }
    
    public void addNodeFirst(ObjectNode objectNode)
    {
	objectNode.setPrev(null);
	objectNode.setNext(_top);
	if(_top != null)
	    _top.setPrev(objectNode);
	
	_top = objectNode;
	
	_size++;
    }
    
    public ObjectNode getLastNode()
    {
	return _last;
    }
    
    public void deleteNode(ObjectNode objectNode)
    {
	if(objectNode == null)
	    return;
	
	ObjectNode nextObjectNode = objectNode.getNext();
	ObjectNode prevObjectNode = objectNode.getPrev();
	if(prevObjectNode != null)
	    prevObjectNode.setNext(nextObjectNode);
	if(nextObjectNode != null)
	    nextObjectNode.setPrev(prevObjectNode);
	_size--;
    }
    
    public String deleteLastNode()
    {
	
	if(_last == null || _last.getPrev() == null)
	    return null;
	
	_last.getPrev().setNext(null);
	_last = _last.getPrev();
	
	_size--;

	Object obj = _last.getObject();

	if(obj instanceof Nameable)
	    return ((Nameable)obj).getName();

	return obj.toString();
    }
    
    public int size()
    {
	return _size;
    }
    
    public Vector getVectorList()
    {
	if(_top == null)
	    return null;
	
	Vector list = new Vector();
	_top.getVectorList(list);
	
	return list;
    }	
}


class ObjectNode
{
    private ObjectNode _prev;
    private ObjectNode _next;
    private Object _obj;
    
    public ObjectNode() {
    }
    
    public ObjectNode(Object obj) {
	this(null, null, obj);
    }
    
    public ObjectNode(ObjectNode prev, ObjectNode next, Object obj)
    {
	_prev = prev;
	_next = next;
	_obj = obj;
    }
    
    public void setPrev(ObjectNode prev)
    {
	_prev = prev;
    }
    public void setNext(ObjectNode next)
    {
	_next = next;
    }

    public void setObject(NamedObject obj)
    {
	_obj = obj;
    }

    public ObjectNode getPrev()
    {
	return _prev;
    }
    public ObjectNode getNext()
    {
	return _next;
    }

    public Object getObject()
    {
	return _obj;
    }
    
    public void getVectorList(Vector list)
    {
	list.add(this);
	if(this.getNext() == null)
	    return;
	this.getNext().getVectorList(list);	
    }

}
