/*
 * ParseXml.java: 
 *
 * Copyright (c) 2002 Hiroyuki Okawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ParseXml.java,v 1.10 2002/08/31 17:38:28 nozomi Exp $
 * $Log: ParseXml.java,v $
 * Revision 1.10  2002/08/31 17:38:28  nozomi
 * translate comments and modify coding style
 *
 * Revision 1.9  2002/06/28 12:54:40  ryo
 * fix misspelling
 *
 * Revision 1.8  2002/06/27 13:14:32  ryo
 * modify specification of getRegisterResult()
 *
 * Revision 1.7  2002/04/16 03:53:37  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.6  2002/03/29 08:03:04  okawa
 * linkFlag
 *
 * Revision 1.5  2002/03/28 06:28:35  okawa
 * add copyright
 *
 * Revision 1.4  2002/02/21 01:56:47  okawa
 * add register function
 *
 * Revision 1.3  2002/02/14 08:34:44  okawa
 * add parse function from InputSource
 *
 * Revision 1.2  2002/02/07 06:22:47  okawa
 * add another parse function
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 *
 */

package org.nomencurator.controller;

import org.apache.xerces.parsers.DOMParser;
import ui.DOMParserSaveEncoding;

import org.xml.sax.Parser;
import org.xml.sax.helpers.ParserFactory;
import org.xml.sax.InputSource;

import java.io.StringReader;

import java.util.Vector;

import org.nomencurator.NameUsage;
import org.nomencurator.NamedObject;
import org.nomencurator.Appearance;
import org.nomencurator.Annotation;
import org.nomencurator.Publication;
import org.nomencurator.Author;
import org.nomencurator.editor.NameUsageListRow;
import org.nomencurator.awt.model.NameUsageTreeNode;

import org.nomencurator.awt.tree.TextTreeModel;

public class ParseXml
{
    /** DOM parser */
    private DOMParser parser_;

    /**
     * flag for linked object operation.
     * If true, this creates linked object for 1 to 1 relationship.
     * If false, this does not create linked record but set as a
     * propary of NamedObject.
     * Default value is true.
     */
    private boolean linkFlag;

    public static final String DOCTYPE = "NOM-XML";

    /** Constractor */
    public ParseXml()
    {
	this(true);
    }

    /** Constractor */
    public ParseXml(boolean linkFlag)
    {
        parser_ = new DOMParserSaveEncoding();
        try {
            parser_.setFeature("http://apache.org/xml/features/dom/defer-node-expansion", false); // otherwise parser.getCurrentNode() == null
            parser_.setFeature("http://apache.org/xml/features/continue-after-fatal-error", true);
            parser_.setFeature("http://apache.org/xml/features/allow-java-encodings", true);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        this.linkFlag = linkFlag;
    }
    
    /**
     * get NameUsage information
     * @param     String url
     * @return    Vector<NameUsageListRow>
     */
    public Vector getNameUsageList(String url)
    {
	Vector v = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                v = parseNameUsageList(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	if(v == null)
	    v = new Vector();
	return v;
    }

    /**
     * get NameUsage information
     * @param     InputSource source
     * @return    Vector<NameUsageListRow>
     */
    public Vector getNameUsageList(InputSource source)
    {
	Vector v = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                v = parseNameUsageList(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	if(v == null)
	    v = new Vector();
	return v;
    }

    /**
     * @param   org.w3c.dom.Document parsed document
     * @return  Vector<NameUsageListRow>
     */
    private Vector parseNameUsageList(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getNameUsageList();
    }
    
    /**
     * get NameUsage Tree information
     * @param   String url
     * @return  Vector<NameUsageTreeNode>
     */
    public Vector getTreeModel(String url)
    {
	Vector v = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                v = parseTreeModel(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	if(v == null)
	    v = new Vector();
	return v;
    }

    /**
     * get NameUsage Tree information
     * @param   InputSource source
     * @return  Vector<NameUsageTreeNode>
     */
    public Vector getTreeModel(InputSource source)
    {
	Vector v = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                v = parseTreeModel(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	if(v == null)
	    v = new Vector();
	return v;
    }

    /**
     * @param   org.w3c.dom.Document parsed document
     * @return  Vector<NameUsageTreeNode>
     */
    private Vector parseTreeModel(org.w3c.dom.Document doc)
    {
        Vector v = null;
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        v = nomParser.getTreeModel();
        return v;
    }

    /**
     * get NameUsage
     * @param   String url
     * @return  NameUsage
     */
    public NameUsage getNameUsage(String url)
    {
	NameUsage nameUsage = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
   			String docType = doc.getDoctype().getName();

            if( docType.equals(DOCTYPE)) {
                nameUsage = parseNameUsage(doc);
            }
        } catch (Exception e) {

            e.printStackTrace();
        }
	return nameUsage;
    }
    
    /**
     * get NameUsage
     * @param   InputSource source
     * @return  NameUsage
     */
    public NameUsage getNameUsage(InputSource source)
    {
	NameUsage nameUsage = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                nameUsage = parseNameUsage(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return nameUsage;
    }
    
    /**
     * parse a NameUsage object
     * @param   org.w3c.dom.Document parsed document
     * @return  NameUsage
     */
    private NameUsage parseNameUsage(org.w3c.dom.Document doc)
    {
	
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getNameUsage();
    }
    
    /**
     * get NamedObject
     * @param   String url
     * @return  NamedObject (abstract object)
     */
    public NamedObject getNamedObject(String url)
    {
	
	NamedObject namedObject = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
            if (doc.getDoctype() == null)
            	return null;
   			String docType = doc.getDoctype().getName();

            if( docType.equals(DOCTYPE)) {
                namedObject = parseNamedObject(doc);
            }
        } catch (Exception e) {

            e.printStackTrace();
        }
	return namedObject;
    }

    /**
     * get NamedObject
     * @param   InputSource source
     * @return  NamedObject (abstract object)
     */
    public NamedObject getNamedObject(InputSource source)
    {
	NamedObject namedObject = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                namedObject = parseNamedObject(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return namedObject;
    }

    /**
     * parse a NamedObject
     * @param   org.w3c.dom.Document parsed document
     * @return  NamedObject
     */
    private NamedObject parseNamedObject(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getNamedObject();
    }
    
    /**
     * get NamedObject List
     * @param   String url
     * @return  Vector<NamedObject> (abstract object)
     */
    public Vector getNamedObjectList(String url)
    {
	Vector v = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                v = parseNamedObjectList(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return v;
    }
    
    /**
     * get NamedObject List
     * @param   InputSource source
     * @return  Vector<NamedObject> (abstract object)
     */
    public Vector getNamedObjectList(InputSource source)
    {
	Vector v = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                v = parseNamedObjectList(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return v;
    }
    
    /**
     * parse NamedObject List
     * @param   org.w3c.dom.Document parsedDocument
     * @return  Vector <NamedObject>
     */
    private Vector parseNamedObjectList(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getNamedObjectList();
    }
    
    /**
     * get Appearance
     * @param   String url
     * @return  Appearance
     */
    public Appearance getAppearance(String url)
    {
	Appearance appearance = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                appearance = parseAppearance(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return appearance;
    }
    
    /**
     * get Appearance
     * @param   InputSource source
     * @return  Appearance
     */
    public Appearance getAppearance(InputSource source)
    {
	Appearance appearance = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                appearance = parseAppearance(doc);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
	return appearance;
    }
    
    /**
     * parse an Appearance object
     * @param   org.w3c.dom.Document parsed document
     * @return  Appearance
     */
    private Appearance parseAppearance(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getAppearance();
    }
    
    /**
     * get Annotation
     * @param   String url
     * @return  Annotation
     */
    public Annotation getAnnotation(String url)
    {
	Annotation annotation = null;
	try {
	    parser_.parse(url);
	    org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                annotation = parseAnnotation(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return annotation;
    }

    /**
     * get Annotation
     * @param   InputSource source
     * @return  Annotation
     */
    public Annotation getAnnotation(InputSource source) {
	Annotation annotation = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                annotation = parseAnnotation(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return annotation;
    }
    
    /**
     * parse an Annotation object
     * @param   org.w3c.dom.Document parsed document
     * @return  Annotation
     */
    private Annotation parseAnnotation(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getAnnotation();
    }
    
    /**
     * get Publication
     * @param   String url
     * @return  Publication
     */
    public Publication getPublication(String url)
    {
	
	Publication publication = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                publication = parsePublication(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return publication;
    }
    
    /**
     * get Publication
     * @param   InputSource source
     * @return  Publication
     */
    public Publication getPublication(InputSource source)
    {
	Publication publication = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                publication = parsePublication(doc);
            }
        }
	catch (Exception e) {
            e.printStackTrace();
        }
	return publication;
    }
    
    /**
     * parse a Publication object
     * @param   org.w3c.dom.Document parsed document
     * @return  Publication
     */
    private Publication parsePublication(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getPublication();
    }
    
    /**
     * get Author
     * @param   String url
     * @return  Author
     */
    public Author getAuthor(String url)
    {
	Author author = null;
        try {
            parser_.parse(url);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                author = parseAuthor(doc);
            }
        }
	catch (Exception e) {
            e.printStackTrace();
        }
	return author;
    }

    /**
     * get Author
     * @param   InputSource source
     * @return  Author
     */
    public Author getAuthor(InputSource source)
    {
	Author author = null;
        try {
            parser_.parse(source);
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                author = parseAuthor(doc);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
	return author;
    }

    /**
     * parse an Author object
     * @param   org.w3c.dom.Document parsed document
     * @return  Author
     */
    private Author parseAuthor(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getAuthor();
    }

    /**
     * get Register result
     * @param result result XML code.
     * @return int
     */
    public int getRegisterResult(String result)
    {
	int ret = 0;
        try {
            parser_.parse(new InputSource(new StringReader(result)));
            org.w3c.dom.Document doc = parser_.getDocument();
	    String docType = doc.getDoctype().getName();
	    
            if( docType.equals(DOCTYPE)) {
                ret = parseRegisterResult(doc);
            }
        } catch (Exception e) {
	    
            e.printStackTrace();
        }
	return ret;
    }
    
    /**
     * parse an Register result
     * @param   org.w3c.dom.Document parsed document
     * @return  int result
     */
    private int parseRegisterResult(org.w3c.dom.Document doc)
    {
        org.w3c.dom.Node node = doc.getDocumentElement();
        // parse
        // ParseNomXml nomParser = new ParseNomXml(node);
        ParseNomXml nomParser = new ParseNomXml(node, linkFlag);
        return nomParser.getRegisterResult();
    }

    // FOR TEST
    /*
    public static void main(String args[]) {
        ParseXml parse = new ParseXml();
        // NameUsage nameUsage = parse.getNameUsage("file:///nameusge.xml");
        String buf = "<?xml version=\"1.0\"?><!DOCTYPE NOM-XML><response status=\"0\" text=\"OK\"><nameusage><oid>101</oid><rank>Kingdum</rank><name>Name101</name><authority>authority1</authority><recorder>recorder1</recorder><higher>NameUsage::higher1</higher><lower>NameUsage::lower1</lower><lower>NameUsage::lower2</lower><type>NameUsage::type1</type><typeof>NameUsage::typeof1</typeof></namerecord></response>";
        NameUsage nameUsage = parse.getNameUsage(new InputSource(new StringReader(buf)));
        System.out.println("rank:"+nameUsage.getRank());
        System.out.println("higher:"+nameUsage.getHigherTaxon().persistentID());
    }
    */
}
