/*
 * LinkTable.java: 
 *
 * Copyright (c) 2002 Hiroyuki Okawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: LinkTable.java,v 1.7 2002/06/07 14:07:51 ryo Exp $
 * $Log: LinkTable.java,v $
 * Revision 1.7  2002/06/07 14:07:51  ryo
 * add id and its accessor
 *
 * Revision 1.6  2002/04/17 13:54:08  ryo
 * modify specification
 *
 * Revision 1.5  2002/04/01 06:48:31  nozomi
 * use java.util.Vector
 *
 * Revision 1.4  2002/03/28 06:29:41  okawa
 * add copyright
 *
 * Revision 1.3  2002/03/11 11:00:00  okawa
 * add clear()
 *
 * Revision 1.2  2002/02/22 10:54:51  okawa
 * initial import into CVS
 */

package org.nomencurator.controller.linktable;

import java.util.Hashtable;
import java.util.Vector;

public class LinkTable {

	private Hashtable _fromHashList;
	private Hashtable _toHashList;

	/**
	 * constractor
	 */
	public LinkTable() {
		_fromHashList = new Hashtable();
		_toHashList = new Hashtable();
	}

	/**
	 * add LinkRecord to List
	 * @param LinkRecord
	 */
	public void addLinkRecord(LinkRecord rec) {
		Hashtable hash;
		hash = (Hashtable)_fromHashList.get(rec.fromOid);
		if (hash == null) hash = new Hashtable();
		hash.put(rec.toOid, rec);
		_fromHashList.put(rec.fromOid, hash);
		hash = (Hashtable)_toHashList.get(rec.toOid);
		if (hash == null) hash = new Hashtable();
		hash.put(rec.fromOid, rec);
		_toHashList.put(rec.toOid, hash);
	}

	/**
	 * get From Object
	 * @param String TO objectID
	 * @return String 
	 */
	public String getFrom(String toOid) {
		Vector v = getFromList(toOid);
		if (v != null) {
			return v.elementAt(0).toString();
		}
		return null;
	}

	/**
	 * get From Object List
	 * @param NameRecord TO object
	 * @return Vector<String> 
	 */
	public Vector getFromList(String toOid) {
		return getVectorFromHashList(_toHashList, toOid);
	}

	/**
	 * get From LinkRecords
	 * @param NameRecord TO object
	 * @return Vector<String> 
	 */
	public Vector getFromLinkRecords(String toOid) {
		return getLinkRecordsFromHashList(_toHashList, toOid);
	}

	/**
	 * get To Object
	 * @param String 'FROM' objectID
	 * @return String 
	 */
	public String getTo(String fromOid) {
		Vector v = getToList(fromOid);
		if (v != null) {
			return v.elementAt(0).toString();
		}
		return null;
	}

	/**
	 * get To Object List
	 * @param String 'FROM' objectID
	 * @return Vector<String> 
	 */
	public Vector getToList(String fromOid) {
		return getVectorFromHashList(_fromHashList, fromOid);
	}

	/**
	 * get To LinkRecord
	 * @param String 'FROM' objectID
	 * @return Vector<String> 
	 */
	public Vector getToLinkRecords(String fromOid) {
		return getLinkRecordsFromHashList(_fromHashList, fromOid);
	}

	/**
	 * Get Vector from Hashtable
	 * @param Hashtable 'FROM' or 'TO' Hashtable
	 * @param String 'FROM' or 'TO' objectID
	 * @return Vector<String>
	 */
	private Vector getVectorFromHashList(Hashtable hashList, String oid) {
		Hashtable hash = (Hashtable)hashList.get(oid);
		if (hash != null) {
			return new Vector(hash.keySet());
		}
		return null;
	}

	/**
	 * Get LinkRecords from Hashtable
	 * @param Hashtable 'FROM' or 'TO' Hashtable
	 * @param String 'FROM' or 'TO' objectID
	 * @return Vector<String>
	 */
	private Vector getLinkRecordsFromHashList(Hashtable hashList, String oid) {
		Hashtable hash = (Hashtable)hashList.get(oid);
		if (hash != null) {
			return new Vector(hash.values());
//			return new Vector(hash.entrySet());
		}
		return null;
	}

	/**
	 * clear link record
	 */
	public void clear() {
		_fromHashList.clear();
		_toHashList.clear();
	}

/*
	public static void main(String[] args) {
		// stress test.(begin)
		int number_of_testdata = 10000;
		LinkRecord[] lr = new LinkRecord[number_of_testdata];
		String[] s = new String[number_of_testdata];
		int i;
		for (i=0; i<s.length; i++) s[i] = new String("" + i);
		for (i=0; i<lr.length; i++) lr[i] = new LinkRecord(s[i], s[i]);
		LinkTable table = new LinkTable();
		long start_time = System.currentTimeMillis();
		for (i=0; i<lr.length; i++) {
			table.addLinkRecord(lr[i]);
		}
		long put_end_time = System.currentTimeMillis();
		for (i=0; i<s.length; i++) {
			table.getFromList(s[i]);
			table.getFrom(s[i]);
			table.getToList(s[i]);
			table.getTo(s[i]);
		}
		long get_end_time = System.currentTimeMillis();
		System.out.println("stress test result");
		System.out.println("put cost: " + (put_end_time - start_time) + "(ms)");
		System.out.println("get cost: " + (get_end_time - put_end_time) + "(ms)");
		// stress test.(end)

		// function test.(begin)
		LinkTable table = new LinkTable();
		table.addLinkRecord(new LinkRecord("1", "4"));
		table.addLinkRecord(new LinkRecord("1", "4"));	// overlap test.
		table.addLinkRecord(new LinkRecord("2", "4"));
		table.addLinkRecord(new LinkRecord("2", "1"));
		table.addLinkRecord(new LinkRecord("3", "1"));
		table.addLinkRecord(new LinkRecord("3", "1"));	// overlap test.
		table.addLinkRecord(new LinkRecord("3", "2"));
		table.addLinkRecord(new LinkRecord("4", "1"));
		table.addLinkRecord(new LinkRecord("4", "2"));
		for (int i=1; i<=5; i++) {
			String s = new Integer(i).toString();
			System.out.println("(List) " + i + " from " + table.getFromList(s));
			System.out.println("(List) " + i + " to   " + table.getToList(s));
			System.out.println("(Oid)  " + i + " from " + table.getFrom(s));
			System.out.println("(Oid)  " + i + " to   " + table.getTo(s));
		}
		// function test.(end)

	}
*/
}
