/*
 * AbstractPublicationEditPanel.java:  an abstract class to be used
 * as superclass of PublicationEditor Panel
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 *
 * $Id: AbstractPublicationEditPanel.java,v 1.31 2002/09/17 08:46:39 nozomi Exp $
 * $Log: AbstractPublicationEditPanel.java,v $
 * Revision 1.31  2002/09/17 08:46:39  nozomi
 * re-organise initialisation
 *
 * Revision 1.30  2002/09/07 17:15:12  nozomi
 * fix bug in unsaved object putting methods
 *
 * Revision 1.29  2002/09/07 03:09:27  nozomi
 * use modified flag
 *
 * Revision 1.28  2002/09/03 05:06:42  nozomi
 * remove unnecessary error msg. generating line
 *
 * Revision 1.27  2002/09/03 02:51:47  nozomi
 * use CardLayout to switch editors
 *
 * Revision 1.26  2002/09/02 16:29:07  nozomi
 * note TextArea is also focuse controlled
 *
 * Revision 1.25  2002/08/20 13:17:22  t.okada
 * add notes column
 *
 * Revision 1.24  2002/08/20 09:07:20  nozomi
 * fix new allocation of unused Components
 *
 * Revision 1.23  2002/08/13 02:44:53  t.okada
 * Size adjustment of TextField
 *
 * Revision 1.22  2002/08/05 03:09:38  nozomi
 * change appearance list arrangement
 *
 * Revision 1.21  2002/06/23 13:38:42  nozomi
 * aware of FocusController
 *
 * Revision 1.20  2002/05/22 23:42:39  nozomi
 * use NamedObjectListModel#getTitle to create appearance list
 *
 * Revision 1.19  2002/05/21 08:07:56  nozomi
 * enable dummy model addition when appearance entries in the TableList exceeds number of presented rows
 *
 * Revision 1.18  2002/05/21 04:42:35  nozomi
 * add getAuthoList
 *
 * Revision 1.17  2002/05/21 03:30:19  nozomi
 * authorList returned to List
 *
 * Revision 1.16  2002/05/16 05:03:26  ryo
 * modify createAuthorComponents()
 *
 * Revision 1.15  2002/05/14 10:13:35  ryo
 * change type of authorList into TableList
 *
 * Revision 1.14  2002/04/28 08:23:20  nozomi
 * fix volume and nubmer text model bug
 *
 * Revision 1.13  2002/04/17 21:00:16  nozomi
 * fix scondary author bug
 *
 * Revision 1.12  2002/04/08 02:00:04  nozomi
 * refuse to setModel other than an instance of PublicationEditModel
 *
 * Revision 1.11  2002/03/20 13:57:19  okawa
 * no message
 *
 * Revision 1.10  2002/03/03 23:48:53  nozomi
 * setModel() synchronized
 *
 * Revision 1.9  2002/02/24 18:13:36  nozomi
 * re-implementation of setModel() to be used switching between models
 *
 * Revision 1.8  2002/02/19 20:53:03  nozomi
 * default createContainerField() is provided
 *
 * Revision 1.7  2002/02/19 19:27:39  nozomi
 * add BookSectionEditPanel
 *
 * Revision 1.6  2002/02/18 23:09:31  nozomi
 * commented out code was removed
 *
 * Revision 1.5  2002/02/05 18:05:52  nozomi
 * Add summary's visibility accessor/modifier
 *
 * Revision 1.4  2002/02/05 06:48:44  nozomi
 * Use java.util.Vector
 *
 * Revision 1.3  2002/02/04 04:51:05  nozomi
 * Add layout methods
 *
 * Revision 1.2  2002/02/04 01:28:33  nozomi
 * Implement components creation
 *
 * Revision 1.1  2002/02/02 19:36:14  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import java.awt.Component;
import java.awt.GridBagConstraints;

import java.util.Vector;

import jp.kyasu.awt.Label;
import jp.kyasu.awt.TextArea;
import jp.kyasu.awt.TextField;

import jp.kyasu.editor.DocumentEditor;
import jp.kyasu.editor.Editor;

import jp.kyasu.graphics.TextStyle;
import jp.kyasu.graphics.VTitledPaneBorder;

import org.nomencurator.Publication;

import org.nomencurator.editor.NamedObjectEditPanel;

import org.nomencurator.editor.model.ObjectEditModel;
import org.nomencurator.editor.model.PublicationEditModel;
import org.nomencurator.editor.model.AppearanceListModel;

/**
 * <code>Panel</code> to represent a <code>PublicationEditModel</code>
 * via an <code>AbstractPublicationEditPanel</code>.
 *
 * @see org.nomencurator.editor.model.PublicationEditModel
 * @see org.nomencurator.Publication
 *
 * @version 	17 Sep 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class AbstractPublicationEditPanel
    extends NamedObjectEditPanel
    implements PublicationEditor
{
    /** Default width of <code>TextField</code>s representing publication's title */
    protected static int defaultTitleTextFieldSize = 16;

    /** <code>PublicationEditModel</code> represented by this editor */
    protected PublicationEditModel        model;

    /** <code>TextField</code> represents authors' names */
    protected TextField       authors;

    /** <code>TableList</code> controlling switch between <code>AuthorEditor</code>s */
    protected List       authorList;

    /** Title of author components */
    protected static String authorTitle = "Author";

    /**
     * <code>TextField</code> represents sencondary authors' names
     * such as name of editor
     */
    protected TextField       secondaryAuthors;

    /** <code>TableList</code> controlling switch between secondary <code>AuthorEditor</code>s */
    protected List       secondaryAuthorList;

    /**
     * <code>Panel</code> holdes <code>Component</code>s representing a
     * <code>Publication</code>
     */
    protected Panel citationPanel;

    /** <code>TextField</code> represents year of the publication */
    protected TextField          year;
    
    /** Title of year component */
    protected static String yearTitle = "Year";

     /**
     * Title of <code>TextArea</code> representing note
     */
    protected static String notesTitle = "Note";

   /**
     * <code>TextField</code> represents title of contents,
     * e.g. title of an article
     */
    protected TextField      contentsField;

    /**
     * <code>TextField</code> represents title of container,
     * e.g. journal name
     */
    protected TextField      containerField;

    /**
     * <code>TableList</code> represents title of contents,
     * e.g. each chapter of a book
     */
    protected TableList      tableOfContents;
    
    /** <code>TextField</code> represents volume of the publication */
    protected TextField      volume;

    /**
     * <code>TextField</code> represents number of the publication
     */
    protected TextField      number;

    /** <code>TextField</code> represents first page of the publication */
    protected TextField      firstPage;

    /** <code>TextField</code> represents last page of the publication */
    protected TextField      lastPage;

    /** <code>TextField</code> representing ISBN of the book */
    protected TextField          isxn;

    /** <code>TextField</code> representing publisher of the book */
    protected TextField          publisher;

    /** <code>TextField</code> representing published place of the book */
    protected TextField          place;

    /** <code>TextArea</code> representing publication */
    protected TextArea      notes;

    /** <code>DocumentEditor</code> represents summmary of the publication */
    //    protected DocumentEditor      summary;
    protected TextArea      summary;

    /** Specifies whether to show summary */
    protected boolean            showSummary; 

    /** <code>BorderedPanel</code> holds <code>summary</code> */
    protected BorderedPanel summaryPanel;

    /** Title of <code>summaryPanel</code> */
    protected static String summaryPanelTitle = "Abstruct";


    /** <code>TableList</code> represents appearances in this publication */
    protected TableList        appearanceList;

    /** <code>BorderedPanel</code> holds <code>appearanceList</code> */
    protected BorderedPanel appearanceListPanel;

    /** Title of <code>appearanceListPanel</code> */
    protected static String appearanceListPanelTitle = "Appearances";

    protected TextStyle          defaultTextStyle;
    protected TextStyle          boldStyle;
    protected TextStyle          italicStyle;

    /**
     * Creates <code>PublicationEditPanel</code> with specified
     * <code>model</code>, <code>editable</code>, <code>authorEditors</code>
     * <code>appearanceEditors</code>
     */
    protected AbstractPublicationEditPanel(PublicationEditModel model,
					   boolean editable,
					   boolean showSummary,
					   Vector authorEditors,
					   Vector appearanceEditors)
    {
	super(model, editable);
	this.showSummary = showSummary;
	addModelComponentsLater();
    }

    /**
     * Creates components representing the model
     */
    protected void createModelComponents()
    {
	model = (PublicationEditModel)getModel();
	createCitationPanel();
	createSummaryPanel();
	createAppearancePanel();
    }

    /**
     * Creates citation container <code>Panel</code> and layout 
     * <code>Component</code>s representing citation information 
     */
    protected void createCitationPanel()
    {
	citationPanel = new Panel();
	createCitationComponents();
	layoutCitationComponents();
    }

    /**
     * Creates <code>Component</code>s representing citation information 
     */
    protected void createCitationComponents()
    {
	createAuthorComponents();
	createYearComponent();
	createContentsField();
	createVolume();
	createNumber();
	createPages();
	createNotes();
	createContainerComponents();
	createSecondaryAuthorComponents();
    }

    /**
     * Creates <code>Component</code>s representing container information 
     */
    protected abstract void createContainerComponents();


    /**
     * Creates <code>Component</code>s representing <code>container</code> information 
     */
    protected void createContainerComponents(PublicationEditModel container)
    {
	createContainerField(container);
	createISXN(container);
	createPublisher(container);
	createPlace(container);
    }


    /**
     * Layouts <code>Component</code>s representing citation information 
     */
    protected void layoutCitationComponents()
    {
	layoutAuthorComponents();
	layoutYearComponent();
	layoutContentsField();
	layoutContainerField();
	layoutVolume();
	layoutNumber();
	layoutPages();
	layoutNotesComponent();
    }

    /**
     * Creates <code>Component</code>s representing author of the publication
     */
    protected void createAuthorComponents()
    {
	authors = new TextField(model.getAuthorTextEditModel(), defaultTitleTextFieldSize);
	authors.addTextListener(this);
	authorList = new List(model.getAuthorList(), 3);
    }

    /**
     * Returns <code>List</code> representing authors
     *
     * @return List representing list of <code>AuthorEditModel</code>s
     */
    public List getAuthorList()
    {
	return authorList;
    }

    /**
     * Creates <code>Component</code>s representing secondary author 
     * (e.g. editor of a book) of the publication
     */
    protected void createSecondaryAuthorComponents()
    {
	secondaryAuthors = new TextField(model.getSecondaryAuthorTextEditModel(), defaultTitleTextFieldSize);
	secondaryAuthors.addTextListener(this);
	secondaryAuthorList = new List(model.getSecondaryAuthorList(), 3);
    }

    /**
     * Layouts <code>Component</code>s representing author(s) of the publication
     */
    protected void layoutAuthorComponents()
    {
	if(authors != null)
	    citationPanel.add(authorTitle, authors, GridBagConstraints.HORIZONTAL);

	if(authorList != null)
	    citationPanel.add("", authorList, GridBagConstraints.HORIZONTAL);
    }

    /**
     * Creates <code>Component</code> representing year of the publication
     */
    protected void createYearComponent()
    {
	year = new TextField(model.getYearTextModel(), 5);
	year.addTextListener(this);
    }

    /**
     * Layouts <code>Component</code> representing year of the publication
     */
    protected void layoutYearComponent()
    {
	if(year != null)
	    citationPanel.add(new Label(yearTitle), year);
    }


    /**
     * Creates <code>Component</code>s representing contents title of the publication
     */
    protected void createContentsField()
    {
	contentsField = new TextField(model.getContentsTitleTextModel(), defaultTitleTextFieldSize);
	contentsField.addTextListener(this);
    }

    /**
     * Layouts <code>Component</code>s representing contents title of the publication
     */
    protected abstract void layoutContentsField();

    /**
     * Creates <code>Component</code>s representing container title of the publication
     */
    protected void createContainerField()
    {
	createContainerField(model);
    }

    /**
     * Creates <code>Component</code>s representing container title of the publication
     * specified by <code>model</code>
     *
     * @param model <code>PublicationEditModel</code> representing the publication
     */
    protected void createContainerField(PublicationEditModel model)
    {
	containerField = new TextField(model.getContainerTitleTextModel(), defaultTitleTextFieldSize);
	containerField.addTextListener(this);
    }

    /**
     * Layouts <code>Component</code>s representing container title of the publication
     */
    protected abstract void layoutContainerField();

    /**
     * Creates <code>Component</code>s representing table of contents
     */
    protected void createTableOfContents()
    {
    }

    /**
     * Layouts <code>Component</code>s representing table of contents
     */
    protected abstract void layoutTableOfContents();

    /**
     * Creates <code>Component</code> representing volume of container
     */
    protected void createVolume()
    {
	volume = new TextField(model.getVolumeTextModel(), 5);
	volume.addTextListener(this);
    }

    /**
     * Layouts <code>Component</code> representing volume of container
     */
    protected abstract void layoutVolume();

    /**
     * Creates <code>Component</code> representing number of container
     */
    protected void createNumber()
    {
	number = new TextField(model.getNumberTextModel(), 3);
	number.addTextListener(this);
    }

    /**
     * Layouts <code>Component</code> representing number of container
     */
    protected abstract void layoutNumber();

    /**
     * Creates <code>Component</code>s pages
     */
    protected void createPages()
    {
	createFirstPage();
	createLastPage();
    }

    /**
     * Layouts <code>Component</code>s pages
     */
    protected abstract void layoutPages();

    /**
     * Layouts <code>Component</code>s pages
     */
    protected void layoutPages(String title)
    {
	citationPanel.add(new Label(title), 
			  new Component[]{firstPage, new Label("-"), lastPage});
    }

    /**
     * Layouts <code>Component</code>s pages
     */
    protected void layoutPage(String title)
    {
	citationPanel.add(new Label(title), firstPage);
    }

    /**
     * Creates <code>Component</code> representing first page
     */
    protected void createFirstPage()
    {
	firstPage = new TextField(model.getFirstPageTextModel(), 7);
	firstPage.addTextListener(this);
    }

    /**
     * Creates <code>Component</code> representing last page
     */
    protected void createLastPage()
    {
	lastPage = new TextField(model.getLastPageTextModel(), 7);
	lastPage.addTextListener(this);
    }

    /**
     * Creates <code>Component</code> representing notes of the publication
     */
    protected void createNotes()
    {
    	notes = new TextArea(model.getNotesTextModel(), 3, 25);
	notes.addTextListener(this);
    }

    /**
     * Layouts <code>Component</code> representing notes of the publication
     */
    protected void layoutNotesComponent()
    {
	if(notes != null)
	    citationPanel.add(new Label(notesTitle), notes);
    }

    /**
     * Creates <code>TextField</code> representing publisher
     */
    protected void createISXN(PublicationEditModel model)
    {
	isxn = new TextField(model.getISXNTextModel(), defaultTitleTextFieldSize);
	isxn.addTextListener(this);
    }

    /**
     * Creates <code>TextField</code> representing publisher
     */
    protected void createPublisher(PublicationEditModel model)
    {
	publisher = new TextField(model.getPublisherTextModel(), defaultTitleTextFieldSize);
	publisher.addTextListener(this);
    }

    /**
     * Creates <code>TextField</code> representing published place
     */
    protected void createPlace(PublicationEditModel model)
    {
	place = new TextField(model.getPlaceTextModel(), defaultTitleTextFieldSize);
	place.addTextListener(this);
    }

    /**
     * Creates <code>Component</code>s representing summary of the publication
     * and its container <code>Panel</code>
     */
    protected void createSummaryPanel()
    {
	summaryPanel = new BorderedPanel(new VTitledPaneBorder(summaryPanelTitle));
	//	summary = new AcronymTextEditor(5, defaultTitleTextFieldSize, model.getContentsSummaryTextModel());
	summary = new TextArea(model.getContentsSummaryTextModel(), 5, defaultTitleTextFieldSize);
	summary.addTextListener(this);
	summaryPanel.add(summary);
    }

    /**
     * Creates <code>Component</code>s representing citation information 
     * and its container <code>Panel</code>
     */
    protected void createAppearancePanel()
    {
	appearanceListPanel = new BorderedPanel(new VTitledPaneBorder(appearanceListPanelTitle));
	appearanceList = new TableList(model.getAppearanceTextListModel(), 3,
				       model.getAppearanceTextListModel().getTitle());
	//	appearanceListPanel.add(appearanceList);
    }

    /**
     * Sets <CODE>ed</CODE> as <CODE>Editor</CODE> of <CODE>TextComponent</CODE>s
     * to provide <CODE>Text</CODE> handling using GUI.
     *
     * @param ed <CODE>Editor</CODE> to be linked to <CODE>TextComponent</CODE>s
     * of this
     */
    public void setEditor(Editor ed)
    {
	if(focusedEditor == ed)
	    return;

	if(focusedEditor != null) {
	    focusedEditor.unlisten(authors);
	    focusedEditor.unlisten(secondaryAuthors);
	    focusedEditor.unlisten(year);
	    focusedEditor.unlisten(contentsField);
	    focusedEditor.unlisten(containerField);
	    focusedEditor.unlisten(volume);
	    focusedEditor.unlisten(number);
	    focusedEditor.unlisten(firstPage);
	    focusedEditor.unlisten(lastPage);
	    if(isxn != null) {
		focusedEditor.unlisten(isxn);
		focusedEditor.unlisten(publisher);
		focusedEditor.unlisten(place);
	    }
	    focusedEditor.unlisten(summary);
	    focusedEditor.unlisten(notes);
	}

	focusedEditor = ed;

	if(focusedEditor != null) {
	    focusedEditor.listen(authors);
	    focusedEditor.listen(secondaryAuthors);
	    focusedEditor.listen(year);
	    focusedEditor.listen(contentsField);
	    focusedEditor.listen(containerField);
	    focusedEditor.listen(volume);
	    focusedEditor.listen(number);
	    focusedEditor.listen(firstPage);
	    focusedEditor.listen(lastPage);
	    if(isxn != null) {
		focusedEditor.listen(isxn);
		focusedEditor.listen(publisher);
		focusedEditor.listen(place);
	    }
	    focusedEditor.listen(summary);
	    focusedEditor.listen(notes);
	}

    }


    /**
     * Returns <code>TableList</code> representing appearances
     *
     * @return TableList representing list of <code>AppearanceEditModel</code>s
     */
    public TableList getAppearanceList()
    {
	return 	appearanceList;
    }

    /**
     * Adds components to this <code>Panel</code>
     *
     */
    protected void addModelComponents()
    {
	add(citationPanel);
    }

    /**
     * Removes components from this <code>Panel</code>
     *
     */
    protected void removeModelComponents()
    {
	remove(citationPanel);
	
    }

    /**
     * Adds components to this <code>Panel</code>
     *
     */
    protected void addModelComponentsLater()
    {
	if(showSummary)
	    add(summaryPanel);
	/*
	add(appearanceListPanel);
	*/
	citationPanel.add(appearanceListPanelTitle, appearanceList, GridBagConstraints.HORIZONTAL);
    }

    /**
     * Gets <code>PublicationEditModel</code> accompanying with this editor
     *
     * @return PublicationEditModel accompanying with this editor
     */
    public PublicationEditModel getPublicationEditModel()
    {
	return (PublicationEditModel)getModel();
    }

    /**
     * Setups components to use given <CODE>model</CODE>.
     * This method is called from setModel(ObjectEditModel)
     * internally.
     */
    protected void setComponents(ObjectEditModel objectEditModel)
    {
	if(model == objectEditModel)
	    return;

	if(!(objectEditModel instanceof PublicationEditModel))
	    return;

	model = (PublicationEditModel)objectEditModel;

	/*
	boolean visibility = isVisible();
	setVisible(false);
	*/
	//	invalidate();
	authors.setModel(model.getAuthorTextEditModel());

	authorList.setModel(model.getAuthorList());
	secondaryAuthors.setModel(model.getSecondaryAuthorTextEditModel());
	secondaryAuthorList.setModel(model.getSecondaryAuthorList());
	year.setModel(model.getYearTextModel());
	contentsField.setModel(model.getContentsTitleTextModel());
	containerField .setModel(model.getContainerTitleTextModel());
	if(volume != null) {
	    volume.setModel(model.getVolumeTextModel());
	}
	number.setModel(model.getNumberTextModel());
	firstPage.setModel(model.getFirstPageTextModel());
	lastPage.setModel(model.getLastPageTextModel());
	notes.setModel(model.getNotesTextModel());
	if(isxn != null) {
	    isxn.setModel(model.getISXNTextModel());
	    publisher.setModel(model.getPublisherTextModel());
	    place.setModel(model.getPlaceTextModel());
	}
	summary.setModel(model.getContentsSummaryTextModel());
	appearanceList.setModel(model.getAppearanceTextListModel());

	//validate();
	//setVisible(true);
	//	setVisible(visibility);
    }

    /**
     * A utility method to get <code>Publication</code> under edition
     *
     * @return Publication under edition
     */
    public Publication getPublication()
    {
	return model.getPublication();
    }

    /**
     * A utility method to set <code>publication</code> as
     * target of edition by accompanying <code>PublicationEditModel</code>
     *
     * @param publication to be set as target <code>Publication</code>
     */
    public void setPublication(Publication publication)
    {
	//model.setPublication(publication);
	model.setObject(publication);
    }
    
    /**
     * Creates an <code>ObjectEditModel</code> appropriate to 
     * the subclass of <code>ObjectEditPanel</code> which is used
     * in <code>Button</code> pressed event handling.
     *
     */
    protected ObjectEditModel createObjectEditModel()
    {
	return new PublicationEditModel();
    }

    /**
     * Creates a <code>Panel</code> with its containing <code>Button</code>s 
     * to control the editor.
     *
     */
    protected void createButtonPanel()
    {
	//do nothing; its job of wrapping panel...
    }

    /**
     * Adds a <code>Panel</code> containing <code>Button</code>s 
     * to control the editor.
     *
     */
    protected void addButtonPanel()
    {
	//do nothing; its job of wrapping panel...
    }

    /**
     * Returns boolean determining summary's visibility
     *
     * @return true if summary of the <code>Publication</code> is visible
     */
    public boolean isSummaryVisible()
    {
	return showSummary;
    }

    /**
     * Sets <code>visibility</code> of summary
     *
     * @param visibility true to make summary of the <code>Publication</code> visible
     */
    public void setSummaryVisible(boolean visibility)
    {
	showSummary = visibility;
    }
}
