/*
 *
 * AcronymTextEditor.java:  an editor for text with acronym attribute.
 *
 * Copyright (c) 2000 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: AcronymTextEditor.java,v 1.2 2002/02/19 06:21:30 nozomi Exp $
 *	$Log: AcronymTextEditor.java,v $
 *	Revision 1.2  2002/02/19 06:21:30  nozomi
 *	AcronymTextEditor accepts TextEditModel as contructor's paramter
 *	
 *	Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 *	initial import into CVS
 *	
 *	
 *
 */

package org.nomencurator.editor;

import java.awt.Color;
import java.awt.Component;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;

import java.util.Hashtable;

import jp.kyasu.awt.Button;
import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.ToggleButton;
import jp.kyasu.awt.ToolBar;

import jp.kyasu.awt.event.TextPositionEvent;
import jp.kyasu.awt.event.TextPositionListener;

import jp.kyasu.editor.ColorButton;
import jp.kyasu.editor.TextEditor;
import jp.kyasu.editor.RichTextEditor;
import jp.kyasu.editor.DocumentEditor;

import jp.kyasu.graphics.BasicTSModifier;
import jp.kyasu.graphics.ExtendedFont;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.VImage;

import org.nomencurator.editor.GreekButton;
import org.nomencurator.editor.Latin1Button;
import org.nomencurator.editor.TextConverter;


public class AcronymTextEditor extends DocumentEditor
    implements ActionListener, ItemListener, TextListener, TextPositionListener {

    protected CharacterSetButton greek;
    protected CharacterSetButton latin1;
    protected ToggleButton  acronymButton;
    protected ToolBar       attributeBar;
    protected Hashtable     acronyms;
    protected Color         acronymColor;
    protected Color         plainColor;
    protected int           selectionStart;
    protected int           selectionEnd;
    protected String        selectedString;

    static public final String I_ACRONYM      = "Acronym";
    static public final String A_ACRONYM      = I_ACRONYM;

    public AcronymTextEditor ()
    {
	this(null);
    }

    public AcronymTextEditor (TextEditModel model)
    {
	this(true, model);
    }

    public AcronymTextEditor (boolean showToolBar)
    {
	this(20, 65, showToolBar);
    }

    public AcronymTextEditor (boolean showToolBar, TextEditModel model)
    {
	this(20, 65, showToolBar, model);
    }

    public AcronymTextEditor (int row, int columns)
    {
	this(row, columns, true);
    }

    public AcronymTextEditor (int row, int columns, TextEditModel model)
    {
	this(row, columns, true, model);
    }

    public AcronymTextEditor (int row, int columns, boolean showToolBar) 
    {
	this(row, columns, showToolBar, null);
    }

    public AcronymTextEditor (int row, int columns, boolean showToolBar, TextEditModel model) 
    {
	this(row, columns, showToolBar, model, Color.green.darker());
    }

    public AcronymTextEditor (int row, int columns, boolean showToolBar, TextEditModel model, Color acronymColor)
    {
	super(row, columns, showToolBar, model);
	plainColor = editModel.getRichText().getRichTextStyle().getTextStyle().getExtendedFont().getColor();
	this.acronymColor = acronymColor;
	acronyms = new Hashtable();
    }

    protected ToolBar createToolBar(boolean showToolBar,
				    ActionListener openActionListener,
				    ActionListener saveActionListener,
				    ActionListener printActionListener)
    {
	Component[] edit   = createEditComponents();
	Component[] find   = createFindComponents(false);
	Component[] charTable = createCharTableComponents();
	Component[] slant  = createFontComponents();
	Component[] script = createScriptComponents();
	Component[] attribute = createAttributeComponents();
	return new ToolBar(new Component[][][]{{edit, find, charTable, slant, script, attribute}}, showToolBar);
    }

    protected Component[] createCharTableComponents()
    {
	//	greek = new GreekButton();
	greek = (CharacterSetButton)setupButton(new GreekButton(), "greek", "greek");
	greek.addItemListener(this);
	//	addCaretDisableComp(greek);

	//latin1 = new Latin1Button();
	latin1 = (CharacterSetButton)setupButton(new Latin1Button(), "latin1", "diacritics");
	latin1.addItemListener(this);
	//	addCaretDisableComp(latin1);

	return new Component[]{ greek, latin1 };
    }

    protected Component[] createFontComponents()
    {
	italicButton = createIconToggleButton(I_ITALIC);
	italicButton.addItemListener(this);

	italicButton.setEnabled(false);

	addCaretDisableComp(italicButton);

	return new Component[]{ italicButton };
    }

    protected Component[] createAttributeComponents()
    {
	acronymButton = createToggleButton(A_ACRONYM);
	acronymButton.addItemListener(this);

	ColorButton color = new ColorButton();
	color.addItemListener(this);
	color.setToolTipText(getToolTip("select a color for acronymized word"));

	acronymButton.setEnabled(false);
	addCaretDisableComp(acronymButton);

	//	color.setEnabled(true);

	return new Component[]{ acronymButton /*, color*/ };
    }

    /**
     * Invoked when an action occurs.
     * @see java.awt.event.ActionListener
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	if (command.equals(A_ACRONYM)) {
	    make_selection_acronym();
	}
	else {
	    super.actionPerformed(e);
	}
    }

    boolean acronymMade = false;

    public void make_selection_acronym()
    {
	/*
	 * Put selected String to acronym dictionary
	 * if it has not yet been put.
	 */

	String candidate = getSelectedText();
	if(candidate == null || candidate.length() == 0) {
	    candidate = selectedString;
	}
	selectedString = null;
	
	Integer count = (Integer)acronyms.get(candidate);
	if(count == null) {
	    acronyms.put(candidate, new Integer(1));
	}
	else {
	    acronyms.put(candidate, new Integer(count.intValue() + 1));
	}

	/*
	 * Save current color and selected region indexes
	 */ 
	selectionStart = getSelectionStart();
	selectionEnd   = getSelectionEnd();

	/*
	 * Set color of the selected text, flush selection,
	 * restore color of text and then re-select the text.
	 */
	//	set_font_color(acronymColor);
    }

    public void unmake_selection_acronym()
    {
	/*
	 * Put selected String to acronym dictionary
	 * if it has not yet been put.
	 */

	String candidate = getSelectedText();
	if(candidate == null || candidate.length() == 0)
	    return;
	
	Integer count = (Integer)acronyms.get(candidate);
	if(count != null) {
	    if(count.intValue() > 1) {
		acronyms.put(candidate, new Integer(count.intValue() - 1));
	    }
	    else{
		acronyms.remove(candidate);
	    }
	}

	/*
	 * Set color of the selected text, flush selection,
	 * restore color of text and then re-select the text.
	 */
	set_font_color(plainColor);

	acronymMade = false;
	//	reset_font_color(acronymColor, selectionStart, selectionEnd);
    }

    /**
     * Invoked when the value of the text has changed.
     * @see java.awt.event.TextListener
     */
    public void textValueChanged(TextEvent e) {
	if(acronymMade){
	    
	}
	super.textValueChanged(e);
    }

    /**
     * Invoked when an item's state has been changed.
     * @see java.awt.event.ItemListener
     */
    public void itemStateChanged(ItemEvent e) {
	Object obj = e.getItem();
	
	if (obj == null || (obj instanceof Color)) {
	    acronymColor = (Color)obj;
	    return;
	}
	else if (obj instanceof Character) {
	    insert(new String(new char[] {((Character)obj).charValue()}),
		   getCaretPosition());
	    return;
	}
	else if (!(obj instanceof String)) {
	    super.itemStateChanged(e);
	    return;
	}
	String command = (String)obj;
	boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
	if (command.equals(A_ACRONYM)) {
	    if(selected)
		make_selection_acronym();
	    else{
		unmake_selection_acronym();
	    }
	}
	else {
	    super.itemStateChanged(e);
	}
    }

    public VImage getIcon(String name) {
	String file = "icons/" + name + ".gif";
	return jp.kyasu.awt.AWTResources.getIcon(getClass(), file);
    }


    /**
     * Sets the font color of the selected text to be the specified color.
     */
    public void reset_font_color(Color color, int start, int end) {
	BasicTSModifier modifier = new BasicTSModifier();
	if (color != null)
	    modifier.put(BasicTSModifier.COLOR, color);
	else
	    modifier.put(BasicTSModifier.COLOR, BasicTSModifier.NULL);
	getModel().modifyTextStyle(start, end, modifier);
    }

    public void textPositionChanged(TextPositionEvent event) {
	if(selectionIsCaret())
		acronymButton.setState(false);
	else{
	    String selected = getSelectedText();
	    selectedString = selected;
	    if(selected != null && selected.length() != 0){
		selectedString = selected;
		Integer count = (Integer)acronyms.get(selected);
		if(count == null || count.intValue() == 0){
		    acronymButton.setState(false);
		}
		else{
		    acronymButton.setState(true);
		}
	    }
	}
	super.textPositionChanged(event);
    }

    public String getXML()
    {
	return TextConverter.TextToXML(getTEXT());
    }

    public String getHTML()
    {
	return TextConverter.TextToHTML(getTEXT());
    }

    public String getTeX()
    {
	return TextConverter.TextToTeX(getTEXT());
    }

    public String getBibTeX()
    {
	return TextConverter.TextToBibTeX(getTEXT(), acronyms);
    }

}

