/*
 * AuthorEditPanel.java:  a Panel implementation of AuthorEditor
 * for TaxoNote, based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: AuthorEditPanel.java,v 1.26 2002/10/18 04:56:39 nozomi Exp $
 * $Log: AuthorEditPanel.java,v $
 * Revision 1.26  2002/10/18 04:56:39  nozomi
 * use getDummyModel()
 *
 * Revision 1.25  2002/09/17 08:46:39  nozomi
 * re-organise initialisation
 *
 * Revision 1.24  2002/09/08 14:16:29  nozomi
 * support Author search
 *
 * Revision 1.23  2002/09/07 17:15:12  nozomi
 * fix bug in unsaved object putting methods
 *
 * Revision 1.22  2002/09/07 03:09:27  nozomi
 * use modified flag
 *
 * Revision 1.21  2002/09/05 11:16:03  t.okada
 * publicationList is set as Observer of surname.
 *
 * Revision 1.20  2002/08/15 12:57:35  ryo
 * add the processing for extending Choice
 *
 * Revision 1.19  2002/08/05 08:01:05  t.okada
 * DummyModel addition processing is corrected.
 *
 * Revision 1.18  2002/06/23 13:38:42  nozomi
 * aware of FocusController
 *
 * Revision 1.17  2002/05/31 21:42:11  nozomi
 * fix feudality spelling
 *
 * Revision 1.16  2002/05/26 21:03:46  nozomi
 * modify double click handling
 *
 * Revision 1.15  2002/05/23 06:05:00  nozomi
 * re-fix list name bug in mouseDoubleClicked
 *
 * Revision 1.14  2002/05/23 06:02:50  nozomi
 * fix list name bug in mouseDoubleClicked
 *
 * Revision 1.13  2002/05/23 04:52:22  nozomi
 * simplify double click handling
 *
 * Revision 1.12  2002/05/22 23:47:32  nozomi
 * change double click handling
 *
 * Revision 1.11  2002/05/17 10:53:25  ryo
 * modify *ButtonClicked()
 *
 * Revision 1.10  2002/05/14 10:02:43  ryo
 * add updateView()
 *
 * Revision 1.9  2002/05/10 13:42:37  ryo
 * add the processing which should be performed when double clicking a list
 *
 * Revision 1.8  2002/05/08 20:48:36  nozomi
 * add mouseDoiubleClicked to handle douible click
 *
 * Revision 1.7  2002/05/08 10:56:53  ryo
 * disable OK, Clear and Cancel button in *EditPanel
 *
 * Revision 1.6  2002/04/20 23:29:27  nozomi
 * improve mutual linkage between EditPanels
 *
 * Revision 1.5  2002/04/16 23:43:51  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.4  2002/03/10 08:48:44  okawa
 * modify 1 line in setModel()
 *
 * Revision 1.3  2002/02/24 23:48:36  nozomi
 * implements ItemSelectable
 *
 * Revision 1.2  2002/02/24 20:01:23  nozomi
 * modified to use setModel()
 *
 * Revision 1.1  2002/01/28 05:57:31  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import java.awt.AWTEventMulticaster;
import java.awt.GridBagConstraints;
import java.awt.ItemSelectable;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseEvent;

import java.util.Date;
import java.util.Vector;

import jp.kyasu.awt.BorderedPanel;
import jp.kyasu.awt.Choice;
import jp.kyasu.awt.Frame;
import jp.kyasu.awt.Label;
import jp.kyasu.awt.TextField;

import jp.kyasu.editor.Editor;

import jp.kyasu.graphics.VTitledPaneBorder;

import org.nomencurator.Author;

import org.nomencurator.broker.NamedObjectBroker;

import org.nomencurator.controller.PreferenceManager;

import org.nomencurator.editor.TableList;

import org.nomencurator.editor.model.AuthorEditModel;
import org.nomencurator.editor.model.NamedObjectListModel;
import org.nomencurator.editor.model.ObjectEditModel;
import org.nomencurator.editor.model.PublicationEditModel;

/**
 * <code>Panel</code> to modify a <code>Author</code> object via <code>AuthorEditModel</code> 
 *
 * @see 	org.nomencurator.Author
 * @see 	org.nomencurator.editor.AuthorEditor
 * @see 	org.nomencurator.editor.model.AuthorEditModel
 *
 * @version 	18 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class AuthorEditPanel
    extends NamedObjectEditPanel
    implements AuthorEditor, ItemSelectable, ItemListener
	       //ComponentListener, TextListener
{
    /** <code>AuthorEditModel</code> used by this <code>Panel</code> */
    protected AuthorEditModel model;

    protected static int defaultAuthorTextFieldSize = 16;

    /**
     * <code>org.nomencurator.editor.BorderdPanel</code> to contain
     * name editing components
     */
    protected org.nomencurator.editor.BorderedPanel namePanel;

    /** Title of <code>Panel</code> containing name <code>Component</code>s */
    protected static String namePanelTitle = "Name";

    /** <code>Choice</code> to choose title of the author */
    protected Choice title;

    /** Title of authors name title chooser */
    protected static String titlesTitle = "Title";

    /** <code>TextField</code> to enter and show first name of the author */
    protected TextField firstName;

    /** Title of first name <code>TextField</code> of the author */
    protected static String firstNameTitle = "First name";

    /** <code>TextField</code> to enter and show middle name of the author */
    protected TextField middleName;

    /** Title of middle name <code>TextField</code> of the author */
    protected static String middleNameTitle = "Middle name";

    /** <code>Choice</code> to choose feudality of the author */
    protected Choice feudality;

    /** Title of authors name feudality chooser */
    protected static String feudalityTitle = "Feudality";

    /** <code>TextField</code> to enter and show surname of the author */
    protected TextField surname;

    /** Title of first name <code>TextField</code> of the author */
    protected static String surnameTitle = "Surname";

    /** <code>Choice</code> to choose epithet of the author */
    protected Choice epithet;

    /** Title of authors name epithet chooser */
    protected static String epithetTitle = "Epithet";


    /** <code>BorderedPanel</code> to show publicaiont list */
    protected BorderedPanel publicationPanel;

    /** Title of <code>Panel</code> containing publication list */
    protected static String publicationPanelTitle = "Publications";

    /** default rows of publication list */
    protected static int defaultPublicationListRows = 5;

    /** <code>List</code> to enter and show author's publications */
    protected TableList publicationList;

    /** <code>BorderedPanel</code> to show affiliation list */
    protected BorderedPanel affiliationPanel;

    /** Title of <code>Panel</code> containing affiliation list */
    protected static String affiliationPanelTitle = "Affiliations";

    /** default rows of affiliation list */ 
    protected static int defaultAffiliationListRows = 3;

    /** <code>TableList</code> to enter and show author's affiliations */
    protected TableList affiliationList;

    protected static String[] affiliationListTitle = {
	"institute", "since", "until"
    };

    /** <code>BorderedPanel</code> to show curriculum vitae list */
    protected BorderedPanel curriculumVitaePanel;

    /** Title of <code>Panel</code> containing curriculum vitae list */
    protected static String curriculumVitaePanelTitle = "Curriculum Vitae";

    /** Rows of curriculumVitae */
    protected static final int curriculumVitaeRows = 2;

    /** <code>TableList</code> to enter and show author's date of birth and death */
    protected TableList curriculumVitae;

    protected static String[] curriculumVitaeTitle = {
	"Date", ""
    };

    protected PublicationEditPanel publicationEditPanel;
    
    //    protected AffiliationList affiliationList;
    protected Vector affiliationEditors;

    /** Listeners of this <code>Component</code> */
    transient protected ItemListener itemListener;

    /** <code>ItemEvent</code> to be passed to listeners */
    protected ItemEvent itemEvent;

    /** Constracts an <code>AuthorEditPanel</code> with new <code>Author</code> */
    public AuthorEditPanel ()
    {
	this(new Author());
    }

    /**
     * Constracts an <code>AuthorEditPanel</code> to edit <code>author</code>
     *
     * @param author <code>Author</code> to be edited by this <code>Panel</code>
     */
    public AuthorEditPanel(Author author)
    {
	this(new AuthorEditModel(author));
    }

    /**
     * Constracts an <code>AuthorEditPanel</code> to edit <code>model</code>
     *
     * @param model <code>AuthorEditModel</code> to be used by this <code>Panel</code>
     */
    public AuthorEditPanel(AuthorEditModel model)
    {
	this(model, true);
    }

    /**
     * Constracts an <code>AuthorEditPanel</code> for <code>model</code>
     * with <code>editble</code> status.  
     * The instance may modify <code>model</code>'s attributes when
     * <code>editable</code> is true.
     *
     * @param model <code>AuthorEditModel</code> to be used by this <code>Panel</code>
     * @param editable true if this object is allowed to edit <code>model</code>
     */
    public AuthorEditPanel (AuthorEditModel model, boolean editable)
    {
	super(model, editable);
    }

    /**
     * Creates an empty <code>AuthoerEditModel</code>.
     */
    protected ObjectEditModel createObjectEditModel()
    {
	return new AuthorEditModel();
    }

    /**
     * Method to create components corresponding to given <code>model</code>.
     *
     * @param model <code>ObjectEditModel</code> to be shared by this editor
     */
    protected void createModelComponents()
    {
	model = (AuthorEditModel)getModel();
	createNamePanel();
	createPublicationListPanel();
	createAffiliationListPanel();
	createCurriculumVitaePanel();
    }

    /**
     * Method to create author's name components corresponding to given <code>model</code>.
     *
     * @param model <code>AuthorEditModel</code> to be shared by this editor
     */
    protected void createNamePanel()
    {
	title = new Choice(model.getTitleModel());
	title.select(model.getTitleIndex());
	title.addItemListener(this);

	firstName = new TextField(model.getFirstName(), defaultAuthorTextFieldSize);
	firstName.addTextListener(this);
	firstName.addMouseListener(this);

	middleName = new TextField(model.getMiddleName(), defaultAuthorTextFieldSize);
	middleName.addTextListener(this);
	middleName.addMouseListener(this);

	feudality  = new Choice(model.getFeudalityModel());
	feudality.select(model.getFeudalityIndex());
	feudality.addItemListener(this);

	surname = new TextField(model.getSurname(), defaultAuthorTextFieldSize);
	surname.addTextListener(this);
	surname.addMouseListener(this);

	epithet = new Choice(model.getEpithetModel());
	epithet.select(model.getEpithetIndex());
	epithet.addItemListener(this);

	namePanel = new org.nomencurator.editor.BorderedPanel(new VTitledPaneBorder(namePanelTitle));

	namePanel.add(new Label(titlesTitle), title);
	namePanel.add(new Label(firstNameTitle), firstName, GridBagConstraints.HORIZONTAL);
	namePanel.add(new Label(middleNameTitle), middleName, GridBagConstraints.HORIZONTAL);
	namePanel.add(new Label(feudalityTitle), feudality);
	namePanel.add(new Label(surnameTitle), surname, GridBagConstraints.HORIZONTAL);
	namePanel.add(new Label(epithetTitle), epithet);
    }

    /**
     * Method to create components corresponding to given <code>model</code>.
     *
     * @param model <code>AuthorEditModel</code> to be shared by this editor
     */
    protected void createPublicationListPanel()
    {
	publicationList = new TableList(model.getPublicationList(),
					defaultPublicationListRows,
					model.getPublicationListTitle());
	publicationList.addMouseListener(this);
	model.addObserver(publicationList);
	publicationPanel = new BorderedPanel(new VTitledPaneBorder(publicationPanelTitle));
	publicationPanel.add(publicationList);
    }

    /**
     * Method to create components corresponding to affiliation list of given <code>model</code>.
     *
     * @param model <code>AuthorEditModel</code> to be shared by this editor
     */
    protected void createAffiliationListPanel()
    {
	affiliationList = new TableList(model.getAffiliationList(),
					defaultAffiliationListRows,
					affiliationListTitle);
	affiliationPanel = new BorderedPanel(new VTitledPaneBorder(affiliationPanelTitle));
	affiliationPanel.add(affiliationList);
    }

    /**
     * Method to create curriculum vitae components corresponding to given <code>model</code>.
     *
     * @param model <code>AuthorEditModel</code> to be shared by this editor
     */
    protected void createCurriculumVitaePanel()
    {
	curriculumVitae = new TableList(model.getCurriculumVitae(),
				 curriculumVitaeRows,
				 curriculumVitaeTitle);
	curriculumVitaePanel = new BorderedPanel(new VTitledPaneBorder(curriculumVitaePanelTitle));
	curriculumVitaePanel.add(curriculumVitae);
    }

    /**
     * Sets <CODE>ed</CODE> as <CODE>Editor</CODE> of <CODE>TextComponent</CODE>s
     * to provide <CODE>Text</CODE> handling using GUI.
     *
     * @param ed <CODE>Editor</CODE> to be linked to <CODE>TextComponent</CODE>s
     * of this
     */
    public void setEditor(Editor ed)
    {
	if(focusedEditor == ed)
	    return;

	if(focusedEditor != null) {
	    focusedEditor.unlisten(firstName);
	    focusedEditor.unlisten(middleName);
	    focusedEditor.unlisten(surname);
	}

	focusedEditor = ed;

	if(focusedEditor != null) {
	    focusedEditor.listen(firstName);
	    focusedEditor.listen(middleName);
	    focusedEditor.listen(surname);
	}

    }


    /**
     * Method to add components to this <code>Panel</code>.
     */
    protected void addModelComponents()
    {
	add(namePanel);
	add(publicationPanel);
	add(affiliationPanel);
	//	add(curriculumVitaePanel);
    }

    /**
     * Method to add components to this <code>Panel</code>.
     */
    protected void removeModelComponents()
    {
	remove(namePanel);
	remove(publicationPanel);
	remove(affiliationPanel);
	//	remove(curriculumVitaePanel);
    }

    /**
     * Gets <code>AuthorEditModel</code> accompanying with this editor
     *
     * @return AuthorEditModel accompanying with this editor
     */
    public AuthorEditModel getAuthorEditModel()
    {
	return (AuthorEditModel)getModel();
    }

    /**
     * Setups components to use given <CODE>model</CODE>.
     * This method is called from setModel(ObjectEditModel)
     * internally.
     */
    protected void setComponents(ObjectEditModel objectEditModel)
    {
	model = (AuthorEditModel)objectEditModel;

	title.select(model.getTitleIndex());

	firstName.setModel(model.getFirstName());

	middleName.setModel(model.getMiddleName());

	feudality.select(model.getFeudalityIndex());

	surname.setModel(model.getSurname());

	epithet.select(model.getEpithetIndex());

	publicationList.setModel(model.getPublicationList());

	affiliationList.setModel(model.getAffiliationList());

	//	curriculumVitae.setModel(model.getCurriculumVitae());

	if(itemListener != null) {
	    if(itemEvent == null)
		itemEvent = 
		    new ItemEvent(this, 
				  ItemEvent.ITEM_STATE_CHANGED,
				  this,
				  ItemEvent.ITEM_STATE_CHANGED
				  );
	    itemListener.itemStateChanged(itemEvent);
	}

    }

    /**
     * A utility method to get <code>Author</code> under edition
     *
     * @return Author under edition
     */
    public Author getAuthor()
    {
	return (Author)getObject();
    }

    /**
     * A utility method to set <code>author</code> as
     * target of edition by accompanying <code>AuthorEditModel</code>
     *
     * @param author to be set as target <code>Author</code>
     */
    public void setAuthor(Author author)
    {
	setObject(author);
    }

    public String getSummary()
    {
	return getAuthorEditModel().getFullName();
    }

    /*
    public String getInitializedName()
    {
	StringBuffer s = new StringBuffer(surname);
	if(epithet.length() > 0)
	    s.append(" " + epithet);
	if(firstname.length() > 0)
	    s.append(", " + firstname);
	if(middlename.length() > 0)
	    s.append(" " + middlename);
	if(title.length() > 0)
	    s.append(", " + title);
	return s.substring(0);
    }
    */

    static public void main(String args[]) {
	jp.kyasu.awt.Frame f = new jp.kyasu.awt.Frame("AuthorEditPanel");
	f.add(new AuthorEditPanel());
	f.pack();
	f.setVisible(true);
    }


    public String getAuthorNameWithLifePeriod()
    {
	Author author = (Author)getNamedObject();
	if(author == null)
	    return "";
	StringBuffer s = new StringBuffer(author.getFullname());
	Date birth = author.getBirthDate();
	Date death = author.getDeathDate();
	if(birth != null || death != null) {
	    s.append(" (");
	    if(birth == null)
		s.append("?");
	    else
		s.append(birth.toString());
	    if(death == null)
		s.append("?");
	    else
		s.append(death.toString());
	    s.append(")");
	}
	return s.toString();
    }

    public void saveAttributes()
    {
	if(isEditable())
	    model.saveAttributes();
    }

    public void loadAttributes()
    {
	model.loadAttributes();
    }

    /**
     * Adds the specified item listener to receive item events from this list.
     *
     * @param l the item listener.
     */
    public synchronized void addItemListener(ItemListener l)
    {
	itemListener = AWTEventMulticaster.add(itemListener, l);
	enableEvents(0); // mark newEventsOnly
    }

    /**
     * Removes the specified item listener so that it no longer receives
     * item events from this list.
     * @param l the item listener.
     */
    public synchronized void removeItemListener(ItemListener l)
    {
	itemListener = AWTEventMulticaster.remove(itemListener, l);
    }

    /**
     * Returns the selected items on the list in an array of Objects.
     * @see java.awt.ItemSelectable
     */
    public Object[] getSelectedObjects()
    {
	return new Object[]{model};
    }


    /**
     * Invoked when the mouse has been clicked on the Ok button.
     */
//    public void okButtonClicked()
//    {
//	super.okButtonClicked();
//    }

    /**
     * Returns <code>PublicationEditPanel</code> related to this
     * <code>AuthorEditPanel</code>
     */
    public PublicationEditPanel getPublicationEditPanel()
    {
	return publicationEditPanel;
    }

    /**
     * Sets <code>publicationEditPanel</code> as the <code>PublicationEditPanel</code>
     * linked to this <code>AuthorEditPanel</code>
     *
     * @param publicationEditPanel to be linked to <code>AuthorEditPanel</code>
     */
    public void setPublicationEditPanel(PublicationEditPanel publicationEditPanel)
    {
	if(this.publicationEditPanel == publicationEditPanel)
	    return;

	if(this.publicationEditPanel != null)
	    this.publicationEditPanel.setAuthorEditPanel(null);

	this.publicationEditPanel = publicationEditPanel;
	if(this.publicationEditPanel != null)
	    this.publicationEditPanel.setAuthorEditPanel(this);
    }

    public void mouseDoubleClicked(MouseEvent e) {
	Object source = e.getSource();
	if (source == publicationList) {
	    model.updateSummary();
	    if (source == publicationList) {
		selectPublication(isEditable() &&
				  (publicationList.getSelectedModel() == publicationList.getDummyModel() ||
				   e.getY() >=  publicationList.getLastBaseLine()));
	    }
	}
	else if(source instanceof TextField) {
	    AuthorSearchDialog asd = new AuthorSearchDialog(getFrame());
	    asd.pack();
	    asd.show();

	    if (asd.isOK()) {
		AuthorEditModel aem = asd.getSelectedAuthorEditModel();
		if(aem != null)
		    setModel(aem);
	    }
	}
    }
    
    public void updateView() {
	getAuthorEditModel().updateList();
    }
    
	public void itemStateChanged(ItemEvent e) {
	    super.itemStateChanged(e);
	    Object source = e.getSource();
	    if (source == title ||
		source == epithet ||
		source == feudality) {
		Choice choice = (Choice)source;
		String item = null;
		if (choice.getSelectedIndex() == choice.getItemCount() - 1) {
		    ChoiceDialog dlg = new ChoiceDialog(choice);
		    dlg.show();
		    if (dlg.isOK()) {
			item = dlg.getString();
			//					choice.insert(item, choice.getItemCount() - 1);
			if (source == title)
			    model.addTitle(item);
			else if (source == epithet)
			    model.addEpithet(item);
			else if (source == feudality)
			    model.addFeudality(item);
		    }
		    choice.select(choice.getItemCount() - 2);
		    PreferenceManager.savePreference();
		}
		if (source == title) {
		    model.setTitleIndex(title.getSelectedIndex());
		} else if (source == epithet) {
		    model.setEpithetIndex(epithet.getSelectedIndex());
		} else if (source == feudality) {
		    model.setFeudalityIndex(feudality.getSelectedIndex());
		}
	    }
	}

    /*
    public void newButtonClicked() {
	super.newButtonClicked();
	AuthorEditModel currentEditModel = (AuthorEditModel)objectEditModel;
	AuthorEditModel previousEditModel = (AuthorEditModel)previousModel;
	PublicationEditModel publicationEditModel =
	    (PublicationEditModel)this.publicationEditPanel.getModel();
	currentEditModel.add(publicationEditModel);
	publicationEditModel.addAuthor(currentEditModel);
	this.updateView();
    }
    */

    protected void selectPublication() {
	selectPublication(false);
    }
    
    protected void selectPublication(boolean forceNewCreate) {
	PublicationEditModel publicationEditModel = (PublicationEditModel)publicationList.getSelectedModel();
	if (publicationEditModel == publicationList.getDummyModel() ||
	    publicationEditModel == null)
	    forceNewCreate = true;

	if (forceNewCreate) {
	    // create new record
	    publicationList.deselect(publicationList.getSelectedIndex());
	    publicationEditModel = new PublicationEditModel();
	    NamedObjectListModel publicationListModel = model.getPublicationList();
	    if(publicationList.getRows() == publicationListModel.getItemCount() + 1){
		publicationList.invalidate();
		publicationListModel.enableDummyModel(isEditable());
		publicationList.validate();
	    }
	}
	model.add(publicationEditModel);
	publicationEditModel.addAuthor(model);
	publicationEditPanel.setModel(publicationEditModel);
	getTabbedPane().setSelectedIndex(TabbedPane.V_TABINDEX_PUBLICATION);
    }

    public void putUnsavedObject()
    {
	if(isModified())
	    NamedObjectBroker.getInstance().putUnsavedObjects(getAuthor().getPublications());
	super.putUnsavedObject();
    }

}
