/*
 * BorderedPanel.java:  a ClosableBorderedPanel with a method to open given Window.
 * It can be used as base class for editor frames for navigation of Nomencurator.
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: BorderedPanel.java,v 1.5 2002/03/26 04:04:07 nozomi Exp $
 * $Log: BorderedPanel.java,v $
 * Revision 1.5  2002/03/26 04:04:07  nozomi
 * add BorderedPanel(String) constructor
 *
 * Revision 1.4  2002/03/10 09:55:39  nozomi
 * add utiltiy methods
 *
 * Revision 1.3  2002/02/05 03:45:00  nozomi
 * Add fill parameter to Label-utility functions
 *
 * Revision 1.2  2002/02/05 03:33:14  nozomi
 * Add Label-making utility methods and default alignment
 *
 * Revision 1.1  2002/01/29 06:40:35  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.LayoutManager;

import jp.kyasu.awt.Label;

import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.V3DBorder;
import jp.kyasu.graphics.VBorder;
import jp.kyasu.graphics.Visualizable;
import jp.kyasu.graphics.VTitledPaneBorder;

/**
 * <code>BorderedPanel</code> with <code>GridBagConstraints</code>
 *
 * @see 	jp.kyasu.awt.BorderedPanel
 * @see 	org.nomencurator.awt.BorderedPanel
 * @see 	org.nomencurator.editor.GridBagContainer
 *
 * @version 	10 Mar 2002
 * @author 	Nozomi `James' Ytow
 */
public class BorderedPanel
    extends org.nomencurator.awt.BorderedPanel
    implements GridBagContainer
{
    /**
     * <code>GridBagContainerAdaptor</code> to handle layout
     */
    protected GridBagContainerAdaptor adaptor;

    /**
     * Default alignment value in <code>Label</code>
     */
    protected int defaultLabelAlignment = Label.RIGHT;

    /**
     * Creates a new panel using the default layout manager and border.
     */
    public BorderedPanel()
    {
	this(new V3DBorder(false));
    }

    /**
     * Creates a new panel using spefied title
     *
     * @param title in <code>String</code>
     */
    BorderedPanel(String title)
    {
	this(new VTitledPaneBorder(title));
    }

    /**
     * Creates a new panel with the specified border.
     *
     * @param border the visual border of the panel.
     */
    public BorderedPanel(VBorder border)
    {
	this(new GridBagLayout(), border);
    }

    /**
     * Creates a new panel with the specified layout manager.
     *
     * @param layout the layout manager for the panel.
     */
    public BorderedPanel(LayoutManager layout)
    {
	this(layout, new V3DBorder(false));
    }

    /**
     * Creates a new panel with the specified layout manager and border.
     *
     * @param layout the layout manager for the panel.
     * @param border the visual border of the panel.
     */
    public BorderedPanel(LayoutManager layout, VBorder border)
    {
	super(layout, border);

	if(getLayout() instanceof GridBagLayout)
	    adaptor = new GridBagContainerAdaptor(this);

	initialize();
    }

    /**
     * Initializes <code>Component</code>s in this container.
     * It may be overriden by subclasses.
     */
    protected void initialize()
    {
	createComponents();
	addComponents();
    }

    /**
     * Creates <code>Component</code>s in this container.
     * It may be overriden by subclasses.
     */
    protected void createComponents()
    {
    }

    /**
     * Adds <code>Component</code>s to this container.
     * It may be overriden by subclasses.
     */
    protected void addComponents()
    {
    }

    /**
     * Sets <code>manager</code> as <code>LayoutManager</code>
     * to be used 
     *
     * @param manager <code>GridBagLayout</code> to be used
     */
    public void setGridBagLayout(GridBagLayout manager)
    {
	setLayout(manager);
    }

    /**
     * Gets the <code>GridBagLayout</code> in use
     *
     * @return GridBagLayout in use
     */
    public GridBagLayout getGridBagLayout()
    {
	return (GridBagLayout)getLayout();
    }


    /**
     * Sets <code>constraints</code> as <code>GridBagConstraints</code>
     * to be used
     *
     * @param constratins <code>GridBagConstraints</code> to be used
     */
    public void setConstraints(GridBagConstraints constraints)
    {
	adaptor.setConstraints(constraints);
    }

    /**
     * Gets the <code>GridBagConstraints</code> in use
     *
     * @return GridBagConstraints in use
     */
    public GridBagConstraints getConstraints()
    {
	return adaptor.getConstraints();
    }

    /**
     * Sets width and height of <code>Insets</code> used
     * when layout <code>Components</code> to be <code>value</code>
     *
     * @param value withd and height of <code>Insets</code>
     */
    public void setInsetsValue(int value)
    {
	adaptor.setInsetsValue(value);
    }

    /**
     * Gets width and height of <code>Insets</code> to be used
     *
     * @return int width and height of <code>Insets</code>
     */
    public int getInsetsValue()
    {
	return adaptor.getInsetsValue();
    }

    /**
     * Returns default alignment value in <code>Label</code>
     *
     * @return default alignment value in <code>Label</code>
     */
    public int getDefaultLabelAlignment()
    {
	return defaultLabelAlignment;
    }

    /**
     * Sets <code>alignment</code> as default alignment value in <code>Label</code>
     *
     * @param alignment default alignment value in <code>Label</code>
     */
    public void setDefaultLabelAlignment(int alignment)
    {
	defaultLabelAlignment = alignment;
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>String</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(String label, Component component, int fill)
    {
	adaptor.add(new Label(label, defaultLabelAlignment), component, fill);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>String</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(String label, int alignment, Component component)
    {
	adaptor.add(new Label(label, alignment), component);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>String</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(String label, int alignment, Component component, int fill)
    {
	adaptor.add(new Label(label, alignment), component, fill);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Text</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(Text label, Component component)
    {
	add(label, defaultLabelAlignment, component);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Text</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(Text label, Component component, int fill)
    {
	add(label, defaultLabelAlignment, component, fill);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Text</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(Text label, int alignment,  Component component)
    {
	adaptor.add(new Label(label, alignment), component);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Text</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(Text label, int alignment,  Component component, int fill)
    {
	adaptor.add(new Label(label, alignment), component, fill);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Visualizable</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(Visualizable label, Component component)
    {
	add(label, defaultLabelAlignment, component);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Visualizable</code> to be used as contents of <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(Visualizable label, Component component, int fill)
    {
	add(label, defaultLabelAlignment, component, fill);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Visualizable</code> to be used as contents of <code>Label</code>
     * @param alignment alignment of <code>label</code> in the <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(Visualizable label, int alignment, Component component)
    {
	adaptor.add(new Label(label, alignment), component);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Visualizable</code> to be used as contents of <code>Label</code>
     * @param alignment alignment of <code>label</code> in the <code>Label</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(Visualizable label, int alignment, Component component, int fill)
    {
	adaptor.add(new Label(label, alignment), component, fill);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(Label label, Component component)
    {
	adaptor.add(label, component);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(Label label, Component component, int fill)
    {
	adaptor.add(label, component, fill);
    }

    /**
     * Adds <code>Component</code>s in <code>component</code> array to a <code>Container</code>
     * implementing this interface with <code>label</code>
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param components an array of <code>Component</code>s to be added to a <code>Container</code>
     */
    public void add(Label label, Component[] components)
    {
	adaptor.add(label, components);
    }

    /**
     * Adds <code>Component</code>s in <code>component</code> array to a <code>Container</code>
     * implementing this interface with <code>label</code>
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param components an array of <code>Component</code>s to be added to a <code>Container</code>
     */
    public void add(Label label, Component[] components, int fill)
    {
	adaptor.add(label, components, fill);
    }

    /**
     * Adds <code>Component</code>s in <code>component</code> array to a <code>Container</code>
     * implementing this interface with <code>label</code>
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param components an array of <code>Component</code>s to be added to a <code>Container</code>
     */
    public void add(Label label, Component[] components, int[] fill)
    {
	adaptor.add(label, components, fill);
    }

}

