/*
 *
 * CharactorChooser.java:  a class to pick up a character from a table.
 *
 * Copyright (c) 2000 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: CharacterChooser.java,v 1.1.1.1 2002/01/16 12:33:33 ryo Exp $
 *	$Log: CharacterChooser.java,v $
 *	Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 *	initial import into CVS
 *	
 *	
 *
 */

package org.nomencurator.editor;

import java.awt.AWTEventMulticaster;
import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.ItemSelectable;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.Hashtable;

import jp.kyasu.awt.BorderedPanel;
import jp.kyasu.awt.Button;
import jp.kyasu.awt.PopupPanel;

import jp.kyasu.graphics.V3DBorder;
import jp.kyasu.graphics.VActiveButton;


class CharacterChooser extends PopupPanel 
    implements ActionListener, ItemSelectable
{
    public char[][] characterTable;
    protected char selectedChar;

    protected Hashtable charMap;


    /** The item listener. */
    transient protected ItemListener itemListener;

    CharacterChooser()
    {
	selectedChar = 0;
	itemListener = null;
	setCharTable();
	initializePanel();
    }

    protected void setCharTable()
    {
	characterTable = getCharTable(new char[][][] {
	    greekCharTable, latin1CharTable, xLatinACharTable
	    });
    }

    protected void initializePanel() {


	GridBagLayout gridbag = new GridBagLayout();
	GridBagConstraints c = new GridBagConstraints();
	c.ipadx = 0;
	c.ipady = 0;
	c.insets.bottom = 0;
	c.insets.left = 0;
	c.insets.top = 0;
	c.insets.right = 0;

	BorderedPanel bp = new BorderedPanel(new V3DBorder(true));
	bp.setLayout(gridbag);

	int allChars = 0;
	int maxChars = 0;
	for(int i = 0 ; i < characterTable.length; i++) {
	    allChars += characterTable[i].length;
	    if(maxChars < characterTable[i].length)
		maxChars = characterTable[i].length;
	}

	charMap = new Hashtable(allChars);

	int size = (int)(Math.sqrt((double)maxChars * 2.0) + 0.5);

	int y = 0;
	for(int i = 0 ; i < characterTable.length; i++) {
	    allChars = characterTable[i].length;
	    int charIndex = 0;
	    for (; charIndex < allChars; y++) {
		for (int x = 0; x < size && charIndex < allChars; x++) {
		    char ch = characterTable[i][charIndex++];
		    Button b = new Button(new VActiveButton(new String(new char[]{ch})));
		    charMap.put(b, new Character(ch));
		    b.setFocusTraversable(false);
		    b.addActionListener(this);
		    c.gridx = x;
		    c.gridy = y;
		    gridbag.setConstraints(b, c);
		    bp.add(b);
		}
	    }
	}

	add(bp, BorderLayout.CENTER);
	setSize(getPreferredSize());
    }
    
    protected char[][] getCharTable(char[][][] charTable)
    {
	int subsets = 0;
	for(int i = 0; i < charTable.length; i++){
	    if(subsets < charTable[i].length)
		subsets = charTable[i].length;
	}

	char[][] table = new char[subsets][];
	int sizes[] = new int[subsets];
	for(int j = 0; j < subsets; j++) {
	    sizes[j] = 0;
	}

	for(int i = 0; i < charTable.length; i++){
	    for(int j = 0; j < charTable[i].length; j++) {
		sizes[j] += charTable[i][j].length;
	    }
	}

	for(int j = 0; j < subsets; j++) {
	    table[j] = new char[sizes[j]];
	    int index = 0;
	    for(int i = 0; i < charTable.length; i++){
		for(int k = 0; k < charTable[i][j].length; k++) {
		    table[j][index++] = charTable[i][j][k];
		}
	    }
	}

	return table;
    }

    /**
     * Returns the selected character.
     */
    public char getChar() {
	return selectedChar;
    }

    /**
     * Returns the selected items or null if no items are selected.
     * @see java.awt.ItemSelectable
     */
    public Object[] getSelectedObjects() {
	return new Character[]{ new Character(selectedChar)};
    }

    /**
     * Add a listener to recieve item events when the state of an item changes.
     * @see java.awt.ItemSelectable
     */
    public void addItemListener(ItemListener l) {
	itemListener = AWTEventMulticaster.add(itemListener, l);
    }

    /**
     * Removes an item listener.
     * @see java.awt.ItemSelectable
     */
    public void removeItemListener(ItemListener l) {
	itemListener = AWTEventMulticaster.remove(itemListener, l);
    }

    /** Notifies the item event to the item listeners. */
    protected void notifyItemListeners() {
	if (itemListener != null) {
	    ItemEvent e = new ItemEvent(this, ItemEvent.ITEM_STATE_CHANGED,
					new Character(selectedChar), ItemEvent.SELECTED);
	    itemListener.itemStateChanged(e);
	}
    }

    public static char[][] latin1CharTable =
    {
	{
	'a', '\u00E0', '\u00E1', '\u00E2', '\u00E3', '\u00E4', '\u00E5', '\u00E6',
	'b',
	'c', '\u00E7',
	'd', '\u00F0',
	'e', '\u00E8', '\u00E9', '\u00EA', '\u00EB',
	'f',
	'g',
	'h',
	'i', '\u00EC', '\u00ED', '\u00EE', '\u00EF',
	'j',
	'k',
	'l',
	'm',
	'n', '\u00F1',
	'o', '\u00F2', '\u00F3', '\u00F4', '\u00F5', '\u00F6', '\u00F8',
	'p',
	'q',
	'r',
	's', '\u00DF',
	't',
	'u', '\u00F9', '\u00FA', '\u00FB', '\u00FC',
	'v',
	'w',
	'x',
	'y', '\u00FD', '\u00FF',
	'z',
	'\u00FE',
	},

	{
	'A', '\u00C0', '\u00C1', '\u00C2', '\u00C3', '\u00C4', '\u00C5', '\u00C6',
	'B',
	'C', '\u00C7',
	'D', '\u00D0',
	'E', '\u00C8', '\u00C9', '\u00CA', '\u00CB',
	'F',
	'G',
	'H',
	'I', '\u00CC', '\u00CD', '\u00CE', '\u00CF',
	'J',
	'K',
	'L',
	'M',
	'N', '\u00D1',
	'O', '\u00D2', '\u00D3', '\u00D4', '\u00D5', '\u00D6', '\u00D8',
	'P',
	'Q',
	'R',
	'S',
	'T',
	'U', '\u00D9', '\u00DA', '\u00DB', '\u00DC',
	'V',
	'W',
	'X',
	'Y', '\u00DD',
	'Z',
	'\u00DE',
	}

    };

    public static char[][] greekCharTable = {
	{
	//smalls
	'\u03B1', '\u03AC',
	'\u03B2', '\u03B3','\u03B4',
	'\u03B5', '\u03AD',
	'\u03B6',
	'\u03B7', '\u03AE',
	'\u03B8',
	'\u03B9', '\u03AF', '\u03CA',
	'\u03BA', '\u03BB', '\u03BC','\u03BD','\u03BE',
	'\u03BF', '\u03CC',
	'\u03C0', '\u03C1', '\u03C3','\u03C4',
	'\u03C5', '\u03CB', '\u03CD', '\u03B0',
	'\u03C6', '\u03C7', '\u03C8',
	'\u03C9', '\u03CE'
	},
	{
	//capitals
	'\u0391', '\u0386', //GREEK CAPITAL LETTER ALPHA, and it WITH TONOS
	'\u0392', '\u0393','\u0394', //GREEK CAPITAL LETTER BETA, GAMA, DELTA
	'\u0395', '\u0388', //GREEK CAPITAL LETTER EPSILON, and it WITH TONOS
	'\u0396', //GREEK CAPITAL LETTER ZETA
	'\u0397', '\u0389', //GREEK CAPITAL LETTER ETA, and it WITH TONOS
	'\u0398', //GREEK CAPITAL LETTER THETA
	'\u0399', '\u038A', '\u03AA', //GREEK CAPITAL IOTA, it WITH TONOS and it WITH DIALYTIKA
	'\u039A', '\u039B', '\u039C','\u039D','\u039E',
	//GREEK CAPITAL KAPPA, LAMDA, MU, NU, XI
	'\u039F', '\u038C', //GREEK CAPITAL OMICRON and it with TONOS
	'\u03A0', '\u03A1', '\u03A3','\u03A4',
	'\u03A5', '\u038E', '\u03AB', '\u03D2', '\u03D3', '\u03D4', 
	'\u03A6', '\u03A7', '\u03A8',
	'\u03A9', '\u038F',
	}
    };

    public static char[][] xLatinACharTable = {
	{	
	'\u0100',
	},
    };

    public void actionPerformed(ActionEvent e)
    {
	Character ch = (Character)charMap.get(e.getSource());
	if(ch != null){
	    selectedChar = ch.charValue();
	    hidePopup();
	    notifyItemListeners();
	}
    }

    public void showPopup(java.awt.Component component, int x, int y)
    {
	super.showPopup(component, x, y);
    }
}
