/*
 * GridBagContainerAdaptor.java:  an implementation of GridBagContainer
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: GridBagContainerAdaptor.java,v 1.5 2002/03/27 02:25:26 nozomi Exp $
 * $Log: GridBagContainerAdaptor.java,v $
 * Revision 1.5  2002/03/27 02:25:26  nozomi
 * minor change in comment style
 *
 * Revision 1.4  2002/03/10 09:55:39  nozomi
 * add utiltiy methods
 *
 * Revision 1.3  2002/02/24 18:12:43  nozomi
 * filling parameter can be specified when add Components
 *
 * Revision 1.2  2002/02/22 22:15:28  nozomi
 * Remove unnecessary debugging message
 *
 * Revision 1.1  2002/01/29 10:00:36  nozomi
 * initial import into CVS
 */

package org.nomencurator.editor;

import java.awt.Component;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

import jp.kyasu.awt.Label;

/**
 * An adaptor implementation of <code>GridBacContainer</code> arranging
 * <code>Component</code> with specified <code>Label</code>
 * using <code>GridBagLayout</code>.
 *
 * @see org.nomencurator.editor.GridBagContainer
 * @see java.awt.GridBagConstraints
 * @see java.awt.GridBagLayout
 *
 * @version 	10 Mar 2002
 * @author 	Nozomi `James' Ytow
 */
public class GridBagContainerAdaptor
    implements GridBagContainer
{
    /** <code>Container</code> to be interfaced by this adaptor */
    protected Container container;

    /** <code>GridBagConstraints</code> to be used by this adaptor */
    protected GridBagConstraints constraints;

    /** Width and height of <code>Insets</code> */
    protected static int insetsValue = 12;

    protected static Insets topInsets = new Insets(insetsValue, 0, 0, 0);
    protected static Insets leftInsets = new Insets(0, insetsValue, 0, 0);
    protected static Insets bottomInsets = new Insets(0, 0, insetsValue - 1, 0);
    protected static Insets rightInsets = new Insets(0, 0, 0, insetsValue);
    protected static Insets leftRightInsets = new Insets(0, insetsValue, 0, insetsValue);
    protected static Insets topRightInsets = new Insets(insetsValue, 0, 0, insetsValue);

    /**
     * Constracts a <code>GridBagContaierAdaptor</code>
     * with <code>containter</code>
     *
     * @param containter <code>Container</code> to be adapted
     */
    public GridBagContainerAdaptor(Container container)
    {
	this.container = container;
	if(!(container.getLayout() instanceof GridBagLayout))
	    setGridBagLayout(new GridBagLayout());
	constraints    = new GridBagConstraints();
    }
    
    /**
     * Sets <code>manager</code> as <code>LayoutManager</code>
     * to be used 
     *
     * @param manager <code>GridBagLayout</code> to be used
     */
    public void setGridBagLayout(GridBagLayout manager)
    {
	container.setLayout(manager);
    }

    /**
     * Gets the <code>GridBagLayout</code> in use
     *
     * @return GridBagLayout in use
     */
    public GridBagLayout getGridBagLayout()
    {
	return (GridBagLayout)container.getLayout();
    }

    /**
     * Sets <code>constraints</code> as <code>GridBagConstraints</code>
     * to be used
     *
     * @param constratins <code>GridBagConstraints</code> to be used
     */
    public void setConstraints(GridBagConstraints constraints)
    {
	if(this.constraints == constraints)
	    return;
	this.constraints = constraints;
    }

    /**
     * Gets the <code>GridBagConstraints</code> in use
     *
     * @return GridBagConstraints in use
     */
    public GridBagConstraints getConstraints()
    {
	return constraints;
    }

    /**
     * Sets width and height of <code>Insets</code> used
     * when layout <code>Components</code> to be <code>value</code>
     *
     * @param value withd and height of <code>Insets</code>
     */
    public void setInsetsValue(int value)
    {
	insetsValue = value;
	updateInsets();
    }

    /**
     * Sets values of static <code>Insets</code>
     */
    protected void updateInsets()
    {
	topInsets.top         = insetsValue;
	leftInsets.left       = insetsValue;
	bottomInsets.bottom   = insetsValue - 1;
	rightInsets.right     = insetsValue;
	leftRightInsets.left  = insetsValue;
	leftRightInsets.right = insetsValue;
	topRightInsets.top    = insetsValue;
	topRightInsets.right  = insetsValue;
    }

    /**
     * Gets width and height of <code>Insets</code> to be used
     *
     * @return int width and height of <code>Insets</code>
     */
    public int getInsetsValue()
    {
	return insetsValue;
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     */
    public void add(Label label, Component component)
    {
	add(label, component, GridBagConstraints.NONE);
    }

    /**
     * Adds <code>component</code> to a <code>Container</code>
     * implementing this interface with <code>label</code>protected static Insets 
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param component <code>Component</code> to be added to a <code>Container</code>
     * @param fill filling paramter of <code>GridBagConstraints</code>
     */
    public void add(Label label, Component component, int fill)
    {
	GridBagConstraints saved = (GridBagConstraints)constraints.clone();
	addLabel(label);
	constraints.gridx  = GridBagConstraints.RELATIVE;
	constraints.insets = rightInsets;
	constraints.anchor = GridBagConstraints.WEST;
	constraints.fill = fill;
	constraints.gridwidth  = GridBagConstraints.REMAINDER;
	container.add(component, constraints);
	restoreConstraints(saved);
    }

    /**
     * Adds <code>Component</code>s in <code>component</code> array to a <code>Container</code>
     * implementing this interface with <code>label</code>
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param components an array of <code>Component</code>s to be added to a <code>Container</code>
     */
    public void add(Label label, Component[] components)
    {
	add(label, components, GridBagConstraints.NONE);
    }

    /**
     * Adds <code>Component</code>s in <code>component</code> array to a <code>Container</code>
     * implementing this interface with <code>label</code>
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param components an array of <code>Component</code>s to be added to a <code>Container</code>
     */
    public void add(Label label, Component[] components, int fill)
    {
	//save parameters
	GridBagConstraints saved = (GridBagConstraints)constraints.clone();
	/*
	int gridwidth = constraints.gridwidth;
	int gridx = constraints.gridx;
	int ipadx = constraints.ipadx;
	int anchor = constraints.anchor;
	int fill = constraints.fill;
	Insets insets  = constraints.insets;
	*/
	constraints.gridwidth = 1;
	constraints.gridx = 0;
	constraints.fill = fill;
	//	constraints.fill = GridBagConstraints.NONE;
	constraints.anchor = GridBagConstraints.EAST;
	constraints.insets = leftRightInsets;

	container.add(label, constraints);

	constraints.gridx  = GridBagConstraints.RELATIVE;
	constraints.insets = rightInsets;
	constraints.anchor = GridBagConstraints.CENTER;
	if (components != null) {
	    int counts = components.length - 2;
	    for(int i = 0; i < counts; i++) {
		container.add(components[i], constraints);
	    }
	    if(counts > -1){
		//		constraints.fill = GridBagConstraints.HORIZONTAL;
		constraints.gridwidth  = GridBagConstraints.RELATIVE;
		container.add(components[counts++], constraints);
	    }
	    //		constraints.fill = GridBagConstraints.NONE;
	    constraints.gridwidth  = GridBagConstraints.REMAINDER;
	    container.add(components[counts], constraints);
	}

	//restore parameters
	restoreConstraints(saved);
	/*
	constraints.fill = fill;
	constraints.anchor = anchor;
	constraints.ipadx = ipadx;
	constraints.gridx = gridx;
	constraints.gridwidth = gridwidth;
	constraints.insets = insets;
	*/
    }

    /**
     * Adds <code>Component</code>s in <code>component</code> array to a <code>Container</code>
     * implementing this interface with <code>label</code>
     *
     * @param label <code>Label</code> to be added to a <code>Container</code>
     * @param components an array of <code>Component</code>s to be added to a <code>Container</code>
     */
    public void add(Label label, Component[] components, int[] fill)
    {
	//save parameters
	GridBagConstraints saved = (GridBagConstraints)constraints.clone();

	constraints.gridwidth = 1;
	constraints.gridx = 0;
	constraints.anchor = GridBagConstraints.EAST;
	constraints.insets = leftRightInsets;

	container.add(label, constraints);

	constraints.gridx  = GridBagConstraints.RELATIVE;
	constraints.insets = rightInsets;
	constraints.anchor = GridBagConstraints.CENTER;
	if (components != null) {
	    int counts = components.length - 2;
	    for(int i = 0; i < counts; i++) {
		constraints.fill = fill[i];
		container.add(components[i], constraints);
	    }
	    if(counts > -1){
		constraints.gridwidth  = GridBagConstraints.RELATIVE;
		constraints.fill = fill[counts];
		container.add(components[counts++], constraints);
	    }
	    constraints.gridwidth  = GridBagConstraints.REMAINDER;
	    constraints.fill = fill[counts];
	    container.add(components[counts], constraints);
	}

	//restore parameters
	restoreConstraints(saved);
    }

    protected void restoreConstraints(GridBagConstraints saved)
    {
	constraints.fill = saved.fill;
	constraints.anchor = saved.anchor;
	constraints.ipadx = saved.ipadx;
	constraints.gridx = saved.gridx;
	constraints.gridwidth = saved.gridwidth;
	constraints.insets = saved.insets;
    }

    protected void addLabel(Label label)
    {
	constraints.gridwidth = 1;
	constraints.gridx = 0;
	constraints.fill = GridBagConstraints.NONE;
	constraints.anchor = GridBagConstraints.EAST;
	constraints.insets = leftRightInsets;

	container.add(label, constraints);

    }
}
