/*
 * List.java:  a List implementing EditorSelector
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: List.java,v 1.13 2002/10/11 03:42:54 nozomi Exp $
 * $Log: List.java,v $
 * Revision 1.13  2002/10/11 03:42:54  nozomi
 * add up(int), down(int) and moveModel(int, int) methods)
 *
 * Revision 1.12  2002/09/17 08:47:25  nozomi
 * add/removeModel() methods return boolean
 *
 * Revision 1.11  2002/09/06 02:11:46  nozomi
 * add getNamedObjects()
 *
 * Revision 1.10  2002/05/26 15:19:51  nozomi
 * fix dummy model handling
 *
 * Revision 1.9  2002/05/25 16:55:19  nozomi
 * fix model selection bug
 *
 * Revision 1.8  2002/05/22 22:18:12  nozomi
 * add createModelSelector for simpler maintenance
 *
 * Revision 1.7  2002/05/21 03:31:25  nozomi
 * support getLastBaseLine()
 *
 * Revision 1.6  2002/05/20 02:01:47  nozomi
 * Re-implemented to utilise ModelSelector
 *
 * Revision 1.5  2002/05/20 01:58:18  nozomi
 * ModelSelector re-defind to be based on TextListModel
 *
 * Revision 1.4  2002/05/17 17:40:04  nozomi
 * ModelSelectors moved to org.nomencurator.editor.model
 *
 * Revision 1.3  2002/02/26 16:31:26  nozomi
 * switched to ModelSelector
 *
 * Revision 1.2  2002/02/14 01:53:13  okawa
 * modify parameter from ObjectEditor to NamedObjectEditor
 *
 * Revision 1.1  2002/01/29 10:00:36  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import java.util.Vector;

import jp.kyasu.awt.DefaultTextListModel;
import jp.kyasu.awt.ScrollPanel;
import jp.kyasu.awt.TextListModel;

import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.V3DBorder;
import jp.kyasu.graphics.VBorder;

import jp.kyasu.graphics.text.TextLineInfo;

import org.nomencurator.editor.model.ModelSelectAdaptor;
import org.nomencurator.editor.model.ModelSelector;
import org.nomencurator.editor.model.NamedObjectEditModel;

/**
 * A <code>List</code> enables slection of editors
 * 
 * @see jp.kyasu.awt.List
 * @see org.nomencurator.editor.EditorSelector
 *
 * @version 	11 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class List
    extends jp.kyasu.awt.List
    implements ModelSelector
{
    /**
     * <code>ModelSelector</code> linking a line in this <code>List</code>
     * and models
     */
    ModelSelector modelSelector;

    /**
     * Creates a new scrolling list. Initially there are no visible
     * lines, and only one item can be selected from the list.
     */
    public List()
    {
	this(0, false);
    }

    /**
     * Creates a new scrolling list initialized with the specified
     * number of visible lines. By default, multiple selections are
     * not allowed.
     * @param rows the number of items to show.
     */
    public List(int rows)
    {
    	this(rows, false);
    }

    /**
     * Creates a new scrolling list initialized to display the specified
     * number of rows. If the value of <code>multipleMode</code> is
     * <code>true</code>, then the user can select multiple items from
     * the list. If it is <code>false</code>, only one item at a time
     * can be selected.
     * @param rows         the number of items to show.
     * @param multipleMode if <code>true</code>, then multiple selections
     *                     are allowed; otherwise, only one item can be
     *                     selected at a time.
     */
    public List(int rows, boolean multipleMode)
    {
	this(rows, multipleMode, ScrollPanel.SCROLLBARS_BOTH);
    }

    /**
     * Creates a new scrolling list initialized to display the specified
     * number of rows, multipleMode, and scroll bar visibility..
     * @param rows         the number of items to show.
     * @param multipleMode if <code>true</code>, then multiple selections
     *                     are allowed; otherwise, only one item can be
     *                     selected at a time.
     * @param scrollbars   a constant that determines what scrollbars are
     *                     created to view the list.
     */
    public List(int rows, boolean multipleMode, int scrollbars)
    {
	this(RichTextStyle.DEFAULT_LIST_STYLE,
	     rows, multipleMode, scrollbars);
    }

    /**
     * Creates a new scrolling list with the specified style and number of rows.
     * @param richTextStyle the style of the text list model.
     * @param rows          the number of items to show.
     */
    public List(RichTextStyle richTextStyle, int rows) {
	this(richTextStyle, rows, false);
    }

    /**
     * Creates a new scrolling list with the specified style, number of rows,
     * and multipleMode.
     * @param richTextStyle the style of the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     */
    public List(RichTextStyle richTextStyle, int rows, boolean multipleMode)
    {
	this(richTextStyle, rows, multipleMode, ScrollPanel.SCROLLBARS_BOTH);
    }

    /**
     * Creates a new scrolling list with the specified style, number of rows,
     * multipleMode, and scroll bar visibility.
     * @param richTextStyle the style of the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     */
    public List(RichTextStyle richTextStyle, int rows, boolean multipleMode,
		int scrollbars)
    {
	this(new DefaultTextListModel(richTextStyle), rows, multipleMode,
	     scrollbars, new V3DBorder(false));
    }

    /**
     * Creates a new scrolling list with the specified model and number of rows.
     * @param textListModel the text list model.
     * @param rows          the number of items to show.
     */
    public List(TextListModel textListModel, int rows)
    {
	this(textListModel, rows, false);
    }

    /**
     * Creates a new scrolling list with the specified model, number of rows,
     * and multipleMode.
     * @param textListModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     */
    public List(TextListModel textListModel, int rows, boolean multipleMode)
    {
	this(textListModel, rows, multipleMode, ScrollPanel.SCROLLBARS_BOTH);
    }

    /**
     * Creates a new scrolling list with the specified model, number of rows,
     * multipleMode, and scroll bar visibility.
     * @param textListModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     */
    public List(TextListModel textListModel, int rows, boolean multipleMode,
		int scrollbars)
    {
	this(textListModel, rows, multipleMode, scrollbars,
	     new V3DBorder(false));
    }

    /**
     * Creates a new scrolling list with the specified model, number of rows,
     * multipleMode, scroll bar visibility, and border visual.
     * @param textListModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     * @param border        the border visual of the list.
     */
    public List(TextListModel textListModel, int rows, boolean multipleMode,
		int scrollbars, VBorder border)
    {
	this(textListModel, rows, multipleMode, scrollbars, border, false);
    }

    /**
     * Creates a new scrolling list with the specified model, number of rows,
     * multipleMode, scroll bar visibility, and border visual.
     * @param textListModel the text list model.
     * @param rows          the number of items to show.
     * @param multipleMode  if <code>true</code>, then multiple selections
     *                      are allowed; otherwise, only one item can be
     * @param scrollbars    a constant that determines what scrollbars are
     *                      created to view the list.
     * @param border        the border visual of the list.
     * @param enable        true to add dummy modal automatically
     */
    public List(TextListModel textListModel, int rows, boolean multipleMode,
		int scrollbars, VBorder border, boolean enable)
    {
	super(textListModel, rows, multipleMode, scrollbars, border);
	//createModelSelector(textListModel, enable);
    }

    /**
     * Create appropriate <code>ModelSelecotr</code> handling
     * <code>textListModel</code> with dummy model addition mode
     * status of <code>enable</code>.
     *
     * @param textListModel <code>TextListModel</code> to be handled by the <code>ModelSelector</code>
     * @param enable true to enable dummy model addition of the <code>ModelSelector</code>
     */
    protected void createModelSelector(TextListModel textListModel, boolean enable)
    {
	if(modelSelector != null &&
	   modelSelector.getModel() == textListModel)
	    return;

	if(textListModel instanceof ModelSelector) {
	    modelSelector = (ModelSelector)textListModel;
	    //	    modelSelector.enableDummyModel(enable);
	}
	else {
	    modelSelector = new ModelSelectAdaptor(textListModel, enable);
	}
    }

    /**
     * Adds <code>model</code> at end of model list
     *
     * @param model <code>NamedObjectEditModel</code> to be added
     */
    public boolean addModel(NamedObjectEditModel model)
    {
	return modelSelector.addModel(model);
    }

    /**
     * Inserts <code>model</code> at <code>index</code> of model list.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     */
    public boolean addModel(int index, NamedObjectEditModel model)
    {
	return modelSelector.addModel(index, model);
    }

    /**
     * Inserts <code>model</code> at <code>index</code> of model list
     * with its representation given by <code>items</code>.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     * @param items  the representaition of the model
     */
    public boolean addModel(int index, NamedObjectEditModel model, Object[][] items)
    {
	return modelSelector.addModel(index, model, items);
    }

    /**
     * Remove all <code>NamedObjectEditModel</code>s from <code>NamedObjectEditModel</code> list
     *
     */
    public void removeAllModels()
    {
	modelSelector.removeAllModels();
    }


    /**
     * Release <code>model</code> from this adaptor's control
     */
    public boolean removeModel(NamedObjectEditModel model)
    {
	return modelSelector.removeModel(model);
    }

    /**
     * Returns <code>NamedObjectEditModel</code> at <code>index</code>
     *
     * @param index  where <code>model</code> is
     * @return <code>NamedObjectEditModel</code> at  <code>index</code>
     */
    public NamedObjectEditModel getModel(int index)
    {
	return modelSelector.getModel(index);
    }


    /**
     * Selects <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModel(NamedObjectEditModel model)
    {
	modelSelector.selectModel(model);
    }

    /**
     * Selects <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be selected
     */
    public void select(int index)
    {
	selectModelAt(index);
    }

    /**
     * Selects an <code>NamedObjectEditModel</code> at <code>index</code> of list
     *
     * @param index of <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModelAt(int index)
    {
	modelSelector.selectModelAt(index);
    }

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel</code> selected
     */
    public NamedObjectEditModel getSelectedModel()
    {
	return modelSelector.getSelectedModel();
    }

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel[]</code> selected
     */
    public NamedObjectEditModel[] getSelectedModels()
    {
	return modelSelector.getSelectedModels();
    }


    /**
     * Returns index of <code>model</code>
     *
     * @param model <code>NamedObjectEditModel</code> to be searched
     * @return int index of <code>model</code>
     */
    public int indexOf(NamedObjectEditModel model)
    {
	return modelSelector.indexOf(model);
    }

    /**
     * Enables or desables automatic addition of a dummy
     * <code>NamedObjectEditModel</code> to the list according to
     * value of <code>enable</code>
     *
     * @param enable <code>boolean</code> enables 
     * automatic addition of a dummy <code>NamedObjectEditModel</code>
     * if true or disables if false
     */
    public void enableDummyModel(boolean enable)
    {
	modelSelector.enableDummyModel(enable);
    }

    /**
     * Returns true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     *
     * @return true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     */
    public boolean isDummyModelEnabled()
    {
	return modelSelector.isDummyModelEnabled();
    }

    /**
     * Returns a dummy <code>NamedObjectEditModel</code> instance for double-clicking
     * operation.
     *
     * @return null to be considered as a dummy <code>NamedObjectEditModel</code>
     */
    public NamedObjectEditModel getDummyModel()
    {
	return modelSelector.getDummyModel();
    }

    /**
     * Returns Text array representing empty summary 
     * can be used to represent a dummy <code>NamedObjectEditModel</code>.
     *
     * @return Text[][] representing a summary of an empty <code>NamedObjectEditModel</code>.
     */
    public Text[][] getEmptySummary()
    {
	return modelSelector.getEmptySummary();
    }

    /**
     * Returns <code>Vector</code> of <code>NamedObjectEditModel</code>
     * accompanying to the <code>ListModel</code>
     */
    public Vector getModels()
    {
	return modelSelector.getModels();
    }

    /**
     * Sets <code>models</code> as <code>NamedObjectEditModel</code> list
     * accompanying to the <code>ListModel</code>
     */
    public void setModels(Vector models)
    {
	modelSelector.setModels(models);
    }

    /**
     * Updates summary text of all specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update()
    {
	modelSelector.update();
    }


    /**
     * Updates summary text of specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update(NamedObjectEditModel model)
    {
	modelSelector.update(model);
    }

    /**
     * Updates summary text of model specified by <code>index</code>
     *
     * @param index index of <code>NamedObjectEditModel</code> to be updated
     */
    public void update(int index)
    {
	modelSelector.update(index);
    }

    /**
     * Sets <code>textListModel</code> as a summarized representation
     * <code>NamedObjectEditModel</code> handled by this selector
     *
     * @param textListModel <code>TextListModel</code> representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     */
    public void setModel(TextListModel textListModel)
    {
	super.setModel(textListModel);

	boolean enable = false;
	if(modelSelector != null)
	    enable = modelSelector.isDummyModelEnabled();
	else if(textListModel instanceof ModelSelector)
	    enable = ((ModelSelector)textListModel).isDummyModelEnabled();
	createModelSelector(textListModel, enable);
	//createModelSelector(textListModel, modelSelector.isDummyModelEnabled());
    }

    /**
     * Returns position of the last line
     *
     * @return position of the last line
     */
    public int getLastBaseLine()
    {
	TextLineInfo line  = listView.getVisibleEnd();
	return (line.y + line.lineHeight);
//	return (line.y == 0)? 0: (line.y + line.lineHeight);
    }

    /**
     * Returns true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     *
     * @return true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     */
    public boolean contains(NamedObjectEditModel model)
    {
	return modelSelector.contains(model);
    }

    /**
     * Returns <code>Vector</code> of <code>NamedObject</code>s
     * accompanying to the <code>ListModel</code>
     */
    public Vector getNamedObjects()
    {
	return modelSelector.getNamedObjects();
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one above
     */
    public boolean up(int index)
    {
	return modelSelector.up(index);
    }


    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one below
     */
    public boolean down(int index)
    {
	return modelSelector.down(index);
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>source</CODE> to <CODE>destination</CODE>
     */
    public boolean moveModel(int source, int destination)
    {
	return modelSelector.moveModel(source, destination);
    }
}
