/*
 * ObjectEditPanel.java:  an editor of Object
 * for the Nomencurator, a Nomenclature Heuristic Model.
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ObjectEditPanel.java,v 1.28 2002/10/18 07:48:36 nozomi Exp $
 * $Log: ObjectEditPanel.java,v $
 * Revision 1.28  2002/10/18 07:48:36  nozomi
 * remove debug infro printing line
 *
 * Revision 1.27  2002/10/11 03:43:23  nozomi
 * add markButton
 *
 * Revision 1.26  2002/10/08 00:37:03  nozomi
 * remove redundant loadAttributes() call
 *
 * Revision 1.25  2002/09/17 08:46:39  nozomi
 * re-organise initialisation
 *
 * Revision 1.24  2002/09/08 14:23:57  nozomi
 * support model creation
 *
 * Revision 1.23  2002/09/07 04:02:38  nozomi
 * be capable to listen to more event to set modified state
 *
 * Revision 1.22  2002/09/07 03:09:27  nozomi
 * use modified flag
 *
 * Revision 1.21  2002/09/06 11:44:58  nozomi
 * use specialised HTMLStyle
 *
 * Revision 1.20  2002/09/05 02:29:12  nozomi
 * modify addButtonPanel() to allow a EditPanel without common buttons
 *
 * Revision 1.19  2002/06/23 13:38:42  nozomi
 * aware of FocusController
 *
 * Revision 1.18  2002/06/06 05:50:34  t.okada
 * add taxoNote member
 *
 * Revision 1.17  2002/06/03 01:43:52  nozomi
 * add Vector to hold not-yet-saved models
 *
 * Revision 1.16  2002/05/21 12:41:05  nozomi
 * remove Summarizble relating code
 *
 * Revision 1.15  2002/05/17 10:53:25  ryo
 * modify *ButtonClicked()
 *
 * Revision 1.14  2002/05/10 13:32:22  ryo
 * add tabbedPane accessor
 *
 * Revision 1.13  2002/05/08 20:48:36  nozomi
 * add mouseDoiubleClicked to handle douible click
 *
 * Revision 1.12  2002/05/08 10:56:53  ryo
 * disable OK, Clear and Cancel button in *EditPanel
 *
 * Revision 1.11  2002/04/19 22:56:21  nozomi
 * change changeButtonState()
 *
 * Revision 1.10  2002/04/19 21:44:14  nozomi
 * mouse event handling modified
 *
 * Revision 1.9  2002/04/16 23:43:51  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.8  2002/04/04 12:22:49  ryo
 * add *ButtonClicked() and changeButtonState()
 *
 * Revision 1.7  2002/03/29 08:05:16  okawa
 * change isEditable
 *
 * Revision 1.6  2002/03/06 00:47:43  okawa
 * restore the last modification
 *
 * Revision 1.5  2002/03/04 00:42:25  okawa
 * modify mousePressed() okButton case
 *
 * Revision 1.4  2002/03/03 23:48:53  nozomi
 * setModel() synchronized
 *
 * Revision 1.3  2002/02/27 23:50:42  nozomi
 * add repaint()
 *
 * Revision 1.2  2002/02/23 19:19:58  nozomi
 * Simplify setModel()
 *
 * Revision 1.1  2002/01/29 06:38:44  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import java.awt.AWTEventMulticaster;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.ItemSelectable;
import java.awt.LayoutManager;

import java.util.Vector;
import java.util.Observable;

import jp.kyasu.awt.Button;
import jp.kyasu.awt.ColumnarLayout;
import jp.kyasu.awt.Panel;
import jp.kyasu.awt.TextComponent;
import jp.kyasu.awt.TextModel;
import jp.kyasu.awt.Dialog;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;

import jp.kyasu.editor.Editor;

import jp.kyasu.graphics.html.DefaultHTMLReaderTarget;
import jp.kyasu.graphics.html.HTMLReader;
import jp.kyasu.graphics.html.HTMLReaderTarget;
import jp.kyasu.graphics.html.HTMLStyle;
import jp.kyasu.graphics.html.HTMLText;
import jp.kyasu.graphics.html.HTMLWriter;
import jp.kyasu.graphics.html.VAnchor;

import org.nomencurator.NamedObject;

import org.nomencurator.broker.NamedObjectBroker;

import org.nomencurator.editor.ObjectEditor;

import org.nomencurator.editor.model.ObjectEditModel;
import org.nomencurator.editor.model.NamedObjectEditModel;

/**
 * A <code>Panel</code> providing graphical component wrapping
 * <code>ObjectEditModel</code>.
 *
 * @see org.nomencurator.editor.model.ObjectEditModel
 *
 * @version 	18 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class ObjectEditPanel 
    extends Panel
    implements ObjectEditor, 
	       ActionListener,
	       FocusListener,
	       ItemListener,
	       MouseListener,
	       TextListener,
	       Modifiable
{
    /** Model of this editor */
    protected ObjectEditModel  objectEditModel;

    /** Model edited by this editor to handle cancelation */
    protected ObjectEditModel  previousModel;

    /** Editable status of this editor */
    protected boolean editable;

    /** Modification state holder, if necessary.... */
    protected boolean modified;

    /** <code>Panel</code> to hold controlling <code>Button</code>s */
    protected Panel buttonPanel;

    /** Ok <code>Button</code> */
    protected Button okButton;

    /** Title of ok <code>Button</code> */
    protected static String okButtonTitle = "OK";

    
    /** Cancel <code>Button</code> */
    protected Button cancelButton;

    /** Title of cancel <code>Button</code> */
    protected static String cancelButtonTitle = "Cancel";

    /** New <code>Button</code> */
    protected Button newButton;

    /** Title of new <code>Button</code> */
    protected static String newButtonTitle = "New";

    /** Copy <code>Button</code> */
    protected Button copyButton;

    /** Title of copy <code>Button</code> */
    protected static String copyButtonTitle = "Copy";

    /** Copy <code>Button</code> */
    protected Button markButton;

    /** Title of copy <code>Button</code> */
    protected static String markButtonTitle = "Mark";

    /**
     * Clear <code>Button</code>
     */
    protected Button clearButton;

    protected transient Vector forcusListeners;

    /**
     * Title of clear <code>Button</code>
     */
    protected static String clearButtonTitle = "Clear";

    /**
     *
     */
    protected static LayoutManager buttonPanelLayout = new FlowLayout(FlowLayout.TRAILING);

    /** List of <code>ItemListener</code>s */
    transient protected ItemListener itemListeners;

    /** <CODE>TabbedPane</CODE> containting this <CODE>Panel</CODE> */
    protected TabbedPane tabbedPane;
	
    /**
     * 
     */
    protected TaxoNote taxoNote;

    /** <CODE>Editor</CODE> handling <CODE>TextComponent</CODE>s */
    protected Editor focusedEditor;

    /**
     * Constructor of <code>ObjectEditPanel</code> can be called
     * by only subclasses.
     * 
     * @param model <code>Object</code> to be edited by this model
     * @param editable boolean indicating whether this editor is allowed to 
     * manipulate the <code>Object</code>
     */
    protected ObjectEditPanel(ObjectEditModel model, boolean editable)
    {
	super(new ColumnarLayout());

	objectEditModel = model;
	this.editable = editable;

	modified = false;

	setupComponents();
    }

    /**
     * Sets up <code>Components</code> used in this editor.
     * The subclasses may be override it.
     *
     * @param model <code>Object</code> to be edited by this model
     */
    protected void setupComponents()
    {
	createModelComponents();
	createButtonPanel();
	
	addModelComponents();
	addButtonPanel();
    }

    /**
     * Sets <CODE>ed</CODE> as <CODE>Editor</CODE> of <CODE>TextComponent</CODE>s
     * to provide <CODE>Text</CODE> handling using GUI.
     *
     * @param ed <CODE>Editor</CODE> to be linked to <CODE>TextComponent</CODE>s
     * of this
     */
    public abstract void setEditor(Editor ed);

    /**
     * Creates an <code>ObjectEditModel</code> appropriate to 
     * the subclass of <code>ObjectEditPanel</code> which is used
     * in <code>Button</code> pressed event handling.
     *
     */
    protected abstract ObjectEditModel createObjectEditModel();


    /**
     * Method to craete components corresponding to given <code>objectEditModel</code>.
     *
     * @param objectEditModel <code>ObjectEditModel</code> to be shared by this editor
     */
    protected abstract void createModelComponents();


    /**
     * Creates a <code>Panel</code> with its containing <code>Button</code>s 
     * to control the editor.
     *
     */
    protected void createButtonPanel()
    {
	okButton = new Button(okButtonTitle);
	newButton = new Button(newButtonTitle);
	copyButton = new Button(copyButtonTitle);
	clearButton = new Button(clearButtonTitle);
	cancelButton = new Button(cancelButtonTitle);
	markButton = new Button(markButtonTitle);

	buttonPanel = new Panel(buttonPanelLayout);

	buttonPanel.add(okButton);
	buttonPanel.add(newButton);
	buttonPanel.add(copyButton);
	buttonPanel.add(clearButton);
	buttonPanel.add(cancelButton);
	buttonPanel.add(markButton);

	okButton.addMouseListener(this);
	newButton.addMouseListener(this);
	copyButton.addMouseListener(this);
	clearButton.addMouseListener(this);
	cancelButton.addMouseListener(this);
	markButton.addMouseListener(this);
	
	changeButtonState();
    }

    /**
     * Adds <code>Component</code>s representing the model to this <code>Panel</code>.
     *
     */
    protected abstract void addModelComponents();

    /**
     * Removes <code>Component</code>s representing the model from this <code>Panel</code>.
     *
     */
    protected abstract void removeModelComponents();


    /**
     * Adds a <code>Panel</code> containing <code>Button</code>s 
     * to control the editor.
     *
     */
    protected void addButtonPanel()
    {
	if(buttonPanel != null)
	    add(buttonPanel);
    }

    /**
     * Returns true if this editor allows modification of
     * accompanying model.
     * It is part of <code>ObjectEditor</code> interface.
     *
     * @return true if this editor allows modification of
     * accompanying model, or false if not
     *
     * @see org.nomencurator.editor.ObjectEditor.isEditable()
     */
    public boolean isEditable()
    {
	return editable && getModel().isEditable();
    }

    /**
     * Sets this editor editable according to given parameter <code>editable</code>.
     * It is part of <code>ObjectEditor</code> interface.
     *
     * @param editable true to allow modification of
     * accompanying model by this edior, or false to disable
     *
     * @see org.nomencurator.editor.ObjectEditor.setEditable()
     */
    public void setEditable(boolean editable)
    {
	if(this.editable == editable)
	    return;

	this.editable = editable;
	changeButtonState();
    }

    /**
     * Returns <code>ObjectEditModel</code> under edition
     * by this editor.
     * It is part of <code>ObjectEditor</code> interface.
     *
     * @return ObjectEditModel under edition
     *
     * @see org.nomencurator.editor.ObjectEditor.getModel()
     */
    public ObjectEditModel getModel()
    {
	return objectEditModel;
    }


    /**
     * Sets given <code>model</code> to be edited by this editor
     * It is part of <code>ObjectEditor</code> interface.
     *
     * @param model <code>ObjectEditModel</code> to be edited
     *
     * @see org.nomencurator.editor.ObjectEditor.setModel()
     */
    public synchronized void setModel(ObjectEditModel model)
    {
	if(objectEditModel == model)
	    return;

	if(model == null)
	    model = createObjectEditModel();

	if(objectEditModel != null) {
	    objectEditModel.deleteObserver(this);
	    if(isModified()) {
		objectEditModel.saveAttributes();
		Object o = objectEditModel.getObject();
		NamedObjectBroker.getInstance().putUnsavedObject((NamedObject)o);
		NamedObjectBroker.getInstance().putModel((NamedObjectEditModel)objectEditModel);
		setModified(false);
	    }
	}

	//model.loadAttributes();
	objectEditModel = model;
	setComponents(objectEditModel);

	setModified(false);

	if(objectEditModel != null) {
	    //objectEditModel.loadAttributes();
	    objectEditModel.addObserver(this);
	}
	changeButtonState();
    }

    /**
     * Setups components to use given <CODE>model</CODE>.
     * This method is called from setModel(ObjectEditModel)
     * internally.
     */
    protected abstract void setComponents(ObjectEditModel model);

    public void update(Observable obs, Object arg)
    {
	setModified(true);
    }

    /**
     * A utility method to get <code>Object</code> under edition
     *
     * @return Object under edition
     *
     * @see org.nomencurator.editor.model.ObjectEditModel.getObject()
     */
    public Object getObject()
    {
	return objectEditModel.getObject();
    }

    /**
     * A utility method to set <code>object</code> as
     * target of edition by accompanying <code>ObjectEditModel</code>
     *
     * @return Object under edition
     *
     * @see org.nomencurator.editor.model.ObjectEditModel.setObject()
     */
    public void setObject(Object object)
    {
	objectEditModel.setObject(object);
    }

    /**
     * A utility method to load attributes of <code>Object</code>
     * to the model
     *
     * @see ObjectEditModel.loadAttributes()
     */
    public void loadAttributes()
    {
	objectEditModel.loadAttributes();
    }

    /**
     * A utility method to save attributes to <code>Object</code>
     * from the model
     *
     * @see ObjectEditModel.saveAttributes()
     */
    public synchronized void saveAttributes()
    {
	if(modified == false)
	    return;

	objectEditModel.saveAttributes();

	modified = false;
    }

    /**
     * Invoked when mouse clicked on <code>Button</code>s
     *
     * @param event <code>MouseEvent</code> to be handled
     */
    public void mouseClicked(MouseEvent event) {
	int clicks = event.getClickCount();
	if(clicks == 2) {
	    mouseDoubleClicked(event);
	    return;
	}

	Object source = event.getSource();
	if (!(source instanceof Button))
	    return;

	Button button = (Button) source;
	if (button.isEnabled()) {
	    if (button == okButton) {
//		okButtonClicked();
	    } else if (button == newButton) {
		newButtonClicked();
	    } else if (button == copyButton) {
		copyButtonClicked();
	    } else if (button == clearButton) {
//		clearButtonClicked();
	    } else if (button == cancelButton) {
//		cancelButtonClicked();
	    } else if (button == markButton) {
		markButtonClicked();
	    }
	    
	}
    }

    /**
     * Invoked when mouse double clicked
     *
     * @param event <code>MouseEvent</code> to be handled
     */
    public void mouseDoubleClicked(MouseEvent e)
    {
    }

    /**
     * Invoked when mouse released on <code>Button</code>s
     *
     * @param event <code>MouseEvent</code> to be handled
     */
    public void mousePressed(MouseEvent event)
    {
	//do nothing
    }

    /**
     * Invoked when mouse released on <code>Button</code>s
     *
     * @param event <code>MouseEvent</code> to be handled
     */
    public void mouseReleased(MouseEvent event)
    {
	//do nothing
    }

    /**
     * Invoked when mouse enters to <code>Button</code>s area
     *
     * @param event <code>MouseEvent</code> to be handled
     */
    public void mouseEntered(MouseEvent event)
    {
	//do nothing
    }

    /**
     * Invoked when mouse exits from <code>Button</code>s area
     *
     * @param event <code>MouseEvent</code> to be handled
     */
    public void mouseExited(MouseEvent event)
    {
	//do nothing
    }

    public static void repaint(Component component)
    {
	Dimension d = component.getSize();
	Graphics g = component.getGraphics();
	g.setColor(component.getBackground());
	g.fillRect(0, 0, d.width, d.height);
	component.paintAll(g);
    }
    
    /**
     * Invoked when the mouse has been clicked on the OK button.
     */
//    public void okButtonClicked()
//    {
//	//	if (modified) {
//	objectEditModel.saveAttributes();
//	jp.kyasu.awt.Dialog.message("saved attributes.");
//	System.out.println("saved attributes.");
//	//	}
//	setEditable(false);
//	changeButtonState();
//    }

    /**
     * Invoked when the mouse has been clicked on the New button.
     */
    public void newButtonClicked()
    {
	previousModel = objectEditModel;
	//	objectEditModel = createObjectEditModel();
	setEditable(true);
	setModel(createObjectEditModel());
	changeButtonState();
    }
    
    /**
     * Invoked when the mouse has been clicked on the Copy button.
     */
    public void copyButtonClicked()
    {
	previousModel = objectEditModel;
	objectEditModel = (ObjectEditModel) previousModel.clone();
	objectEditModel.setEditable(true);
	setEditable(true);
	changeButtonState();
    }

    /**
     * Invoked when the mouse has been clicked on the Clear button.
     */
//    public void clearButtonClicked()
//    {
//	objectEditModel.clear();
//    }
    
    /**
     * Invoked when the mouse has been clicked on the Cancel button.
     */
//    public void cancelButtonClicked()
//    {
//	if (previousModel != null) {
//	    objectEditModel = previousModel;
//	    previousModel = null;
//	}
//	modified = false;
//	setEditable(false);
//	changeButtonState();
//    }

    /**
     * Invoked when the mouse has been clicked on the Cancel button.
     */
    public void markButtonClicked() {
	taxoNote = getTaxoNote();
	if(taxoNote == null)
	    return;

	ShoppingListDialog shoppingList
	    = taxoNote.getShoppingListDialog();
	NamedObjectEditModel model = (NamedObjectEditModel)getModel();
	if (!shoppingList.contains(model))
	    shoppingList.addModel(model);
	markButton.setEnabled(false);
	shoppingList.show();
    }
    
    /**
     * Change button state.
     */
    public void changeButtonState() {
	if (okButton != null) 
	    okButton.setEnabled(false);
	    //okButton.setEnabled(editable);
	if (newButton != null) 
	    newButton.setEnabled(true);
	    //newButton.setEnabled(!editable);
	if (copyButton != null)
	    copyButton.setEnabled(true);
	    //copyButton.setEnabled(!editable);
	if (clearButton != null)
	    //clearButton.setEnabled(false);
	    clearButton.setEnabled(editable);
	if (cancelButton != null)
	    cancelButton.setEnabled(false);
	    //cancelButton.setEnabled(editable);
	if (markButton != null) {
	    TaxoNote taxoNote = getTaxoNote();
	    if(taxoNote == null) {
		markButton.setEnabled(false);
	    }
	    else {
		ShoppingListDialog shoppingList = taxoNote.getShoppingListDialog();
		markButton.setEnabled(shoppingList == null || !shoppingList.contains((NamedObjectEditModel)getModel()));
	    }

	}
	    
    }

    /**
     * Adds the specified item listener to receive item events from this list.
     * @param listener the item listener.
     */
    public synchronized void addItemListener(ItemListener listener)
    {
	itemListeners = AWTEventMulticaster.add(itemListeners, listener);
	enableEvents(0); // mark newEventsOnly
    }

    /**
     * Removes the specified item listener so that it no longer receives
     * item events from this list.
     * @param listener the item listener.
     */
    public synchronized void removeItemListener(ItemListener listener)
    {
	itemListeners = AWTEventMulticaster.remove(itemListeners, listener);
    }

    public TabbedPane getTabbedPane() {
	return tabbedPane;
    }
    
    public void setTabbedPane(TabbedPane tabbedPane) {
	this.tabbedPane = tabbedPane;
    }
    
    
    public TaxoNote getTaxoNote() {
	return taxoNote;
    }
    
    public void setTaxoNote(TaxoNote taxoNote) {
	this.taxoNote = taxoNote;
    }

    /**
     * Adds a <code>listener</code> as <code>ForcusListener</code>
     * to this.
     *
     * @param listener <code>FocusListener</code> to be added
     */
    public synchronized void addFocusdListener(FocusListener listener)
    {
	if(forcusListeners == null)
	    forcusListeners = new Vector();

	forcusListeners.addElement(listener);
    }

    /**
     * Removes a <code>listener</code> as <code>ForcusListener</code>
     * to this.
     *
     * @param listener <code>FocusListener</code> to be removed
     */
    public void removeFocusdListener(FocusListener listener)
    {
	if(forcusListeners == null)
	    return;

	forcusListeners.remove(listener);
    }

    /**
     * Invoked when a component gained focus
     * @see java.awt.event.FocusListener
     */
    public void focusGained(FocusEvent event)
    {
	if(event.isTemporary() ||
	   forcusListeners == null)
	    return;

	Object[] listeners;
	synchronized (forcusListeners) {
	    listeners = forcusListeners.toArray();
	}
	for(int i = 0; i < listeners.length; i++) {
	    ((FocusListener)listeners[i]).focusGained(event);
	}
    }

    /**
     * Invoked when a component lost focus
     * @see java.awt.event.FocusListener
     */
    public void focusLost(FocusEvent event)
    {
	if(event.isTemporary() ||
	   forcusListeners == null)
	    return;

	Object[] listeners;
	synchronized (forcusListeners) {
	    listeners = forcusListeners.toArray();
	}
	for(int i = 0; i < listeners.length; i++) {
	    ((FocusListener)listeners[i]).focusLost(event);
	}
    }

    //HTML handling utilities taken from HTMLEditor of KFC
    /**
     * Sets the html text of this text component.
     * @see #getHTMLText()
     */
    public void setHTMLText(TextComponent component, HTMLText htmlText)
    {
	if (htmlText == null)
	    throw new NullPointerException();
	component.setBackground(htmlText.getBackgroundColor());
	component.setForeground(htmlText.getTextColor());

	component.setRichText(htmlText);
    }

    /**
     * Returns the html text of this text component.
     * @see #setHTMLText(jp.kyasu.graphics.html.HTMLText)
     */
    public HTMLText getHTMLText(TextComponent component)
    {
	return (HTMLText)component.getRichText();
    }

    public void textValueChanged(TextEvent e)
    {
	setModified(true);
    }

    public void itemStateChanged(ItemEvent e)
    {
	setModified(true);
    }

    public void actionPerformed(ActionEvent e)
    {
	setModified(true);
    }


    public void setModified(boolean modified)
    {
	//	new Error().printStackTrace();
	if(this.modified == modified)
	    return;
	
	this.modified = modified;
    }
	
    public boolean isModified()
    {
    	return modified;
    };
}
