/*
 * PreferenceView.java: 
 *
 * Copyright (c) 2002 Hiroyuki Okawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: PreferenceView.java,v 1.9 2002/08/13 02:34:32 ryo Exp $
 * $Log: PreferenceView.java,v $
 * Revision 1.9  2002/08/13 02:34:32  ryo
 * change call function to PreferenceManager.savePreference()
 *
 * Revision 1.8  2002/07/16 07:29:10  ryo
 * fix NullPointerException bug when new record added
 *
 * Revision 1.7  2002/07/16 07:09:26  ryo
 * modify connection setting list control
 *
 * Revision 1.6  2002/07/12 16:53:54  ryo
 * add database setting panel
 *
 * Revision 1.5  2002/06/05 12:32:02  ryo
 * set the top directory of localdb to NamedObjectBroker when OK button is pressed
 *
 * Revision 1.4  2002/06/04 13:47:23  ryo
 * add localdb path panel and remove priority panel
 *
 * Revision 1.3  2002/03/28 06:32:03  okawa
 * add copyright
 *
 * Revision 1.2  2002/03/06 13:39:04  okawa
 * add search / register priority
 *
 * Revision 1.1  2002/02/14 01:59:15  okawa
 * initial import into CVS
 *
 * Revision 1.1  2002/02/01 10:05:11  okawa
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import jp.kyasu.awt.Frame;
import jp.kyasu.awt.Panel;
import jp.kyasu.awt.Label;
import jp.kyasu.awt.Choice;
import jp.kyasu.awt.BorderedPanel;
import jp.kyasu.awt.TabbedPane;
import jp.kyasu.awt.Dialog;
import jp.kyasu.graphics.VTitledPaneBorder;
import jp.kyasu.awt.Button;
import jp.kyasu.awt.TextField;

import java.awt.Checkbox;
import java.awt.CheckboxGroup;
import java.awt.FileDialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.util.Vector;
import java.util.Enumeration;

import org.nomencurator.Preference;
import org.nomencurator.controller.PreferenceManager;
import org.nomencurator.broker.NamedObjectBroker;
import org.nomencurator.editor.model.ConnectionEditModel;
import org.nomencurator.editor.model.ConnectionListModel;

public class PreferenceView
extends Panel
implements MouseListener, ActionListener
{

	protected Preference preference;
	protected Choice higherChoice;
	protected Choice lowerChoice;
	protected TableList connectionList;
	protected ConnectionListModel connectionListModel;

	protected Dialog dialog;
	protected Button okButton;
	protected Button cancelButton;

	protected GridBagLayout gbl = new GridBagLayout();
	protected GridBagConstraints c = new GridBagConstraints();

    static public final String L_PREFERENCE = "Preference";
    static public final String L_SEARCH_GENERATION = "Search Generation";
    static public final String L_CONNECTION_LIST   = "Connection List";
    static public final String L_HIGHER = "Higher";
    static public final String L_LOWER  = "Lower";
    static public final String L_OK     = "OK";
    static public final String L_CANCEL = "Cancel";

    static public final String A_OK     = "A_OK";
    static public final String A_CANCEL = "A_Cancel";

	/** <CODE>MenuItem</CODE> for add record */
	protected MenuItem addRecordMenuItem;

	/** Menu title string for add record */
    protected static String addRecordTitle = "add record";

	/** <CODE>MenuItem</CODE> for delete record */
	protected MenuItem deleteRecordMenuItem;

	/** Menu title string for delete record */
    protected static String deleteRecordTitle = "delete record";

	/**
	 * Constructor
	 */
    public PreferenceView() {
        super();
        preference = PreferenceManager.getInstance();
        initialize();
    }

	/**
     * Initialize this panel
	 */
	protected void initialize() {

		setLayout(gbl);

	    BorderedPanel generationPanel = new BorderedPanel(new VTitledPaneBorder(L_SEARCH_GENERATION));
	    BorderedPanel connectionListPanel = new BorderedPanel(new VTitledPaneBorder(L_CONNECTION_LIST));
	    generationPanel.setLayout(gbl);
	    connectionListPanel.setLayout(gbl);

	    higherChoice = new Choice();
		higherChoice.add("0");
		higherChoice.add("1");

	    lowerChoice = new Choice();
		lowerChoice.add("0");
		lowerChoice.add("1");
		lowerChoice.add("2");
		lowerChoice.add("3");

		c.gridx = 0;
		c.gridy = 0;
		c.fill = c.BOTH;
		c.anchor = c.CENTER;
		c.insets = new Insets(2, 20, 2, 10);
	    generationPanel.add(new Label(L_HIGHER), c);

		c.gridx = 1;
		c.insets = new Insets(2, 10, 2, 20);
	    generationPanel.add(new Label(L_LOWER), c);

		c.gridx = 0;
		c.gridy = 1;
		c.insets = new Insets(2, 20, 2, 10);
	    generationPanel.add(higherChoice, c);

		c.gridx = 1;
		c.insets = new Insets(2, 10, 2, 20);
	    generationPanel.add(lowerChoice, c);

		connectionListModel = new ConnectionListModel();
		connectionList = new TableList(
			connectionListModel,
			10,
			connectionListModel.getColumnTitles());
		connectionList.setSelectionMode(TableList.SHIFT_MULTIPLE_SELECTIONS);
		connectionList.addMouseListener(this);
		connectionList.setScrollbarDisplayPolicy(jp.kyasu.awt.ScrollPanel.SCROLLBARS_ALWAYS);

		c.gridx = 0;
		c.gridy = 0;
		c.fill = c.BOTH;
		c.anchor = c.CENTER;
		c.insets = new Insets(2, 2, 2, 2);
		connectionListPanel.add(connectionList, c);

		c.gridx = 0;
		c.gridy = 0;
		add(generationPanel, c);
		c.gridx = 0;
		c.gridy = 1;
		add(connectionListPanel, c);
		
		// create popup menu
		PopupMenu pm = connectionList.getPopupMenu();
		if (pm.getItemCount() > 0)
			pm.addSeparator();
		addRecordMenuItem = new MenuItem(addRecordTitle);
		addRecordMenuItem.addActionListener(this);
		pm.add(addRecordMenuItem);
		deleteRecordMenuItem = new MenuItem(deleteRecordTitle);
		deleteRecordMenuItem.addActionListener(this);
		pm.add(deleteRecordMenuItem);
		connectionList.setPopupMenu(pm);
    }

	/**
     * show
	 */
	public void show(Frame frame) {

	    higherChoice.select((new Integer(preference.getHigherCount())).toString());
	    lowerChoice.select((new Integer(preference.getLowerCount())).toString());
		connectionListModel.setAbstractConnections(NamedObjectBroker.getInstance().getConnectionInterfaces());

	    dialog = new Dialog(frame, L_PREFERENCE);
	    Panel mainPanel = new Panel();
	    mainPanel.setLayout(gbl);

	    TabbedPane pane = new TabbedPane(TabbedPane.TOP);
	    Panel dummyPanel = new Panel();
	    pane.addTab(L_PREFERENCE, this);
	    pane.addTab("", dummyPanel);
	    //pane.setSize(300, 250);

		c.gridx = 0;
		c.gridy = 0;
		c.fill = c.BOTH;
		c.anchor = c.CENTER;
		c.insets = new Insets(2, 2, 2, 2);
	    mainPanel.add(pane, c);

        okButton = new Button(L_OK);
        cancelButton = new Button(L_CANCEL);
	    okButton.addActionListener(new PreferenceViewActionListener(this));
	    okButton.setActionCommand(A_OK);
	    cancelButton.addActionListener(new PreferenceViewActionListener(this));
	    cancelButton.setActionCommand(A_CANCEL);

		Panel buttonPanel = new Panel();
	    buttonPanel.add(okButton);
	    buttonPanel.add(cancelButton);

		c.gridy++;
	    mainPanel.add(buttonPanel, c);
        dialog.add(mainPanel);
        dialog.pack();
	    dialog.show(); 
    }

	/**
     * setting Preference
	 */
	protected void ok() {
	    preference.setHigherCount((new Integer(higherChoice.getSelectedItem())).intValue());
	    preference.setLowerCount((new Integer(lowerChoice.getSelectedItem())).intValue());
		Vector v = connectionListModel.getModels();
		Vector connections = new Vector();
		for (Enumeration e = v.elements(); e.hasMoreElements(); )
			connections.addElement(((ConnectionEditModel)e.nextElement()).getModel());
		NamedObjectBroker.getInstance().setConnectionInterfaces(connections);
	    PreferenceManager.savePreference();
    }

	public void mouseClicked(MouseEvent e) {
		int clickCount = e.getClickCount();
		if(clickCount == 2) {
			mouseDoubleClicked(e);
			return;
		}
	}

	/**
	 * Invoked when mouse double clicked
	 * @param e <code>MouseEvent</code> to be handled
	 */
	public void mouseDoubleClicked(MouseEvent e) {
		Object source = e.getSource();
		if (source == connectionList) {
			// edit model
			ConnectionListModel listModel = (ConnectionListModel)connectionList.getModel();
			ConnectionEditModel model = (ConnectionEditModel)listModel.getSelectedModel();
			int index = listModel.getSelectedIndex();
			if (e.getY() >= connectionList.getLastBaseLine() || model == null) {
				// create new model
				index = -1;
				model = new ConnectionEditModel();
			}
			ConnectionSettingDialog dlg =
				new ConnectionSettingDialog(getFrame(), model);
			dlg.show();
			if (dlg.isOK()) {
				model = dlg.getModel();
				listModel.addModel(index, model);
				listModel.update(model);
			}
		}
	}

	public void mousePressed(MouseEvent e) {
	}

	public void mouseReleased(MouseEvent e) {
	}

	public void mouseEntered(MouseEvent e) {
	}

	public void mouseExited(MouseEvent e) {
	}

	/**
	 * Invoked when an action occurs.
	 * @param e <code>ActionEvent</code> to be handled
	 * @see java.awt.event.ActionListener
	 */
	public void actionPerformed(ActionEvent e) {
		Object source = e.getSource();
		ConnectionListModel listModel = (ConnectionListModel)connectionList.getModel();
		if (source == addRecordMenuItem) {
			// add new record
			ConnectionEditModel model = new ConnectionEditModel();
			ConnectionSettingDialog dlg =
				new ConnectionSettingDialog(getFrame(), model);
			dlg.show();
			if (dlg.isOK()) {
				model = dlg.getModel();
				connectionList.invalidate();
				listModel.addModel(model);
				connectionList.validate();
			}
		} else if (source == deleteRecordMenuItem) {
			// delete existent record
			ConnectionEditModel model = (ConnectionEditModel)listModel.getSelectedModel();
			if (model != null) {
				listModel.removeModel(model);
			}
		}
		if(connectionList.getRows() == listModel.getItemCount() + 1) {
		    connectionList.invalidate();
		    listModel.enableDummyModel(true);
		    connectionList.validate();
		}
	}

}

/**
 * An ActionListener for the dialog utilities.
 */
class PreferenceViewActionListener implements ActionListener {
    PreferenceView view;

    PreferenceViewActionListener(PreferenceView view) {
	    this.view      = view;
    }

    public void actionPerformed(ActionEvent e) {
	    if(e.getActionCommand().equals(view.A_OK)){
	        view.ok();
	    }
	    view.dialog.setVisible(false);
    }
}
