/*
 * RankChoice.java:  a Choice to select Rank
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: RankChoice.java,v 1.15 2002/10/11 03:50:21 nozomi Exp $
 * $Log: RankChoice.java,v $
 * Revision 1.15  2002/10/11 03:50:21  nozomi
 * override select(int)
 *
 * Revision 1.14  2002/08/21 14:52:10  nozomi
 * use KFC's Choice
 *
 * Revision 1.13  2002/08/21 06:41:54  nozomi
 * use Java AWT's Choice
 *
 * Revision 1.12  2002/08/13 02:34:57  ryo
 * call PreferenceManager.savePreference() after user added rank
 *
 * Revision 1.11  2002/07/31 06:57:19  nozomi
 * accept orphan rank
 *
 * Revision 1.10  2002/06/06 14:58:25  nozomi
 * modify equiv. handling
 *
 * Revision 1.9  2002/06/06 09:09:12  nozomi
 * handle equivalent
 *
 * Revision 1.8  2002/06/05 11:14:14  nozomi
 * modify layout
 *
 * Revision 1.7  2002/05/20 02:02:24  nozomi
 * fix a mistake in a comment
 *
 * Revision 1.6  2002/05/14 04:52:30  nozomi
 * fix recursive ItemEvent handling
 *
 * Revision 1.5  2002/05/09 23:57:10  nozomi
 * fix dialog related bug
 *
 * Revision 1.4  2002/05/08 20:43:32  nozomi
 * fix infra-rank support
 *
 * Revision 1.3  2002/05/01 03:08:30  nozomi
 * support infra-rank
 *
 * Revision 1.2  2002/04/16 03:53:37  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.1  2002/03/26 03:56:38  nozomi
 * use RankChoice to handle additional ranks
 *
 */

package org.nomencurator.editor;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Insets;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;

import jp.kyasu.awt.AWTResources;
import jp.kyasu.awt.Button;
import jp.kyasu.awt.Checkbox;
import jp.kyasu.awt.CheckboxGroup;
import jp.kyasu.awt.Choice;
import jp.kyasu.awt.Dialog;
import jp.kyasu.awt.Label;
import jp.kyasu.awt.TextField;

import org.nomencurator.Rank;

import org.nomencurator.controller.PreferenceManager;
import org.nomencurator.editor.model.RankListModel;

/**
 * <code>Panel</code> to view and modify an <code>NameUsage</code>
 * accompanying to an <code>NameUsageEditModel</code>.
 *
 * @see org.nomencurator.editor.model.NameUsageEditModel
 * @see org.nomencurator.NameUsage
 *
 * @version 	11 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class RankChoice
    extends Choice
{
    /** A short-cut to <code>RankListModel</code> used by this <code>Choice</code> */
    protected RankListModel rankListModel;

    /** <code>RankDialog</code> handling new <code>Rank</code> */
    protected RankDialog rankDialog;

    /**
     * Creates an <code>RankChoice</code> using specified <code>RankListModel</code>
     */
    public RankChoice()
    {
	this(new RankListModel());
    }

    /**
     * Creates an <code>RankChoice</code> using specified <code>RankListModel</code>
     */
    public RankChoice(RankListModel model)
    {
	this(model, true);
    }

    /**
     * Creates an <code>RankChoice</code> using specified <code>RankListModel</code>
     */
    public RankChoice(RankListModel model, boolean enableDialog)
    {
	super(model);
	rankListModel = model;
	if(enableDialog)
	    addItemListener(this);
    }

    public void addItemListener(ItemListener listener)
    {
	super.addItemListener(listener);
    }

    /**
     * Invoked when an item's state has been changed.
     * @see java.awt.event.ItemListener
     */
    public void itemStateChanged(ItemEvent e)
    {
	if(e.getSource() != this) {
	    super.itemStateChanged(e);
	    return;
	}

	if(getSelectedIndex() != rankListModel.getItemCount() - 1)
	    return;

	rankDialog = new RankDialog(this);
	rankDialog.pack();
	rankDialog.show();
    }

    public RankListModel getRankListModel()
    {
	return rankListModel;
    }

    /**
     * Selects the item at the specified index in the scrolling list.
     * @param index the position of the item to select.
     */
    public synchronized void select(int index) {
	listController.select(index, -1, false, false);
    }
}

class RankDialog
    extends Dialog
    implements ActionListener, ItemListener, TextListener
{
    TextField  rankNameField;
    RankChoice higherRank;
    RankChoice lowerRank;
    RankChoice equivRank;
    Checkbox   higherCheckbox;
    Checkbox   lowerCheckbox;
    Checkbox   equivCheckbox;
    CheckboxGroup checkboxes;
    RankChoice owner;
    GridBagContainerAdaptor adaptor;
    GridBagConstraints constraints;
    BorderedPanel rankNamePanel;

    Button okButton;
    Button cancelButton;
    Panel buttonPanel;

    public RankDialog(RankChoice owner)
    {
	super(getFrame(owner), true);
	this.owner = owner;
	rankNameField   = new TextField(10);
	rankNameField.addTextListener(this);
	higherRank = new RankChoice(owner.getRankListModel(), false);
	lowerRank  = new RankChoice(owner.getRankListModel(), false);
	equivRank  = new RankChoice(owner.getRankListModel(), false);
	checkboxes = new CheckboxGroup();
	higherCheckbox = new Checkbox("higher", checkboxes, false /*true*/);
	higherCheckbox.addItemListener(this);
	lowerCheckbox  = new Checkbox("lower", checkboxes, false);
	lowerCheckbox.addItemListener(this);
	equivCheckbox  = new Checkbox("equiv.", false);
	equivCheckbox.addItemListener(this);
	equivRank.setEnabled(false/*equivCheckbox.getState()*/);
	//	lowerRank.setEnabled(false);
	disableAnyLink();	
	okButton = new Button(AWTResources.getResourceString("kfc.dialog.okLabel", "OK"));
	okButton.addActionListener(this);
	cancelButton = new Button(AWTResources.getResourceString("cancelLabel", "Cancel"));
	cancelButton.addActionListener(this);
	buttonPanel = new Panel(new FlowLayout(FlowLayout.RIGHT));
	buttonPanel.add(okButton);
	buttonPanel.add(cancelButton);

	adaptor = new GridBagContainerAdaptor(this);
	constraints = adaptor.getConstraints();

	Insets insets = constraints.insets;
	constraints.gridx = 0;
	constraints.gridy = 0;
	constraints.fill = GridBagConstraints.VERTICAL;
	constraints.gridheight = 1;
	constraints.anchor = GridBagConstraints.WEST;
	constraints.insets = new Insets(0, 6, 0, 0);
	add(new Label("rank name"), constraints);

	constraints.insets = insets;
	constraints.gridx++;
	add(higherCheckbox, constraints);
	constraints.insets = new Insets(0, 0, 0, 6);
	constraints.gridx++;
	add(higherRank, constraints);
	constraints.gridy++;

	constraints.insets = new Insets(0, 6, 0, 0);
	constraints.gridx = 0;
	add(rankNameField, constraints);
	constraints.gridx++;

	constraints.insets = insets;
	constraints.anchor = GridBagConstraints.WEST;
	add(equivCheckbox, constraints);
	constraints.gridx++;
	constraints.insets = new Insets(0, 0, 0, 6);
	add(equivRank, constraints);
	constraints.gridy++;

	constraints.insets = insets;
	constraints.gridx = 1;
	constraints.anchor = GridBagConstraints.WEST;
	add(lowerCheckbox, constraints);
	constraints.gridx++;
	constraints.insets = new Insets(0, 0, 0, 6);
	add(lowerRank, constraints);
	constraints.gridy++;

	constraints.insets = insets;
	constraints.gridx = 0;
	constraints.gridwidth = GridBagConstraints.REMAINDER;
	constraints.fill = GridBagConstraints.HORIZONTAL;
	add(buttonPanel, constraints);
	/*
	constraints.anchor = GridBagConstraints.EAST;
	add(okButton, constraints);
	constraints.gridx++;
	add(cancelButton, constraints);
	*/
    }

    public void show()
    {
	super.show();
	rankNameField.requestFocus();
    }

    public static Frame getFrame(Component component)
    {
	while(!(component instanceof Frame))
	    component = component.getParent();
	return (Frame)component;
    }

    public void itemStateChanged(ItemEvent e)
    {
	Object source = e.getSource();
	if(source == higherCheckbox) {
	    selectHigher();
	}
	else if(source == lowerCheckbox) {
	    selectLower();
	}
	else if(source == equivCheckbox) {
	    equivRank.setEnabled(equivCheckbox.getState());
	}

    }

    public void selectHigher()
    {
	allowPartialLink(true);		
	lowerRank.setEnabled(false);
	lowerCheckbox.setEnabled(false);
	higherCheckbox.setState(true);
	higherRank.setEnabled(true);
    }

    public void selectHigher(boolean higher)
    {
	//	allowPartialLink(true);	
	lowerRank.setEnabled(!higher);
	higherCheckbox.setState(higher);
	higherCheckbox.setEnabled(!higher);
	lowerCheckbox.setEnabled(!higher);
	lowerCheckbox.setState(!higher);
	
	higherRank.setEnabled(higher);
    }

    public void selectLower()
    {
	allowPartialLink(true);
	higherRank.setEnabled(false);
	higherCheckbox.setEnabled(false);
	lowerCheckbox.setState(true);
	lowerRank.setEnabled(true);
    }

    public void allowPartialLink(boolean allowance)
    {
	lowerCheckbox.setEnabled(allowance);
	higherCheckbox.setEnabled(allowance);
	equivCheckbox.setEnabled(!allowance);

	lowerCheckbox.setState(false);
	higherCheckbox.setState(false);
	equivCheckbox.setState(false);
	/*
	if(allowance)
	    selectHigher(higherCheckbox.getState());
	*/
    }


    public void allowFullLink()
    {
	allowFullLink(true);
    }

    public void allowFullLink(boolean allowance)
    {
	allowPartialLink(false);
	higherRank.setEnabled(allowance);
	lowerRank.setEnabled(allowance);
	equivRank.setEnabled(equivCheckbox.getState() & !allowance);
    }

    public void disableAnyLink()
    {
	allowFullLink(false);
    }

    /**
     * Invoked when the value of the text has changed.
     * @see java.awt.event.TextListener
     */
    public void textValueChanged(TextEvent e) {
	String rankName = rankNameField.getText().toLowerCase();
	int prefixLength = 0;
	RankChoice baseRank = null;
	String nextRankName = null;
	if(rankName.startsWith("super")) {
	    allowPartialLink(true);
	    selectLower();
	    prefixLength = 5;
	    baseRank = lowerRank;
	}
	else if (rankName.startsWith("sub")) {
	    allowPartialLink(true);
	    selectHigher();
	    prefixLength = 3;
	    baseRank = higherRank;
	}
	else if(rankName.startsWith("infra")) {
	    allowPartialLink(true);
	    selectHigher();
	    prefixLength = 5;
	    String baseRankName = rankName.substring(prefixLength);
	    Rank rank = Rank.get(baseRankName);
	    if(rank == null) {
		return;
	    }
	    nextRankName = new String("sub" + baseRankName);
	    if(Rank.get(nextRankName) == null) {
		rank = Rank.get(baseRankName);
		if(rank != null) {
		    Rank subRank = new Rank(nextRankName, null, null);
		    subRank.setHigher(rank);
		    Rank.put(subRank);
		}
	    }
	    baseRank = higherRank;
	}
	else {
	    disableAnyLink();
	    higherRank.select(higherRank.getItemCount() - 1);
	    lowerRank.select(lowerRank.getItemCount() - 1);
	    return;
	}

	if(baseRank != null) {
	    RankChoice otherRank = null;
	    if(nextRankName == null) {
		nextRankName = rankName.substring(prefixLength);
	    }

	    allowPartialLink(true);
	
	    if(baseRank == higherRank) {
		selectHigher();
		otherRank = lowerRank;
	    }
	    else if(baseRank == lowerRank) {
		selectLower();
		otherRank = higherRank;
	    }	

	    otherRank.select(otherRank.getItemCount() - 1);
	    baseRank.select(nextRankName);
	}
    }

    /**
     * Invoked when the button has been triggered by the controller.
     */
    public void actionPerformed(ActionEvent e)
    {
	Object source = e.getSource();
	hide();
	if(source == okButton) {
	    Rank rank = new Rank(rankNameField.getText(), null, null);
	    if(higherCheckbox.isEnabled()) {
		if(higherCheckbox.getState()) {
		    rank.setHigher(Rank.get(higherRank.getSelectedItem()));
		}
		else {
		    rank.setLower(Rank.get(lowerRank.getSelectedItem()));
		}
	    }
	    else if(equivCheckbox.getState()) {
		rank.setEquivalent(Rank.get(equivRank.getSelectedItem()));
	    }
	    else {
		rank.setHigher(rank);
		rank.setLower(rank);
	    }
	    Rank.put(rank);
	    owner.select(rank.getName());
	    PreferenceManager.savePreference();
	}
	else if (source == cancelButton) {
	}
    }

}

