/*
 * TableList.java:  a TableList implementing EditorSelector
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: TableList.java,v 1.18 2002/10/11 03:42:54 nozomi Exp $
 * $Log: TableList.java,v $
 * Revision 1.18  2002/10/11 03:42:54  nozomi
 * add up(int), down(int) and moveModel(int, int) methods)
 *
 * Revision 1.17  2002/09/19 06:59:34  nozomi
 * add/removeModel() methods return boolean; add removeAll()
 *
 * Revision 1.16  2002/09/06 02:11:46  nozomi
 * add getNamedObjects()
 *
 * Revision 1.15  2002/09/05 11:12:45  t.okada
 * add Observer interface
 *
 * Revision 1.14  2002/09/03 04:02:02  nozomi
 * modify initialize() method
 *
 * Revision 1.13  2002/05/25 16:55:19  nozomi
 * fix model selection bug
 *
 * Revision 1.12  2002/05/22 22:15:56  nozomi
 * fix ModelSelctor bug
 *
 * Revision 1.11  2002/05/20 02:01:47  nozomi
 * Re-implemented to utilise ModelSelector
 *
 * Revision 1.10  2002/05/17 17:40:04  nozomi
 * ModelSelectors moved to org.nomencurator.editor.model
 *
 * Revision 1.9  2002/05/14 10:06:40  ryo
 * modify specification of getLastBaseLine()
 *
 * Revision 1.8  2002/03/06 00:09:50  nozomi
 * uses own view
 *
 * Revision 1.7  2002/03/05 01:49:10  nozomi
 * add getLastBaseLine()
 *
 * Revision 1.6  2002/03/03 23:49:50  nozomi
 * unuse ModeSelector
 *
 * Revision 1.5  2002/02/26 16:31:26  nozomi
 * switched to ModelSelector
 *
 * Revision 1.4  2002/02/16 15:58:10  nozomi
 * Accept TableModel as subinterface of TextListModel
 *
 * Revision 1.3  2002/02/08 01:06:37  nozomi
 * Accepts TableModel as model
 *
 * Revision 1.2  2002/02/02 19:35:26  nozomi
 * Chaned from ObjectEditor to NamedObjectEditor to implement EditorSelector
 *
 * Revision 1.1  2002/01/29 10:02:30  nozomi
 * initial import into CVS
 *
 */

package org.nomencurator.editor;

import java.awt.Point;

import java.util.Enumeration;
import java.util.Vector;
import java.util.Observer;
import java.util.Observable;

import jp.kyasu.awt.Button;
import jp.kyasu.awt.TextListModel;

import jp.kyasu.awt.text.TableListView;

import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.RichTextStyle;

import jp.kyasu.graphics.text.TextLineInfo;
import jp.kyasu.graphics.text.TextPositionInfo;

import org.nomencurator.editor.model.DefaultTextListModel;
import org.nomencurator.editor.model.ModelSelectAdaptor;
import org.nomencurator.editor.model.ModelSelector;
import org.nomencurator.editor.model.NamedObjectEditModel;
//import org.nomencurator.editor.model.NamedObjectListModel;
import org.nomencurator.editor.model.TableModel;
//import org.nomencurator.editor.model.TextListModel;



/**
 * A <code>TableList</code> enables slection of editors
 * 
 * @see jp.kyasu.awt.TableList
 * @see org.nomencurator.editor.Modelselector
 *
 * @version 	11 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class TableList
    extends jp.kyasu.awt.TableList
    implements ModelSelector, Observer
{
    /**
     * model of this list
     */
    protected TextListModel textListModel;

    /**
     * <code>ModelSelector</code> linking a line in this <code>List</code>
     * and models
     */
    protected ModelSelector modelSelector;

    /**
     * Constructs an empty table list with the specified column titles.
     * @param columnTitles the column titles.
     */
    public TableList(String columnTitles[])
    {
	this(0, columnTitles);
    }

    /**
     * Constructs an empty table list with the specified number of visual
     * lines and column titles.
     * @param rows         the number of visual lines in the table list.
     * @param columnTitles the column titles.
     */
    public TableList(int rows, String columnTitles[])
    {
	this(rows, columnTitles, null);
    }

    /**
     * Constructs an empty table list with the specified number of visual
     * lines, column titles, and widths of the columns.
     * @param rows         the number of visual lines in the table list.
     * @param columnTitles the column titles.
     * @param columnWidths  the widths of the columns, or null.
     * @see jp.kyasu.awt.Label#LEFT
     * @see jp.kyasu.awt.Label#CENTER
     * @see jp.kyasu.awt.Label#RIGHT
     */
    public TableList(int rows, String columnTitles[], int columnWidths[])
    {
	this(rows, columnTitles, columnWidths, DEFAULT_TABLE_STYLE);
    }

    /**
     * Constructs an empty table list with the specified column titles,
     * widths of the columns, and the rich text style of the table list.
     * @param rows          the number of visual lines in the table list.
     * @param columnTitles  the column titles.
     * @param columnWidths  the widths of the columns, or null.
     * @param richTextStyle the rich text style of the table list.
     * @see jp.kyasu.awt.Label#LEFT
     * @see jp.kyasu.awt.Label#CENTER
     * @see jp.kyasu.awt.Label#RIGHT
     */
    public TableList(int rows, String columnTitles[], int columnWidths[],
		     RichTextStyle richTextStyle)
    {
	this(rows, columnTitles, null, columnWidths, richTextStyle);
    }

    /**
     * Constructs an empty table list with the specified column titles,
     * alignments of the column titles, widths of the columns, and the
     * rich text style of the table list.
     * @param rows          the number of visual lines in the table list.
     * @param columnTitles  the column titles.
     * @param columnAligns  the alignments of the column titles, or null.
     * @param columnWidths  the widths of the columns, or null.
     * @param richTextStyle the rich text style of the table list.
     * @see jp.kyasu.awt.Label#LEFT
     * @see jp.kyasu.awt.Label#CENTER
     * @see jp.kyasu.awt.Label#RIGHT
     */
    public TableList(int rows, String columnTitles[], int columnAligns[],
		     int columnWidths[], RichTextStyle richTextStyle)
    {
	this(rows, columnTitles, columnAligns,
	      columnWidths, richTextStyle, false);
    }

    /**
     * Constructs an empty table list with the specified column titles,
     * alignments of the column titles, widths of the columns, and the
     * rich text style of the table list.
     * @param rows          the number of visual lines in the table list.
     * @param columnTitles  the column titles.
     * @param columnAligns  the alignments of the column titles, or null.
     * @param columnWidths  the widths of the columns, or null.
     * @param richTextStyle the rich text style of the table list.
     * @see jp.kyasu.awt.Label#LEFT
     * @see jp.kyasu.awt.Label#CENTER
     * @see jp.kyasu.awt.Label#RIGHT
     */
    public TableList(int rows, String columnTitles[], int columnAligns[],
		     int columnWidths[], RichTextStyle richTextStyle,
		     boolean enable)
    {
	super(rows, columnTitles, columnAligns,
	      columnWidths, richTextStyle);
	createModelSelector(getModel(), enable);
    }

    /**
     * Constructs a table list with the specified model, number of visual
     * lines, and column titles.
     * @param model        the text list model.
     * @param rows         the number of visual lines in the table list.
     */
    public TableList(TableModel model, int rows)
    {
	this(model, rows, model.getTitle(), model.getAlignments());
    }

    /**
     * Constructs a table list with the specified model, number of visual
     * lines, and column titles.
     * @param model        the text list model.
     * @param rows         the number of visual lines in the table list.
     * @param columnAligns the alignments of the column titles, or null.
     * @see jp.kyasu.awt.Label#LEFT
     * @see jp.kyasu.awt.Label#CENTER
     * @see jp.kyasu.awt.Label#RIGHT
     */
    public TableList(TableModel model, int rows, int columnAligns[])
    {
	this(model, rows, model.getTitle(), columnAligns);
    }


    /**
     * Constructs a table list with the specified model, number of visual
     * lines, and column titles.
     * @param model        the text list model.
     * @param rows         the number of visual lines in the table list.
     * @param columnTitles the column titles.
     */
    public TableList(TextListModel model, int rows, String columnTitles[])
    {
	this(model, rows, columnTitles, null);
    }

    /**
     * Constructs a table list with the specified model, number of visual
     * lines, column titles, and alignments of the column titles.
     * @param model        the text list model.
     * @param rows         the number of visual lines in the table list.
     * @param columnTitles the column titles.
     * @param columnAligns the alignments of the column titles, or null.
     * @see jp.kyasu.awt.Label#LEFT
     * @see jp.kyasu.awt.Label#CENTER
     * @see jp.kyasu.awt.Label#RIGHT
     */
    public TableList(TextListModel model, int rows,
		    String columnTitles[], int columnAligns[])
    {
	this(model, rows,
	     createColumnButtons(
		columnTitles, columnAligns,
		model.getTextList().getRichTextStyle().getTextStyle()));
    }

    /**
     * Constructs a table list with the specified model, number of visual
     * lines, and column buttons.
     * @param model   the text list model.
     * @param rows    the number of visual lines in the table list.
     * @param buttons the buttons for column titles.
     */
    public TableList(TextListModel model, int rows, Button buttons[])
    {
	this(model, rows, buttons, false);
    }

    /**
     * Constructs a table list with the specified model, number of visual
     * lines, and column buttons.
     * @param model   the text list model.
     * @param rows    the number of visual lines in the table list.
     * @param buttons the buttons for column titles.
     */
    public TableList(TextListModel model, int rows, Button buttons[],
		     boolean enable)
    {
	super(model, rows, buttons);
	createModelSelector(model, enable);
    }

    /**
     * Initializes the table list with the specified model, number of visual
     * lines, and column buttons.
     * @param model   the text list model.
     * @param rows    the number of visual lines in the table list.
     * @param buttons the buttons for column titles.
     */
    protected void initialize(jp.kyasu.awt.TextListModel model, int rows, Button buttons[])
    {
	super.initialize(createTableListView(model, buttons), rows);
	textListModel = (TextListModel)model;
    }

    /**
     * Create appropriate <code>ModelSelecotr</code> handling
     * <code>textListModel</code> with dummy model addition mode
     * status of <code>enable</code>.
     *
     * @param textListModel <code>TextListModel</code> to be handled by the <code>ModelSelector</code>
     * @param enable true to enable dummy model addition of the <code>ModelSelector</code>
     */
    protected void createModelSelector(TextListModel textListModel, boolean enable)
    {
	if(modelSelector != null &&
	   modelSelector.getModel() == textListModel)
	    return;

	if(textListModel instanceof ModelSelector) {
	    modelSelector = (ModelSelector)textListModel;
	    //	    modelSelector.enableDummyModel(enable);
	}
	else {
	    modelSelector = new ModelSelectAdaptor(textListModel, enable);
	}
    }

    /**
     * Returns <code>TableListView</code> constructed with <code>model</code>
     * and <code>buttons</code>.  The subclasses may override this method.
     *
     * @param model   the text list model.
     * @param buttons the buttons for column titles.
     *
     * @return TableListView constructed with <code>model</code> and <code>buttons</code>
     *
     */
    protected TableListView createTableListView(jp.kyasu.awt.TextListModel model, Button buttons[])
    {
	return new org.nomencurator.editor.view.TableListView(model, buttons);
    }

    /**
     * Returns <code>TextListModel</code> with specified
     * <code>columnWidths</code> and <code>richTextStyle</code>
     *
     * @param columnWidths array of int representing width of columns
     * @param richTextStyle <code>RichTextStyle</code> to be applied to the model
     *
     * @return TextListModel created with specified parameters
     */
    protected jp.kyasu.awt.TextListModel createTextListModel(int columnWidths[],
							     RichTextStyle richTextStyle)
    {
	return new DefaultTextListModel(columnWidths.length,
					columnWidths,
					richTextStyle);
    }

    /**
     * Returns <code>Vector</code> of <code>NamedObjectEditModel</code>
     * accompanying to the <code>ListModel</code>
     */
    public Vector getModels()
    {
	//	return textListModel.getModels();
	return modelSelector.getModels();
    }

    /**
     * Sets <code>models</code> as <code>NamedObjectEditModel</code> list
     * accompanying to the <code>ListModel</code>
     */
    public void setModels(Vector models)
    {
	//textListModel.setModels(models);
	modelSelector.setModels(models);
    }

    /**
     * Adds <code>model</code> to <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be added to this selector
     */
    public boolean addModel(NamedObjectEditModel model)
    {
	//	textListModel.addModel(model);
	return modelSelector.addModel(model);
    }

    /**
     * Inserts <code>model</code> at <code>index</code> of <code>NamedObjectEditModel</code> list.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     */
    public boolean addModel(int index, NamedObjectEditModel model)
    {
	//	textListModel.addModel(index, model);
	return modelSelector.addModel(index, model);
    }

    /**
     * Inserts <code>model</code> at <code>index</code> of model list
     * with its representation given by <code>items</code>.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     * @param items  the representaition of the model
     */
    public boolean addModel(int index, NamedObjectEditModel model, Object[][] items)
    {
	return modelSelector.addModel(index, model, items);
    }

    /**
     * Remove all <code>NamedObjectEditModel</code>s from <code>NamedObjectEditModel</code> list
     *
     */
    public void removeAllModels()
    {
	modelSelector.removeAllModels();
    }

    /**
     * Adds <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be removed from this selector
     */
    public boolean removeModel(NamedObjectEditModel model)
    {
	//	textListModel.removeModel(model);
	return modelSelector.removeModel(model);
    }

    /**
     * Returns <code>NamedObjectEditModel</code> at <code>index</code>
     *
     * @param index  where <code>model</code> is
     * @return <code>NamedObjectEditModel</code> at  <code>index</code>
     */
    public NamedObjectEditModel getModel(int index)
    {
	return modelSelector.getModel(index);
    }

    /**
     * Returns <code>NamedObjectEditModel</code> at <code>index</code>
     *
     * @param index  where <code>model</code> is
     * @return <code>NamedObjectEditModel</code> at  <code>index</code>
     */
    /*
    public NamedObjectEditModel getModel(int index)
    {
	//	return textListModel.getModel(index);
	return modelSelector.getModel(index);
    }
    */

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel</code> selected
     */
    public NamedObjectEditModel getSelectedModel()
    {
	//	return textListModel.getSelectedModel();
	return modelSelector.getSelectedModel();
    }

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel[]</code> selected
     */
    public NamedObjectEditModel[] getSelectedModels()
    {
	//	return textListModel.getSelectedModels();
	return modelSelector.getSelectedModels();
    }

    /**
     * Returns index of selected <code>NamedObjectEditModel</code>
     *
     * @return int index of selected <code>NamedObjectEditModel</code>
     */
    public int getSelectedIndex()
    {
	//	return textListModel.getSelectedIndex();
	return modelSelector.getSelectedIndex();
    }

    /**
     * Returns index of <code>model</code>
     *
     * @param model <code>NamedObjectEditModel</code> to be searched
     * @return int index of <code>model</code>
     */
    public int indexOf(NamedObjectEditModel model)
    {
	return modelSelector.indexOf(model);
    }

    /**
     * Updates summary text of specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    /*
    public void update(NamedObjectEditModel model)
    {
	//	textListModel.update(model);
	modelSelector.update(model);
    }
    */

    /**
     * Updates summary text of model specified by <code>index</code>
     *
     * @param index index of <code>NamedObjectEditModel</code> to be updated
     */
    /*
    public void update(int index)
    {
	//	textListModel.update(index);
	modelSelector.update(index);
    }
    */

    /**
     * Returns position of the last line
     *
     * @return position of the last line
     */
    public int getLastBaseLine()
    {
	TextLineInfo line  = listView.getVisibleEnd();
	return (line.y + line.lineHeight);
//	return (line.y == 0)? 0: (line.y + line.lineHeight);
    }

    /**
     * Enables or desables automatic addition of a dummy
     * <code>NamedObjectEditModel</code> to the list according to
     * value of <code>enable</code>
     *
     * @param enable <code>boolean</code> enables 
     * automatic addition of a dummy <code>NamedObjectEditModel</code>
     * if true or disables if false
     */
    public void enableDummyModel(boolean enable)
    {
	modelSelector.enableDummyModel(enable);
    }

    /**
     * Returns true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     *
     * @return true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     */
    public boolean isDummyModelEnabled()
    {
	return modelSelector.isDummyModelEnabled();
    }

    /**
     * Returns a dummy <code>NamedObjectEditModel</code> instance for double-clicking
     * operation.
     *
     * @return null to be considered as a dummy <code>NamedObjectEditModel</code>
     */
    public NamedObjectEditModel getDummyModel()
    {
	return modelSelector.getDummyModel();
    }

    /**
     * Returns Text array representing empty summary 
     * can be used to represent a dummy <code>NamedObjectEditModel</code>.
     *
     * @return Text[][] representing a summary of an empty <code>NamedObjectEditModel</code>.
     */
    public Text[][] getEmptySummary()
    {
	return modelSelector.getEmptySummary();
    }

    /**
     * Selects <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModel(NamedObjectEditModel model)
    {
	modelSelector.selectModel(model);
    }

    /**
     * Selects an <code>NamedObjectEditModel</code> at <code>index</code> of list
     *
     * @param index of <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModelAt(int index)
    {
	modelSelector.selectModelAt(index);
    }

    /**
     * Updates summary text of all specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update()
    {
	modelSelector.update();
    }


    /**
     * Updates summary text of specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update(NamedObjectEditModel model)
    {
	modelSelector.update(model);
    }

    /**
     * Updates summary text of model specified by <code>index</code>
     *
     * @param index index of <code>NamedObjectEditModel</code> to be updated
     */
    public void update(int index)
    {
	modelSelector.update(index);
    }

    /**
     * Sets <code>textListModel</code> as a summarized representation
     * <code>NamedObjectEditModel</code> handled by this selector
     *
     * @param textListModel <code>TextListModel</code> representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     */
    public void setModel(TextListModel textListModel)
    {
	super.setModel(textListModel);
	createModelSelector(textListModel, modelSelector.isDummyModelEnabled());
    }

    /**
     * Returns true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     *
     * @return true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     */
    public boolean contains(NamedObjectEditModel model)
    {
	return modelSelector.contains(model);
    }

    /**
     * Returns <code>Vector</code> of <code>NamedObjectEditModel</code>
     * accompanying to the <code>ListModel</code>
     */
    public Vector getNamedObjects()
    {
	Enumeration e = modelSelector.getModels().elements();
	if(!e.hasMoreElements()) {
	    return null;
	}

	Vector objects = new Vector();
	while(e.hasMoreElements()) {
	    objects.addElement(((NamedObjectEditModel)e.nextElement()).getNamedObject());
	}
	return objects;
    }

    public void update(Observable o, Object arg) {
    	this.update();
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one above
     */
    public boolean up(int index)
    {
	return modelSelector.up(index);
    }


    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one below
     */
    public boolean down(int index)
    {
	return modelSelector.down(index);
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>source</CODE> to <CODE>destination</CODE>
     */
    public boolean moveModel(int source, int destination)
    {
	return modelSelector.moveModel(source, destination);
    }
}

