/*
 *
 * TextConverter.java:  a Text converter used by editors of 
 * the Navigator for the Nomencurator.
 *
 * Copyright (c) 2000 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: TextConverter.java,v 1.1.1.1 2002/01/16 12:33:33 ryo Exp $
 *	$Log: TextConverter.java,v $
 *	Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 *	initial import into CVS
 *	
 *	
 *
 */

package org.nomencurator.editor;

import java.awt.Color;

import java.util.Enumeration;
import java.util.Hashtable;

import jp.kyasu.graphics.ExtendedFont;
import jp.kyasu.graphics.FontModifier;
import jp.kyasu.graphics.ModTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextStyle;
import jp.kyasu.graphics.html.HTMLStyle;
import jp.kyasu.graphics.html.HTMLText;

public class TextConverter {

    public static String TextToXML(Text text)
    {
	return TextToHTML(text); //!
    }

    public static Text XMLToText(String xml)
    {
	return new Text(xml);
    }

    public static String TextToHTML(Text text)
    {
	return HTMLToHTMLString(new HTMLText(text, new HTMLStyle(text.getTextStyleAt(0))));
    }

    public static String HTMLToHTMLString(HTMLText html)
    {
	Text text = html.getText();
	return TextToHTML(0, text.length(), text, html.getTextStyleAt(0), html);
    }

    public static String TextToTeX(Text text)
    {
	return TextToBibTeX(text, new Hashtable());
    }

    public static Text TexToText(String tex)
    {
	return new Text(tex);
    }

    public static String TextToBibTeX(Text text, Hashtable acronyms)
    {
	return TextToBibTeX(0, text.length(), text, text.getTextStyleAt(0), acronyms);
    }

    public static Text TextToCanonicalText(Text text)
    {
	return TextToCanonicalText(text, new Hashtable());
    }

    public static Text TextToCanonicalText(Text text, Hashtable acronyms)
    {
	return TextToCanonicalText(text, acronyms, 0, text.length());
    }

    public static Text TextToCanonicalText(Text text, Hashtable acronyms, int begin, int end)
    {
	Text textBuffer = new Text("");
	int index = begin;
	while (index < end) {
	    TextStyle textStyle = text.getTextStyleAt(index);
	    int runEnd = index + text.getRunLengthAt(index);
	    if (runEnd > end) runEnd = end;
	    textBuffer.append(new Text(getCanonicalTextData(index, runEnd, text, acronyms),
				       text.getTextStyleAt(index)));
	    index = runEnd;
	}
	return textBuffer;
    }

    /**
     * Returns the specified range of the specified text as a HTML with
     * the specified base text style.
     * <p>
     * It was taken from <code>jp.kyasu.graphics.html.HTMWriter.java</code> of the KFC
     * written by Kazuki YASUMATSU.
     *
     * @param begin     the beginning index of the text, inclusive.
     * @param end       the ending index of the text, exclusive.
     * @param text      the text.
     * @param baseStyle the base text style.
     */

    public static String TextToHTML(int begin, int end, Text text, TextStyle baseStyle, HTMLText html)
    {
	StringBuffer buffer = new StringBuffer("");
	boolean firstData = true;
	int index = begin;
	while (index < end) {
	    TextStyle textStyle = text.getTextStyleAt(index);
	    int runEnd = index + text.getRunLengthAt(index);
	    if (runEnd > end) runEnd = end;
	    String[] tags = getXMLFontTags(baseStyle, textStyle, html);
	    buffer.append(tags[0]);
	    buffer.append(getHTMLData(index, runEnd, text, firstData));
	    buffer.append(tags[1]);
	    index = runEnd;
	    firstData = false;
	}
	return buffer.substring(0);
    }

    /**
     * Returns a pair of the XML/HTML font tag with a difference of the specified current
     * text style from the specified base text style.
     *<p>
     * It was taken from <code>jp.kyasu.graphics.html.HTMLWriter.java</code>.
     *
     * @param  baseStyle the base text style.
     * @param  textStyle the current text style.
     * @return an array of font specifiers
     */

    protected static String[] getXMLFontTags(TextStyle baseStyle, TextStyle textStyle, HTMLText html)
    {
	StringBuffer[] buffers = new StringBuffer[2];
	buffers[0] = new StringBuffer();
	buffers[1] = new StringBuffer();
	HTMLStyle htmlStyle = html.getHTMLStyle();

	boolean isA = (textStyle.getClickableTextAction() != null);
	if (isA) {
	    buffers[1].insert(0, "</A>");
	    buffers[0].append("<A href=\"");
	    buffers[0].append(textStyle.getClickableTextAction().getActionCommand());
	    buffers[0].append("\">");
	}

	if (!(textStyle instanceof ModTextStyle)) {
	    ExtendedFont exFont = textStyle.getExtendedFont();
	    if (exFont.getName().equalsIgnoreCase("Monospaced")) {
		buffers[1].insert(0, "</TT>");
		buffers[0].append("<TT>");
	    }
	    if (exFont.isBold()) {
		buffers[1].insert(0, "</B>");
		buffers[0].append("<B>");
	    }
	    if (exFont.isItalic()) {
		buffers[1].insert(0, "</I>");
		buffers[0].append("<I>");
	    }
	    if (!isA && exFont.isUnderline()) {
		buffers[1].insert(0, "</U>");
		buffers[0].append("<U>");
	    }
	    Color color = exFont.getColor();

	    if (color != null && isA && color.equals(html.getLinkColor())) {
		color = null;
	    }

	    int htmlFontIndex = 0;
	    if (exFont.getSize() != baseStyle.getFont().getSize()) {
		htmlFontIndex =
		    htmlStyle.getHTMLFontIndex(
			exFont.getSize() - baseStyle.getFont().getSize());
	    }

	    if (color != null || htmlFontIndex != 0) {
		buffers[1].insert(0, "</FONT>");
		buffers[0].append("<FONT");
		if (color != null) {
		    appendHTMLcolor(buffers[0], color);
		}

		if (htmlFontIndex != 0) {
		    buffers[0].append(" size=" + htmlFontIndex);
		}
		buffers[0].append(">");
	    }
	}

	else {
	    FontModifier modifier = ((ModTextStyle)textStyle).getFontModifier();
	    if (modifier != null) {
		Object value;
		if ((value = modifier.get(FontModifier.NAME)) != null &&
		    (value instanceof String))
		    {
			String name = (String)value;
			if (name.equalsIgnoreCase("Monospaced")) {
			    buffers[1].insert(0, "</TT>");
			    buffers[0].append("<TT>");
			}
		    }
		if ((value = modifier.get(FontModifier.BOLD)) != null &&
		    (value instanceof Boolean))
		    {
			boolean isBold = ((Boolean)value).booleanValue();
			if (isBold) {
			    buffers[1].insert(0, "</B>");
			    buffers[0].append("<B>");
			}
		    }
		if ((value = modifier.get(FontModifier.ITALIC)) != null &&
		    (value instanceof Boolean))
		    {
			boolean isItalic = ((Boolean)value).booleanValue();
			if (isItalic) {
			    buffers[1].insert(0, "</I>");
			    buffers[0].append("<I>");
			}
		    }
		if ((value = modifier.get(FontModifier.UNDERLINE)) != null &&
		    (value instanceof Boolean))
		    {
			boolean isUnderline = ((Boolean)value).booleanValue();
			if (isUnderline && !isA) {
			    buffers[1].insert(0, "<U>");
			    buffers[0].append("<U>");
			}
		    }
		Color color = null;
		int htmlFontIndex = 0;
		if ((value = modifier.get(FontModifier.COLOR)) != null &&
		    (value instanceof Color))
		    {
			color = (Color)value;
			if (isA && color.equals(html.getLinkColor())) {
			    color = null;
			}
		    }
		if ((value = modifier.get(FontModifier.SIZE_DIFF)) != null &&
		    (value instanceof Integer))
		    {
			int size = ((Integer)value).intValue();
			htmlFontIndex = htmlStyle.getHTMLFontIndex(size);
		    }
		if (color != null || htmlFontIndex != 0) {
		    buffers[1].insert(0, "</FONT>");
		    buffers[0].append("<FONT");
		    if (color != null) {
			appendHTMLcolor(buffers[0], color);
		    }
		    if (htmlFontIndex != 0) {
			buffers[0].append(" size=" + htmlFontIndex);
		    }
		    buffers[0].append(">");
		}
	    }
	}

	return new String[]{buffers[0].substring(0), buffers[1].substring(0)};
    }

    /**
     * Writes the specified range of the specified text as a HTML data.
     *
     * @param begin     the beginning index of the text, inclusive.
     * @param end       the ending index of the text, exclusive.
     * @param text      the text.
     * @param firstData true if the data written is a first data in the
     *                  paragraph.
     */

    protected static String getHTMLData(int begin, int end, Text text, boolean firstData)
    {
	StringBuffer buffer = new StringBuffer("");
	boolean inPreFormatted = false;

	char lastChar = (firstData ? ' ' : 'a');
	for (int i = begin; i < end; i++) {
	    char c = text.getChar(i);
	    switch (c) {
	    case '<':
		buffer.append("&lt;");
		break;
	    case '>':
		buffer.append("&gt;");
		break;
	    case '&':
		buffer.append("&amp;");
		break;
	    case ' ':
		if (inPreFormatted || lastChar != ' ') {
		    buffer.append(' ');
		}
		else {
		    buffer.append("&nbsp;");
		    c = 'a'; // lastChar = 'a';
		}
		break;
	    case Text.LINE_SEPARATOR_CHAR:
		break;
	    case Text.LINE_BREAK_CHAR:
		if (inPreFormatted) {
		    buffer.append("\n");
		}
		else {
		    buffer.append("<BR>");
		}
		break;
	    case Text.ATTACHMENT_CHAR:
		//buffer.appendTextAttachment(text.getAttachmentAt(i));
		break;
	    default:
		buffer.append(c);
		break;
	    }
	    lastChar = c;
	}

	return buffer.substring(0);
    }

    public static String TextToTeX(int begin, int end, Text text, TextStyle baseStyle)
    {
	return TextToBibTeX(begin, end, text, baseStyle, new Hashtable());
    }

    /**
     * Returns the specified range of the specified text as a HTML with
     * the specified base text style.
     * <p>
     * It was taken from <code>jp.kyasu.graphics.html.HTMWriter.java</code> of the KFC
     * written by Kazuki YASUMATSU.
     *
     * @param begin     the beginning index of the text, inclusive.
     * @param end       the ending index of the text, exclusive.
     * @param text      the text.
     * @param baseStyle the base text style.
     */
    public static String TextToBibTeX(int begin, int end, Text text, TextStyle baseStyle, Hashtable acronyms)
    {
	StringBuffer buffer = new StringBuffer("");
	int index = begin;
	while (index < end) {
	    TextStyle textStyle = text.getTextStyleAt(index);
	    int runEnd = index + text.getRunLengthAt(index);
	    if (runEnd > end) runEnd = end;
	    String[] tags = getTeXFontTags(baseStyle, textStyle);
	    String body = getBibTeXData(index, runEnd, text, acronyms);
	    boolean acronym = false;
	    if(!acronyms.isEmpty()){
		Enumeration e = acronyms.keys();
		while(e.hasMoreElements()){
		    if(body.indexOf((String)e.nextElement()) != -1)
			acronym |= true;
		}
	    }
	    if(tags[0].length() != 0){
		if(acronym)
		    buffer.append("{");
		buffer.append(tags[0]);
	    }
	    buffer.append(body);
	    if(tags[1].length() != 0){
		buffer.append(tags[1]);
		if(acronym)
		    buffer.append("}");
	    }
	    index = runEnd;
	}
	return buffer.substring(0);
    }

    private static String[] scaler = {
	    "\\tiny", "\\scriptsize", "\\footnotesize", "\\small",
	    "\\normalsize",
	    "\\large", "\\Large", "\\LARGE", "\\huge", "\\Huge"
	};
    protected static String getTeXscaler(int scaling) {

	if(scaling < -4)
	    scaling = -4;
	if(scaling > 5)
	    scaling = 5;
	return scaler[scaling + 4];
    }

    /**
     * Returns a pair of the TeX font tag with a difference of the specified current
     * text style from the specified base text style.
     *<p>
     * It was taken from <code>jp.kyasu.graphics.html.HTMLWriter.java</code> and
     * adapted to TeX.
     *
     * @param  baseStyle the base text style.
     * @param  textStyle the current text style.
     * @return an array of font specifiers
     */
    protected static String[] getTeXFontTags(TextStyle baseStyle, TextStyle textStyle)
    {
	StringBuffer[] buffers = new StringBuffer[2];
	buffers[0] = new StringBuffer();
	buffers[1] = new StringBuffer();

	boolean isA = (textStyle.getClickableTextAction() != null);
	if (!(textStyle instanceof ModTextStyle)) {
	    ExtendedFont exFont = textStyle.getExtendedFont();
	    if (exFont.getSize() != baseStyle.getFont().getSize()) {
		buffers[0].append(getTeXscaler((exFont.getSize() - baseStyle.getFont().getSize())/2));
		buffers[1].append("}");
	    }
	    if (exFont.isBold()) {
		buffers[0].append("{\\bf ");
		buffers[1].append("}");
	    }
	    if (exFont.isItalic()) {
		buffers[0].append("{\\it ");
		buffers[1].append("}");
	    }
	    if (exFont.isSuperscript()) {
		buffers[0].append("$^{");
		buffers[1].append("}$");
	    }
	    if (exFont.isSubscript()) {
		buffers[0].append("$_{");
		buffers[1].append("}$");
	    }
	    if (!isA && exFont.isUnderline()) {
		buffers[0].append("\\underline{");
		buffers[1].insert(0, "}");
	    }
	}

	else {
	    FontModifier modifier = ((ModTextStyle)textStyle).getFontModifier();
	    if (modifier != null) {
		Object value;
		if ((value = modifier.get(FontModifier.BOLD)) != null &&
		    (value instanceof Boolean))
		    {
			boolean isBold = ((Boolean)value).booleanValue();
			if (isBold) {
			    buffers[0].append("{\\bf ");
			    buffers[1].insert(0, "}");
			}
		    }
		if ((value = modifier.get(FontModifier.ITALIC)) != null &&
		    (value instanceof Boolean))
		    {
			boolean isItalic = ((Boolean)value).booleanValue();
			if (isItalic) {
			    buffers[0].append("{\\it ");
			    buffers[1].insert(0, "}");
			}
		    }
		if ((value = modifier.get(FontModifier.UNDERLINE)) != null &&
		    (value instanceof Boolean))
		    {
			boolean isUnderline = ((Boolean)value).booleanValue();
			if (isUnderline && !isA) {
			    buffers[0].append("\\underline{");
			    buffers[1].insert(0, "}");
			}
		    }
		if ((value = modifier.get(FontModifier.SIZE_DIFF)) != null &&
		    (value instanceof Integer))
		    {
			buffers[0].append(getTeXscaler((((Integer)value).intValue() - baseStyle.getFont().getSize())/2));
			buffers[1].append("}");
		    }
	    }
	}

	return new String[]{buffers[0].substring(0), buffers[1].substring(0)};
    }

    protected static String getTeXData(int begin, int end, Text text)
    {
	return getBibTeXData(begin, end, text, new Hashtable());
    }

    /**
     * Writes the specified range of the specified text as a HTML data.
     *
     * @param begin     the beginning index of the text, inclusive.
     * @param end       the ending index of the text, exclusive.
     * @param text      the text.
     * @param firstData true if the data written is a first data in the
     *                  paragraph.
     */
    protected static String getBibTeXData(int begin, int end, Text text, Hashtable acronyms)
    {
	StringBuffer buffer = new StringBuffer("");
	StringBuffer word   = new StringBuffer("");

	for (int i = begin; i < end; i++) {
	    char c = text.getChar(i);
	    switch (c) {
	    case ' ':
	    case Text.LINE_SEPARATOR_CHAR:
	    case Text.LINE_BREAK_CHAR:
		boolean acronym = acronyms.contains(word.substring(0));
		if(acronym){
		    word.insert(0, "{");
		    word.append("}");
		}
		buffer.append(word.substring(0));
		buffer.append(c);
		word.delete(0, word.length());
		break;
	    case '&':
		word.append("\\&");
		break;
	    default:
		word.append(c);
		break;
	    }
	}

	return buffer.substring(0);
    }

    protected static String getCanonicalTextData(int begin, int end, Text text, Hashtable acronyms)
    {
	StringBuffer buffer = new StringBuffer("");
	StringBuffer word   = new StringBuffer("");
	char lastSeparator  = '.';
	char headChar       = ' ';

	for (int i = begin; i < end; i++) {
	    char c = text.getChar(i);
	    switch (c) {
	    case ' ':
	    case ',':
	    case '.':
	    case ';':
	    case ':':
	    case '&':
	    case '?':
	    case '!':
	    case Text.LINE_SEPARATOR_CHAR:
	    case Text.LINE_BREAK_CHAR:
		boolean acronym = acronyms.contains(word.substring(0));
		if(acronym){
		    buffer.append(word.substring(0));
		}
		else{
		    switch (lastSeparator) {
		    case '.':
		    case ':':
		    case '?':
		    case '!':
			headChar = word.charAt(0);
			headChar = Character.isTitleCase(headChar)?
			    Character.toTitleCase(headChar):Character.toUpperCase(headChar);
			buffer.append(headChar);
			word.delete(0, 1);
		    default:
			break;
		    }
		    buffer.append(word.substring(0).toLowerCase());
		}

		buffer.append(c);

		switch(c) {
		case ' ':
		case Text.LINE_SEPARATOR_CHAR:
		case Text.LINE_BREAK_CHAR:
		    if(i != 0) {
			int j = i;
			do{
			    headChar = text.getChar(j--);
			}while(headChar == ' ' && j >= 0);
			switch(headChar) {
			case '.':
			case ':':
			case '?':
			case '!':
			    c = headChar;
			    break;
			}
		    }
		    break;
		default:
		    break;
		}
		lastSeparator = c;

		word.delete(0, word.length());
		break;
	    default:
		word.append(c);
		break;
	    }
	}

	return buffer.substring(0);
    }

    static protected void appendHTMLcolor(StringBuffer buffer, Color color)
    {
	buffer.append(" color=\"");
	buffer.append("#");
	buffer.append(color.getRed());
	buffer.append(color.getGreen());
	buffer.append(color.getBlue());
	buffer.append("\"");
    }
}

