/*
 * ObjectEditModelEvent.java:  an EventObject thrown when 
 * an ObjectEditModel was changed.
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ObjectEditModelEvent.java,v 1.2 2002/02/01 11:33:55 nozomi Exp $
 * $Log: ObjectEditModelEvent.java,v $
 * Revision 1.2  2002/02/01 11:33:55  nozomi
 * minor modifications
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.editor.event;

import java.util.EventObject;

import jp.kyasu.awt.event.TextModelEvent;

import org.nomencurator.editor.model.ObjectEditModel;

/**
 * The ObjectEditModel event is originated from a <code>ObjectEditModel</code> to
 * <code>ObjectEditModelListener</code>s.
 * It is very similar to <code>TextModelEvent</code> because edition by a 
 * <code>ObjectEdior</code> is very similar to <code>TextModel</code> edition.
 *
 * @see         org.nomencurator.editor.model.ObjectEditModel
 * @see         org.nomencurator.editor.event.ObjectEditModelListener
 * @see 	jp.kyasu.awt.TextModel
 * @see 	jp.kyasu.awt.event.TextModelEvent
 *
 * @version 	31 Jan 2002
 * @author 	Nozomi `James' Ytow
 */
public class ObjectEditModelEvent
    extends EventObject
{
    /**
     * EventObject wrapped by this event
     */
    protected Object sourceObject;

    /**
     * type of this event
     */
    protected int eventType;

    /**
     * The edit model is edited.
     */
    static public final int EDIT_MODEL_EDITED
	= TextModelEvent.TEXT_MODEL_EDITED;

    /**
     * The edit model is updated.
     */
    static public final int EDIT_MODEL_UPDATED
	= TextModelEvent.TEXT_MODEL_UPDATED;

    /**
     * The edit model is updated.
     */
    static public final int EDIT_OBJECT_UPDATED
	= EDIT_MODEL_UPDATED + 1;

    /**
     * Marks the first integer id for the range of text model event ids.
     */
    static public final int EDIT_MODEL_FIRST
	= EDIT_MODEL_EDITED;

    /**
     * Marks the last integer id for the range of text model event ids.
     */
    static public final int EDIT_MODEL_LAST
	= EDIT_OBJECT_UPDATED;


    /**
     * Constructs an edit model event of EDIT_MODEL_EDITED type with the 
     * specified edit model (event source) and source event.
     *
     * @param model       the text model (event source).
     * @param sourceObject the event or Object should be wrapped.
     */
    public ObjectEditModelEvent(ObjectEditModel model,
				     EventObject sourceObject)
    {
	this(model, EDIT_MODEL_EDITED, sourceObject);
    }

    /**
     * Constructs an edit model event of EDIT_MODEL_UPDATED type with the 
     * specified edit model (event source).
     *
     * @param model       the text model (event source).
     */
    public ObjectEditModelEvent(ObjectEditModel model)
    {
	this(model, EDIT_MODEL_UPDATED, null);
    }

    /**
     * Constructs an edit model event of EDIT_MODEL_EDITED type with the 
     * specified edit model (event source) and source event.
     *
     * @param model       the text model (event source).
     * @param sourceObject the event or Object should be wrapped.
     */
    public ObjectEditModelEvent(ObjectEditModel model,
				     Object sourceObject)
    {
	this(model, EDIT_OBJECT_UPDATED, sourceObject);
    }


    /**
     * Constructs an edit model event with the specified edit model (event
     * source), event type, and source event should be wrapped by this event.
     * The source event may be null.
     *
     * @param model      the text model (event source).
     * @param eventType  the event type.
     * @param event      the event should be wrapped.
     */
    protected ObjectEditModelEvent(ObjectEditModel model,
			      int eventType,
			      Object object)
    {
	super(model);
	this.eventType = eventType;
	sourceObject   = object;
    }

    /**
     * Returns the type of this edit model event.
     */
    public int getType()
    {
	return eventType;
    }

    /**
     * Returns the edit model (event source) of this edit model event.
     */
    public ObjectEditModel getModel()
    {
	return (ObjectEditModel)source;
    }

    /**
     * Returns the source event of this edit model event.
     */
    public Object getSourceObject()
    {
	return sourceObject;
    }

}
