/*
 * AppearanceEditModel.java:  an Appearance editor model 
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: AppearanceEditModel.java,v 1.40 2002/11/07 06:16:21 nozomi Exp $
 * $Log: AppearanceEditModel.java,v $
 * Revision 1.40  2002/11/07 06:16:21  nozomi
 * set Publication when Appearance created
 *
 * Revision 1.39  2002/10/08 21:37:40  nozomi
 * add line field to summary text array
 *
 * Revision 1.38  2002/10/08 00:40:29  nozomi
 * introduce listenSubModels()
 *
 * Revision 1.37  2002/09/27 19:55:59  nozomi
 * getSummaryTextArray() uses isNominal()
 *
 * Revision 1.36  2002/09/21 19:35:02  nozomi
 * set Publiation to the Appearance when EditModel was set
 *
 * Revision 1.35  2002/09/17 05:46:02  nozomi
 * re-organise initialisation methods
 *
 * Revision 1.34  2002/09/09 16:41:05  nozomi
 * implement updateSumamry()
 *
 * Revision 1.33  2002/09/06 17:01:47  ryo
 * link NamedObjects when *EditModel linked
 *
 * Revision 1.32  2002/09/06 11:43:10  nozomi
 * improve saveAttributes()
 *
 * Revision 1.31  2002/09/05 05:23:10  nozomi
 * support pseudo HTML
 *
 * Revision 1.30  2002/09/04 07:47:55  t.okada
 * duplication display bug of RelatedNames in AppearanceEditPanel is corrected.
 *
 * Revision 1.29  2002/08/27 08:52:58  t.okada
 * addTextModelListener is added.
 *
 * Revision 1.28  2002/08/27 02:50:19  nozomi
 * null pointer handling for publicationEditModel
 *
 * Revision 1.27  2002/08/20 13:14:14  t.okada
 * add lines column
 *
 * Revision 1.26  2002/07/03 07:57:13  nozomi
 * change default text model relating methods
 *
 * Revision 1.25  2002/06/21 23:22:57  nozomi
 * use RichText in TextComponents
 *
 * Revision 1.24  2002/05/27 14:18:55  ryo
 * remove NameUsageEditModel from nameUsageList when removeNameUsageEditModel() is called
 *
 * Revision 1.23  2002/05/26 21:06:25  nozomi
 * automatic dummy addition support
 *
 * Revision 1.22  2002/05/22 23:51:45  nozomi
 * add/remove EditModels mutually
 *
 * Revision 1.21  2002/05/17 19:11:55  ryo
 * modify function about Annotation
 *
 * Revision 1.20  2002/05/17 11:47:46  ryo
 * fix bug
 *
 * Revision 1.19  2002/05/17 11:27:37  ryo
 * fix bug
 *
 * Revision 1.18  2002/05/16 09:14:54  ryo
 * modify addAnnotation()
 *
 * Revision 1.17  2002/05/14 13:34:20  ryo
 * modify saveAttributes()
 *
 * Revision 1.16  2002/05/14 10:14:42  ryo
 * add getSummaryTextArray() and updateList()
 *
 * Revision 1.15  2002/05/10 13:45:17  ryo
 * add new constructor
 *
 * Revision 1.14  2002/05/10 12:20:34  t.okada
 * modify addNameUsageEditModel()
 *
 * Revision 1.13  2002/05/09 00:00:55  nozomi
 * improve publication handling
 *
 * Revision 1.12  2002/04/16 23:43:52  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.11  2002/04/16 03:53:38  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.10  2002/04/09 03:33:37  nozomi
 * accomodate to change of get method names
 *
 * Revision 1.9  2002/04/08 02:04:57  nozomi
 * add dummy methods to be compiled with NameUsage
 *
 * Revision 1.8  2002/04/01 07:06:58  nozomi
 * adapt to more protective Annotation
 *
 * Revision 1.7  2002/03/08 14:07:20  okawa
 * set appearance TextEditModel
 *
 * Revision 1.6  2002/02/28 17:11:02  nozomi
 * use TextListModel of the package
 *
 * Revision 1.5  2002/02/24 18:13:37  nozomi
 * re-implementation of setModel() to be used switching between models
 *
 * Revision 1.4  2002/02/21 01:55:24  okawa
 * modification due to the integration with TabbedPane, EditPanel
 *
 * Revision 1.3  2002/02/07 20:59:13  nozomi
 * Use AnnotationListModel
 *
 * Revision 1.2  2002/01/29 07:12:02  nozomi
 * Add some methods supporting the interface.
 * Title of Buttons of nameRecordList were introduced.
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Vector;

import jp.kyasu.awt.TextEditModel;
//import jp.kyasu.awt.DefaultTextEditModel;
//import jp.kyasu.awt.TextField;

import jp.kyasu.awt.event.TextModelEvent;

import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;

import jp.kyasu.graphics.html.HTMLText;

import org.nomencurator.Annotation;
import org.nomencurator.Appearance;
import org.nomencurator.NamedObject;
import org.nomencurator.NameUsage;
import org.nomencurator.Publication;

import org.nomencurator.broker.NamedObjectBroker;

import org.nomencurator.graphics.html.HTMLWriter;

/**
 * The model interface for an object that acts as a named object edit model
 *
 * @see 	org.nomencurator.editor.event.ObjectEditModelEvent;
 * @see 	org.nomencurator.editor.event.ObjectEditModelListener;
 *
 * @version 	08 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class AppearanceEditModel
    extends NamedObjectEditModel
    implements Cloneable
{
    /**
     * <CODE>PublicationEditModel</CODE> of the publication
     * containing this apperance
     */
    protected PublicationEditModel publicationEditModel;

    /**
     * <CODE>TextEditModel</CODE> retaining summary of the 
     * <CODE>PublicationEditModel</CODE> of the publication
     * containing this apperance
     */
    protected TextEditModel publication;

    /** <CODE>TextEditModel</CODE> of precie page where this appeared */
    protected TextEditModel pageTextModel;

    /** <CODE>TextEditModel</CODE> of precie lines where this appeared */
    protected TextEditModel linesTextModel;

    /** <CODE>TextModel</CODE> of the appearance itself */
    protected TextEditModel appearance;

    /**
     * <CODE>TextListModel</CODE> of <CODE>NameUsage</CODE> list
     * encoded in this appearance
     */
    protected NameUsageListModel nameUsageList;

    /**
     * <CODE>Vector</CODE> of <CODE>NameUsageEditModel</CODE>s
     * encoded in this appearance
     */
    //    protected Vector nameUsageEditModels;

    /**
     * <CODE>AnnotationListModel</CODE> of annotaions
     * encoded in this appearance
     */
    protected AnnotationListModel annotationsList;

    /**
     * a list of <CODE>AnnotationEditModel</CODE>s
     * encoded in this appearance
     */
    //    protected Vector annotationEditModels;

    protected static AppearanceEditModel template;

    protected boolean annotationsModified;

    protected boolean nameUsagesModified;

    /**
     * The constructor of AppearanceEditModel
     *
     */
    public AppearanceEditModel()
    {
	this(true);
    }

    /**
     * The constructor of AppearanceEditModel
     *
     * @param editable boolean determining whether the model is editable
     */
    public AppearanceEditModel(boolean editable)
    {
	this(new Appearance(), editable);
    }

    /**
     * The constructor of AppearanceEditModel
     *
     * @param object <CODE>Appearance</CODE> to be edited
     */
    public AppearanceEditModel(Appearance object)
    {
	this(object, true);
    }

    /**
     * The constructor of AppearanceEditModel
     *
     * @param object <CODE>Object</CODE> to be edited
     * @param editable boolean determining whether the model is editable
     */
    public AppearanceEditModel(Appearance object, boolean editable)
    {
	this(object, editable,
	     new PublicationEditModel());
    }

    /**
     * The constructor of AppearanceEditModel
     *
     * @param object <CODE>Object</CODE> to be edited
     * @param editable boolean determining whether the model is editable
     */
    public AppearanceEditModel(PublicationEditModel publicationEditModel)
    {
	this(new Appearance(publicationEditModel.getPublication()), true, publicationEditModel);
    }

    /**
     * The constructor of AppearanceEditModel
     *
     * @param object <CODE>Object</CODE> to be edited
     * @param editable boolean determining whether the model is editable
     * @param publicationEditModel <CODE>PameRecordEditModel</CODE> containing
     * appearance edited by this model
     */
    public AppearanceEditModel(Appearance object, 
			       boolean editable, 
			       PublicationEditModel publicationEditModel)
    {
	this(object, editable,
	     publicationEditModel, null, null);
    }

    /**
     * The constructor of AppearanceEditModel
     *
     * @param object <CODE>Object</CODE> to be edited
     * @param editable boolean determining whether the model is editable
     * @param publicationEditModel <CODE>PameRecordEditModel</CODE> containing
     * appearance edited by this model
     * @param nameUsageEditModels <CODE>Vector</CODE> of <CODE>NameUsageEditModel</CODE>s
     * encoded in the appearance
     * @param annotationEditModels <CODE>Vector</CODE> of <CODE>AnnotationEditModel</CODE>s
     * encoded in the appearance
     */
    public AppearanceEditModel(Appearance object, 
			       boolean editable,
			       PublicationEditModel publicationEditModel,
			       Vector nameUsageEditModels,
			       Vector annotationEditModels
			       )
    {
	super(object, editable);
    }

    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void createSubModels()
    {
	super.createSubModels();

	createPageTextModel();
	createLinesTextModel();
	createAppearanceTextModel();
	createNameUsageListTextModel();
	createAnnotationsTextModel();

	setAnnotationsModified(false);

	setNameUsagesModified(false);
   }

    /**
     * Listens submodles
     */
    protected void listenSubModels()
    {
	listenAppearanceTextModel();
	lisetnPageTextModel();
	lisetnLinesTextModel();
    }

    protected void createAppearanceTextModel()
    {
	appearance = getDefaultHTMLEditModel();
    }

    protected void listenAppearanceTextModel()
    {
	appearance.addTextModelListener(this);
    }

    protected void createPageTextModel()
    {
	pageTextModel = getDefaultTextEditModel();
    }

    protected void lisetnPageTextModel()
    {
	pageTextModel.addTextModelListener(this);
    }

    protected void createLinesTextModel()
    {
	linesTextModel = getDefaultTextEditModel();
    }

    protected void lisetnLinesTextModel()
    {
	linesTextModel.addTextModelListener(this);
    }

    protected void createNameUsageListTextModel()
    {
	nameUsageList = new NameUsageListModel(isEditable());
    }

    protected void createAnnotationsTextModel()
    {
	annotationsList = new AnnotationListModel(isEditable());
    }


    public boolean addNameUsageEditModel(NameUsageEditModel nameUsageEditModel)
    {
	if(nameUsageEditModel == null)
	    return false;

	return nameUsageList.addModel(nameUsageEditModel);
    }

    public boolean removeNameUsageEditModel(NameUsageEditModel nameUsageEditModel)
    {
    	return nameUsageList.removeModel(nameUsageEditModel);
    }


    public boolean addAnnotationEditModel(AnnotationEditModel annotationEditModel)
    {
	return annotationsList.addModel(annotationEditModel);
    }


    public boolean removeAnnotationEditModel(AnnotationEditModel annotationEditModel)
    {
	return annotationsList.removeModel(annotationEditModel);
    }


    /**
     * Gets <CODE>Appearance</CODE> to be edited by this model.
     *
     * @return Appearance under edition by this model
     * 
     */
    public Appearance getAppearance()
    {
	return (Appearance)getObject();
    }

    /**
     * Sets <CODE>apperance</CODE> to be edited by this model.
     *
     * @param appearance <CODE>Appearance</CODE> to be edited by this model.
     * 
     */
    public void setAppearance(Appearance appearance)
    {
	setObject(appearance);
    }

    /**
     * Gets <CODE>PublicationEditModel</CODE> of the publication
     * containing this apperance
     *
     */
    public PublicationEditModel getPublicationEditModel()
    {
	if(publicationEditModel == null)
	    setPublicationEditModel((PublicationEditModel)getEditModel(getAppearance().getPublication(), PublicationEditModel.getInstance()));
	return publicationEditModel;
    }


    /**
     * Sets <CODE>model</CODE> as <CODE>PublicationEditModel</CODE> of the publication
     * containing this apperance
     */
    public void setPublicationEditModel(PublicationEditModel model)
    {
	if(publicationEditModel == model &&
	   model != null)
	    return;

	if(publication != null)
	    publication.removeTextModelListener(this);

	if(publicationEditModel != null) {
	    publicationEditModel.removeAppearance(this);
	    publication = null;
	}

	publicationEditModel = model;

	if(publicationEditModel == null) {
	    publication = getDefaultTextEditModel();
	}
	else {
	    publicationEditModel.addAppearance(this);
	    publication = 
		publicationEditModel.getSummaryTextModel();
	    getAppearance().setPublication(publicationEditModel.getPublication());
	}

	publication.addTextModelListener(this);

	updateSummary();

	setChanged();
	notifyObservers(publicationEditModel);
    }

    /**
     * Returns <CODE>TextModel</CODE> representing contents of 
     * <CODE>publicationEditModel</CODE> containing this apperance
     */
    public TextEditModel getPublicationTextModel()
    {
	if(publication == null)
	    getPublicationEditModel();

	return publication;
    }

    /**
     * Returns <CODE>TextEditModel</CODE> of precie page where this appeared
     */
    public TextEditModel getPageTextModel()
    {
	return pageTextModel;
    }

    /**
     * Sets <CODE>model</CODE> as <CODE>TextEditModel</CODE> of the page
     */
    public void setPageTextModel(TextEditModel model)
    {
	pageTextModel = model;
    }

    /**
     * Sets <CODE>model</CODE> as <CODE>TextEditModel</CODE> of the lines
     */
    public void setLinesTextModel(TextEditModel model)
    {
	linesTextModel = model;
    }

    /**
     * Returns <CODE>TextEditModel</CODE> of precie page where this appeared
     */
    public TextEditModel getLinesTextModel()
    {
	return linesTextModel;
    }

    /**
     * Returns <CODE>TextModel</CODE> of the appearance itself
     */
    public TextEditModel getAppearanceTextModel()
    {
	return appearance;
    }

    /**
     * Returns <CODE>TextListModel</CODE> of <CODE>NameUsage</CODE> list
     * encoded in this appearance
     */
    public TextListModel getNameUsageList()
    {
	return nameUsageList;
    }

    /**
     * Returns <CODE>Vector</CODE> of <CODE>NameUsageEditModel</CODE>s
     * encoded in this appearance
     */
    public Vector getNameUsageEditModels()
    {
	return nameUsageList.getModels();
    }

    /**
     * Returns <CODE>AnnnotatinListModel</CODE> representing annotaions
     * encoded in this appearance
     */
    public AnnotationListModel getAnnotationList()
    {
	return annotationsList;
    }

    /**
     * Returns <CODE>Vector</cdoe> of <CODE>AnnotationEditModel</CODE>s
     * encoded in this appearance
     */
    public Vector annotationEditModels()
    {
	return annotationsList.getModels();
    }
    
    public void clear()
    {
	setPublicationEditModel(null);
	annotationsList.removeAllModels();
	nameUsageList.removeAllModels();
	pageTextModel.setRichText(new RichText("", DEFAULT_FIELD_STYLE));
	linesTextModel.setRichText(new RichText("", DEFAULT_FIELD_STYLE));
	appearance.setRichText(new HTMLText(new Text(), DEFAULT_HTML_STYLE));
    }

    /**
     * Returns clone of this object
     */
    public Object clone()
    {
	AppearanceEditModel model = 
	    new AppearanceEditModel(getAppearance(), isEditable(),
				    getPublicationEditModel());
	model.nameUsageList = 
	    (NameUsageListModel)nameUsageList.clone();
	model.annotationsList = 
	    (AnnotationListModel)annotationsList.clone();

	return model;
    }

    /**
     * Loades attributes from the <CODE>Appearance</CODE>
     */
    public void loadAttributes()
    {
	super.loadAttributes();

	Appearance app = getAppearance();
	if(app == null)
	    app = new Appearance();

	setString(pageTextModel, app.getPage());
	setString(linesTextModel, app.getLines());

	String as = app.getAppearance();
	if(as == null)
	    as = "";
	HTMLText html = htmlReader.toHTMLText(as);
	if(html.length() == 0)
	    html = new HTMLText(new Text(as), DEFAULT_HTML_STYLE);
	appearance.setRichText(html);

	nameUsageList.setObjects(app.getNameUsages());
	annotationsList.setObjects(app.getAnnotations());

	getPublicationEditModel();

    }

    /**
     * Saves attributes to the <CODE>Appearance</CODE>
     */
    public void saveAttributes()
    {
	Appearance app = getAppearance();
	app.setPage(getPageTextModel().getRichText().getText().toString());
	app.setLines(getLinesTextModel().getRichText().getText().toString());
	HTMLWriter htmlWriter = new HTMLWriter((HTMLText)getAppearanceTextModel().getRichText());
	app.setAppearance(htmlWriter.bodyToHTMLString());
	if(isAnnotationsModified())
	   app.setAnnotations(annotationsList.getNamedObjects());
	if(isNameUsagesModified())
	   app.setNameUsages(nameUsageList.getNamedObjects());
    }

    /**
     * Gets summary contents in <CODE>Text</CODE>
     *
     * @return Text[] representing summary of the model
     */
    public Text[] getSummaryTextArray()
    {
	StringBuffer buffer = new StringBuffer();

	Appearance a = getAppearance();

	if(a.isNominal()) {
	    return new Text[] {
		new Text(a.getPage()),
		new Text(a.getLines()),
		new Text()
		    };
	}

	Enumeration e = nameUsageList.getModels().elements();
	while(e.hasMoreElements()) {
	    if(buffer.length() > 0)
		buffer.append(", ");
	    buffer.append(((NameUsageEditModel)e.nextElement()).getAscribedName());
	}
	
	return new Text[] {
	    pageTextModel.getRichText().getText(),
	    linesTextModel.getRichText().getText(),
	    //appearance.getRichText().getText()
	    new Text(buffer.toString())
		};
    }
    
    public void updateList()
    {
	/*
	nameUsageList.update();
	annotationsList.update();
	*/
    }

    public void updateSummary()
    {
	Text text = (Text)getPublicationTextModel().getRichText().getText();
	if(text != null)
	    text = (Text)text.clone();
	else
	    text = new Text();
	String s = getPageTextModel().getRichText().getText().toString();
	if(s.length() != 0) {
	    if(text.length() != 0)
		text.append(", p. ");
	    text.append(s);
	    text.append(".");
	}
	summary.setRichText(new RichText(text, RichTextStyle.DEFAULT_DOCUMENT_STYLE));
    }

    /**
     * Invoked when the text model has been changed.
     */
    public void textModelChanged(TextModelEvent event)
    {
	Object source = event.getSource();
	if(source == publication) {
	    updateSummary();
	}
	else if(source == annotationsList) {
	    setAnnotationsModified(true);
	}
	else if(source == nameUsageList) {
	    setNameUsagesModified(true);
	}
	else
	    super.textModelChanged(event);
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an instance of <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    NamedObjectEditModel createEditModel(NamedObject object)
    {
	if(!(object instanceof Appearance))
	    throw new IllegalArgumentException(object.getClass().getName()
						+ " is not an instance of org.nomencurator.Appearance");
	return new AppearanceEditModel((Appearance)object);
    }

    public static NamedObjectEditModel getInstance()
    {
	if(template == null)
	    template = new AppearanceEditModel();
	return template;
    }

    protected boolean isAnnotationsModified()
    {
	return annotationsModified;
    }

    protected void setAnnotationsModified(boolean status)
    {
	annotationsModified = status;
	if(status)
	    modified = status;
    }

    protected boolean isNameUsagesModified()
    {
	return nameUsagesModified;
    }

    protected void setNameUsagesModified(boolean status)
    {
	nameUsagesModified = status;
	if(status)
	    modified = status;
    }

}
