/*
 * AuthorEditModel.java:  an Author editor model 
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: AuthorEditModel.java,v 1.32 2002/10/17 06:24:04 nozomi Exp $
 * $Log: AuthorEditModel.java,v $
 * Revision 1.32  2002/10/17 06:24:04  nozomi
 * fix summary of author name
 *
 * Revision 1.31  2002/10/16 13:46:35  nozomi
 * null string handle in summary text creation
 *
 * Revision 1.30  2002/10/08 00:40:29  nozomi
 * introduce listenSubModels()
 *
 * Revision 1.29  2002/09/27 15:11:50  nozomi
 * getSummaryTextArray() uses isNominal()
 *
 * Revision 1.28  2002/09/26 10:49:25  ryo
 * return null string if the title is not selected
 *
 * Revision 1.27  2002/09/26 05:16:29  ryo
 * remove publicationEditModels member
 *
 * Revision 1.26  2002/09/20 10:20:00  nozomi
 * fix bugs in save/loadAttributes()
 *
 * Revision 1.25  2002/09/17 05:46:02  nozomi
 * re-organise initialisation methods
 *
 * Revision 1.24  2002/09/09 16:40:32  nozomi
 * implement updateSumamry()
 *
 * Revision 1.23  2002/09/06 17:01:47  ryo
 * link NamedObjects when *EditModel linked
 *
 * Revision 1.22  2002/09/06 11:44:09  nozomi
 * dummy saveAttributes(), shoudl be improved
 *
 * Revision 1.21  2002/08/27 08:53:14  t.okada
 * addTextModelListener is added.
 *
 * Revision 1.20  2002/08/20 06:14:26  ryo
 * make addFeudality and addEpithet methods to be static and add setTitle, setFeudality and setEpithet methods
 *
 * Revision 1.19  2002/08/15 12:57:35  ryo
 * add the processing for extending Choice
 *
 * Revision 1.18  2002/07/02 08:29:08  nozomi
 * change default text model relating methods
 *
 * Revision 1.17  2002/06/21 23:22:57  nozomi
 * use RichText in TextComponents
 *
 * Revision 1.16  2002/05/22 23:55:04  nozomi
 * add/remove EditModels mutually
 *
 * Revision 1.15  2002/05/14 13:34:20  ryo
 * modify saveAttributes()
 *
 * Revision 1.14  2002/05/14 10:14:42  ryo
 * add getSummaryTextArray() and updateList()
 *
 * Revision 1.13  2002/05/10 12:13:57  t.okada
 * comment out publicationList.replaceItems
 *
 * Revision 1.12  2002/05/08 11:04:17  ryo
 * add accessor for chooser index
 *
 * Revision 1.11  2002/04/16 23:43:52  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.10  2002/04/16 03:53:38  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.9  2002/04/10 07:44:48  nozomi
 * members are protected, change some access methods' name
 *
 * Revision 1.8  2002/03/25 13:55:28  okawa
 * modify add()
 *
 * Revision 1.7  2002/03/25 11:32:20  okawa
 * addAffiliation()
 *
 * Revision 1.6  2002/03/20 15:12:57  okawa
 * set selection of Choices
 *
 * Revision 1.5  2002/02/28 17:11:02  nozomi
 * use TextListModel of the package
 *
 * Revision 1.4  2002/02/24 20:01:23  nozomi
 * modified to use setModel()
 *
 * Revision 1.3  2002/02/21 01:55:24  okawa
 * modification due to the integration with TabbedPane, EditPanel
 *
 * Revision 1.2  2002/01/29 07:14:35  nozomi
 * Add some methods.
 * Introduction of mechanisms supporting TableList.
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */


package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Vector;
import java.util.StringTokenizer;

import jp.kyasu.awt.DefaultTextEditModel;
import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.TextField;

import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextBuffer;

import jp.kyasu.awt.event.TextModelEvent;

import org.nomencurator.Author;
import org.nomencurator.Affiliation;
import org.nomencurator.NamedObject;
import org.nomencurator.Publication;

import org.nomencurator.editor.TableList;

/**
 * The model interface for an object that acts as a named object edit model
 *
 * @see 	org.nomencurator.Author
 * @see 	org.nomencurator.editor.model.NamedObjectEditModel
 * @see 	org.nomencurator.editor.event.ObjectEditModelEvent
 * @see 	org.nomencurator.editor.event.ObjectEditModelListener
 *
 * @version 	17 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class AuthorEditModel
    extends NamedObjectEditModel
{
    /**
     * <CODE>TextListModel</CODE> to hold title of the author
     */
    protected static jp.kyasu.awt.TextListModel titleList = new jp.kyasu.awt.DefaultTextListModel();

    /** Default contents of title */
    protected static String[][]  defaultTitles = {
	{" "}, {"Sir"}, {"Lord"}, {"other..."}
    };

    /** Extended contents of title */
	protected static Vector extendedTitles;

    /** Index of author's title in <CODE>titleList</CODE> */
    protected int title;

  
    /** <CODE>TextEditModel</CODE> to hold first name of the author */
    protected TextEditModel firstName;

    /**
     * <CODE>TextEditModel</CODE> to hold middle name of the author
     */
    protected TextEditModel middleName;

    /**
     * <CODE>TextListModel</CODE> to hold feudality of the author
     */
    protected static jp.kyasu.awt.TextListModel feudalityList = new jp.kyasu.awt.DefaultTextListModel();

    /**
     * Default contents of feudality
     */
    protected static String defaultFeudalities[][] = {
	{" "}, {"van"}, {"von"}, {"de"}, {"other..."}
    };

    /**
     * Extended contents of feudality
     */
    protected static Vector extendedFeudalities;

    /**
     * Index of author's feudality in <CODE>feudalityList</CODE>
     */
    protected int feudality;


    /**
     * <CODE>TextEditModel</CODE> to hold surname of the author
     */
    protected TextEditModel surname;

    /**
     * <CODE>TextListModel</CODE> to hold epithet of the author,
     * e.g. Jr., Sen
     */
    protected static jp.kyasu.awt.TextListModel epithetList = new jp.kyasu.awt.DefaultTextListModel();

    /**
     * Default contents of author's epithet e.g. Jr., Sen
     */
    protected static String defaultEpithets[][] = {
	{" "}, {"Jr"}, {"Sen"}, {"other..."}
    };

    /**
     * Extended contents of author's epithet e.g. Jr., Sen
     */
    protected static Vector extendedEpithets;

    /**
     * Index of author's epithet in <CODE>epithetList</CODE>
     */
    protected int epithet;

    /**
     * <CODE>TextListModel</CODE> to hold birth and death date
     */
    protected jp.kyasu.awt.TextListModel curriculumVitae;

    protected static int curriculumVitaeColumnWidths[] = {4, 20};

    /**
     * <CODE>TextListModel</CODE> to hold affilications of the author
     */
    protected TextListModel affiliationList;

    protected static String[] affiliationListTitle = {
	"institute", "since", "until"
    };

    protected static int affiliationListColumnWidths[] = 
	TableList.columnWidths(affiliationListTitle);

    /**
     * <CODE>TextListModel</CODE> to hold the author's publications
     */
    protected PublicationListModel publicationList;

    protected static String[] publicationListTitle = {
	"author", "year", "title"
    };

    protected static int publicationListColumnWidths[] =
 	TableList.columnWidths(publicationListTitle);

    /**
     * a list of <CODE>PublicationEditModel</CODE>s
     * of this author's publication
     */
//    protected Vector publicationEditModels;

    static {
	titleList.replaceItems(0, 0, defaultTitles);

	feudalityList.replaceItems(0, 0, defaultFeudalities);

	epithetList.replaceItems(0, 0, defaultEpithets);

	affiliationListColumnWidths[0] *= 2;
	publicationListColumnWidths[0] *= 2;
	publicationListColumnWidths[2] *= 2;
    }

    protected static AuthorEditModel template;

    protected boolean publicationsModified;

    /**
     * The constructor of AuthorEditModel
     *
     */
    public AuthorEditModel()
    {
	this(true);
    }

    /**
     * The constructor of AuthorEditModel
     *
     * @param editable boolean determining whether the model is editable
     */
    public AuthorEditModel(boolean editable)
    {
	this(new Author(), editable);
    }

    /**
     * The constructor of AuthorEditModel
     *
     * @param object <CODE>Author</CODE> to be edited
     */
    public AuthorEditModel(Author object)
    {
	this(object, true);
    }

    /**
     * The constructor of AuthorEditModel
     *
     * @param object <CODE>Object</CODE> to be edited
     * @param editable boolean determining whether the model is editable
     */
    public AuthorEditModel(Author object, boolean editable)
    {
	this(object, editable,
	     null);
    }

    /**
     * The constructor of AuthorEditModel
     *
     * @param object <CODE>Object</CODE> to be edited
     * @param editable boolean determining whether the model is editable
     * @param higherEditModel <CODE>NameUsageEditModel</CODE> for higher <CODE>NameUsage</CODE>
     * of the <CODE>NameUsage</CODE> edited by this model
     * @param lowerEditModels <CODE>Vector</CODE> of <CODE>NameUsageEditModel</CODE> for
     * lower <CODE>NameUsage</CODE>s of the <CODE>NameUsage</CODE> edited by this model
     * @param authority <CODE>NameUsageEditModel</CODE> for authority <CODE>NameUsage</CODE>
     * of the <CODE>NameUsage</CODE> edited by this model
     */
    public AuthorEditModel(Author object, 
			   boolean editable,
			   Vector publicationEditModels
			   )
    {
	super(object, editable);
	/*
	createAuthorNameTextEditModel();

	publicationList = new PublicationListModel();
	affiliationList = new DefaultTextListModel(affiliationListColumnWidths);
	curriculumVitae = new DefaultTextListModel(curriculumVitaeColumnWidths);
	
	if(object != null) {
	    setFirstName(new DefaultTextEditModel(new RichText(object.getFirstName(), RichTextStyle.DEFAULT_DOCUMENT_STYLE)));
	    setMiddleName(new DefaultTextEditModel(new RichText(object.getMiddleName(), RichTextStyle.DEFAULT_DOCUMENT_STYLE)));
	    setSurname(new DefaultTextEditModel(new RichText(object.getSurname(), RichTextStyle.DEFAULT_DOCUMENT_STYLE)));
		setTitle(object.getTitle());
		setFeudality(object.getFaudality());
		setEpithet(object.getEpithet());
	    if(object.getAffiliations() != null) {
	        Enumeration e = object.getAffiliations().elements();
	        while(e.hasMoreElements()) {
		    addAffiliation((Affiliation)(e.nextElement()));
		}
	    }
	    
	}
	
    firstName.addTextModelListener(this);
    middleName.addTextModelListener(this);
    surname.addTextModelListener(this);
    titleList.addListModelListener(this);
    feudalityList.addListModelListener(this);
    epithetList.addListModelListener(this);

	this.publicationEditModels = new Vector();
	
	if(publicationEditModels != null &&
	   publicationEditModels.size() != 0) {
	    Enumeration e = publicationEditModels.elements();
	    while(e.hasMoreElements()) {
		add((PublicationEditModel)(e.nextElement()));
	    }
	}
	*/
    }

    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void createSubModels()
    {
	super.createSubModels();
	createAuthorNameTextEditModel();

	publicationList = new PublicationListModel();
	affiliationList = new DefaultTextListModel(affiliationListColumnWidths);
	curriculumVitae = new DefaultTextListModel(curriculumVitaeColumnWidths);

	setPublicationsModified(false);
    }

    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void listenSubModels()
    {
	listenAuthorNameTextEditModel();
    }

    /**
     * Method to create <CODE>TextEditModel</CODE>s
     */
    protected void createAuthorNameTextEditModel()
    {
	firstName = getDefaultTextEditModel();
	middleName = getDefaultTextEditModel();
	surname = getDefaultTextEditModel();
    }

    /**
     * Method to create <CODE>TextEditModel</CODE>s
     */
    protected void listenAuthorNameTextEditModel()
    {
	firstName.addTextModelListener(this);
	middleName.addTextModelListener(this);
	surname.addTextModelListener(this);
    }

    /**
     * Adds <CODE>publicationEditModel</CODE> as <CODE>Publication</CODE> of this
     * <CODE>Author</CODE>
     *
     * @param publicationEditModel a <CODE>PublicationEditModel</CODE> to be added
     */
    public void add(PublicationEditModel publicationEditModel)
    {
	if(publicationList.contains(publicationEditModel))
	    return;

	publicationList.addModel(publicationEditModel);
	//setPublicationsModified(true);
	if(publicationEditModel != null) {
		publicationEditModel.addAuthor(this);
		//this.getAuthor().addPublication(publicationEditModel.getPublication());
	}
	setPublicationsModified(true);
    }

    /**
     * Adds <CODE>publicationEditModel</CODE> as <CODE>Publication</CODE> of this
     * <CODE>Author</CODE>
     *
     * @param publicationEditModel a <CODE>PublicationEditModel</CODE> to be added
     */
    public void add(int index, PublicationEditModel publicationEditModel) {
	/*
		if (publicationEditModels.contains(publicationEditModel))
			return;
		publicationEditModels.insertElementAt(publicationEditModel, index);
		publicationList.addModel(-1, publicationEditModel,
			new String[][]{{publicationEditModel.getSummary(), "", ""}});
//		new String[][] { {
//			getAuthor().getFullname(),
//			Integer.toString(publicationEditModel.getPublication().getYear()),
//			publicationEditModel.getPublication().getCitationTitle()}
//		});
	*/
	if(publicationList.contains(publicationEditModel))
	    return;
	publicationList.addModel(index, publicationEditModel);
	if(publicationEditModel != null)
	    publicationEditModel.addAuthor(this);
	//setPublicationsModified(true);
    }


    /**
     * Removes <CODE>publicationEditModel</CODE> as <CODE>Publication</CODE> of this
     * <CODE>Author</CODE>
     *
     * @param publicationEditModel a <CODE>PublicationEditModel</CODE> to be removeed
     */
    public void remove(PublicationEditModel publicationEditModel)
    {
	publicationList.removeModel(publicationEditModel);
	if(publicationEditModel != null) {
	    publicationEditModel.removeAuthor(this);
	    //	    this.getAuthor().removePublication(publicationEditModel.getPublication());
	}
	//setPublicationsModified(true);
    }

    /**
     * Removes <CODE>publicationEditModel</CODE> as <CODE>Publication</CODE> of this
     * <CODE>Author</CODE>
     *
     * @param publicationEditModel a <CODE>PublicationEditModel</CODE> to be removeed
     *
     * @return <CODE>PublicationEditModel</CODE> removed
     */
    public void removePublicationEditModel(PublicationEditModel publicationEditModel)
    {
	remove(publicationEditModel);
    }

    /**
     * Removes <CODE>publicationEditModel</CODE> as <CODE>Publication</CODE> of this
     * <CODE>Author</CODE>
     *
     * @param publicationEditModel a <CODE>PublicationEditModel</CODE> to be removeed
     *
     * @return <CODE>PublicationEditModel</CODE> removed
     */
    public void removePublicationEditModel(int index)
    {
	publicationList.removeModel(publicationList.getModel(index));
	//setPublicationsModified(true);
    }

    /**
     * Removes <CODE>publicationEditModel</CODE> as <CODE>Publication</CODE> of this
     * <CODE>Author</CODE>
     *
     * @param publicationEditModel a <CODE>PublicationEditModel</CODE> to be removeed
     */
    public void removePublicationEditModelAt(int index)
    {
	removePublicationEditModel(index);
    }

    /**
     * Gets <CODE>Author</CODE> under edit by this model.
     *
     */
    public Author getAuthor()
    {
	return (Author)getObject();
    }

    /**
     * Sets <CODE>author</CODE> to be edited by this model.
     *
     * @param author <CODE>Author</CODE> to be edited
     *
     */
    public void setAuthor(Author author)
    {
	setObject(author);
    }

    /**
     * Gets <CODE>TextListModel</CODE> for title of the author
     *
     */
    public int getTitleIndex()
    {
	return title;
    }

    /**
     * Gets <CODE>TextListModel</CODE> for title of the author
     *
     */
    public String getTitle()
    {
	if(title == 0)
	    return "";

	return titleList.getItem(title, 0).toString();
    }

    /**
     * Gets <CODE>TextListModel</CODE> for title of the author
     *
     */
    public jp.kyasu.awt.TextListModel getTitleModel()
    {
	return titleList;
    }

    /**
     * Sets given <CODE>TextListModel</CODE> to title of the author
     *
     * @param model <CODE>TextListModel</CODE> to be title of the author
     *
     */
    public void setTitleModel(TextListModel model)
    {
	if(titleList == model)
	    return;
	titleList = model;
    }

	/**
	 * Add <CODE>titleName</CODE> to the title list
	 *
	 * @param titleName name of the title to be add;
	 */
	public static void addTitle(String titleName) {
		if (extendedTitles == null)
			extendedTitles = new Vector();
		extendedTitles.addElement(titleName);
		int index = titleList.getItemCount() - 1;
		titleList.replaceItems(index, index,
			new String[][] {{ titleName }});
	}

    /**
     * Gets <CODE>TextEditModel</CODE> for first name of the author
     *
     */
    public TextEditModel getFirstName()
    {
	return firstName;
    }

    /**
     * Sets given <CODE>TextEditModel</CODE> to first name of the author
     *
     */
    public void setFirstName(TextEditModel model)
    {
	if(firstName == model)
	    return;
	firstName = model;
    }

    /**
     * Gets <CODE>TextEditModel</CODE> for middle name of the author
     *
     */
    public TextEditModel getMiddleName()
    {
	return middleName;
    }

    /**
     * Sets given <CODE>TextEditModel</CODE> to middle name of the author
     *
     */
    public void setMiddleName(TextEditModel model)
    {
	if(middleName == model)
	    return;
	middleName = model;
    }

    /**
     * Gets <CODE>TextListModel</CODE> for feudality of the author's name
     *
     */
    public int getFeudalityIndex()
    {
	    return feudality;
    }

    /**
     * Gets <CODE>TextListModel</CODE> for feudality of the author's name
     *
     */
    public String getFeudality()
    {
	if(feudality == 0)
	    return "";

	return feudalityList.getItem(feudality, 0).toString();
    }

    /**
     * Gets <CODE>TextListModel</CODE> for feudality of the author's name
     *
     */
    public jp.kyasu.awt.TextListModel getFeudalityModel()
    {
	return feudalityList;
    }

    /**
     * Sets given <CODE>TextListModel</CODE> to feudality of the author's name
     *
     */
    public void setFeudality(TextListModel model)
    {
	if(feudalityList == model)
	    return;
	feudalityList = model;
    }

	/**
	 * Add <CODE>feudalityName</CODE> to the feudality list
	 *
	 * @param feudalityName name of feudality to be add;
	 */
	public static void addFeudality(String feudalityName) {
		if (extendedFeudalities == null)
			extendedFeudalities = new Vector();
		extendedFeudalities.addElement(feudalityName);
		int index = feudalityList.getItemCount() - 1;
		feudalityList.replaceItems(index, index,
			new String[][] {{ feudalityName }});
	}

    /**
     * Gets <CODE>TextEditModel</CODE> for surname of the author
     *
     */
    public TextEditModel getSurname()
    {
	return surname;
    }

    /**
     * Sets given <CODE>TextEditModel</CODE> to surname of the author
     *
     */
    public void setSurname(TextEditModel model)
    {
	if(surname == model)
	    return;
	surname = model;
    }

    /**
     * Gets <CODE>TextListModel</CODE> for epithet of the author's name
     *
     */
    public int getEpithetIndex()
    {
	return epithet;
    }

    /**
     * Gets <CODE>TextListModel</CODE> for epithet of the author's name
     *
     */
    public String getEpithet()
    {
	if(epithet == 0)
	    return "";	

	return epithetList.getItem(epithet, 0).toString();
    }

    /**
     * Gets <CODE>TextListModel</CODE> for epithet of the author's name
     *
     */
    public jp.kyasu.awt.TextListModel getEpithetModel()
    {
	return epithetList;
    }

    /**
     * Sets given <CODE>TextListModel</CODE> to epithet of the author's name
     *
     */
    public void setEpithet(TextListModel model)
    {
	if(epithetList == model)
	    return;
	epithetList = model;
    }

	/**
	 * Add <CODE>epithetName</CODE> to the epithet list
	 *
	 * @param epithetName name of epithet to be add;
	 */
	public static void addEpithet(String epithetName) {
		if (extendedEpithets == null)
			extendedEpithets = new Vector();
		extendedEpithets.addElement(epithetName);
		int index = epithetList.getItemCount() - 1;
		epithetList.replaceItems(index, index,
			new String[][] {{ epithetName }});
	}

    /**
     * Gets <CODE>TextListModel</CODE> for life time of the author
     *
     */
    public jp.kyasu.awt.TextListModel getCurriculumVitae()
    {
	return curriculumVitae;
    }

    /**
     * Sets given <CODE>TextListModel</CODE> to life time of the author
     *
     */
    public void setCurriculumVitae(TextListModel model)
    {
	if(curriculumVitae == model)
	    return;
	curriculumVitae = model;
    }


    /**
     * Adds <CODE>Affiliation</CODE> of this <CODE>Author</CODE>
     *
     * @param Affiliation a <CODE>Affiliation</CODE> to be added
     */
    public void addAffiliation(Affiliation affiliation)
    {
	affiliationList.replaceItems(0, 0, new String[][]{{affiliation.persistentID(),
	                                                   affiliation.getFrom().toString(),
	                                                   affiliation.getUntil().toString()}});
    }

    /**
     * Gets <CODE>TextListModel</CODE> for affilications of the author
     *
     */
    public TextListModel getAffiliationList()
    {
	return affiliationList;
    }

    /**
     * Sets given <CODE>TextListModel</CODE> to affiliationList of the author
     *
     */
    public void setAffiliationList(TextListModel model)
    {
	if(affiliationList == model)
	    return;
	affiliationList = model;
    }

    /**
     * Gets <CODE>TextListModel</CODE> for publicationList of the author
     *
     */
    public PublicationListModel getPublicationList()
    {
	return publicationList;
    }

    /**
     * Sets given <CODE>TextListModel</CODE> to publicationList of the author
     *
     */
    public void setPublicationList(PublicationListModel model)
    {
	if(publicationList == model)
	    return;
	publicationList = model;
    }

    public String getFullName()
    {
	return getAuthor().getFullname();
    }

    public String getFullName(boolean abbreviate)
    {
	return getAuthor().getFullname(abbreviate);
    }

    static public String[] getPublicationListTitle()
    {
	return publicationListTitle;
    }

    static public String[] getAffiliationListTitle()
    {
	return affiliationListTitle;
    }

    public void clear()
    {
    }

    public Object clone()
    {
	AuthorEditModel model = new AuthorEditModel((Author)getAuthor().clone());
	model.firstName.setRichText((RichText)(firstName.getRichText().clone()));
	model.middleName.setRichText((RichText)(middleName.getRichText().clone()));
	model.surname.setRichText((RichText)(surname.getRichText().clone()));
	model.publicationList = (PublicationListModel)publicationList.clone();
	model.affiliationList = (DefaultTextListModel)((DefaultTextListModel)affiliationList).clone();
	model.curriculumVitae = (DefaultTextListModel)((DefaultTextListModel)curriculumVitae).clone();
	
//	model.publicationEditModels = (Vector)publicationEditModels.clone();

	return model;
    }


	/**
	 * Returns <CODE>Vector</CODE> containing <CODE>PublicationEditModel</CODE>s of the publications
	 *
	 * @return <CODE>Vector</CODE> containing <CODE>PublicationEditModel</CODE>s of the publications
	 */
	public Vector getPublicationEditModels() {
		return publicationList.getModels();
//		return publicationEditModels;
	}

    /**
     * Saves attributes to the <CODE>Author</CODE>
     */
    public void saveAttributes() {
	Author author = getAuthor();
	author.setSurname(this.getSurname().getRichText().getText().toString());
	author.setFirstName(this.getFirstName().getRichText().getText().toString());
	author.setMiddleName(this.getMiddleName().getRichText().getText().toString());
	author.setTitle(getTitle());
	author.setEpithet(getEpithet());
	author.setFeudality(getFeudality());
	//		author.setBirthDate();
	//		author.setDeathDate();
	author.setPublications(publicationList.getNamedObjects());
	setPublicationsModified(false);
    }
    
    /**
     * Sets index for title of the author's name
     * @param i index for title of the author's name
     */
    public void setTitleIndex(int i) {
	title = i;
    }
    
    /**
     * Sets index for title of the author's name
     * @param s string for title of the author's name
     */
    public void setTitle(String s) {
	if (s == null) {
	    setTitleIndex(0);
	    return;
	}
	int i;
	for (i = 0; i < titleList.getItemCount(); i++) {
	    if (s.equals(((Text) titleList.getItem(i, 0)).toString())) {
		setTitleIndex(i);
	    }
	}
	if (s != null && !s.equals("") && i == titleList.getItemCount()) {
	    addTitle(s);
	    setTitleIndex(i-1);
	}
    }
    
    /**
     * Sets index for feudality of the author's name
     * @param i index for feudality of the author's name
     */
    public void setFeudalityIndex(int i) {
	feudality = i;
    }
    
    /**
     * Sets index for feudality of the author's name
     * @param s string for feudality of the author's name
     */
    public void setFeudality(String s) {
	if (s == null) {
	    setFeudalityIndex(0);
	    return;
	}
	int i;
	for (i = 0; i < feudalityList.getItemCount(); i++) {
	    if (s.equals(((Text) feudalityList.getItem(i, 0)).toString())) {
		setFeudalityIndex(i);
	    }
	}
	if (s != null && !s.equals("") && i == feudalityList.getItemCount()) {
	    addFeudality(s);
	    setFeudalityIndex(i-1);
	}
    }
    
    /**
     * Sets index for epithet of the author's name
     * @param i index for epithet of the author's name
     */
    public void setEpithetIndex(int i) {
	epithet = i;
    }
    
    /**
     * Sets index for epithet of the author's name
     * @param s string for epithet of the author's name
     */
    public void setEpithet(String s) {
	if (s == null){
	    setEpithetIndex(0);
	    return;
	}
	int i;
	for (i = 0; i < epithetList.getItemCount(); i++) {
	    if (s.equals(((Text) epithetList.getItem(i, 0)).toString())) {
		setEpithetIndex(i);
	    }
	}
	if (s != null && !s.equals("") && i == epithetList.getItemCount()) {
	    addEpithet(s);
	    setEpithetIndex(i-1);
	}
    }
    
    /**
     * Gets summary contents in <CODE>Text</CODE>
     *
     * @return Text[] representing summary of the model
     */
    public Text[] getSummaryTextArray() {
	TextBuffer buffer = new TextBuffer();
	if(!getNamedObject().isNominal()) {
	    appendTo(buffer, firstName.getRichText().getText());
	    appendTo(buffer, middleName.getRichText().getText());
	    appendTo(buffer, surname.getRichText().getText());
	}
	else {
	    Author a = getAuthor();
	    String nameString = a.getFirstName();
	    if(nameString != null)
		buffer.append(nameString);
	    nameString = a.getMiddleName();
	    if(nameString != null)
		buffer.append(nameString);
	    nameString = a.getSurname();
	    if(nameString != null)
		buffer.append(nameString);
	}
	return new Text[] {buffer.toText()};
    }
    
    public void updateList()
    {
	publicationList.update();
	affiliationList.update();
    }
    
    /**
     * Gets extended contents of title
     * @return Vector extended contents of title
     */
    public static Vector getExtendedTitles() {
	return extendedTitles;
    }
    
    /**
     * Gets extended contents of epithet
     * @return Vector extended contents of epithet
     */
    public static Vector getExtendedEpithets() {
	return extendedEpithets;
    }
    
    /**
     * Gets extended contents of feudality
     * @return Vector extended contents of feudality
     */
    public static Vector getExtendedFeudalities()
    {
	return extendedFeudalities;
    }
    
    /**
     * Loeds attributes of <CODE>Object</CODE> to the model
     */
    public void loadAttributes()
    {
	super.loadAttributes();

	Author author = getAuthor();
	setString(surname, author.getSurname());
	setString(firstName, author.getFirstName());
	setString(middleName, author.getMiddleName());

	/*
	title = 0;
	String s = author.getTitle();
	if(s != null && s.length() != 0) {
	    while(title < defaultTitles.length &&
		  !defaultTitles[title].equals(s))
		title++;
	    if(title == defaultTitles.length)
		title = 0;
	}
	*/
	setTitle(author.getTitle());

	/*
	epithet = 0;
	String s = author.getEpithet();
	if(s != null && s.length() != 0) {
	    while(epithet < defaultEpithets.length &&
		  !defaultEpithets[epithet].equals(s))
		epithet++;
	    if(epithet == defaultEpithets.length)
		epithet = 0;
	}
	*/
	setEpithet(author.getEpithet());

	/*
	feudality = 0;
	s = author.getFeudality();
	if(s != null && s.length() != 0) {
	    while(feudality < defaultFeudalities.length &&
		  !defaultFeudalities[feudality].equals(s))
		feudality++;
	    if(feudality == defaultFeudalities.length)
		feudality = 0;
	}
	*/
	setFeudality(author.getFeudality());

	if(isPublicationsModified())
	    publicationList.setObjects(author.getPublications());
	setPublicationsModified(false);
    }

    public void updateSummary()
    {
	Text text = new Text();
	if(title != 0)
	    text.append(defaultTitles[title][0]);

	if(feudality != 0) {
	    if(text.length() > 0)
		text.append(' ');
	    text.append(defaultFeudalities[feudality][0]);
	}

	String s = surname.getRichText().getText().toString();
	if(s.length() > 0) {
	    if(text.length() > 0)
		text.append(' ');
	    text.append(s);
	}

	if(epithet != 0) {
	    if(text.length() > 0)
		text.append(' ');
	    text.append(defaultEpithets[epithet][0]);
	}

	s = firstName.getRichText().getText().toString();
	if(s.length() > 0) {
	    if(text.length() > 0)
		text.append(", ");
	    text.append(s);
	}

	s = middleName.getRichText().getText().toString();
	if(s.length() > 0) {
	    /*
	    if(text.length() > 0)
		text.append(", ");
	    */
	    StringTokenizer tokens = new StringTokenizer(s);
	    int i = 0;
	    while (tokens.hasMoreTokens()) {
		s = tokens.nextToken();
		if(s.length() > 0) {
		    if(i > 0)
			text.append(' ');
		    text.append(s.charAt(0));
		    text.append('.');
		}
	    }
	}
	summary.setRichText(new RichText(text, RichTextStyle.DEFAULT_DOCUMENT_STYLE));
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an instance of <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    NamedObjectEditModel createEditModel(NamedObject object)
    {
	if(!(object instanceof Author))
	    throw new IllegalArgumentException(object.getClass().getName()
						+ " is not an instance of org.nomencurator.Author");

	return new AuthorEditModel((Author)object);
    }

    public static NamedObjectEditModel getInstance()
    {
	if(template == null)
	    template = new AuthorEditModel();
	return template;
    }

    protected boolean isPublicationsModified()
    {
	return publicationsModified;
    }

    protected void setPublicationsModified(boolean status)
    {
	publicationsModified = status;
	if(status)
	    modified = status;
    }

    /**
     * Invoked when the text model has been changed.
     */
    public void textModelChanged(TextModelEvent event)
    {
	Object source = event.getSource();
	if (source == publicationList) {
	    setPublicationsModified(true);
	}
	else {
	    super.textModelChanged(event);
	}
    }
}
