/*
 * ConnectionEditModel.java : 
 *
 * Copyright (c) 2002 Ryo FUJIMOTO
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ConnectionEditModel.java,v 1.6 2002/10/10 19:48:30 nozomi Exp $
 * $Log: ConnectionEditModel.java,v $
 * Revision 1.6  2002/10/10 19:48:30  nozomi
 * add DiGIR button
 *
 * Revision 1.5  2002/10/09 08:50:56  nozomi
 * implement dummy listenSubModels()
 *
 * Revision 1.4  2002/09/17 08:51:17  nozomi
 * re-organise initialisation
 *
 * Revision 1.3  2002/09/09 16:38:29  nozomi
 * implement dummy updateSumamry()
 *
 * Revision 1.2  2002/09/06 02:18:32  nozomi
 * add empty load/saveAttributes()
 *
 * Revision 1.1  2002/07/16 07:10:14  ryo
 * initial import into CVS
 *
 */

package org.nomencurator.editor.model;

import jp.kyasu.awt.TableList;
import jp.kyasu.awt.DefaultTextListModel;
import jp.kyasu.graphics.Text;

import org.nomencurator.NamedObject;

import org.nomencurator.broker.AbstractConnection;
import org.nomencurator.broker.JdbcConnection;
import org.nomencurator.broker.HttpConnection;
import org.nomencurator.broker.LocaldbConnection;

import java.util.Vector;
import java.util.Enumeration;

public class ConnectionEditModel
extends NamedObjectEditModel
{

	/** Index of JDBC connection type */
	static public final int TYPE_JDBC = 0;

	/** Index of HTTP connection type */
	static public final int TYPE_HTTP = TYPE_JDBC + 1;

	/** Index of LOCAL connection type */
	static public final int TYPE_LOCAL = TYPE_HTTP + 1;

	/** Index of DIGER connection type */
	static public final int TYPE_DIGIR = TYPE_LOCAL + 1;

	/** Index of DIGER connection type */
	static public final int TYPE_MAX = TYPE_DIGIR + 1;

	/** connection type string */
	protected static String[] connectionTypeStrings = {
		"JDBC",
		"HTTP",
		"LOCAL",
		"DiGIR"
	};

	/**
	 * connection type.
	 */
	protected int type = -1;

	/**
	 * connection name.
	 */
	protected String name = null;

	/**
	 * driver class name used for access to a database.
	 */
	protected String driverName = null;

	/**
	 * URL string used for access to a database or a Nomencurator servlet.
	 */
	protected String url = null;

	/**
	 * user name used for access to a database.
	 */
	protected String username = null;

	/**
	 * password used for access to a database.
	 */
	protected String password = null;

	/**
	 * file system path.
	 */
	protected String path = null;

	/**
	 * validity of connection.
	 */
	protected boolean valid = false;

	/**
	 * validity of writing of connection.
	 */
	protected boolean writable = false;

    static ConnectionEditModel template = new ConnectionEditModel();

	/**
	 * Constractor.
	 */
	public ConnectionEditModel() {
		super(null, true);
	}
	
	/**
	 * Constractor.
	 */
	public ConnectionEditModel(AbstractConnection c) {
		super(null, true);
		setModel(c);
	}
	
	/**
	 * Get connection type.
	 * @return int type.
	 */
	public int getType() {
		return type;
	}

	/**
	 * Get connection type string.
	 * @return int type.
	 */
	public String getTypeString() {
		int i = getType();
		if (0 <= i && i < connectionTypeStrings.length) {
			return connectionTypeStrings[i];
		}
		return null;
	}

	/**
	 * 
	 */
	public static String[] getConnectionTypeStrings() {
		return connectionTypeStrings;
	}

	/**
	 * Get connection name.
	 * @return String connection name.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Get driver class name used for access to a database.
	 * @return String driver class name.
	 */
	public String getDriverName() {
		return driverName;
	}

	/**
	 * Get URL string used for access to a database or a Nomencurator servlet.
	 * @return String URL string.
	 */
	public String getUrl() {
		return url;
	}

	/**
	 * Get user name used for access to a database.
	 * @return String user name.
	 */
	public String getUsername() {
		return username;
	}

	/**
	 * Get password used for access to a database.
	 * @return String password.
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * Set connection type.
	 * @param type connection type.
	 */
	public void setType(int type) {
		this.type = type;
	}

	/**
	 * Set connection name.
	 * @param name connection name.
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Set driver class name used for access to a database.
	 * @param driverName driver class name.
	 */
	public void setDriverName(String driverName) {
		this.driverName = driverName;
	}

	/**
	 * Set URL string used for access to a database or a Nomencurator servlet.
	 * @param url URL string.
	 */
	public void setUrl(String url) {
		this.url = url;
	}

	/**
	 * Set user name used for access to a database.
	 * @param username user name.
	 */
	public void setUsername(String username) {
		this.username = username;
	}

	/**
	 * Set password used for access to a database.
	 * @param password password.
	 */
	public void setPassword(String password) {
		this.password = password;
	}

	/**
	 * Set top directory for local database.
	 * @param path the top directory for local database.
	 */
	public void setPath(String path) {
		this.path = path;
	}
	
	/**
	 * Get top directory for local database.
	 * @return String the top directory for local database.
	 */
	public String getPath() {
		return path;
	}

	/**
	 * Set validity of connection.
	 * @param b validity of connection
	 */
	public void setValid(boolean b) {
		valid = b;
	}
	
	/**
	 * Get validity of connection.
	 * @return true if the connection is valid, false if not.
	 */
	public boolean getValid() {
		return valid;
	}
	
	/**
	 * Set validity of writing of connection.
	 * @param b validity of writing of connection
	 */
	public void setWritable(boolean b) {
		writable = b;
	}
	
	/**
	 * Get validity of writing of connection.
	 * @return true if the connection is writable, false if not.
	 */
	public boolean getWritable() {
		return writable;
	}

	/**
	 * Gets summary contents in <code>Text</code>
	 *
	 * @return Text[] representing summary of the model
	 */
	public Text[] getSummaryTextArray() {
		return new Text[] {
			new Text((name == null) ? "" : name),
			new Text((getTypeString() == null) ? "" : getTypeString()),
			new Text((driverName == null) ? "" : driverName),
			new Text((url == null) ? "" : url),
			new Text((username == null) ? "" : username),
			new Text((password == null) ? "" : password),
			new Text((path == null) ? "" : path),
			new Text(new Boolean(valid).toString()),
			new Text(new Boolean(writable).toString())
		};
	}

	/**
	 * Gets model.
	 * @return AbstractConnection model
	 */
	public AbstractConnection getModel() {
		AbstractConnection model;
		if (type == TYPE_JDBC) {
			return getJdbcConnectionModel();
		} else if (type == TYPE_HTTP ||
			   type == TYPE_DIGIR ) {
			return getHttpConnectionModel();
		} else if (type == TYPE_LOCAL) {
			return getLocaldbConnectionModel();
		} else {
			return null;
		}
	}

	/**
	 * Gets model as <code>JdbcConnection</code>.
	 * @return <code>JdbcConnection</code> model
	 */
	public JdbcConnection getJdbcConnectionModel() {
		JdbcConnection c = new JdbcConnection();
		c.setName(getName());
		c.setDriverName(getDriverName());
		c.setUrl(getUrl());
		c.setUser(getUsername());
		c.setPassword(getPassword());
		c.setValid(getValid());
		c.setWritable(getWritable());
		return c;
	}

	/**
	 * Gets model as <code>HttpConnection</code>.
	 * @return <code>HttpConnection</code> model
	 */
	public HttpConnection getHttpConnectionModel() {
		HttpConnection c = new HttpConnection();
		c.setName(getName());
		c.setUrl(getUrl());
		c.setValid(getValid());
		c.setWritable(getWritable());
		return c;
	}

	/**
	 * Gets model as <code>LocaldbConnection</code>.
	 * @return <code>LocaldbConnection</code> model
	 */
	public LocaldbConnection getLocaldbConnectionModel() {
		LocaldbConnection c = new LocaldbConnection();
		c.setName(getName());
		c.setPath(getPath());
		c.setValid(getValid());
		c.setWritable(getWritable());
		return c;
	}

	/**
	 * Sets model.
	 * @param model model
	 */
	public void setModel(AbstractConnection model) {
		setName(model.getName());
		setValid(model.getValid());
		setWritable(model.getWritable());
		if (model instanceof JdbcConnection) {
			setType(TYPE_JDBC);
			setDriverName(((JdbcConnection)model).getDriverName());
			setUrl(((JdbcConnection)model).getUrl());
			setUsername(((JdbcConnection)model).getUsername());
			setPassword(((JdbcConnection)model).getPassword());
		} else if (model instanceof HttpConnection) {
			setType(TYPE_HTTP);
			setUrl(((HttpConnection)model).getUrl());
		} else if (model instanceof LocaldbConnection) {
			setType(TYPE_LOCAL);
			setPath(((LocaldbConnection)model).getPath());
		}
	}

	public Object clone() {
		return new ConnectionEditModel(getModel());
	}

	public void clear() {
	}

	public void updateList() {
	}

    public void saveAttributes(){}
    public void loadAttributes(){}

    public void updateSummary()
    {
    }

        /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an instance of <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    NamedObjectEditModel createEditModel(NamedObject object)
    {
	return new ConnectionEditModel();
    }

    /**
     * Listens submodles' <CODE>Event</CODE>s.
     */
    protected void listenSubModels()
    {
    }

}
