/*
 * ModelSelectAdaptor.java:  an adaptor class to choose NamedObjectEditModels
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ModelSelectAdaptor.java,v 1.19 2002/10/11 03:42:54 nozomi Exp $
 * $Log: ModelSelectAdaptor.java,v $
 * Revision 1.19  2002/10/11 03:42:54  nozomi
 * add up(int), down(int) and moveModel(int, int) methods)
 *
 * Revision 1.18  2002/10/02 02:54:06  nozomi
 * use equals of Objects
 *
 * Revision 1.17  2002/09/27 06:00:55  t.okada
 * Double registration prevention processing addition .
 *
 * Revision 1.16  2002/09/20 13:19:18  nozomi
 * eliminate NamedObject of dummy model from return value of getNamedObject()
 *
 * Revision 1.15  2002/09/17 08:29:56  nozomi
 * add/removeModel() methods return boolean; add removeAll()
 *
 * Revision 1.14  2002/09/06 02:11:46  nozomi
 * add getNamedObjects()
 *
 * Revision 1.13  2002/09/05 15:09:28  nozomi
 * add getNamedObjects() method
 *
 * Revision 1.12  2002/06/03 00:10:07  nozomi
 * use getEmptySummary() to show a dummy model
 *
 * Revision 1.11  2002/05/26 15:18:18  nozomi
 * handle single colum with empty contents
 *
 * Revision 1.10  2002/05/25 16:55:20  nozomi
 * fix model selection bug
 *
 * Revision 1.9  2002/05/23 02:47:40  nozomi
 * modify enableDummyModel to add dummy model at the end of the list
 *
 * Revision 1.8  2002/05/22 23:56:38  nozomi
 * add contains method
 *
 * Revision 1.7  2002/05/21 03:35:29  nozomi
 * emptyText changed to EMPTY_TEXT
 *
 * Revision 1.5  2002/05/20 23:10:56  nozomi
 * handle single column, empty summary
 *
 * Revision 1.4  2002/05/20 05:02:39  nozomi
 * ModelSelector re-defind to be based on TextListModel
 *
 * Revision 1.3  2002/05/17 17:39:20  nozomi
 * ModelSelectors moved to org.nomencurator.editor.model
 *
 * Revision 1.2  2002/02/27 23:52:32  nozomi
 * chnge selected model handling
 *
 * Revision 1.1  2002/02/26 16:31:19  nozomi
 * initial import
 *
 */

package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import jp.kyasu.awt.List;
import jp.kyasu.awt.TableList;

import jp.kyasu.awt.TextListModel;

import jp.kyasu.graphics.Text;

import org.nomencurator.NamedObject;

/**
 * An adaptor implementation of <code>ModelSelector</code> interface.
 * @see org.nomencurator.editor.ModelSelector
 *
 * @version 	11 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class ModelSelectAdaptor
    extends Observable
    implements ModelSelector,
	       Observer
{
    /**
     * <code>TextListModel</code> representing summary of 
     * <code>NamedObjectEditModel</code> handled by this selector
     */
     protected TextListModel textListModel;

    /** <code>Vector</code> of models to be selected */
    protected Vector models;

    /** A dummy <code>NamedObjectEditModel</code> */
    protected NamedObjectEditModel dummyModel;

    /** A dummy <code>Object</code> */
    //    protected static final Object dummyObject = new Object();

    /** <code>boolean</code> whether enable dummy model addition */
    protected boolean dummyModelEnabled;

    protected Text[][] summaryTextArray;

    public static final Text EMPTY_TEXT = new Text("");

    public static final Text LIST_SEPARATOR_TEXT = new Text(new String(new char[]{Text.LINE_SEPARATOR_CHAR}));

    /**
     * Constructs an <code>ModelSelectAdaptor</code>
     * using <code>list</code> as its reprentation
     */
    public ModelSelectAdaptor(List list)
    {
	this(list, false);
    }

    /**
     * Constructs an <code>ModelSelectAdaptor</code>
     * using <code>list</code> as its reprentation
     */
    public ModelSelectAdaptor(List list,
			      boolean enable)
    {
	this(list.getModel(), enable);
    }

    /**
     * Constructs an <code>ModelSelectAdaptor</code>
     * using <code>list</code> as its reprentation
     */
    public ModelSelectAdaptor(TableList list)
    {
	this(list, false);
    }

    /**
     * Constructs an <code>ModelSelectAdaptor</code>
     * using <code>list</code> as its reprentation
     */
    public ModelSelectAdaptor(TableList list,
			      boolean enable)
    {
	this(list.getModel(), enable);
    }

    /**
     * Constructs an <code>ModelSelectAdaptor</code>
     * using <code>controller</code> as its reprentation
     */
    public ModelSelectAdaptor(TextListModel model)
    {
	this(model, false);
    }

    /**
     * Constructs an <code>ModelSelectAdaptor</code>
     * using <code>controller</code> as its reprentation
     */
    public ModelSelectAdaptor(TextListModel model,
			      boolean enable)
    {
	setModel(model);
	models = new Vector();
	if(enable)
	    addDummyModel(enable);
	/*
	{
	    dummyModelEnabled = false;
	    addModel(0, getDummyModel(), getEmptySummary());
	    dummyModelEnabled = enable;
	}
	*/

    }

    /**
     * Returns <code>TextListModel</code> representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     *
     * @return TextListModel representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     */
    public TextListModel getModel()
    {
	return textListModel;
    }

    /**
     * Sets <code>list</code> as representation of this selector
     */
    public void setModel(TextListModel model)
    {
	if(textListModel == model)
	    return;

	textListModel = model;
	if(textListModel instanceof ModelSelector)
	    dummyModel = ((ModelSelector)textListModel).getDummyModel();
	else
	    dummyModel = /*dummyObject*/null;
	if(dummyModelEnabled) {
	    summaryTextArray = createSummaryTextArray();
	}
    }

    /**
     * Adds <code>model</code> at end of model list
     *
     * @param model <code>NamedObjectEditModel</code> to be added
     */
    public boolean addModel(NamedObjectEditModel model)
    {
	int index = -1;
	if(dummyModelEnabled)
	    index = models.size() - 1;
	return addModel(index, model);
    }

    /**
     * Inserts <code>model</code> at <code>index</code> of model list.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     */
    public boolean addModel(int index, NamedObjectEditModel model)
    {
	return addModel(index, model, getSummaryTextArray(model));
    }

    /**
     * Inserts <code>model</code> at <code>index</code> of model list
     * with its representation given by <code>items</code>.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     * @param items an array of <CODE>Object</CODE>s representing contents of the <CODE>model</CODE>
     */
    public boolean addModel(int index, NamedObjectEditModel model, Object[][] items)
    {
	if(models.contains(model))
	    return false;

	/*
	if(model != null) {
		NamedObject no = (NamedObject)((NamedObjectEditModel)model).getObject();
		if(no != null) {
			no = (NamedObject)no.getEntity();
			int size = models.size();
			for(int i = 0; i < size; i++) {
				if(no == (NamedObject)((NamedObjectEditModel)models.elementAt(i)).getObject()) {
					return false;
				}
			}
		}
	}
	*/
	
	int itemCount = models.size();
	if (index <= -1 || index > itemCount) {
	    index = itemCount;
	    if(dummyModelEnabled)
		index--;
	}
	models.insertElementAt(model, index);
	if(model != null)
	    model.addObserver(this);

	textListModel.replaceItems(index, index, trimSingleColumn(items));

	return true;
    }

    
    protected Object[][] trimSingleColumn(Object[][] items)
    {
	if(textListModel.getColumnCount() == 1) {
	    for(int row = 0; row < items.length; row++)
		if(((Text)items[row][0]).length() == 0)
		    items[row][0] = LIST_SEPARATOR_TEXT;
	}
	return items;
    }

    /**
     * Release <code>model</code> from this adaptor's control
     */
    public boolean removeModel(NamedObjectEditModel model)
    {
	if(!models.contains(model))
	    return false;

	int index = models.indexOf(model);
	if(index < -1 ||
	   (dummyModelEnabled && 
	    index == models.size() - 1 ))
	    return false;
	models.removeElementAt(index);
	if(model != null)
	    model.deleteObserver(this);
	textListModel.replaceItems(index, index + 1,
			   new String[0][textListModel.getColumnCount()]);
	return true;
    }

    /**
     * Remove all <code>NamedObjectEditModel</code>s from <code>NamedObjectEditModel</code> list
     *
     */
    public void removeAllModels()
    {
	Enumeration e = models.elements();
	while(e.hasMoreElements()) {
	    ((NamedObjectEditModel)e.nextElement()).deleteObserver(this);
	}
	models.clear();
	textListModel.replaceItems(0, textListModel.getItemCount(),
			   new String[0][textListModel.getColumnCount()]);
	if(dummyModelEnabled)
	    addDummyModel(dummyModelEnabled);
    }

    protected void addDummyModel(boolean enabled)
    {
	dummyModelEnabled = false;
	addModel(textListModel.getItemCount(), getDummyModel(), getEmptySummary());
	dummyModelEnabled = enabled;
    }


    /**
     * Returns <code>NamedObjectEditModel</code> at <code>index</code>
     *
     * @param index  where <code>model</code> is
     * @return <code>NamedObjectEditModel</code> at  <code>index</code>
     */
    public NamedObjectEditModel getModel(int index)
    {
	if(index < 0 || index >= models.size()) 
	    return null;
	return (NamedObjectEditModel)models.elementAt(index);
    }

    /**
     * Selects <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModel(NamedObjectEditModel model)
    {
	selectModelAt(models.indexOf(model));
    }

    /**
     * Selects an <code>NamedObjectEditModel</code> at <code>index</code> of list
     *
     * @param index of <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModelAt(int index)
    {
	textListModel.setSelection(index, index);
    }

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel</code> selected
     */
    public NamedObjectEditModel getSelectedModel()
    {
	int index = getSelectedIndex();
	int itemCount = textListModel.getItemCount();
	if(dummyModelEnabled)
	    itemCount--;
	if(index == -1 || index >= models.size())
	    //if(index == -1 || index >= textListModel.getItemCount())
	    //	if(index == -1 || index >= itemCount)
	    return null;

	return (NamedObjectEditModel)(models.elementAt(index));
    }

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel</code> selected
     */
    public NamedObjectEditModel[] getSelectedModels()
    {
	int indices[] = getSelectedIndexes();
	if(indices.length == 0)
	    return null;
	
	NamedObjectEditModel[] selectedModels = 
	    new NamedObjectEditModel[indices.length];

	for(int i = 0; i < indices.length; i++)
	    selectedModels[i] = 
		(NamedObjectEditModel)models.elementAt(indices[i]);

	return selectedModels;
    }

    /**
     * Returns index of selected <code>NamedObjectEditModel</code>
     *
     * @return int index of selected <code>NamedObjectEditModel</code>
     */
    public int getSelectedIndex()
    {
	int[] selected = getSelectedIndexes();

	if(selected.length > 0) {
	    return selected[0];
	}
	else {
	    return -1;
	}
    }

    /**
     * Returns index of selected <code>NamedObjectEditModel</code>
     *
     * @return int index of selected <code>NamedObjectEditModel</code>
     */
    public int[] getSelectedIndexes()
    {
	return textListModel.getSelectedIndexes();
    }

    /**
     * Returns index of <code>model</code>
     *
     * @param model <code>NamedObjectEditModel</code> to be searched
     * @return int index of <code>model</code>
     */
    public int indexOf(NamedObjectEditModel model)
    {
	return models.indexOf(model);
    }

    /**
     * Returns index of <code>model</code>
     *
     * @param model <code>NamedObjectEditModel</code> to be searched
     * @return int index of <code>model</code>
     */
    public int indexOfModel(NamedObjectEditModel model)
    {
	return models.indexOf(model);
    }

    /**
     * Enables or desables automatic addition of a dummy
     * <code>NamedObjectEditModel</code> to the list according to
     * value of <code>enable</code>
     *
     * @param enable <code>boolean</code> enables 
     * automatic addition of a dummy <code>NamedObjectEditModel</code>
     * if true or disables if false
     */
    public void enableDummyModel(boolean enable)
    {
	if(dummyModelEnabled == enable)
	    return;

	int index = models.size();

	if(enable) {
	    summaryTextArray = createSummaryTextArray();
	    addModel(index, getDummyModel(), getEmptySummary());
	}
	else {
	    index--;
	    summaryTextArray = null;
	    models.removeElementAt(index);
	    textListModel.replaceItems(index, index + 1,
				       new String[0][textListModel.getColumnCount()]);
	}

	dummyModelEnabled = enable;
    }

    /**
     * Returns true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     *
     * @return true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     */
    public boolean isDummyModelEnabled()
    {
	return dummyModelEnabled;
    }

    /**
     * Returns a dummy <code>NamedObjectEditModel</code> instance for double-clicking
     * operation.
     *
     * @return null to be considered as a dummy <code>NamedObjectEditModel</code>
     */
    public /*Object*/NamedObjectEditModel getDummyModel()
    {
	return dummyModel;
    }

    /**
     * Returns Text array representing empty summary 
     * can be used to represent a dummy <code>NamedObjectEditModel</code>.
     *
     * @return Text[][] representing a summary of an empty <code>NamedObjectEditModel</code>.
     */
    public Text[][] getEmptySummary()
    {
	if(summaryTextArray == null) 
	    summaryTextArray = createSummaryTextArray();

	return summaryTextArray;
    }

    protected Text[][] createSummaryTextArray()
    {
	if(textListModel instanceof ModelSelector)
	    return ((ModelSelector)textListModel).getEmptySummary();

	int columns = textListModel.getColumnCount();
	Text[] textArray = new Text[columns];
	for (int i = 0; i < columns; i++)
	    textArray[i] = EMPTY_TEXT;
	return new Text[][]{textArray};
    }


    /**
     * Returns <code>Vector</code> of <code>NamedObjectEditModel</code>
     * accompanying to the <code>ListModel</code>
     */
    public Vector getModels()
    {
	//	return models;
	Vector withoutDummy =
	    (Vector)models.clone();
	if(dummyModelEnabled)
	    withoutDummy.removeElementAt(withoutDummy.size() - 1);
	return withoutDummy;
    }

    /**
     * Sets <code>models</code> as <code>NamedObjectEditModel</code> list
     * accompanying to the <code>TextListModel</code> handled by the
     * <code>TextListController</code>
     */
    public void setModels(Vector models)
    {
	if(this.models == models)
	    return;
	if(this.models != null) {
	    Enumeration e = this.models.elements();
	    while(e.hasMoreElements()) {
		NamedObjectEditModel model
		    = (NamedObjectEditModel)e.nextElement();
		if(model != null)
		    model.deleteObserver(this);
	    }
	}
	if(models == null)
	    models = new Vector();
	this.models = models;
	Enumeration e = models.elements();
	Text[][] texts = new Text[models.size()][];
	
	int i = 0;
	while(e.hasMoreElements()) {
	    NamedObjectEditModel model
		= (NamedObjectEditModel)e.nextElement();
	    texts[i++] = model.getSummaryTextArray();
	    model.addObserver(this);
	}
	//	textListModel.removeSelection(0, models.size() - 1);
	textListModel.removeSelection(0, textListModel.getItemCount()-1);
	textListModel.replaceItems(0, textListModel.getItemCount(), trimSingleColumn(texts));
	if(dummyModelEnabled)
	    addModel(models.size(), getDummyModel(), getEmptySummary());
    }

    /**
     * Updates summary text of all specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update()
    {
	for (int i=0; i<models.size(); i++) {
		update(i);
	}
    }

    /**
     * Updates summary text of specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update(NamedObjectEditModel model)
    {
	int index = indexOf(model);
	if(index == -1)
	    addModel(model);
	update(indexOf(model));
    }

    /**
     * Updates summary text of model specified by <code>index</code>
     *
     * @param index index of <code>NamedObjectEditModel</code> to be updated
     */
    public void update(int index)
    {
	int itemCount = models.size();
	if (itemCount == 0)
		return;
	if (index <= -1 || index > itemCount) {
	    index = itemCount;
	    if(dummyModelEnabled)
		index--;
	}

	NamedObjectEditModel model = 
	    (NamedObjectEditModel)models.elementAt(index);
	if(model == getDummyModel())
	    textListModel.replaceItems(index, index + 1, 
				       trimSingleColumn(getEmptySummary()));
	else if(model != null) {
	    textListModel.replaceItems(index, index + 1, 
				       trimSingleColumn(new Text[][]{model.getSummaryTextArray()}));
	}
    }

    /**
     * Returns true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     *
     * @return true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     */
    public boolean contains(NamedObjectEditModel model)
    {
	return models.contains(model);
    }

    /**
     * Returns <code>Vector</code> of <code>NamedObjectEditModel</code>
     * accompanying to the <code>ListModel</code>
     */
    public Vector getNamedObjects()
    {
	Vector v = getModels();
	if(v == null || v.size() == 0)
	    return null;

	Vector objects = new Vector();
	Enumeration e = v.elements();
	while(e.hasMoreElements()) {
	    NamedObjectEditModel model = 
		(NamedObjectEditModel)e.nextElement();
	    if(isDummyModelEnabled() && model == getDummyModel())
		continue;
	    objects.addElement(model.getNamedObject());
	}
	e = null;
	v = null;
	return objects;
	
    }

    public void update(Observable observable, Object arg)
    {
	update((NamedObjectEditModel)observable);
	setChanged();
	notifyObservers();
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one above
     */
    public boolean up(int index)
    {
	return moveModel(index, index - 1);
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one below
     */
    public boolean down(int index)
    {
	return moveModel(index, index + 1);
    }


    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>source</CODE> to <CODE>destination</CODE>
     */
    public boolean moveModel(int source, int destination)
    {
	int items = textListModel.getItemCount() - 1;
	if(source == destination ||
	   destination < 0 || destination > items ||
	   source < 0 || source  >items)
	    return false;

	NamedObjectEditModel model = getModel(source);
	removeModel(model);
	addModel(destination, model);
	return true;
    }


    protected Text[][] getSummaryTextArray(NamedObjectEditModel model)
    {
	if(textListModel.getColumnCount() == 1)
	    return new Text[][]{new Text[]{model.getSummaryText()}};
	else
	    return new Text[][]{model.getSummaryTextArray()};
    }
}
