/*
 * ModelSelector.java:  an iterface class supporting choice of NamedObjectEditModels
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ModelSelector.java,v 1.9 2002/10/11 03:42:54 nozomi Exp $
 * $Log: ModelSelector.java,v $
 * Revision 1.9  2002/10/11 03:42:54  nozomi
 * add up(int), down(int) and moveModel(int, int) methods)
 *
 * Revision 1.8  2002/09/17 08:29:56  nozomi
 * add/removeModel() methods return boolean; add removeAll()
 *
 * Revision 1.7  2002/09/06 02:11:46  nozomi
 * add getNamedObjects()
 *
 * Revision 1.6  2002/09/05 15:09:28  nozomi
 * add getNamedObjects() method
 *
 * Revision 1.5  2002/05/25 16:55:20  nozomi
 * fix model selection bug
 *
 * Revision 1.4  2002/05/22 23:57:00  nozomi
 * add contains method
 *
 * Revision 1.3  2002/05/20 01:57:57  nozomi
 * ModelSelector re-defind to be based on TextListModel
 *
 * Revision 1.2  2002/05/17 17:38:59  nozomi
 * ModelSelectors moved to org.nomencurator.editor.model
 *
 * Revision 1.1  2002/02/26 16:31:19  nozomi
 * initial import
 *
 */

package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Vector;

import jp.kyasu.awt.TextListModel;

import jp.kyasu.graphics.Text;

/**
 * Interface to selcot <code>NamedObjectEditModel</code>s
 *
 * @version 	11 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public interface ModelSelector
{
    /**
     * Adds <code>model</code> to <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be added to this selector
     */
    public boolean addModel(NamedObjectEditModel model);

    /**
     * Inserts <code>model</code> at <code>index</code> of <code>NamedObjectEditModel</code> list.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     */
    public boolean addModel(int index, NamedObjectEditModel model);

    /**
     * Inserts <code>model</code> at <code>index</code> of model list
     * with its representation given by <code>items</code>.
     *
     * @param index  where <code>model</code> to be added
     * @param model <code>NamedObjectEditModel</code> to be added
     * @param items an array of <CODE>Object</CODE>s representing contents of the <CODE>model</CODE>
     */
    public boolean addModel(int index, NamedObjectEditModel model, Object[][] items);

    /**
     * Removes <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be removed from this selector
     */
    public boolean removeModel(NamedObjectEditModel model);

    /**
     * Remove all <code>NamedObjectEditModel</code>s from <code>NamedObjectEditModel</code> list
     *
     */
    public void removeAllModels();

    /**
     * Returns <code>NamedObjectEditModel</code> at <code>index</code>
     *
     * @param index  where <code>model</code> is
     * @return <code>NamedObjectEditModel</code> at  <code>index</code>
     */
    public NamedObjectEditModel getModel(int index);

    /**
     * Selects <code>model</code> from <code>NamedObjectEditModel</code> list
     *
     * @param model <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModel(NamedObjectEditModel model);

    /**
     * Selects an <code>NamedObjectEditModel</code> at <code>index</code> of list
     *
     * @param index of <code>NamedObjectEditModel</code> to be selected
     */
    public void selectModelAt(int index);

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel</code> selected
     */
    public NamedObjectEditModel getSelectedModel();

    /**
     * Returns selected <code>NamedObjectEditModel</code>
     *
     * @return <code>NamedObjectEditModel[]</code> selected
     */
    public NamedObjectEditModel[] getSelectedModels();

    /**
     * Returns index of selected <code>NamedObjectEditModel</code>
     *
     * @return int index of selected <code>NamedObjectEditModel</code>
     */
    public int getSelectedIndex();

    /**
     * Returns index of selected <code>NamedObjectEditModel</code>
     *
     * @return int index of selected <code>NamedObjectEditModel</code>
     */
    public int[] getSelectedIndexes();

    /**
     * Returns index of <code>model</code>
     *
     * @param model <code>NamedObjectEditModel</code> to be searched
     * @return int index of <code>model</code>
     */
    public int indexOf(NamedObjectEditModel model);

    /**
     * Enables or desables automatic addition of a dummy
     * <code>NamedObjectEditModel</code> to the list according to
     * value of <code>enable</code>
     *
     * @param enable <code>boolean</code> enables 
     * automatic addition of a dummy <code>NamedObjectEditModel</code>
     * if true or disables if false
     */
    public void enableDummyModel(boolean enable);

    /**
     * Returns true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     *
     * @return true if automatic addition of a dummy 
     * <code>NamedObjectEditModel</code> is enabled
     * or false if disabled.
     */
    public boolean isDummyModelEnabled();

    /**
     * Returns a dummy <code>NamedObjectEditModel</code> instance for double-clicking
     * operation.
     *
     * @return null to be considered as a dummy <code>NamedObjectEditModel</code>
     */
    public NamedObjectEditModel getDummyModel();

    /**
     * Returns Text array representing empty summary 
     * can be used to represent a dummy <code>NamedObjectEditModel</code>.
     *
     * @return Text[][] representing a summary of an empty <code>NamedObjectEditModel</code>.
     */
    public Text[][] getEmptySummary();

    /**
     * Returns <code>Vector</code> of <code>NamedObjectEditModel</code>
     * accompanying to the <code>ListModel</code>
     */
    public Vector getModels();

    /**
     * Sets <code>models</code> as <code>NamedObjectEditModel</code> list
     * accompanying to the <code>ListModel</code>
     */
    public void setModels(Vector models);

    /**
     * Updates summary text of all specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update();

    /**
     * Updates summary text of specified <code>model</code> in the list
     *
     * @param model <code>NamedObjectEditModel</code> to be updated
     */
    public void update(NamedObjectEditModel model);

    /**
     * Updates summary text of model specified by <code>index</code>
     *
     * @param index index of <code>NamedObjectEditModel</code> to be updated
     */
    public void update(int index);

    /**
     * Returns <code>TextListModel</code> representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     *
     * @return TextListModel representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     */
    public TextListModel getModel();

    /**
     * Sets <code>textListModel</code> as a summarized representation
     * <code>NamedObjectEditModel</code> handled by this selector
     *
     * @param textListModel <code>TextListModel</code> representing summaries of 
     * <code>NamedObjectEditModel</code> handled by this selector
     */
    public void setModel(TextListModel textListModel);

    /**
     * Returns true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     *
     * @return true if this <code>ModelSelector</code> contains <code>model</code>
     * or false if not.
     */
    public boolean contains(NamedObjectEditModel model);

    /**
     * Returns <code>Vector</code> of <code>NamedObject</code>s
     * accompanying to the <code>ListModel</code>
     */
    public Vector getNamedObjects();


    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one above
     */
    public boolean up(int index);

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one below
     */
    public boolean down(int index);

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>source</CODE> to <CODE>destination</CODE>
     */
    public boolean moveModel(int source, int destination);
}
