/*
 * NameHistoryModel.java:  a class to compare multiple hierarchies
 * for TaxoNote based on Nomencurator data model
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: NameHistoryModel.java,v 1.5 2002/09/02 15:58:12 nozomi Exp $
 * $Log: NameHistoryModel.java,v $
 * Revision 1.5  2002/09/02 15:58:12  nozomi
 * display unspecified at last in each history table
 *
 * Revision 1.4  2002/09/01 10:52:13  nozomi
 * change method name from set/setName to set/getUsedName
 *
 * Revision 1.3  2002/08/30 03:29:56  nozomi
 * tentative fix of duplicate selection bug
 *
 * Revision 1.2  2002/08/29 19:49:40  nozomi
 * change method's name from getViewString() to getViewName()
 *
 * Revision 1.1  2002/08/28 00:23:07  nozomi
 * initial import into CVS repository
 *
 */

package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextBuffer;

import org.nomencurator.Rank;
import org.nomencurator.broker.NamedObjectBroker;

/**
 * The <code>NameHistoryModel</code> is a class to compare
 * multiple hierarchies of names
 *
 * @version 	02 Sep 2002
 * @author 	Nozomi `James' Ytow
 */

public class NameHistoryModel
{
    protected static Text emptyText = new Text(" ");
    
    /** <CODE>Vector</CODE> of <CODE>NameTreeModel</CODE>s */
    protected Vector hierarchies;

    protected Hashtable views;

    protected static String unspecifiedParentName = "unspecified";

    /**
     * <CODE>Hashtable</CODE> with key of each taxonomic view's name,
     * of which values are <CODE>Hashtable</CODE>s of 
     * <CODE>NameUsageNodes</CODE>s in the view with their name as the key.
     * It provides a multiple-volumed dictionary of <CODE>NameUsageNode</CODE>
     */
    protected Hashtable names;

    /**
     * <CODE>Hashtable</CODE> of <CODE>Text</CODE>[][]
     * with key of rank names.
     * It is used to store histories of names under ranks
     * represented as <CODE>TextListModel</CODE>s.
     */
    protected Hashtable histories;

    /**
     * <CODE>Hashtable</CODE>  with key of rank names,
     * of which values are <CODE>Hashtable</CODE>
     * with key of parent taxa names,
     * of which values are <CODE>Vector</CODE> of
     * <CODE>String</CODE>s representing names in the rank
     * under the parent taxon.
     */
    protected Hashtable higherTaxaNames;

    /**
     * <CODE>Hashtable</CODE> with key of each taxonomic view's name,
     * of which values are <CODE>Hashtable</CODE>s with key of
     * rank names, of which values are <CODE>Hashtable</CODE>
     * with key of parent <CODE>NameUsageNode</CODE>'s name (including
     * "unspecified" for nodes without parent), of which values are
     * <CODE>Hashtable</CODE> with key of <CODE>NameUsageNode</CODE>'s name
     * at the rank, of which values are <CODE>NameUsageNode</CODE>
     * itself.
     * It is used to create <CODE>TextListModel</CODE>s in <CODE>histories</CODE>
     */
    protected Hashtable relatives;


    /**
     * <CODE>Hashtable</CODE> with key of rank names,
     * of which values are <CODE>Hashtable</CODE>s with key of
     * each taxonomic view's name, of which values are <CODE>Hashtable</CODE>
     * with key of parent <CODE>NameUsageNode</CODE>'s name (including
     * "unspecified" for nodes without parent), of which values are
     * <CODE>Hashtable</CODE> with key of <CODE>NameUsageNode</CODE>'s name
     * at the rank, of which values are <CODE>NameUsageNode</CODE>
     * itself.
     * It is used to create <CODE>TextListModel</CODE>s in <CODE>histories</CODE>
     */
    protected Hashtable siblings;


    public NameHistoryModel()
    {
    }

    public NameUsageNode getNameUsageNode(String name, String view)
    {
	if(names != null)
	    return (NameUsageNode)(((Hashtable)(names.get(view))).get(name));
	return null;
    }

    public void add(NameTreeModel tree)
    {
	add(tree, -1);
    }

    public void add(NameTreeModel tree, int index)
    {
	add(tree, index, (NameUsageNode)(tree.getRoot()));
    }

    public void add(NameTreeModel tree, int index, NameUsageNode root)
    {
	// initialize holders if it is the first time
	if(hierarchies == null) {
	    hierarchies = new Vector();
	    views = new Hashtable();
	    relatives =  new Hashtable();
	    siblings =  new Hashtable();
	    histories = new Hashtable();
	    names = new Hashtable();
	    higherTaxaNames = new Hashtable();
	}

	// if the tree is already contained, do nothing
	if(hierarchies.contains(tree))
	    return;

	//Need to use?
	//NamedObjectBroker broker = NamedObjectBroker.getInstance();

	// name of the view
	String viewName = tree.getViewName();

	NameTreeModel view = (NameTreeModel)views.get(viewName);

	if(view != null) {
	    NameUsageNode viewRoot = (NameUsageNode)view.getRoot();
	    if(viewRoot.isNodeDescendant(root))
		return;

	    else if(root.isNodeDescendant(viewRoot)) {
		index = indexOf(view);
		remove(view);
		add(tree, index, root);
		return;
	    }
	    else {
		//!?
	    }

	}

	// if index == -1, append the tree at the end
	if(index < 0)
	    index = hierarchies.size();
	
	// put the tree at specified index
	hierarchies.insertElementAt(tree, index);

	views.put(viewName, tree);

	Hashtable nameList = 
	    (Hashtable)(names.get(viewName));
	if(nameList == null) {
	    nameList = new Hashtable();
	    names.put(viewName, nameList);
	}

	Hashtable relative =
	    (Hashtable)(relatives.get(viewName));
	if(relative == null) {
	    relative = new Hashtable();
	    relatives.put(viewName, relative);
	}

	//traverse  the tree under root
	Enumeration e = root.breadthFirstEnumeration();
	while(e.hasMoreElements()) {
	    //get the node to be evaluate
	    NameUsageNode node =
		(NameUsageNode)(e.nextElement());

	    String nodeName = node.getName();

	    //put the node to the "dictionary"
	    nameList.put(nodeName, node);
	
	    //rank and its name will be used repeatedly.
	    Rank rank = node.getNameUsageEditModel().getRank();
	    String rankName = rank.getName();

	    // get or create Hashtable with key of
	    // rank name
	    Hashtable relativesOfRank = 
		(Hashtable)(relative.get(rankName));
	    if(relativesOfRank == null) {
		relativesOfRank = new Hashtable();
		relative.put(viewName, relativesOfRank);
	    }

	    // get or create Hashtable with key of
	    // rank name
	    Hashtable sibling = 
		(Hashtable)(siblings.get(rankName));
	    if(sibling == null) {
		sibling = new Hashtable();
		siblings.put(rankName, sibling);
	    }

	    // get or create Hashtable with key of
	    // view name
	    /*
	    Hashtable siblingOfView = 
		(Hashtable)(sibling.get(viewName));
	    if(siblingOfView == null) {
		siblingOfView = new Hashtable();
		sibling.put(viewName, siblingOfView);
	    }
	    */

	    //provide table to put higher taxa's names
	    Hashtable higherTaxa = 
		(Hashtable)(higherTaxaNames.get(rankName));
	    if(higherTaxa == null) {
		higherTaxa = new Hashtable();
		higherTaxaNames.put(rankName, higherTaxa);
	    }

	    NameUsageNode parent = (NameUsageNode)(node.getParent());
	    String parentName = (parent == null)?
		unspecifiedParentName : parent.getUsedName();

	    Hashtable views =
		(Hashtable)(sibling.get(parentName));
	    if(views == null) {
		views = new Hashtable();
		sibling.put(parentName, views);
	    }
	    Hashtable children = 
		(Hashtable)views.get(viewName);
	    if(children == null) {
		children = new Hashtable();
		views.put(viewName, children);
		relativesOfRank.put(parentName, children);
	    }

	    nodeName = node.getUsedName();
	    //	    children.put(node.getUsedName(), node);
	    children.put(nodeName, node);

	    Hashtable higherTaxaList =
		(Hashtable)(higherTaxa.get(parentName));
	    if(higherTaxaList == null) {
		higherTaxaList = new Hashtable();
		higherTaxa.put(parentName, higherTaxaList);
	    }

	    boolean typeNess = node.getNameUsageEditModel().isType();
	    Boolean t = (Boolean)(higherTaxaList.get(nodeName));
	    if(t != null)
		typeNess |= t.booleanValue();
	    //	    higherTaxaList.put(node.getUsedName(), 
	    higherTaxaList.put(nodeName,
			       typeNess? Boolean.TRUE:Boolean.FALSE);
	}

	///	histories = getHistoryTable();

    }

    public Hashtable getHistoryTable()
    {
	histories = new Hashtable(higherTaxaNames.size());
	String[] rankNames = Rank.getRankNames();

	// 1 for higher taxon name column
	int columns = hierarchies.size() + 1;

	int maxNameLength[] = new int[columns];

	for(int i = 0; i < rankNames.length; i++) {
	    Hashtable higherTaxa = 
		(Hashtable)(higherTaxaNames.get(rankNames[i]));
	    if(higherTaxa == null)
		continue;

	    //initialize maxnameLength
	    for(int j = 0; j < columns; j++)
		maxNameLength[j] = 0;

	    //calcurate number of seprators between higher taxa
	    /*
	    int rows = 0;
	    */
	    int rows = higherTaxa.size();
	    System.err.println(rows + " rows");
	    int separatorAt[] = null;
	    if(rows > 1)
		separatorAt = new int[rows - 1];
	    rows--;

	    //count rows to show lower taxa
	    Enumeration enum = higherTaxa.elements();
	    while(enum.hasMoreElements()) {
		rows += ((Hashtable)(enum.nextElement())).size();
	    }

	    Text[][] history = new Text[rows][];
	    histories.put(rankNames[i], history);

	    int row = 0;
	    for( ; row < rows; row++)
		history[row] = new Text[columns];

	    Hashtable sibling =
		(Hashtable)(siblings.get(rankNames[i]));
	    
	    row = 0;
	    int unspecified = -1;
	    enum = higherTaxa.elements();
	    Enumeration parentNames = higherTaxa.keys();
	    int separator = 0;
	    while(enum.hasMoreElements()) {
		String parentName =
		    (String)(parentNames.nextElement());
		if(parentName.equals(unspecifiedParentName)) {
		    unspecified = row;
		    row = history.length - 1;
		}
		Enumeration nodeNames =
		    ((Hashtable)enum.nextElement()).keys();
		boolean putParentName = true;
		int column = 0;

		Hashtable views = 
		    (Hashtable)(sibling.get(parentName));
		if(views == null)
		    continue;

		while(nodeNames.hasMoreElements()) {
		    if(row > 0 && separatorAt != null) {
			separatorAt[separator++] = row++;
		    }
		    if(putParentName) {
			maxNameLength[column] = parentName.length();
			history[row][column++] = new Text(parentName);
			putParentName = false;
		    }
		    else {
			history[row][column++] = emptyText;
		    }

		    String nodeName =
			(String)(nodeNames.nextElement());
		    
		    Enumeration view = names.keys();
		    while(view.hasMoreElements()){
			String viewName =
			    (String)(view.nextElement());
			Hashtable children =
			    (Hashtable)(views.get(viewName));
			if(children == null)
			    history[row][column] = emptyText;
			else {
			    NameUsageNode child =
				(NameUsageNode)(children.get(nodeName));
			    if(child == null)
				history[row][column] = emptyText;
			    else {
				TextBuffer buffer = new TextBuffer();
				if(child.getNameUsageEditModel().isType())
				    buffer.setColor(child.getNameUsageEditModel().getTypeColor());
				buffer.append(child.getUsedName());
				history[row][column] = buffer.toText();
				buffer = null;
			    }
			}
			if(maxNameLength[column] < history[row][column].length())
			    maxNameLength[column] = history[row][column].length();
		    }
		    column = 0;
		    if(parentName.equals(unspecifiedParentName)) {
			row = unspecified;
		    }
		    else
			row++;
		}
	    }
	    if(separatorAt != null) {
		for(int j = 0; j < separator; j++) {
		    for(int k = 0; k < columns; k++) {
			TextBuffer buffer = new TextBuffer('-');
			for(int l = 0; l < maxNameLength[k]; l++)
			    buffer.append('-');
			System.err.println(j + ", " + k + "/" + maxNameLength[k]);
			history[separatorAt[j]][k] =
			    buffer.toText();
		    }
		}
	    }
	}
	return histories;
    }

    public String[] getHistoryTitle()
    {
	String[] title = new String[names.size() + 1];
	int index = 0;
	title[index++] = "higher taxon";
	Enumeration views = names.keys();
	while(views.hasMoreElements()) {
	    title[index++] =
		(String)views.nextElement();
	}
	views = null;
	return title;
    }

    protected void traverse(NameTreeModel tree, NameUsageNode node)
    {
    }

    public void remove(NameTreeModel tree)
    {
	if(!hierarchies.contains(tree))
	    return;
	views.remove(tree.getViewName());
	hierarchies.remove(tree);

	if(hierarchies.size() == 0) {
	    higherTaxaNames = null;
	    names = null;
	    histories = null;
	    siblings = null;
	    relatives = null;
	    views = null;
	    hierarchies = null;
	}
    }

    public int indexOf(NameTreeModel tree)
    {
	return hierarchies.indexOf(tree);
    }

    public NameTreeModel getView(String viewName)
    {
	if(views == null || 
	   viewName == null ||
	   viewName.length() == 0)
	    return null;

	return (NameTreeModel)views.get(viewName);
    }

    public void clear()
    {
	hierarchies.clear();
	views.clear();
	relatives.clear();
	siblings.clear();
	histories.clear();
	names.clear();
	higherTaxaNames.clear();
	/*
	higherTaxaNames = null;
	names = null;
	histories = null;
	siblings = null;
	relatives = null;
	views = null;
	hierarchies = null;
	*/
    }

}
