/*
 * NameUsageEditModel.java:  a NameUsage editor model 
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: NameUsageEditModel.java,v 1.97 2002/10/08 21:38:14 nozomi Exp $
 * $Log: NameUsageEditModel.java,v $
 * Revision 1.97  2002/10/08 21:38:14  nozomi
 * remove unused codes
 *
 * Revision 1.96  2002/10/08 00:40:29  nozomi
 * introduce listenSubModels()
 *
 * Revision 1.95  2002/10/02 06:55:32  nozomi
 * improve type handling; eliminate annotationEditModels
 *
 * Revision 1.94  2002/10/02 06:16:42  t.okada
 * add this.annotationEditModels = new Vector();
 *
 * Revision 1.93  2002/10/02 02:49:05  nozomi
 * fix and improve rank handling
 *
 * Revision 1.92  2002/10/01 11:08:00  nozomi
 * make annotation modification accessors putlic
 *
 * Revision 1.91  2002/09/28 03:15:26  nozomi
 * add NameUsageEditModel(Rank) constructor
 *
 * Revision 1.90  2002/09/27 20:08:30  nozomi
 * getSummaryTextArray() uses isNominal()
 *
 * Revision 1.89  2002/09/26 06:33:31  ryo
 * fix NullPointerException bug
 *
 * Revision 1.88  2002/09/25 11:38:47  ryo
 * fix NullPointerException bug
 *
 * Revision 1.87  2002/09/20 13:14:56  nozomi
 * re-handle AppearanceEditModel given as argument to construcutor
 *
 * Revision 1.86  2002/09/20 09:05:31  nozomi
 * fix null-rank handling in saveAttributes()
 *
 * Revision 1.85  2002/09/17 05:46:02  nozomi
 * re-organise initialisation methods
 *
 * Revision 1.84  2002/09/09 16:59:40  nozomi
 * modify loadAttributes()
 *
 * Revision 1.83  2002/09/06 17:01:47  ryo
 * link NamedObjects when *EditModel linked
 *
 * Revision 1.82  2002/09/06 14:23:33  ryo
 * remove unnecessary line
 *
 * Revision 1.81  2002/09/06 02:20:37  nozomi
 * improve saveAttributes(); colour only name for type
 *
 * Revision 1.80  2002/09/04 02:18:48  nozomi
 * apply type-colouring only on name
 *
 * Revision 1.79  2002/09/03 17:00:36  ryo
 * use getViewName in setAuthority
 *
 * Revision 1.78  2002/09/03 16:39:11  ryo
 * use getViewName in updateSummary
 *
 * Revision 1.77  2002/09/03 16:00:20  ryo
 * comment out clearing lowerTaxa in saveAttributes()
 *
 * Revision 1.76  2002/09/03 13:32:53  t.okada
 * modify updateSummary method
 *
 * Revision 1.75  2002/09/03 02:59:42  nozomi
 * modify getViewName()
 *
 * Revision 1.74  2002/09/02 16:37:03  nozomi
 * use getDefaultRichTextEditModel() for note
 *
 * Revision 1.73  2002/09/01 10:53:35  nozomi
 * change method name from set/setName to set/getUsedName
 *
 * Revision 1.72  2002/08/29 19:49:40  nozomi
 * change method's name from getViewString() to getViewName()
 *
 * Revision 1.71  2002/08/28 00:31:44  nozomi
 * modify getViewString()
 *
 * Revision 1.70  2002/08/27 08:12:27  ryo
 * add the processing for adding type
 *
 * Revision 1.69  2002/08/27 03:15:51  nozomi
 * support getViewName() methods
 *
 * Revision 1.68  2002/08/26 06:44:08  ryo
 * use Rank#isLower to compare rank
 *
 * Revision 1.67  2002/08/23 08:19:36  ryo
 * add genus name abbreviation
 *
 * Revision 1.66  2002/08/23 08:15:31  nozomi
 * modification in athority text relating methods
 *
 * Revision 1.65  2002/08/23 07:33:02  nozomi
 * use setRichText in setAthorityRichText
 *
 * Revision 1.64  2002/08/23 07:22:24  nozomi
 * separate authority summary generator as methods
 *
 * Revision 1.63  2002/08/23 05:03:12  ryo
 * add removeAnnotation()
 *
 * Revision 1.62  2002/08/23 04:03:08  nozomi
 * change colour of summary text if it is type
 *
 * Revision 1.61  2002/08/23 04:00:08  nozomi
 * support type
 *
 * Revision 1.60  2002/08/22 13:26:04  t.okada
 * It corrects for a locale colum addition.
 *
 * Revision 1.59  2002/08/22 08:24:34  nozomi
 * support type
 *
 * Revision 1.58  2002/08/21 12:05:10  ryo
 * change the rule which generates authority string
 *
 * Revision 1.57  2002/08/19 21:40:45  nozomi
 * support partially-linked ranks
 *
 * Revision 1.56  2002/08/14 05:03:31  ryo
 * change the rule which generates authority string
 *
 * Revision 1.55  2002/08/09 05:26:22  nozomi
 * synchronize notifications
 *
 * Revision 1.54  2002/08/05 08:01:43  t.okada
 * DummyModel addition processing is corrected.
 *
 * Revision 1.53  2002/07/18 14:38:30  t.okada
 * note item is added to nameusage
 *
 * Revision 1.52  2002/07/03 13:20:42  t.okada
 * bugfix saveAttributes() method
 *
 * Revision 1.51  2002/07/02 08:29:08  nozomi
 * change default text model relating methods
 *
 * Revision 1.50  2002/06/21 23:22:58  nozomi
 * use RichText in TextComponents
 *
 * Revision 1.49  2002/06/11 12:26:33  ryo
 * add accessor of authority and add getCitationTextEditModel()
 *
 * Revision 1.48  2002/06/09 12:45:06  nozomi
 * change higher/lower handling
 *
 * Revision 1.47  2002/06/03 01:47:46  nozomi
 * setChanged() to call notification in removeLower()
 *
 * Revision 1.46  2002/06/03 01:41:34  nozomi
 * support insertion of higher node
 *
 * Revision 1.45  2002/05/31 21:25:32  nozomi
 * fix higher notification by setChanged()
 *
 * Revision 1.44  2002/05/28 07:33:54  ryo
 * add getLowerEditModelsBelowCurrent()
 *
 * Revision 1.43  2002/05/27 22:46:33  nozomi
 * implement symmetric constraints on higher/lower
 *
 * Revision 1.42  2002/05/27 14:19:37  ryo
 * add removeLower()
 *
 * Revision 1.41  2002/05/26 21:06:25  nozomi
 * automatic dummy addition support
 *
 * Revision 1.40  2002/05/25 17:18:18  nozomi
 * add NameUsageEditModel(NameUsageEditModel)
 *
 * Revision 1.39  2002/05/25 06:46:11  nozomi
 * use NameUsageListModel for lower taxa
 *
 * Revision 1.38  2002/05/21 03:37:15  nozomi
 * remove debug System.out.println()
 *
 * Revision 1.37  2002/05/17 15:53:25  t.okada
 * add getAnnotationEditModels() method
 *
 * Revision 1.36  2002/05/17 09:38:46  ryo
 * modify related to Annotation
 *
 * Revision 1.35  2002/05/16 12:46:36  ryo
 * modify addAnnotation()
 *
 * Revision 1.34  2002/05/16 11:25:36  ryo
 * add setAuthority() in saveAttributes()
 *
 * Revision 1.33  2002/05/16 09:15:22  ryo
 * modify updateList()
 *
 * Revision 1.32  2002/05/15 06:35:37  ryo
 * modify addLower()
 *
 * Revision 1.31  2002/05/14 10:15:34  ryo
 * add updateList()
 *
 * Revision 1.30  2002/05/10 13:45:17  ryo
 * add new constructor
 *
 * Revision 1.29  2002/05/10 12:21:38  t.okada
 * modify NameUsageEditModel constructor
 *
 * Revision 1.28  2002/05/09 00:02:39  nozomi
 * fix getSummaryTextArray
 *
 * Revision 1.27  2002/04/19 21:45:36  nozomi
 * relating model handling
 *
 * Revision 1.26  2002/04/17 16:23:00  nozomi
 * getNameUsage() may return null
 *
 * Revision 1.25  2002/04/16 23:43:52  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.24  2002/04/16 00:41:36  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.23  2002/04/01 07:06:58  nozomi
 * adapt to more protective Annotation
 *
 * Revision 1.22  2002/03/29 08:06:37  okawa
 * implements clone(), loadAttributes(), saveAttributes()
 *
 * Revision 1.21  2002/03/26 03:58:43  nozomi
 * use RankChoice to handle additional ranks
 *
 * Revision 1.20  2002/03/25 13:54:45  okawa
 * modify addAnnotation()
 *
 * Revision 1.19  2002/03/25 12:02:27  okawa
 * no message
 *
 * Revision 1.18  2002/03/20 06:30:43  okawa
 * modify setting TextModels
 *
 * Revision 1.17  2002/03/10 22:00:18  nozomi
 * add authority relating methods
 *
 * Revision 1.16  2002/03/10 06:58:33  nozomi
 * call notfyObservers() of model
 *
 * Revision 1.15  2002/03/09 23:37:44  nozomi
 * add incertae sedis
 *
 * Revision 1.14  2002/03/08 23:08:09  nozomi
 * minor modification of omments
 *
 * Revision 1.13  2002/03/08 14:05:46  okawa
 * avoid NollPointer
 *
 * Revision 1.12  2002/03/05 01:51:43  nozomi
 * change lower/higher handling
 *
 * Revision 1.11  2002/03/03 23:51:25  nozomi
 * add utility mthods to handle lower taxa
 *
 * Revision 1.10  2002/02/28 17:11:02  nozomi
 * use TextListModel of the package
 *
 * Revision 1.9  2002/02/27 23:59:45  nozomi
 * use DefaultTextListModel in the package
 *
 * Revision 1.8  2002/02/23 21:51:45  nozomi
 * Enable climbing up hierarchy
 *
 * Revision 1.7  2002/02/21 01:55:24  okawa
 * modification due to the integration with TabbedPane, EditPanel
 *
 * Revision 1.6  2002/02/16 21:58:31  nozomi
 * Rank names capitalised
 *
 * Revision 1.5  2002/02/07 20:59:13  nozomi
 * Use AnnotationListModel
 *
 * Revision 1.4  2002/02/07 08:36:15  nozomi
 * Add accessor methods
 *
 * Revision 1.3  2002/02/05 18:52:26  nozomi
 * Removed overriding summary field
 *
 * Revision 1.2  2002/01/29 07:23:05  nozomi
 * Made it Clonable
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */	


package org.nomencurator.editor.model;

import java.awt.Color;

import java.util.Enumeration;
import java.util.Vector;

import java.util.Observer;
import java.util.Locale;

import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.TextField;
import jp.kyasu.awt.TextModel;
import jp.kyasu.awt.DefaultTextEditModel;

import jp.kyasu.awt.event.TextModelEvent;

import jp.kyasu.graphics.BasicTSModifier;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextBuffer;
import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;

import org.nomencurator.Annotation;
import org.nomencurator.Appearance;
import org.nomencurator.NamedObject;
import org.nomencurator.NameUsage;
import org.nomencurator.Publication;
import org.nomencurator.Rank;

import org.nomencurator.broker.NamedObjectBroker;

/**
 * The model interface for an object that acts as a named object edit model
 *
 * @see 	org.nomencurator.editor.event.ObjectEditModelEvent;
 * @see 	org.nomencurator.editor.event.ObjectEditModelListener;
 *
 * @version 	08 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class NameUsageEditModel
    extends NamedObjectEditModel
    implements Cloneable 
{
    /** <code>TextListModel</code> for list of rank names */
    protected static jp.kyasu.awt.TextListModel rankList
	= new jp.kyasu.awt.DefaultTextListModel();
    
    /**
     * Initial contents of <code>rank</code>
     */
    protected static String[][] ranks = {
	{"Domain"},
	{"Kingdom"},
	{"Phylum"},
	{"Class"},
	{"Order"},
	{"Family"},
	{"Tribe"},
	{"Genus"},
	{"Section"},
	{"Series"},
	{"Species"}, 
	{"Variety"},
	{"Form"},
	{"other..."},
    };

    static {
	rankList.replaceItems(0, 0, ranks);
    }

    /** <code>Rank</code> of the <code>NameUsage</code> */
    protected Rank rank;

    /** <code>Rank</code> of the <code>NameUsage</code> */
    protected static Rank defaultRank = Rank.SPECIES;

    /** <code>TextEditModel</code> for name */
    protected TextEditModel nameModel;

    /**
     * <code>TextEditModel</code> for generic name 
     * will be used only if rank is species.
     */
    protected TextEditModel epithet; 

    /**
     * <code>TextEditModel</code> for variety name 
     * will be used only if rank is variety.
     */
    protected TextEditModel variety; 

    /**
     * <code>TextEditModel</code> for form name 
     * will be used only if rank is form.
     */
    protected TextEditModel form; 

    /**
     * TextEditModel for higher NameUsage
     */
    protected TextEditModel higher;

    /**
     * NameUsageEditModel of higher NameUsage
     */
    protected NameUsageEditModel higherEditModel;

    /**
     * Incertae sedis status of this NameUsage to higher one
     */
    protected boolean insertaeSedis;

    /** Typenss  of this NameUsage */
    protected boolean type;

    /** Type of type */
    protected String typeOfType;

    /**
     * <code>NameUsageListModel</code> for lower NameUsages
     */
//    protected NameUsageListModel lowerList;
    protected NameUsageListModel lowerList;

    /**
     * a list of lower NameUsageEditModels
     */
    //    protected Vector lowerEditModels;

    /**
     * <code>TextEditModel</code> for authority
     */
    protected TextEditModel authority;

    /**
     * <code>NameUsageEditModel</code> of authority <code>NameUsage</code>
     */
    protected NameUsageEditModel authorityEditModel;

    /** <code>EditModel</code> for appearance*/
    protected AppearanceEditModel appearanceEditModel;

    /** <code>TextEditModel</code> for note */
    protected TextEditModel note;

    /** TextListModel for annotaions */
    protected AnnotationListModel annotationList;

    /** Locale */
    protected Locale locale;

    /** Locale */
    protected static final Locale EMPTY_LOCALE = new Locale(" ", "");

    /** Keep Locale */
    protected static Locale keepLocale;

    protected static Color defaultTypeColor = Color.red;
    protected static Color typeColor = defaultTypeColor;
    protected static BasicTSModifier typeStyleModifier
	= new BasicTSModifier();
    static {
	typeStyleModifier.put(BasicTSModifier.COLOR, typeColor);
    }

    protected static jp.kyasu.awt.TextListModel typeListModel =
	new jp.kyasu.awt.DefaultTextListModel();

    static {
	typeListModel.replaceItems(0, 0, 
				 new Text[][] {
				     { new Text(NameUsage.HOLOTYPE) },
				     { new Text(NameUsage.LECTOTYPE) },
				     { new Text(NameUsage.ALLOTYPE) },
				     { new Text(NameUsage.COTYPE) },
				     { new Text(NameUsage.GENOTYPE) },
				     { new Text(NameUsage.HAPANTOTYPE) },
				     { new Text(NameUsage.NEOTYPE) },
				     { new Text(NameUsage.PARALECTOTYPE) },
				     { new Text(NameUsage.PARATYPE) },
				     { new Text(NameUsage.SYNTYPE) },
				     { new Text(NameUsage.TOPOTYPE) },
				     { new Text(NameUsage.TYPE_SERIES) },
				     { new Text(NameUsage.TYPE_HOST) }
				 });
    }

    protected static NameUsageEditModel template;

    protected boolean annotationsModified;

    /** The constructor of NameUsageEditModel */
    public NameUsageEditModel()
    {
	this(defaultRank);
    }

    /** The constructor of NameUsageEditModel */
    public NameUsageEditModel(Rank rank)
    {
	this(true);
	this.rank = rank;
	getNameUsage().setRank(rank.getName());
    }

    /**
     * The constructor of NameUsageEditModel
     *
     * @param editable boolean determining whether the model is editable
     */
    public NameUsageEditModel(boolean editable)
    {
	this(new NameUsage(), editable);
    }

    /**
     * The constructor of NameUsageEditModel
     *
     * @param object <code>NameUsage</code> to be edited
     */
    public NameUsageEditModel(NameUsage object)
    {
	this(object, true);
    }

    /**
     * The constructor of NameUsageEditModel
     *
     * @param object <code>Object</code> to be edited
     * @param editable boolean determining whether the model is editable
     */
    public NameUsageEditModel(NameUsage object, boolean editable)
    {
	this(object, editable,
	     null, null, null, null, null, false);
    }

    /**
     * The constructor of NameUsageEditModel
     */
    public NameUsageEditModel(AppearanceEditModel appearanceEditModel)
    {
	this(new NameUsage(), true,
	     null, null, null, appearanceEditModel, null, false);
    }

    /**
     * The constructor of NameUsageEditModel
     */
    public NameUsageEditModel(NameUsageEditModel higherEditModel)
    {
	this(new NameUsage(), true,
	     higherEditModel, null, null, 
	     higherEditModel.getAppearanceEditModel(), null, false);
    }

    /**
     * The constructor of NameUsageEditModel
     *
     * @param object <code>Object</code> to be edited
     * @param editable boolean determining whether the model is editable
     * @param higherEditModel <code>NameUsageEditModel</code> for higher <code>NameUsage</code>
     * of the <code>NameUsage</code> edited by this model
     * @param lowerEditModels <code>Vector</code> of <code>NameUsageEditModel</code> for
     * lower <code>NameUsage</code>s of the <code>NameUsage</code> edited by this model
     * @param authority <code>NameUsageEditModel</code> for authority <code>NameUsage</code>
     * of the <code>NameUsage</code> edited by this model
     */
    public NameUsageEditModel(NameUsage object, 
			       boolean editable,
			       NameUsageEditModel higherEditModel,
			       Vector lowerEditModels,
			       NameUsageEditModel authority,
			       AppearanceEditModel appearance,
			       Vector annotationEditModels,
			       boolean insertaeSedis
			       )
    {
	super(object, editable);
	setAppearanceEditModel(appearance);
    }

    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void createSubModels()
    {
	createNameUsageTextModels();
	if(higher == null)
	    higher = getDefaultTextEditModel();

	createLowerTextModel();
	createAnnotationsTextModel();
	setAnnotationsModified(false);
    }
    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void listenSubModels()
    {
	listenNameUsageTextModels();
    }


    protected void createNameUsageTextModels()
    {
	summary = getDefaultTextEditModel();

	nameModel = getDefaultTextEditModel();
	epithet = getDefaultTextEditModel();
	variety = getDefaultTextEditModel();
	form = getDefaultTextEditModel();
	//note = getDefaultTextEditModel();
	note = getDefaultRichTextEditModel();

	locale = EMPTY_LOCALE;
    }

    protected void listenNameUsageTextModels()
    {
	nameModel.addTextModelListener(this);
	epithet.addTextModelListener(this);
	variety.addTextModelListener(this);
	form.addTextModelListener(this);
	note.addTextModelListener(this);
    }

    protected void setNameUsageTextModels(NameUsage nameUsage)
    {
	if(nameUsage == null) {
	    return;
	}
	summary = getDefaultTextEditModel();

	String name = nameUsage.getUsedName();
	if(name != null) {
	    nameModel.setRichText(new RichText(name, RichTextStyle.DEFAULT_DOCUMENT_STYLE));
	    nameModel.addTextModelListener(this);
	    // summary = new DefaultTextEditModel(new RichText(nameUsage.major, TextField.DEFAULT_FIELD_STYLE));
	    summary.setRichText(new RichText(name, RichTextStyle.DEFAULT_DOCUMENT_STYLE));
	}
	epithet = getDefaultTextEditModel();
	epithet.addTextModelListener(this);
	
	variety = getDefaultTextEditModel();
	variety.addTextModelListener(this);

	form = getDefaultTextEditModel();
	form.addTextModelListener(this);

	if(nameUsage.getAppearance() != null) {
	}
    }

    public void setHigherEditModel(NameUsageEditModel higherEditModel)
    {
	if(this.higherEditModel == higherEditModel) {
	    if(higherEditModel != null) {
		if(higherEditModel.lowerList.contains(this))
		    synchronized(this) {
			notifyObservers(getHigherModel());
		    }
		return;
	    }
	}

	if(this.higherEditModel != null) {
	    higherEditModel.removeLower(this);
	    higher = null;
	}

	if(this.higherEditModel != higherEditModel)
	    setChanged();

	this.higherEditModel = higherEditModel;

	if(higherEditModel == null) {
	    if(higher == null)
		higher = getDefaultTextEditModel();
	    return;
	}

	higher = higherEditModel.getSummaryTextModel();
	higherEditModel.addLower(this);

	NameUsage nameUsage = getNameUsage();
	if(nameUsage != null) 
	    nameUsage.setHigherTaxon(higherEditModel.getNameUsage());
    }

    protected void createLowerTextModel()
    {
	lowerList = new NameUsageListModel(isEditable());
    }

    public void addLower(NameUsageEditModel lowerEditModel)
    {
	addLower(-1, lowerEditModel);
    }

    public void addLower(int index, NameUsageEditModel lowerEditModel)
    {
	if(lowerEditModel == null ||
	   lowerList.contains(lowerEditModel)) {
	    //	    notifyObservers(lowerEditModel);
	    return;
	}
	lowerList.addModel(index, lowerEditModel);
	setChanged();
	lowerEditModel.setHigherEditModel(this);
	synchronized(this) {
	    notifyObservers(lowerEditModel);
	}
    }

    public void removeLower(NameUsageEditModel lowerEditModel)
    {
	lowerList.removeModel(lowerEditModel);
	setChanged();
	notifyObservers(lowerEditModel);
    }

    public int getLowerCount()
    {
	return lowerList.getItemCount();
    }

    protected void createAppearanceTextModel(AppearanceEditModel appearance)
    {
	if(appearance == null) {
	    appearance = new AppearanceEditModel();
	}
	appearance.addNameUsageEditModel(this);
    }

    public void setAppearanceEditModel(AppearanceEditModel appearanceEditModel)
    {
	if(this.appearanceEditModel == appearanceEditModel)
	    return;

	if(this.appearanceEditModel != null) 
	    this.appearanceEditModel.removeNameUsageEditModel(this);

	this.appearanceEditModel = appearanceEditModel;

	if(appearanceEditModel != null) {
	    this.appearanceEditModel.addNameUsageEditModel(this);
	    getNameUsage().setAppearance(appearanceEditModel.getAppearance());
	}

	updateSummary();
	setChanged();
	notifyObservers(appearanceEditModel);
    }

    public void setAuthorityEditModel(NameUsageEditModel authorityEditModel)
    {
	if(this.authorityEditModel == authorityEditModel &&
	   authority != null)
	    return;

	if (this.authorityEditModel != null)
	    authority = null;

	if(this.authorityEditModel != authorityEditModel)
	    setChanged();

	this.authorityEditModel = authorityEditModel;

	if (authorityEditModel == null) {
	    if(authority == null)
		authority = getDefaultTextEditModel();
	    return;
	}
	authority = authorityEditModel.getAppearanceEditModel().getPublicationEditModel().getViewNameModel();

	notifyObservers(authorityEditModel);
    }

    protected void setAuthorityEditModel()
    {
	setAuthorityEditModel((NameUsageEditModel)getEditModel(getNameUsage().getAuthority()));
    }

    public String getViewName()
    {
	//return getViewName(false);
	return getNameUsage().getViewName();
    }

    public String getViewName(boolean toSort)
    {/*
	if(appearanceEditModel != null) {
	    return getAuthorityString(appearanceEditModel, toSort);
	}
	else 
     */
	/*
	if(getNameUsage().getAppearance() != null)
	    return getAuthorityString(getNameUsage().getAppearance().getPublication(), toSort);
	else if(appearanceEditModel != null) {
	    return getAuthorityString(appearanceEditModel, toSort);
	}
	else 
	    return toString();
	*/
	return getViewName();
    }

    public String getAuthorityString()
    {
	return getAuthorityString(false);
    }

    public String getAuthorityString(boolean toSort)
    {
	if(authorityEditModel == null)
	    return "";
	return getAuthorityString(authorityEditModel.getAppearanceEditModel(), toSort);
    }

    public static String getAuthorityString(AppearanceEditModel model)
    {
	return getAuthorityString(model, false);
    }

    public static String getAuthorityString(AppearanceEditModel model, boolean toSort)
    {
	if(model == null)
	    return "";
	return getAuthorityString(model.getPublicationEditModel(), toSort);
    }

    public static String getAuthorityString(PublicationEditModel model)
    {
	return getAuthorityString(model, false);
    }

    public static String getAuthorityString(PublicationEditModel model, boolean toSort)
    {
	StringBuffer buffer = new StringBuffer();
	if(model != null) {
	    if(toSort) {
		buffer.append(model.getYearTextModel().getRichText().getText().toString());
		buffer.append(' ');
		buffer.append(model.getAuthorsSummaryTextModel().getRichText().getText().toString());
	    }
	    else {
		buffer.append(model.getAuthorsSummaryTextModel().getRichText().getText().toString());
		buffer.append(' ');
		buffer.append(model.getYearTextModel().getRichText().getText().toString());
	    }
	}
	return buffer.toString();
    }

    public static String getAuthorityString(Publication pub, boolean toSort)
    {
	StringBuffer buffer = new StringBuffer();
	if(pub != null) {
	    if(toSort) {
		buffer.append(pub.getYear());
		buffer.append(' ');
		buffer.append(pub.getAuthorNames());
	    }
	    else {
		buffer.append(pub.getAuthorNames());
		buffer.append(' ');
		buffer.append(pub.getYear());
	    }
	}
	return buffer.toString();
    }

    public static Text getAuthorityText(PublicationEditModel model)
    {
	return new Text(getAuthorityString(model));
    }

    
    public NameUsageEditModel getAuthorityEditModel()
    {
	if(authority == null)
	    setAuthorityEditModel();
	return authorityEditModel;
    }

    /**
     * Get <code>TextEditModel</code> from authority.
     */
    public TextEditModel getAuthorityTextModel()
    {
	if(authority == null)
	    setAuthorityEditModel();
    	return authority;
    }

    /**
     * Set <code>TextEditModel</code> to authority.
     * @param textEditModel
     */
    public void setAuthorityTextModel(TextEditModel textEditModel) {
	authority = textEditModel;
    }
    
    /**
     * Set <code>RichText</code> to authority.
     */
    public void setAuthorityRichText(RichText richText)
    {
	getAuthorityTextModel().setRichText(richText);
    }

    public RichText getAuthorityRichText()
    {
	return authority.getRichText();
    }

    public Text getAuthorityText()
    {
	return getAuthorityTextModel().getRichText().getText();
    }

    public String getAuthority()
    {
	return getAuthorityText().toString();
    }

    protected void createAnnotationsTextModel()
    {
	annotationList = new AnnotationListModel();
    }

    public void addAnnotation(AnnotationEditModel annotationEditModel)
    {
	annotationList.addModel(annotationEditModel);
    }

    public void removeAnnotation(AnnotationEditModel annotationEditModel)
    {
	annotationList.removeModel(annotationEditModel);
    }

    /**
     * Returns <code>Object</code> to be edited by this model.
     *
     */
    public NameUsage getNameUsage()
    {
	return (NameUsage)getObject();
    }

    /**
     * Returns <code>Rank</code> representing rank
     */ 
    public Rank getRank()
    {
	return rank;
    }

    /**
     * Sets <code>rank</code> as rank of the model
     */ 
    public void setRank(Rank rank)
    {
	if(this.rank != null && this.rank.equals(rank))
	    return;
	this.rank = rank;
	setChanged();
	notifyObservers(this.rank);
    }

    /**
     * Sets <code>rankName</code> as rank
     */ 
    public void setRank(String rankName)
    {
	Rank newRank = Rank.get(rankName);
	if(newRank == null)
	    newRank = new Rank(rankName, true);

	setRank(newRank);
    }
    /**
     * Returns default <code>Rank</code>
     */ 
    public static Rank getDefaultRank()
    {
	return defaultRank;
    }

    /**
     * Sets <code>rank</code> as default rank of models
     */ 
    public static void setDefaultRank(Rank rank)
    {
	defaultRank = rank;
    }


    /**
     * Returns <code>TextListModel</code> representing rank list
     */ 
    public static jp.kyasu.awt.TextListModel getRankModel()
    {
	return rankList;
    }

    /**
     * Returns name
     */ 
    public TextEditModel getNameModel()
    {
	return nameModel;
    }

    /**
     * Returns name
     */ 
    public Text getNameText()
    {
    	if (nameModel != null && nameModel.getRichText() != null) {
	    return nameModel.getRichText().getText();
    	}
    	return null;
    }

    /**
     * Returns name
     */ 
    public String getUsedName()
    {
	return getAscribedName();
    }
    /**
     * Returns name
     */ 
    public String getAscribedName()
    {
	/*
	String usedName = getNameText().toString();
	if(usedName == null || usedName.length() == 0)
	    usedName = getNameUsageModel().getUsedName();
	return usedName;
	*/
	return getNameText().toString();
    }

    /**
     * Returns name
     */ 
    /*
    public void setUsedName(String nameString)
    {
	nameModel.getRichText().getText().toString();
    }
    */

    /**
     * Returns epithet
     */ 
    public TextEditModel getEpithetModel()
    {
	return epithet;
    }

    /**
     * Returns epithet
     */ 
    public Text getEpithetText()
    {
	return epithet.getRichText().getText();
    }

    /**
     * Returns epithet
     */ 
    public String getEpithet()
    {
	return getEpithetText().toString();
    }

    /**
     * Returns note
     */ 
    public TextEditModel getNoteModel()
    {
	return note;
    }

    /**
     * Returns note
     */ 
    public Text getNoteText()
    {
	return note.getRichText().getText();
    }

    /**
     * Returns note
     */ 
    public String getNote()
    {
	return getNoteText().toString();
    }

    /**
     * Returns form
     */ 
    public TextEditModel getForm()
    {
	return form;
    }

    /**
     * Returns variety
     */ 
    public TextEditModel getVariety()
    {
	return variety;
    }

    /** Returns higher */ 
    public TextEditModel getHigher()
    {
	if(higher == null)
	    setHigherEditModel();
	return higher;
    }

    /**
     * Returns higherEditModel 
     */ 
    public NameUsageEditModel getHigherModel()
    {
	if(higher == null)
	    setHigherEditModel();
	return higherEditModel;
    }

    /**
     * <code>NameUsageListModel</code> for lower NameUsages
     */
    public NameUsageListModel getLower()
    {
	return lowerList;
    }

    /**
     *
     */
    public AppearanceEditModel getAppearanceEditModel()
    {
	if(appearanceEditModel == null)
	    setAppearanceEditModel((AppearanceEditModel)getEditModel(getNameUsage().getAppearance(), AppearanceEditModel.getInstance()));
	return appearanceEditModel;
    }

    /**
     * Returns <code>AnnotationListModel</code> representing <code>Annotation</code>s
     */
    public AnnotationListModel getAnnotationList()
    {
	return annotationList;
    }

    /**
     * Returns <code>AnnotationEditModels</code>
     */ 
    public Vector getAnnotationEditModels()
    {
	return annotationList.getModels();
    }

    /**
     * Returns <code>Object</code> to be edited by this model.
     *
     */
    public void setNameUsage(NameUsage object)
    {
    setObject(object);
    }

    public void clear()
    {
	createNameUsageTextModels();
    }

    public Object clone()
    {
	// return new NameUsageEditModel();
	NameUsageEditModel nameUsageEditModel = new NameUsageEditModel(getNameUsage());
	nameUsageEditModel.setHigherEditModel(getHigherModel());
	nameUsageEditModel.lowerList = getLower();
	nameUsageEditModel.authority = authority;
	nameUsageEditModel.appearanceEditModel = appearanceEditModel;
	nameUsageEditModel.annotationList = annotationList;
	nameUsageEditModel.insertaeSedis = insertaeSedis;
	nameUsageEditModel.note = note;
	
	return nameUsageEditModel;
    }

    /**
     * Gets summary contents in <code>Text</code>
     *
     * @return Text[] representing summary of the model
     */
    public Text[] getSummaryTextArray()
    {
	NameUsage n = getNameUsage();
	Text nameText = n.isNominal()? new Text(n.getAscribedName()):getNameText();
	
	Text authority = (getAuthorityEditModel() != null)? 
			            getAuthorityText() : new Text();
	Text year = (getAppearanceEditModel() != null 
		     && getAppearanceEditModel().getPublicationEditModel() != null 
		     && getAppearanceEditModel().getPublicationEditModel().getYearTextModel() != null)? 
	    getAppearanceEditModel().getPublicationEditModel().getYearTextModel().getRichText().getText() : new Text();

	Text rankText = n.isNominal()? new Text(n.getRank()):
	    ((rank == null)? new Text():
	     ((rank == Rank.SPECIMEN && isType() && typeOfType != null) ? new Text(typeOfType)
		 : new Text(rank.getName())));
	
	return new Text[]{rankText,
			  nameText,
			  authority,
			  year};
	
    }

    public void updateSummary() {
	if(appearanceEditModel != null)
	    appearanceEditModel.updateSummary();
	Text[] summaryTextArray = getSummaryTextArray();
	TextBuffer buffer = new TextBuffer();
	
	// append "rank"
	/*
	  if(isType())
	  buffer.setColor(typeColor);
	*/
	buffer.append(summaryTextArray[0].toString());
	
	buffer.append(' ');
	// append "name"
	if(isType())
	    buffer.setColor(typeColor);
	String nameString = "";
	if (summaryTextArray[1] != null) {
	    nameString = summaryTextArray[1].toString();
	}
	if(nameString.length() > 0) {
	    String firstWord = (nameString.indexOf(' ') == -1) ? nameString : nameString.substring(0, nameString.indexOf(' '));
	    String followWord = (nameString.indexOf(' ') == -1) ? "" : nameString.substring(nameString.indexOf(' '), nameString.length());
	    if (getHigherModel() != null && getRank() != null && (getRank().equals(Rank.SPECIES) || getRank().isLower(Rank.SPECIES))) {
		String str = getHigherModel().getUsedName();
		str = (str.indexOf(' ') == -1) ? str : str.substring(0, str.indexOf(' '));
		if (str.equals(firstWord) && !followWord.equals("")) {
		    nameString = firstWord.charAt(0) + ". " + followWord;
		}
	    }
	    buffer.append(nameString);
	}
	
	NameUsageEditModel authorityEditModel = getAuthorityEditModel();
	if (authorityEditModel != null) {
	    String word1 = getUsedName();
	    word1 = (word1.indexOf(' ') == -1) ? word1 : word1.substring(0, word1.indexOf(' '));
	    String word2 = authorityEditModel.getUsedName();
	    word2 = (word2.indexOf(' ') == -1) ? word2 : word2.substring(0, word2.indexOf(' '));
	    String beginBracket = "";
	    String endBracket = "";
	    if (getRank()!=null && getRank().equals(Rank.SPECIES)
		&& !word1.equals(word2)) {
		beginBracket = "(";
		endBracket = ")";
	    }
	    if (authorityEditModel != this) {
		buffer.append(' ');
				/*
				  if(isType())
				  buffer.setColor(typeColor);
				*/
		buffer.append(beginBracket);
				/*
				  if(isType())
				  buffer.setColor(typeColor);
				*/
		buffer.append(authorityEditModel.getViewName());
				/*
				  if(isType())
				  buffer.setColor(typeColor);
				*/
		buffer.append(endBracket);
		buffer.append(" (");
	    }
	    Text author = (getAppearanceEditModel() != null
			   && getAppearanceEditModel().getPublicationEditModel() != null
			   && getAppearanceEditModel().getPublicationEditModel().getAuthorsSummaryTextModel() != null) ?
		getAppearanceEditModel().getPublicationEditModel().getAuthorsSummaryTextModel().getRichText().getText() : new Text();
	    buffer.append(' ');
	    // append "author"
	    /*
	      if(isType())
	      buffer.setColor(typeColor);
	    */
	    buffer.append(author);
	    buffer.append(' ');
	    // append "year"
	    /*
	      if(isType())
	      buffer.setColor(typeColor);
	    */
	    buffer.append(summaryTextArray[3].toString());
	    if (authorityEditModel != this) {
		/*
		  if(isType())
		  buffer.setColor(typeColor);
		*/
		buffer.append(" )");
	    }
	}
	setSummaryText(buffer.toText());
    }
    
    public boolean isIncertaeSedis()
    {
	return insertaeSedis;
    }

    public void setIncertaeSedis(boolean b)
    {
	if(b == insertaeSedis)
	    return;

	insertaeSedis = b;
    }

    public boolean isType()
    {
	return type;
    }

    public void setType(boolean b)
    {
	if(b == type)
	    return;

	type = b;
	if(!type)
	    typeOfType = null;
	updateSummary();
	setChanged();
	notifyObservers();
    }

    public void setTypeOfType(String typeOfType)
    {
	if(this.typeOfType == typeOfType ||
	   (this.typeOfType != null &&
	    this.typeOfType.equals(typeOfType)))
	    return;

	this.typeOfType = typeOfType;
	setType(typeOfType != null);
	/*
	updateSummary();
	setChanged();
	notifyObservers();
	*/
    }

    public Color getTypeColor()
    {
	return typeColor;
    }

    public void setTypeColor(Color color)
    {
	if(typeColor.equals(color))
	    return;
	typeColor = color;
	typeStyleModifier.put(BasicTSModifier.COLOR, typeColor);
	updateSummary();
	setChanged();
	notifyObservers();
    }

    public jp.kyasu.awt.TextListModel getTypeListModel()
    {
	return typeListModel;
    }

    /**
     * Loads attributes from the <code>NameUsage</code>
     */
    public void loadAttributes()
    {
	super.loadAttributes();

        NameUsage nameUsage = (NameUsage)getNameUsage().getEntity();
	if(nameUsage == null)
	    nameUsage = new NameUsage();
	setString(nameModel, nameUsage.getAscribedName());
	String r = nameUsage.getRank();
	rank = null;
	if(r != null) {
	    rank = Rank.get(r);
	    if (rank == null)
		rank = new Rank(r);
	}

	setType(nameUsage.isType());
	setTypeOfType(nameUsage.getTypeOfType());
	setIncertaeSedis(false);	
	setIncertaeSedis(nameUsage.isIncertaeSedis());

	setString(note, nameUsage.getNote());

	setAppearanceEditModel((AppearanceEditModel)getEditModel(nameUsage.getAppearance(), AppearanceEditModel.getInstance()));

	setHigherEditModel((NameUsageEditModel)getEditModel(nameUsage.getHigherTaxon()));
	lowerList.setObjects(nameUsage.getLowerTaxa());

	setAuthorityEditModel((NameUsageEditModel)getEditModel(nameUsage.getAuthority()));

	annotationList.setObjects(nameUsage.getAnnotations());

	updateSummary();
    }
    
    /**
     * Saves attributes to the <code>NameUsage</code>
     */
    public void saveAttributes()
    {
	NameUsage nameUsage = getNameUsage();
	nameUsage.setAscribedName(getAscribedName());
	if(rank != null)
	    nameUsage.setRank(rank.getName());
	nameUsage.setNote(getNote());
	if (this.getAuthorityEditModel() != null) {
	    nameUsage.setAuthority(this.getAuthorityEditModel().getNameUsage());
	}
	nameUsage.setAppearance((Appearance)getAppearanceEditModel().getObject());
	if (higherEditModel != null)
	    nameUsage.setHigherTaxon(higherEditModel.getNameUsage());
	else
	    nameUsage.setHigherTaxon(null);

	nameUsage.setType(isType());
	nameUsage.setTypeOfType(typeOfType);
	nameUsage.setIncertaeSedis(isIncertaeSedis());

	nameUsage.setLocale(getLocale());
	
	nameUsage.setLowerTaxa(lowerList.getNamedObjects());

	if(authorityEditModel != null)
	    nameUsage.setAuthority(authorityEditModel.getNameUsage());
	else
	    nameUsage.setAuthority(null);

	if(isAnnotationsModified())
	    nameUsage.setAnnotations(annotationList.getNamedObjects());
	setAnnotationsModified(false);
    }
    
    public Vector getLowerEditModels()
    {
	return lowerList.getModels();
    }

    /**
     * Returns index of <code>model</code>
     *
     * @param model <code>NamedObjectEditModel</code> to be searched
     * @return int index of <code>model</code>
     */
    public int indexOf(NamedObjectEditModel model)
    {
	return lowerList.indexOf(model);
    }
	
    public void setLowerEditModels(Vector lowerEditModels)
    {
	/*
	if (this.lowerEditModels == lowerEditModels &&
	    lowerEditModels != null)
	    return;
	this.lowerEditModels = lowerEditModels;
	*/
	lowerList.setModels(lowerEditModels);
    }

    //    public boolean addLowerEditModel(NameUsageEditModel lowerEditModel)
    public void addLowerEditModel(NameUsageEditModel lowerEditModel)
    {
	/*
	if(lowerEditModels == null)
	    lowerEditModels = new Vector();
	if(lowerEditModels.contains(lowerEditModel))
	    return false;
	
	lowerEditModels.addElement(lowerEditModel);
	return true;
	*/
	addLower(lowerEditModel);
    }

	/**
	 * get the <code>NameUsageEditModel</code> list below the current <code>NameUsageEditModel</code>
	 * @return <code>NameUsageEditModel</code> list
	 */
	public Vector getLowerEditModelsBelowCurrent() {
		return this.getLowerEditModelsBelowCurrent(new Vector(), 10);
	}

    /**
     * get the <code>NameUsageEditModel</code> list below the current <code>NameUsageEditModel</code>
     * @param editModels the current <code>NameUsageEditModel</code> list
     * @param limit the limit of this recursive function
     * @return <code>NameUsageEditModel</code> list
     */
    private Vector getLowerEditModelsBelowCurrent(Vector editModels, int limit) {
	// nest limit check.
	limit--;
	if (limit < 0)
	    return null;
	// get lowers.
	Vector v = this.getLowerEditModels();
	if (v == null || v.size() == 0)
	    return null;
	for (Enumeration e = v.elements(); e.hasMoreElements(); ) {
	    NameUsageEditModel nameUsageEditModel = (NameUsageEditModel)e.nextElement();
	    editModels.addElement(nameUsageEditModel);
	    Vector vv = nameUsageEditModel.getLowerEditModelsBelowCurrent(editModels, limit);
	    if (vv != null)
		editModels.addAll(vv);
	}
	return editModels;
    }

    /**
     * Gets citation.
     */
    public TextEditModel getCitationTextEditModel()
    {
	return appearanceEditModel.getSummaryTextModel();
    }
    
    public void updateList() {
	/*
	lowerList.update();
	annotationList.update();
	*/
    }
    
    public void setLocale(Locale locale)
    {
    	this.locale = locale;
    	this.keepLocale = locale;
    }
    
    public Locale getLocale()
    {
    	return locale;
    }
    
    public void update()
    {
	//?
    }

    protected void setHigherEditModel()
    {
	setHigherEditModel((NameUsageEditModel)getEditModel(getNameUsage().getHigherTaxon()));
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an instance of <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    NamedObjectEditModel createEditModel(NamedObject object)
    {
	if(!(object instanceof NameUsage))
	    throw new IllegalArgumentException(object.getClass().getName()
						+ " is not an instance of org.nomencurator.NameUsage");

	return new NameUsageEditModel((NameUsage)object);
	/*
	NameUsageEditModel model = new NameUsageEditModel();
	model.setObject(object);
	return model;
	*/
    }

    public static NamedObjectEditModel getInstance()
    {
	if(template == null)
	    template = new NameUsageEditModel();
	return template;
    }

    public boolean isAnnotationsModified()
    {
	return annotationsModified;
    }

    public void setAnnotationsModified(boolean status)
    {
	annotationsModified = status;
	if(status)
	    modified = status;
    }

    /**
     * Invoked when the text model has been changed.
     */
    public void textModelChanged(TextModelEvent event)
    {
	Object source = event.getSource();
	if (source == annotationList) {
	    setAnnotationsModified(true);
	}
	else {
	    super.textModelChanged(event);
	}
    }
}
