/*
 * NamedObjectEditModel.java:  a NamedObject editor model 
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: NamedObjectEditModel.java,v 1.23 2002/10/01 02:55:53 nozomi Exp $
 * $Log: NamedObjectEditModel.java,v $
 * Revision 1.23  2002/10/01 02:55:53  nozomi
 * imeplement equals() to examines NamedObjects by equals()
 *
 * Revision 1.22  2002/09/17 05:46:02  nozomi
 * re-organise initialisation methods
 *
 * Revision 1.21  2002/09/09 16:36:20  nozomi
 * make updateSumamry() abstract
 *
 * Revision 1.20  2002/08/29 19:46:39  nozomi
 * implements interface Nameable
 *
 * Revision 1.19  2002/08/29 19:02:08  nozomi
 * add String peristentID() method
 *
 * Revision 1.18  2002/07/02 08:29:08  nozomi
 * change default text model relating methods
 *
 * Revision 1.17  2002/06/21 23:24:43  nozomi
 * use RichText in TextComponents
 *
 * Revision 1.16  2002/06/21 17:49:12  nozomi
 * returns Text*Model of DEFAULT_DOCUMENT_STYLE
 *
 * Revision 1.15  2002/06/04 13:48:13  ryo
 * remove unused import line
 *
 * Revision 1.14  2002/05/14 10:15:34  ryo
 * add updateList()
 *
 * Revision 1.13  2002/05/10 13:45:59  ryo
 * add debug message
 *
 * Revision 1.12  2002/05/09 00:03:26  nozomi
 * make updateSummary public
 *
 * Revision 1.11  2002/03/29 02:23:34  nozomi
 * implements Observer
 *
 * Revision 1.10  2002/03/08 23:06:19  nozomi
 * update when getSummaryText()implements
 *
 * Revision 1.9  2002/03/06 00:48:57  okawa
 * restore the last modification
 *
 * Revision 1.8  2002/03/04 00:36:46  okawa
 * add save function
 *
 * Revision 1.7  2002/02/28 19:00:14  nozomi
 * fix missing import directive
 *
 * Revision 1.6  2002/02/28 17:11:22  nozomi
 * use DefaultTextListModel of the package
 *
 * Revision 1.5  2002/02/27 23:58:17  nozomi
 * add appendTo() method
 *
 * Revision 1.4  2002/02/26 15:43:53  nozomi
 * Add getSummaryTextArray() and getSummaryArray() to models
 *
 * Revision 1.3  2002/02/07 20:58:00  nozomi
 * Changed summary from TextModel to TextEditModel
 *
 * Revision 1.2  2002/01/29 07:17:17  nozomi
 * Made it an abstract class
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */


package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import jp.kyasu.awt.DefaultTextModel;
import jp.kyasu.awt.DefaultTextEditModel;
import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.TextField;
import jp.kyasu.awt.TextModel;

import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextBuffer;

import org.nomencurator.Nameable;
import org.nomencurator.NamedObject;

import org.nomencurator.broker.NamedObjectBroker;

/**
 * The model interface for an object that acts as a named object edit model
 *
 * @see 	org.nomencurator.editor.event.ObjectEditModelEvent;
 * @see 	org.nomencurator.editor.event.ObjectEditModelListener;
 *
 * @version 	01 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class NamedObjectEditModel
    extends ObjectEditModel
    implements Nameable, Observer
{
    /**
     * The constructor of NamedObjectEditModel
     *
     * @param object <code>Object</code> to be edited
     * @param editable boolean determining whether the model is editable
     */
    protected NamedObjectEditModel(NamedObject object, boolean editable)
    {
	super(object, editable);
    }

    /**
     * Loeds attributes of <code>Object</code> to the model
     */
    public void loadAttributes()
    {
	NamedObjectBroker.getInstance().putModel(this);
    }

    /**
     * Sets summary contents to given <code>richText</code>
     *
     * @param richText <code>RichText</code> to be set as contents of <code>summary</code>
     */
    public void setSummary(RichText richText)
    {
	setSummaryRichText(richText);
    }

    /**
     * Sets summary contents to given <code>richText</code>
     *
     * @param richText <code>RichText</code> to be set as contents of <code>summary</code>
     */
    public void setSummaryRichText(RichText richText)
    {
	summary.setRichText(richText);
    }

    /**
     * Gets summary contents in <code>RichText</code>
     *
     * @return RichText
     */
    public RichText getSummaryRichText()
    {
	return summary.getRichText();
    }

    /**
     * Sets summary contents to given <code>text</code>
     *
     * @param text <code>Text</code> to be set as contents of <code>summary</code>
     */
    public void setSummary(Text text)
    {
	setSummaryText(text);
    }

    /**
     * Sets summary contents to given <code>text</code>
     *
     * @param text <code>Text</code> to be set as contents of <code>summary</code>
     */
    public void setSummaryText(Text text)
    {
	setText(summary, text);
    }

    /**
     * Gets summary contents in <code>Text</code>
     *
     * @return Text
     */
    public Text getSummaryText()
    {
	return getText(getSummaryTextModel());
    }

    /**
     * Gets summary contents in <code>Text</code>
     *
     * @return Text
     */
    public Text[] getSummaryTextArray()
    {
	return new Text[]{getSummaryText()};
    }

    /**
     * Sets summary contents to given <code>string</code>
     *
     * @param string <code>String</code> to be set as contents of <code>summary</code>
     */
    public void setSummary(String string)
    {
	setString(summary, string);
    }

    /**
     * Gets summary contents in <code>String</code>
     *
     * @return String
     */
    public String getSummary()
    {
	return getString(summary);
    }

    /**
     * Gets summary contents in <code>String</code>
     *
     * @return String
     */
    public String[] getSummaryArray()
    {
	return new String[]{getSummary()};
    }

    /**
     * Gets persistend ID string of a <code>NamedObject</code> under edition.
     *
     */
    public String getPersistentID()
    {
	NamedObject object = (NamedObject)getObject();

	if(object == null)
	    return null;

	return object.getPersistentID();
    }

    /**
     * Gets <code>NamedObject</code> to be edited by this model.
     *
     */
    public NamedObject getNamedObject()
    {
	return (NamedObject)getObject();
    }

    /**
     * Utility method used to append <code>Text</code> as word(s)
     *
     */
    protected static void appendTo(TextBuffer buffer, Text text)
    {
	if(text.length() > 0) {
	    buffer.append(' ');
	    buffer.append(text);
	}
    }

    public void update(Observable observable, Object arg)
    {
	if(observable == getObject())
	    loadAttributes();
	updateSummary();
	setChanged();
	notifyObservers(arg);
    }

    public abstract void updateList();

    /**
     * Returns peristent ID of wrapped <CODE>NamedObject</CODE> as its name,
     * or result of toString() if no <CODE>NamedObject</CODE> is wrapped
     *
     * @return String rpresenting peristent ID of wrapped <CODE>NamedObject</CODE>
     */
    public String getName()
    {
	NamedObject obj = getNamedObject();
	if(obj == null)
	    return toString();
	return obj.persistentID();
    }

    /**
     * Gives a name as a <CODE>String</CODE>
     *
     * @param String representing a name
     */
    public void setName(String name)
    {
	NamedObject obj = getNamedObject();
	if(obj != null)
	    obj.setName(name);
    }

    /**
     * Sets <code>Object</code> to be edited by this model.
     *
     * @param object <code>Object</code> to be edited by this model
     */
    public void setObject(Object object)
    {
	NamedObject obj = getNamedObject();
	if(obj == object)
	    return;

	if(object != null &&
	   !(object instanceof NamedObject))
	    throw new IllegalArgumentException(object.getClass().getName() +
					       " is not an instance of " +
					       (obj != null? obj.getClass().getName():"NamedObject"));
	if(obj != null)
	    obj.deleteObserver(this);

	super.setObject(object);

	if(object != null)
	    ((NamedObject)object).addObserver(this);
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an instance of <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    abstract NamedObjectEditModel createEditModel(NamedObject object);

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    NamedObjectEditModel getEditModel(NamedObject object)
    {
	return getEditModel(object, this);
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    static NamedObjectEditModel getEditModel(NamedObject object, 
					     NamedObjectEditModel template)
    {
	if(object == null)
	    return null;

	NamedObjectBroker broker = NamedObjectBroker.getInstance();
	String name = object.getName();
	NamedObjectEditModel model = broker.getModel(name);
	if(model == null) {
	    if(object.isNominal())
		object = broker.getNamedObjectEntity(name);
	    model = template.createEditModel(object);
	    //	    broker.putModel(model);
	}
	return model;
    }

    public boolean equals(Object object)
    {
	if(object == this)
	    return true;

	if(object == null)
	    return false;

	if(!(object instanceof NamedObjectEditModel))
	    return false;

	if(getNamedObject() == null) 
	    return false;

	return getNamedObject().equals(((NamedObjectEditModel)object).getNamedObject());

    }
}
