/*
 * NamedObjectListModel.java:  an implementation of TextListModel
 * to be used by TableList of TaxoNote based on Nomencurator
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
/*
 *
 * $Id: NamedObjectListModel.java,v 1.23 2002/10/17 17:09:50 nozomi Exp $
 * $Log: NamedObjectListModel.java,v $
 * Revision 1.23  2002/10/17 17:09:50  nozomi
 * check NamedObjectBroker when create NamedObjectEdtirModel from NamedObject Vector
 *
 * Revision 1.22  2002/10/17 04:16:07  nozomi
 * add setObjects(Vector, boolean)
 *
 * Revision 1.21  2002/10/11 03:42:54  nozomi
 * add up(int), down(int) and moveModel(int, int) methods)
 *
 * Revision 1.20  2002/09/17 08:29:56  nozomi
 * add/removeModel() methods return boolean; add removeAll()
 *
 * Revision 1.19  2002/09/05 15:44:25  nozomi
 * add getNamedObjects() method
 *
 * Revision 1.18  2002/06/03 00:15:20  nozomi
 * remove unused line
 *
 * Revision 1.17  2002/05/31 10:23:25  ryo
 * call initialize() in constructor
 *
 * Revision 1.16  2002/05/31 06:01:07  nozomi
 * add cooments on initialize()
 *
 * Revision 1.15  2002/05/31 05:47:23  nozomi
 * add initialize()
 *
 * Revision 1.14  2002/05/25 16:55:20  nozomi
 * fix model selection bug
 *
 * Revision 1.13  2002/05/22 23:58:00  nozomi
 * add contains method
 *
 * Revision 1.12  2002/05/21 03:35:29  nozomi
 * emptyText changed to EMPTY_TEXT
 *
 * Revision 1.11  2002/05/20 02:01:47  nozomi
 * Re-implemented to utilise ModelSelector
 *
 * Revision 1.10  2002/05/16 22:15:27  nozomi
 * extends org.nomencurartor.editor.model.DefaultTextListModel instead of jp.kyasu.awt's one
 *
 * Revision 1.9  2002/05/16 20:20:52  nozomi
 * add public abstract Text[][] getEmptySummary() to avoid foever loop
 *
 * Revision 1.8  2002/05/15 01:03:48  nozomi
 * ordinary index is restricted to keep the dummy model as the last model
 *
 * Revision 1.7  2002/05/14 23:47:43  nozomi
 * name changed from DefaultTextListModel
 *
 * Revision 1.6  2002/05/14 10:15:09  ryo
 * add addModel()
 *
 * Revision 1.5  2002/03/05 01:51:12  nozomi
 * modify update() methods
 *
 * Revision 1.4  2002/03/03 23:50:39  nozomi
 * change in item insertion
 *
 * Revision 1.3  2002/02/28 17:11:22  nozomi
 * use DefaultTextListModel of the package
 *
 * Revision 1.2  2002/02/28 03:19:24  nozomi
 * add update() methods
 *
 * Revision 1.1  2002/02/27 23:43:18  nozomi
 * initial import to CVS
 *
 */

package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Vector;

import jp.kyasu.awt.TextListModel;

import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextList;

import org.nomencurator.NamedObject;

import org.nomencurator.broker.NamedObjectBroker;

/**
 * Extention of <CODE>jp.kyasu.awt.DefaultTextListModel</CODE>
 * supporting <CODE>NamedObjectEditModel</CODE> selection
 * defined by <CODE>org.nomencurator.editor.model.TextListModel</CODE>
 *
 * @see 	jp.kyasu.awt.DefaultTextListModel
 *
 * @version 	17 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class NamedObjectListModel
    extends jp.kyasu.awt.DefaultTextListModel
    implements TextListModel, ModelSelector
{
    protected static final Text EMPTY_TEXT = ModelSelectAdaptor.EMPTY_TEXT;

    protected static final Text LIST_SEPARATOR_TEXT = ModelSelectAdaptor.LIST_SEPARATOR_TEXT;

    protected ModelSelector modelSelector;

    /**
     * Constructs a one column text list model with the default rich text
     * style.
     */
    protected NamedObjectListModel()
    {
	this(false);
    }

    /**
     * Constructs a one column text list model with the default rich text
     * style.
     */
    protected NamedObjectListModel(boolean enable)
    {
	this(RichTextStyle.DEFAULT_LIST_STYLE, enable);
    }

    /**
     * Constructs a one column text list model with the specified rich text
     * style.
     *
     * @param richTextStyle the rich text style.
     */
    protected NamedObjectListModel(RichTextStyle richTextStyle)
    {
	this(richTextStyle, false);
    }


    /**
     * Constructs a one column text list model with the specified rich text
     * style.
     *
     * @param richTextStyle the rich text style.
     */
    protected NamedObjectListModel(RichTextStyle richTextStyle, boolean enable)
    {
	super(richTextStyle);
	//	modelSelector = new ModelSelectAdaptor(this, enable);
	initialize(enable);
    }


    /**
     * Constructs a text list model with the specified column widths.
     *
     * @param colWidths the column widths.
     */
    protected NamedObjectListModel(int colWidths[])
    {
	this(colWidths, false);
    }

    /**
     * Constructs a text list model with the specified column widths.
     *
     * @param colWidths the column widths.
     */
    protected NamedObjectListModel(int colWidths[], boolean enable)
    {
	this(colWidths.length, colWidths, enable);
    }

    /**
     * Constructs a text list model with the specified number of columns
     * and column widths.
     *
     * @param columns   the number of columns.
     * @param colWidths the column widths.
     */
    protected NamedObjectListModel(int columns, int colWidths[])
    {
	this(columns, colWidths, false);
    }

    /**
     * Constructs a text list model with the specified number of columns
     * and column widths.
     *
     * @param columns   the number of columns.
     * @param colWidths the column widths.
     */
    protected NamedObjectListModel(int columns, int colWidths[],
				   boolean enable)
    {
	this(columns, colWidths, RichTextStyle.DEFAULT_LIST_STYLE, enable);
    }

    /**
     * Constructs a text list model with the specified number of columns,
     * column widths, and rich text style.
     *
     * @param columns       the number of columns.
     * @param colWidths     the column widths.
     * @param richTextStyle the rich text style.
     */
    protected NamedObjectListModel(int columns, int colWidths[],
				RichTextStyle richTextStyle)
    {
	this(columns, colWidths, richTextStyle, false);
    }

    /**
     * Constructs a text list model with the specified number of columns,
     * column widths, and rich text style.
     *
     * @param columns       the number of columns.
     * @param colWidths     the column widths.
     * @param richTextStyle the rich text style.
     */
    protected NamedObjectListModel(int columns, int colWidths[],
				   RichTextStyle richTextStyle,
				   boolean enable)
    {
	super(columns, colWidths, richTextStyle);
	initialize(enable);
    }

    /**
     * Constructs a text list model with the specified <CODE>textList</CODE>
     *
     * @param textList      <CODE>TextList</CODE> to be used by this object
     */
    protected NamedObjectListModel(TextList textList)
    {
	this(textList, false);
    }

    /**
     * Constructs a text list model with the specified <CODE>textList</CODE>
     *
     * @param textList      <CODE>TextList</CODE> to be used by this object
     */
    protected NamedObjectListModel(TextList textList, boolean enable)
    {
	super(textList);
	initialize(enable);
    }

    /**
     * Initializes additional components of this model.
     * The subclasses may override this method to initialize
     * their own members 
     *
     * @param enable true if automatic dummy addition is enabled
     */
    protected void initialize(boolean enable)
    {
	modelSelector = new ModelSelectAdaptor(this, enable);
    }

    /**
     * Adds <CODE>model</CODE> to <CODE>NamedObjectEditModel</CODE> list
     *
     * @param model <CODE>NamedObjectEditModel</CODE> to be added to this selector
     */
    public boolean addModel(NamedObjectEditModel model)
    {
	return modelSelector.addModel(model);
    }

    /**
     * Inserts <CODE>model</CODE> at <CODE>index</CODE> of <CODE>NamedObjectEditModel</CODE> list.
     *
     * @param index  where <CODE>model</CODE> to be added
     * @param model <CODE>NamedObjectEditModel</CODE> to be added
     */
    public boolean addModel(int index, NamedObjectEditModel model)
    {
	return modelSelector.addModel(index, model);
    }

    /**
     * Inserts <CODE>model</CODE> at <CODE>index</CODE> of model list
     * with its representation given by <CODE>items</CODE>.
     *
     * @param index  where <CODE>model</CODE> to be added
     * @param model <CODE>NamedObjectEditModel</CODE> to be added
     * @param items an array of <CODE>Object</CODE>s representing contents of the <CODE>model</CODE>
     */
    public boolean addModel(int index, NamedObjectEditModel model, Object[][] items)
    {
	return modelSelector.addModel(index, model, items);
    }

    /**
     * Removes <CODE>model</CODE> from <CODE>NamedObjectEditModel</CODE> list
     *
     * @param model <CODE>NamedObjectEditModel</CODE> to be removed from this selector
     */
    public boolean removeModel(NamedObjectEditModel model)
    {
	return modelSelector.removeModel(model);
    }

    /**
     * Remove all <CODE>NamedObjectEditModel</CODE>s from <CODE>NamedObjectEditModel</CODE> list
     *
     */
    public void removeAllModels()
    {
	modelSelector.removeAllModels();
    }

    /**
     * Selects an <CODE>NamedObjectEditModel</CODE> at <CODE>index</CODE> of list
     *
     * @param index of <CODE>NamedObjectEditModel</CODE> to be selected
     */
    public NamedObjectEditModel getModel(int index)
    {
	return modelSelector.getModel(index);
    }

    /**
     * Selects <CODE>model</CODE> from <CODE>NamedObjectEditModel</CODE> list
     *
     * @param model <CODE>NamedObjectEditModel</CODE> to be selected
     */
    public void selectModel(NamedObjectEditModel model)
    {
	modelSelector.selectModel(model);
    }

    /**
     * Selects an <CODE>NamedObjectEditModel</CODE> at <CODE>index</CODE> of list
     *
     * @param index of <CODE>NamedObjectEditModel</CODE> to be selected
     */
    public void selectModelAt(int index)
    {
	modelSelector.selectModelAt(index);
    }


    /**
     * Returns selected <CODE>NamedObjectEditModel</CODE>
     *
     * @return <CODE>NamedObjectEditModel</CODE> selected
     */
    public NamedObjectEditModel getSelectedModel()
    {
	return modelSelector.getSelectedModel();
    }


    /**
     * Returns selected <CODE>NamedObjectEditModel</CODE>
     *
     * @return <CODE>NamedObjectEditModel</CODE> selected
     */
    public NamedObjectEditModel[] getSelectedModels()
    {
	return modelSelector.getSelectedModels();
    }


    /**
     * Returns index of selected <CODE>NamedObjectEditModel</CODE>
     *
     * @return int index of selected <CODE>NamedObjectEditModel</CODE>
     */
    public int getSelectedIndex()
    {
	return modelSelector.getSelectedIndex();
    }


    /**
     * Returns index of <CODE>model</CODE>
     *
     * @param model <CODE>NamedObjectEditModel</CODE> to be searched
     * @return int index of <CODE>model</CODE>
     */
    public int indexOf(NamedObjectEditModel model)
    {
	return modelSelector.indexOf(model);
    }


    /**
     * Enables or desables automatic addition of a dummy
     * <CODE>NamedObjectEditModel</CODE> to the list according to
     * value of <CODE>enable</CODE>
     *
     * @param enable <CODE>boolean</CODE> enables 
     * automatic addition of a dummy <CODE>NamedObjectEditModel</CODE>
     * if true or disables if false
     */
    public void enableDummyModel(boolean enable)
    {
	modelSelector.enableDummyModel(enable);
    }

    /**
     * Returns true if automatic addition of a dummy 
     * <CODE>NamedObjectEditModel</CODE> is enabled
     * or false if disabled.
     *
     * @return true if automatic addition of a dummy 
     * <CODE>NamedObjectEditModel</CODE> is enabled
     * or false if disabled.
     */
    public boolean isDummyModelEnabled()
    {
	return modelSelector.isDummyModelEnabled();
    }


    /**
     * Returns a dummy <CODE>NamedObjectEditModel</CODE> instance for double-clicking
     * operation.
     *
     * @return null to be considered as a dummy <CODE>NamedObjectEditModel</CODE>
     */
    /**
     * Returns a dummy <CODE>NamedObjectEditModel</CODE> instance for double-clicking
     * operation.
     *
     * @return null to be considered as a dummy <CODE>NamedObjectEditModel</CODE>
     */
    public NamedObjectEditModel getDummyModel()
    {
	return null;
    }

    /**
     * Returns Text array representing empty summary 
     * can be used to represent a dummy <CODE>NamedObjectEditModel</CODE>.
     *
     * @return Text[][] representing a summary of an empty <CODE>NamedObjectEditModel</CODE>.
     */
    public Text[][] getEmptySummary()
    {
	return modelSelector.getEmptySummary();
    }


    /**
     * Returns <CODE>Vector</CODE> of <CODE>NamedObjectEditModel</CODE>
     * accompanying to the <CODE>ListModel</CODE>
     */
    public Vector getModels()
    {
	return modelSelector.getModels();
    }


    /**
     * Sets <CODE>models</CODE> as <CODE>NamedObjectEditModel</CODE> list
     * accompanying to the <CODE>ListModel</CODE>
     */
    public void setModels(Vector models)
    {
	modelSelector.setModels(models);
    }

    /**
     * Sets <CODE>Vector</CODE> of <CODE>NamedObjectEditModel</CODE>,
     * which is created from <CODE>NamedObject</CODE> contained
     * in <CODE>objects</CODE>, as <CODE>NamedObjectEditModel</CODE> list
     * accompanying to the <CODE>ListModel</CODE>
     *
     * @param objects <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * to be added to this list via <CODE>NamedObjectEditModel</CODE>s
     */
    public void setObjects(Vector objects)
    {
	setObjects(objects, true);
    }

    /**
     * Sets <CODE>Vector</CODE> of <CODE>NamedObjectEditModel</CODE>,
     * which is created from <CODE>NamedObject</CODE> contained
     * in <CODE>objects</CODE>, as <CODE>NamedObjectEditModel</CODE> list
     * accompanying to the <CODE>ListModel</CODE>.
     * If <CODE>allowNominal</CODE> is false, all <CODE>NamedObject</CODE> s
     * examined and entity seted if it is nominal.
     *
     * @param objects <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * to be added to this list via <CODE>NamedObjectEditModel</CODE>s
     * @param allowNominal to be false to prohibit nominal 
     * <CODE>NamedObject</CODE>s
     */
    public void setObjects(Vector objects, boolean allowNominal)
    {
	if(objects == null || objects.isEmpty()) {
	    modelSelector.setModels(null);
	    return;
	}
	if(!allowNominal) {
	    NamedObjectBroker broker = 
		NamedObjectBroker.getInstance();
	    Enumeration e = objects.elements();
	    while(e.hasMoreElements()) {
		NamedObject o = (NamedObject)e.nextElement();
		if(o.isNominal())
		    broker.getNamedObjectEntity(o.getName());
	    }
	}
	modelSelector.setModels(getModels(objects));
    }


    /**
     * Updates summary text of all specified <CODE>model</CODE> in the list
     *
     * @param model <CODE>NamedObjectEditModel</CODE> to be updated
     */
    public void update()
    {
	modelSelector.update();
    }


    /**
     * Updates summary text of specified <CODE>model</CODE> in the list
     *
     * @param model <CODE>NamedObjectEditModel</CODE> to be updated
     */
    public void update(NamedObjectEditModel model)
    {
	modelSelector.update(model);
    }

    /**
     * Updates summary text of model specified by <CODE>index</CODE>
     *
     * @param index index of <CODE>NamedObjectEditModel</CODE> to be updated
     */
    public void update(int index)
    {
	modelSelector.update(index);
    }

    /**
     * Returns <CODE>TextListModel</CODE> representing summaries of 
     * <CODE>NamedObjectEditModel</CODE> handled by this selector
     *
     * @return TextListModel representing summaries of 
     * <CODE>NamedObjectEditModel</CODE> handled by this selector
     */
    public TextListModel getModel()
    {
	return this;
    }

    /**
     * Sets <CODE>textListModel</CODE> as a summarized representation
     * <CODE>NamedObjectEditModel</CODE> handled by this selector
     *
     * @param textListModel <CODE>TextListModel</CODE> representing summaries of 
     * <CODE>NamedObjectEditModel</CODE> handled by this selector
     */
    public void setModel(TextListModel textListModel)
    {
    }

    /**
     * Returns <CODE>TextListModel</CODE> representing summaries of 
     * <CODE>NamedObjectEditModel</CODE> handled by this selector
     *
     * @return TextListModel representing summaries of 
     * <CODE>NamedObjectEditModel</CODE> handled by this selector
     */
    public ModelSelector getModelSelector()
    {
	return modelSelector;
	//	return this; //?
    }

    /**
     * Returns true if this <CODE>ModelSelector</CODE> contains <CODE>model</CODE>
     * or false if not.
     *
     * @return true if this <CODE>ModelSelector</CODE> contains <CODE>model</CODE>
     * or false if not.
     */
    public boolean contains(NamedObjectEditModel model)
    {
	return modelSelector.contains(model);
    }

    /**
     * Returns <CODE>Vector</CODE> of <CODE>NamedObject</CODE>s
     * accompanying to the <CODE>ListModel</CODE>
     */
    public Vector getNamedObjects()
    {
	return modelSelector.getNamedObjects();
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param namedObjects <CODE>Vector</CODE> containing <CODE>NamedObject</CODE>s
     * to be represented by <CODE>NamedObjectEditModel</CODE>s
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    protected abstract Vector getModels(Vector namedObjects);


    /**
     * Creates and returns a <CODE>Vector</CODE> of <CODE>NamedObjectEditModel</CODE>s
     * reprenseting <CODE>NamedObject</CODE>s given as <CODE>namedObjects</CODE>.
     *
     * @param namedObjects <CODE>Vector</CODE> containing <CODE>NamedObject</CODE>s
     * to be represented by <CODE>NamedObjectEditModel</CODE>s
     * @param template a <CODE>NamedObjectEditModel</CODE> providing
     * <CODE>getEditModel(NamedObject)</CODE> methods
     *
     * @return Vector containing  <CODE>NamedObjectEditModel</CODE>s representing
     * <CODE>NamedObject</CODE>s in <CODE>namedObjects</CODE>
     */
    protected static Vector getModels(Vector namedObjects,
				      NamedObjectEditModel template)
    {
	if(namedObjects == null ||
	   namedObjects.isEmpty())
	    return null;

	Vector v = new Vector();

	NamedObjectBroker  broker = 
	    NamedObjectBroker.getInstance();

	Enumeration e = namedObjects.elements();
	while(e.hasMoreElements()) {
	    NamedObject obj = (NamedObject)e.nextElement();
	    NamedObjectEditModel model = 
		broker.getModel(obj.getName());
	    if(model == null)
		model = template.getEditModel(obj);
	    v.addElement(model);
	}
	return v;
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one above
     */
    public boolean up(int index)
    {
	return modelSelector.up(index);
    }


    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>index</CODE> to the place just one below
     */
    public boolean down(int index)
    {
	return modelSelector.down(index);
    }

    /**
     * Moves position of <CODE>NamedObjectEditModel</CODE> at
     * <CODE>source</CODE> to <CODE>destination</CODE>
     */
    public boolean moveModel(int source, int destination)
    {
	return modelSelector.moveModel(source, destination);
    }
}
