/*
 * ObjectEditModel.java:  an object editor model 
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ObjectEditModel.java,v 1.16 2002/10/18 02:47:51 nozomi Exp $
 * $Log: ObjectEditModel.java,v $
 * Revision 1.16  2002/10/18 02:47:51  nozomi
 * add isNotifyEnabled()
 *
 * Revision 1.15  2002/10/08 21:40:22  nozomi
 * add notification controll
 *
 * Revision 1.14  2002/10/08 00:40:30  nozomi
 * introduce listenSubModels()
 *
 * Revision 1.13  2002/09/17 05:46:02  nozomi
 * re-organise initialisation methods
 *
 * Revision 1.12  2002/09/07 04:04:07  nozomi
 * minor modification in method names
 *
 * Revision 1.11  2002/09/06 12:57:18  nozomi
 * modify default HTML style
 *
 * Revision 1.10  2002/09/05 05:24:46  nozomi
 * support pseudo HTML
 *
 * Revision 1.9  2002/08/27 08:30:42  t.okada
 * add modified attribute
 *
 * Revision 1.8  2002/08/05 03:16:45  nozomi
 * change default TextEditModel mode
 *
 * Revision 1.7  2002/07/02 08:29:08  nozomi
 * change default text model relating methods
 *
 * Revision 1.6  2002/06/22 18:58:21  nozomi
 * add missing import files
 *
 * Revision 1.5  2002/06/21 23:24:43  nozomi
 * use RichText in TextComponents
 *
 * Revision 1.4  2002/03/09 23:38:15  nozomi
 * implements ListModelListener and TextModelListener
 *
 * Revision 1.3  2002/03/05 01:52:25  nozomi
 * make Observable
 *
 * Revision 1.2  2002/01/29 07:18:50  nozomi
 * Made it an abstract class
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */


package org.nomencurator.editor.model;

import java.awt.Font;

import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import jp.kyasu.awt.DefaultTextModel;
import jp.kyasu.awt.DefaultTextEditModel;
import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.TextField;
import jp.kyasu.awt.TextModel;

import jp.kyasu.awt.event.ListModelEvent;
import jp.kyasu.awt.event.ListModelListener;
import jp.kyasu.awt.event.TextModelEvent;
import jp.kyasu.awt.event.TextModelListener;

import jp.kyasu.awt.util.HTMLTextEditModel;

import jp.kyasu.graphics.ParagraphStyle;
import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextStyle;
import jp.kyasu.graphics.TextBuffer;

import jp.kyasu.graphics.html.HTMLStyle;

import org.nomencurator.graphics.html.HTMLReader;

import org.nomencurator.editor.Modifiable;
import org.nomencurator.editor.event.ObjectEditModelEvent;
import org.nomencurator.editor.event.ObjectEditModelListener;


/**
 * The model interface for an object that acts as a named object edit model
 *
 * @see 	org.nomencurator.editor.event.ObjectEditModelEvent;
 * @see 	org.nomencurator.editor.event.ObjectEditModelListener;
 *
 * @version 	18 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public abstract class ObjectEditModel
    extends Observable
    implements ListModelListener,
	       TextModelListener,
	       Modifiable
{

    /**
     * The default rich text style for the text field.
     */
    static public final RichTextStyle DEFAULT_FIELD_STYLE =
	new RichTextStyle(
		RichTextStyle.NO_WRAP,
		RichTextStyle.JAVA_LINE_SEPARATOR,
		false,
		new TextStyle("SansSerif", Font.PLAIN, 12),
		new ParagraphStyle(ParagraphStyle.LEFT, 2, 2, 2, 0, 0));

    //    public static final RichTextStyle DEFAULT_FIELD_STYLE = TextField.DEFAULT_FIELD_STYLE;
    public static final RichTextStyle DEFAULT_DOCUMENT_STYLE = RichTextStyle.DEFAULT_DOCUMENT_STYLE;
    public static final HTMLStyle DEFAULT_HTML_STYLE = org.nomencurator.graphics.html.HTMLStyle.DEFAULT_HTML_STYLE;

    protected static HTMLReader htmlReader = new HTMLReader();

    /** An <code>Object</code> to be edited with this model */
    protected Object toBeEdited;

    /** a flag whether this model allows modification */
    protected boolean editable;

    /** a flag whether thsi model */
    protected boolean modified = false;

    //    protected boolean initializing;
    
    protected boolean notify;

    /** <code>TextEditModel</code> to hold a summary of this EditModel */
    TextEditModel summary;

    /**
     * The constructor of ObjectEditModel
     *
     * @param object <code>Object</code> to be edited
     * @param editable boolean determining whether the model is editable
     */
    protected ObjectEditModel(Object object, boolean editable)
    {
	setEditable(editable);
	createSubModels();
	//	initializing = true;
	notify = false;
	setObject(object);
	listenSubModels();
	notify = true;
	//	initializing = false;
    }

    /**
     * Creates submodles representing the <CODE>Object</CODE> under edition.
     * It is called in the constructor to create necessary submodles
     * such as <CODE>TextModel</CODE>s.
     */
    protected void createSubModels()
    {
	craeteSummaryTextModel();
    }

    /**
     * Creates a <code>TextModel</code> to be used as summary holder
     */
    protected void craeteSummaryTextModel()
    {
	summary = getDefaultTextEditModel();
    }

    /**
     * Listens submodles' <CODE>Event</CODE>s.
     */
    protected abstract void listenSubModels();

    /**
     * Sets summary TextModel
     */
    public void setSummaryTextModel(TextEditModel summary)
    {
	if(this.summary == summary)
	    return;

	this.summary = summary;
    }

    /**
     * Gets summary TextModel
     */
    public TextEditModel getSummaryTextModel()
    {
	updateSummary();
	return summary;
    }

    /**
     * Returns whether the model allows modification
     *
     * @return true if the model allows modification
     */
    public boolean isEditable()
    {
	return editable;
    }

    /**
     * Sets whether the model allows modification
     *
     * @param editable true if the model will allow modification
     */
    public void setEditable(boolean editable)
    {
	if(this.editable == editable)
	    return;

	this.editable = editable;
    }

    /**
     * Gets <code>Object</code> to be edited by this model.
     *
     */
    public Object getObject()
    {
	return toBeEdited;
    }

    /**
     * Sets <code>Object</code> to be edited by this model.
     *
     * @param object <code>Object</code> to be edited by this model
     */
    public void setObject(Object object)
    {
	if(toBeEdited == object)
	    return;
	if(toBeEdited != null) {
	    if(isModified())
		saveAttributes();
	}
	toBeEdited = object;
	if(toBeEdited != null) {
	    loadAttributes();
	    updateSummary();
	}
	else
	    setString(summary, "");

	setModified(false);

	//	if(!initializing) {
	    setChanged();
	    notifyObservers(object);
	    //}
	//fireObjectEditModelEvent(new ObjectEditModelEvent(this, toBeEdited));
    }

    public abstract void updateSummary();

    /**
     * Adds the specified named object edit model listener to receive
     * named object edit events from this model.
     *
     * @param listener the text model listener.
     */
    /*
    public synchronized void addObjectEditModelListener(ObjectEditModelListener listener)
    {
	listeners.addElement(listener);
    }
    */

    /**
     * Removes the specified named object edit model listener 
     * so it no longer receives named object edit events from this model.
     *
     * @param listener the text model listener.
     */
    /*
    public synchronized void removeObjectEditModelListener(ObjectEditModelListener listener)
    {
	listeners.removeElement(listener);
    }
    */

    /**
     * Polls <code>ObjectEditModelLister</code>s listing this model
     *
     * @param event ObjectEditModelEvent to be handled to listeners
     */
    /*
    protected void fireObjectEditModelEvent(ObjectEditModelEvent event)
    {
	Vector currentListeners = null;

	// synchronize to capture "fixed" list of listeners
	synchronized(this) {
	    currentListeners = (Vector)listeners.clone();
	}

	Enumeration e = currentListeners.elements();
	while(e.hasMoreElements()) {
	    ((ObjectEditModelListener)(e.nextElement())).editModelChanged(event);
	}
    }
    */

    /**
     * Loeds attributes of <code>Object</code> to the model
     */
    public abstract void loadAttributes();


    /**
     * Saves attributes from the model to <code>Object</code>
     */
    public abstract void saveAttributes();

    /**
     * Clear model's contents
     */
    public abstract void clear();

    /**
     * Clones this object.
     */
    public abstract Object clone();

    /**
     * Invoked when the text model has been changed.
     */
    public void textModelChanged(TextModelEvent event)
    {
	setChanged();
	notifyObservers();
    }

    /**
     * Invoked when the list model has been changed.
     */
    public void listModelChanged(ListModelEvent event)
    {
	setChanged();
	notifyObservers();
    }

    /**
     * set modifid flag when model has been changed.
     */
    public void setChanged()
    {
	if(!notify)
	    return;
	super.setChanged();
	setModified(true);
    }
    
    public void setModified(boolean modified)
    {
	if(this.modified == modified)
	    return;

	this.modified = modified;
    }
    
    public boolean isModified()
    {
    	return modified;
    }

    /**
     * Sets contents of given <code>TextModel</code> to <code>contents</code>
     *
     * @param model <code>TextModel</code> to be set its contens
     * @param contents <code>String</code> to be set as contents of the model
     */
    protected static void setString(TextModel model, String contents)
    {
	if(contents == null)
	    contents = "";
	model.setRichText(new RichText(contents, model.getRichText().getRichTextStyle()));
    }

    /**
     * Gets contents of given <code>TextModel</code> in <code>String</code>
     *
     * @param model <code>TextModel</code> to be set its contens
     *
     * @return contents of the model in <code>String</code>
     */
    protected static String getString(TextModel model)
    {
	return getText(model).toString();
    }

    /**
     * Sets contents of given <code>TextModel</code> to <code>contents</code>
     *
     * @param model <code>TextModel</code> to be set its contens
     * @param contents <code>Text</code> to be set as contents of the model
     */
    protected static void setText(TextModel model, Text contents)
    {
	if(contents == null)
	    contents = new Text();
	model.setRichText(new RichText(contents, model.getRichText().getRichTextStyle()));
    }

    /**
     * Gets contents of given <code>TextModel</code> in <code>Text</code>
     *
     * @param model <code>TextModel</code> to be set its contens
     *
     * @return contents of the model in <code>Text</code>
     */
    protected static Text getText(TextModel model)
    {
	return model.getRichText().getText();
    }

    /**
     * A utility method to create DefaultTextEditModel with
     */
    protected static TextEditModel getDefaultTextEditModel()
    {
	return new DefaultTextEditModel(new RichText(new Text("",
							      TextField.DEFAULT_FIELD_STYLE.getTextStyle()),
					TextField.DEFAULT_FIELD_STYLE));
	//return new DefaultTextEditModel(DEFAULT_FIELD_STYLE);
    }

    /**
     * A utility method to create DefaultTextEditModel with
     */
    protected static TextEditModel getDefaultRichTextEditModel()
    {
	return new DefaultTextEditModel(RichTextStyle.DEFAULT_DOCUMENT_STYLE);
    }

    protected static TextModel getDefaultTextModel()
    {
	//	return new DefaultTextModel(new RichText(TextField.DEFAULT_FIELD_STYLE));
	return new DefaultTextModel(DEFAULT_FIELD_STYLE);
    }

    protected static TextModel getDefaultRichTextModel()
    {
	return new DefaultTextModel(RichTextStyle.DEFAULT_DOCUMENT_STYLE);
    }

    protected static TextEditModel getDefaultHTMLEditModel()
    {
	return new HTMLTextEditModel(DEFAULT_HTML_STYLE);
    }

    /**
     * A utility method to create DefaultTextEditModel with
     */
    /*
    protected static TextEditModel createDefaultTextEditModel()
    {
	return new DefaultTextEditModel(RichTextStyle.DEFAULT_DOCUMENT_STYLE);
	//	return new DefaultTextEditModel(new RichText(TextField.DEFAULT_FIELD_STYLE));
    }

    protected static TextModel createDefaultTextModel()
    {
	//	return new DefaultTextModel(new RichText(TextField.DEFAULT_FIELD_STYLE));
	return new DefaultTextModel(RichTextStyle.DEFAULT_DOCUMENT_STYLE);
    }
    */

    public void setNotifyEnabled(boolean b)
    {
	notify = b;
    }

    public boolean isNotifyEnabled()
    {
	return notify;
    }
}

