/*
 * PublicationEditModel.java:  an Appearanace editor model 
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: PublicationEditModel.java,v 1.64 2002/10/18 02:52:52 nozomi Exp $
 * $Log: PublicationEditModel.java,v $
 * Revision 1.64  2002/10/18 02:52:52  nozomi
 * use setNotifyEnabled() in de-nominalisation
 *
 * Revision 1.63  2002/10/18 02:13:49  nozomi
 * set AuthorEditModels in createAuthorTextModel()
 *
 * Revision 1.62  2002/10/17 17:29:09  nozomi
 * initialise appearance list with edit models
 *
 * Revision 1.61  2002/10/08 00:42:41  nozomi
 * introduce listenSubModels()
 *
 * Revision 1.60  2002/10/02 02:49:51  nozomi
 * load Authors when necessary
 *
 * Revision 1.59  2002/09/27 19:55:59  nozomi
 * getSummaryTextArray() uses isNominal()
 *
 * Revision 1.58  2002/09/26 05:20:26  ryo
 * comment out isAuthorsModified check in saveAttributes()
 *
 * Revision 1.57  2002/09/21 19:36:00  nozomi
 * set Publiation to the Appearance when EditModel was set
 *
 * Revision 1.56  2002/09/17 05:46:02  nozomi
 * re-organise initialisation methods
 *
 * Revision 1.55  2002/09/09 16:37:22  nozomi
 * implement updateSumamry(); change constant name to follow modification in org.nomencurator.NamedObject
 *
 * Revision 1.54  2002/09/06 17:01:47  ryo
 * link NamedObjects when *EditModel linked
 *
 * Revision 1.53  2002/09/06 02:21:33  nozomi
 * add dummy loadAttributes() to compile
 *
 * Revision 1.52  2002/09/02 16:37:03  nozomi
 * use getDefaultRichTextEditModel() for note
 *
 * Revision 1.51  2002/08/27 08:53:06  t.okada
 * addTextModelListener is added.
 *
 * Revision 1.50  2002/08/26 09:25:15  t.okada
 * add publisher and place to publication
 *
 * Revision 1.49  2002/08/20 13:17:34  t.okada
 * add notes column
 *
 * Revision 1.48  2002/08/15 05:16:55  ryo
 * change type of year into String
 *
 * Revision 1.47  2002/08/05 03:17:41  nozomi
 * change getYeatTextModel()
 *
 * Revision 1.46  2002/07/02 08:29:08  nozomi
 * change default text model relating methods
 *
 * Revision 1.45  2002/06/21 23:22:58  nozomi
 * use RichText in TextComponents
 *
 * Revision 1.44  2002/06/07 14:09:24  ryo
 * add authorListSummary which is automatically created from authorList
 *
 * Revision 1.43  2002/06/06 11:29:34  ryo
 * Automatic creation of authorList was stopped
 *
 * Revision 1.42  2002/05/31 19:10:20  nozomi
 * withdraw rev 1.40
 *
 * Revision 1.41  2002/05/31 10:24:33  ryo
 * delete code of creating authors
 *
 * Revision 1.40  2002/05/31 06:02:24  nozomi
 * use AuthorListModel#getAuthorsSummaryModel() to get TextEditModel for authors
 *
 * Revision 1.39  2002/05/24 08:12:23  ryo
 * fix bug of linkage process in addAuthor()
 *
 * Revision 1.38  2002/05/23 03:10:06  nozomi
 * change default status of lists
 *
 * Revision 1.37  2002/05/22 23:59:12  nozomi
 * use NamedObjectListModels
 *
 * Revision 1.36  2002/05/22 05:30:42  ryo
 * get author list from AuthorListModel in updateList()
 *
 * Revision 1.35  2002/05/21 08:37:19  nozomi
 * use NamedObjectListModel to handle linked edit models
 *
 * Revision 1.34  2002/05/21 08:01:26  nozomi
 * disable automatic dummy edit model addition
 *
 * Revision 1.33  2002/05/21 03:37:53  nozomi
 * use NamedObjectListModel
 *
 * Revision 1.32  2002/05/20 06:12:32  nozomi
 * use AppearanceListModel
 *
 * Revision 1.31  2002/05/16 05:06:22  ryo
 * fix bug of authorList
 *
 * Revision 1.30  2002/05/15 10:16:16  ryo
 * fix bug in setPublicationTextModels()
 *
 * Revision 1.29  2002/05/15 09:06:36  t.okada
 * PublicationEditModel()
 *
 * Revision 1.28  2002/05/14 13:34:20  ryo
 * modify saveAttributes()
 *
 * Revision 1.27  2002/05/14 10:15:51  ryo
 * add getSummaryTextArray() and updateList()
 *
 * Revision 1.26  2002/05/10 13:45:40  ryo
 * comment out debug message
 *
 * Revision 1.25  2002/05/10 12:11:42  t.okada
 * modify PublicationEditModel constructor
 *
 * Revision 1.24  2002/05/08 11:05:40  ryo
 * bug fix in saveAttributes()
 *
 * Revision 1.23  2002/04/29 19:42:38  nozomi
 * don't show zero when year is zero, i.e. invalid year
 *
 * Revision 1.22  2002/04/19 21:45:36  nozomi
 * relating model handling
 *
 * Revision 1.21  2002/04/17 21:00:16  nozomi
 * fix scondary author bug
 *
 * Revision 1.20  2002/04/16 23:43:52  nozomi
 * apply ryo's modification on another branch
 *
 * Revision 1.19  2002/04/15 23:41:46  nozomi
 * exclude NullPointer from setPublicationTextModels(Publication)
 *
 * Revision 1.18  2002/04/09 03:33:37  nozomi
 * accomodate to change of get method names
 *
 * Revision 1.17  2002/03/27 06:09:33  nozomi
 * fix number-null bug
 *
 * Revision 1.16  2002/03/22 05:12:23  okawa
 * no message
 *
 * Revision 1.15  2002/03/20 15:12:06  okawa
 * no message
 *
 * Revision 1.14  2002/03/20 13:58:19  okawa
 * modify addAppearance()
 *
 * Revision 1.13  2002/03/20 06:30:43  okawa
 * modify setting TextModels
 *
 * Revision 1.12  2002/03/08 14:04:13  okawa
 * set TextModel
 *
 * Revision 1.11  2002/02/28 17:11:02  nozomi
 * use TextListModel of the package
 *
 * Revision 1.10  2002/02/19 04:59:47  nozomi
 * support ISXN
 *
 * Revision 1.9  2002/02/18 23:11:04  nozomi
 * add table of contents
 *
 * Revision 1.8  2002/02/16 22:31:38  nozomi
 * Fix PublicationEditPanel related issues
 *
 * Revision 1.7  2002/02/04 04:54:14  nozomi
 * Add received/revised/accepted text models
 *
 * Revision 1.6  2002/02/04 01:23:54  nozomi
 * Some access methods were made public
 *
 * Revision 1.5  2002/02/03 09:35:55  nozomi
 * Add get/set methods for TextModels
 *
 * Revision 1.4  2002/02/01 22:17:19  nozomi
 * implementation of appearanceList relating methods
 *
 * Revision 1.3  2002/02/01 11:26:30  nozomi
 * Change comments...
 *
 * Revision 1.2  2002/01/29 07:20:02  nozomi
 * Made it Cloneable
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.editor.model;

import java.util.Enumeration;
import java.util.Vector;

import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.TextModel;
import jp.kyasu.awt.TextListModel;
import jp.kyasu.awt.TextField;
import jp.kyasu.awt.DefaultTextEditModel;

import jp.kyasu.awt.event.TextModelEvent;

import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;

import org.nomencurator.Appearance;
import org.nomencurator.Author;
import org.nomencurator.NamedObject;
import org.nomencurator.Publication;

import org.nomencurator.broker.NamedObjectBroker;

import org.nomencurator.editor.TableList;

/**
 * The model represents a <code>Publication</code>
 *
 * @see 	org.nomencurator.PublicationEditor
 * @see 	org.nomencurator.editor.PublicationEditor
 *
 * @version 	18 Oct 2002
 * @author 	Nozomi `James' Ytow
 */
public class PublicationEditModel
    extends NamedObjectEditModel
    implements Cloneable
{

    /**
     * <code>PublicationEditModel</code> of which
     * <code>Publication</code> containing object of this model
     */
    protected PublicationEditModel container;

    /**
     * <code>Vector</code> of <code>PublicationEditModel</code>s
     * which are contained by <code>Publication</code> of this object
     */
    protected Vector contents;

    /**
     * <code>TextListModel</code> to hold
     * summary of <code>AppearanceEditModel</code>
     */
    protected AppearanceListModel appearanceList;

    /** <code>TextEditModel</code> to hold authors */
    protected TextEditModel authors;

    /** <code>TextListModel</code> to hold list of authors */
    protected AuthorListModel authorList;

    /** <code>TextEditModel</code> to hold secondary authors */
    protected TextEditModel secondaryAuthors;

    /** <code>TextListModel</code> to hold list of secondary authors */
    protected AuthorListModel secondaryAuthorList;

    /** <code>TextEditModel</code> represents year of the <code>Publication</code> */
    protected TextEditModel          year;
    
    /** <code>TextEditModel</code> to edit citationTitle of the <code>Publication</code> */
    protected TextEditModel      containerTitle;
    
    /** <code>TextEditModel</code> to edit year of the <code>Publication</code> */
    protected TextEditModel          volume;
    
    /** <code>TextEditModel</code> to edit year of the <code>Publication</code> */
    protected TextEditModel          number;
    
    /**
     * <code>TextEditModel</code> to edit first page of the <code>Publication</code>
     */
    protected TextEditModel      firstPage;

    /**
     * <code>TextEditModel</code> to edit last page of the <code>Publication</code>
     */
    protected TextEditModel      lastPage;

    /**
     * <code>TextEditModel</code> to edit notes of the <code>Publication</code>
     */
    protected TextEditModel      notes;

    /** <code>TextEditModel</code> to edit contentsTitle of the <code>Publication</code> */
    protected TextEditModel      contentsTitle;

    /** <code>TextEditModel</code> represents ISBN/ISSN of the <code>Publication</code> */
    protected TextEditModel          isxn;
    
    /** <code>TextEditModel</code> represents publisher of the <code>Publication</code> */
    protected TextEditModel          publisher;
    
    /** <code>TextEditModel</code> represents published place of the <code>Publication</code> */
    protected TextEditModel          place;
    
    /** <code>TextEditModel</code> representing received date of the <code>Publication</code> */
    protected TextEditModel      received;

    /** <code>TextEditModel</code> representing received date of revised the <code>Publication</code> */
    protected TextEditModel      revised;

    /** <code>TextEditModel</code> representing accepted date of the <code>Publication</code> */
    protected TextEditModel      accepted;


    /** <code>TextEditModel</code> to edit abstract of the <code>Publication</code> */
    protected TextEditModel      publicationSummary;

    /** <code>TextListModel</code> representing table of contents */
    protected TableModel tableOfContents;

    /**
     * <code>TextEditModel</code> to manage view name of the <code>Publication</code>,
     * which is not used by <CODE>PublicationEditPanel</CODE> but other
     * <CODE>NameUsageEditModel</CODE>s.
     */
    protected TextEditModel      viewName;

    protected TextEditModel      authorsSummaryModel;

    protected static PublicationEditModel template;

    boolean authorsModified;

    boolean appearancesModified;

    /** Constructs a <code>PublicationEditModel</code> */
    public PublicationEditModel()
    {
	this(true);
    }

    /**
     * The constructor of PublicationEditModel
     *
     * @param editable boolean determining whether the model is editable
     */
    public PublicationEditModel(boolean editable)
    {
	this(new Publication(), editable);
    }

    /**
     * The constructor of PublicationEditModel
     *
     * @param object <code>Publication</code> to be edited
     */
    public PublicationEditModel(Publication object)
    {
	this(object, true);
    }

    /**
     * The constructor of PublicationEditModel
     *
     * @param object <code>Object</code> to be edited
     * @param editable boolean determining whether the model is editable
     */
    public PublicationEditModel(Publication object, boolean editable)
    {
	this(object, editable,
	     null, null, null, null);
    }

    /**
     * The constructor of PublicationEditModel
     *
     * @param object <code>Object</code> to be edited
     * @param editable boolean determining whether the model is editable
     * @param container <code>PublicatoinEditModel</code> containing the <code>Publication</code>
     * as part, e.g. book containing this publication as a chapter
     * @param contents <code>Vector</code> of <code>PublicationEditModel</code>s
     * contained in the publication
     * @param authorEditModels <code>Vector</code> of <code>AuthorEditModel</code>s
     * of the publication
     * @param appearanceEditModels <code>Vector</code> of <code>AppearanceEditModel</code>s
     * contained in the publication
     */
    public PublicationEditModel(Publication object, 
				boolean editable,
				PublicationEditModel container,
				Vector contents,
				Vector authorEditModels,
				Vector appearanceEditModels
				)
    {
	super(object, editable);
    }

    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void createSubModels()
    {
	super.createSubModels();
	createPublicationTextModel();
	createAuthorTextModel();
	createTableOfContentsModel();
	createAppearanceTextModel();
	setAuthorsModified(false);
	setAppearancesModified(false);
    }

    /**
     * Creates submodles representing the <CODE>Publication</CODE> under edition.
     */
    protected void listenSubModels()
    {
	listenPublicationTextModel();
	listenAuthorTextModel();
    }

    /**
     * Creates <code>TextModel</code>s of this object.
     * Constructor of subclasses may override this method
     * to provide its own model
     */
    protected void createPublicationTextModel()
    {
	year = getYearTextModel();

	contentsTitle  = getDefaultRichTextEditModel();

	firstPage      = getDefaultTextEditModel();
	lastPage       = getDefaultTextEditModel();


	isxn           = getDefaultTextEditModel();
	publisher      = getDefaultRichTextEditModel();
	place          = getDefaultTextEditModel();

	volume         = getDefaultTextEditModel();
	number         = getDefaultTextEditModel();

	received       = getDefaultTextEditModel();
	revised        = getDefaultTextEditModel();
	accepted       = getDefaultTextEditModel();

	notes	       = getDefaultRichTextEditModel();

	publicationSummary = getDefaultRichTextEditModel();
    }

    /**
     * Creates <code>TextModel</code>s of this object.
     * Constructor of subclasses may override this method
     * to provide its own model
     */
    protected void listenPublicationTextModel()
    {
	year.addTextModelListener(this);
	contentsTitle.addTextModelListener(this);
	firstPage.addTextModelListener(this);
	lastPage.addTextModelListener(this);
	isxn.addTextModelListener(this);
	publisher.addTextModelListener(this);
	place.addTextModelListener(this);
	volume.addTextModelListener(this);
	number.addTextModelListener(this);
	received.addTextModelListener(this);
	revised.addTextModelListener(this);
	accepted.addTextModelListener(this);
	notes.addTextModelListener(this);
    }

    /**
     * Creates a <code>TextModel</code> to be used as summary holder
     */
    protected void craeteSummaryTextModel()
    {
	super.craeteSummaryTextModel();
	viewName = getDefaultTextEditModel();
    }

    protected void setPublicationTextModels(Publication object)
    {
	if(object == null)
	    return;

	String value = object.EMPTY_STRING;

	if(object.getYear() != null)
	    year.setRichText(new RichText(object.getYear(), RichTextStyle.DEFAULT_DOCUMENT_STYLE));
	else
	    year.setRichText(new RichText(object.EMPTY_STRING, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getContentsTitle();
	if(value == null)
	    value = object.EMPTY_STRING;
	contentsTitle.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

        if(object.getContainer() != null)
	    containerTitle.setRichText(new RichText(object.getContainer().getCitationTitle(), RichTextStyle.DEFAULT_DOCUMENT_STYLE));
	else
		if (object.getCitationTitle() != null)
			containerTitle.setRichText(new RichText(object.getCitationTitle(), RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getAuthorNames();
	if(value == null)
	    value = object.EMPTY_STRING;
	authors.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getFirstPage();
	if(value == null)
	    value = object.EMPTY_STRING;
	firstPage.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getLastPage();
	if(value == null)
	    value = object.EMPTY_STRING;
	lastPage.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));
	
	value = object.getNotes();
	if(value == null)
		value = object.EMPTY_STRING;
	notes.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getISXN();
	if(value == null)
	    value = object.EMPTY_STRING;
	isxn.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE)); // HCC

	value = object.getPublisher();
	if(value == null)
	    value = object.EMPTY_STRING;
	publisher.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getPlace();
	if(value == null)
	    value = object.EMPTY_STRING;
	place.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getVolume();
	if(value == null)
	    value = object.EMPTY_STRING;
	volume.setRichText(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE));

	value = object.getNumber();
	if(value == null)
	    value = object.EMPTY_STRING;
	setNumberTextModel(new DefaultTextEditModel(new RichText(value, RichTextStyle.DEFAULT_DOCUMENT_STYLE)));
    }

    /**
     * Returns <code>TextEditModel</code> representing authors of the publication
     *
     * @return <code>TextEditModel</code> representing authors of the publication
     */
    public TextEditModel getAuthorTextEditModel()
    {
	return authors;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel</code> representing authors of the publication
     *
     * @param model <code>TextEditModel</code> to be used as representation of authors of the publication
     */
    public void setAuthorTextEditModel(TextEditModel model)
    {
	if(authors == model)
	    return;

	authors = model;
    }

    /**
     * Returns <code>TextListModel</code> representing authors list of the publication
     *
     * @return <code>TextEditModel</code> representing authors list of the publication
     */
    public AuthorListModel getAuthorList()
    {
	return authorList;
    }

    /**
     * Sets <code>model</code> as <code>TextListModel</code> representing authors list of the publication
     *
     * @param model <code>TextListModel</code> to be used as representation of author list of the publication
     */
    public void setAuthorList(AuthorListModel model)
    {
	if(authorList == model)
	    return;
	
	authorList = model;
    }
    
    /**
     * Returns <code>TextEditModel</code> representing secondary authors of the publication
     *
     * @return <code>TextEditModel</code> representing secondary authors of the publication
     */
    public TextEditModel getSecondaryAuthorTextEditModel()
    {
	return secondaryAuthors;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel</code> representing secondary authors of the publication
     *
     * @param model <code>TextEditModel</code> to be used as representation of secondary authors of the publication
     */
    public void setSecondaryAuthorTextEditModel(TextEditModel model)
    {
	if(secondaryAuthors == model)
	    return;

	secondaryAuthors = model;
    }

    /**
     * Returns <code>TextListModel</code> representing secondary authors list of the publication
     *
     * @return <code>TextEditModel</code> representing secondary authors list of the publication
     */
    public TextListModel getSecondaryAuthorList()
    {
	return secondaryAuthorList;
    }

    /**
     * Sets <code>model</code> as <code>TextListModel</code> representing secondary authors list of the publication
     *
     * @param model <code>TextListModel</code> to be used as representation of secondary author list of the publication
     */
    public void setSecondaryAuthorList(AuthorListModel model)
    {
	if(secondaryAuthorList == model)
	    return;

	secondaryAuthorList = model;
    }

    /**
     * Returns <code>Vector</code> containing <code>AuthorEditModel</code>s of the authors
     *
     * @return <code>Vector</code> containing <code>AuthorEditModel</code>s of the authors
     */
    public Vector getAuthorEditModels()
    {
	return authorList.getModels();
    }

    /**
     * Adds a <code>AuthorEditModel</code> as a author
     *
     * @param authorEditModel to be added to author list of this publication
     */
    public void addAuthor(AuthorEditModel model)
    {
	if(authorList.contains(model))
	    return;

	authorList.addModel(model);

	if(model != null)
	    model.add(this);

	//	getPublication().addAuthor(model.getAuthor());
	//setAuthorsModified(true);
    }

    /**
     * Removes specified <code>AuthorEditModel</code> from author list
     *
     * @param authorEditModel to be removed from author list of this publication
     */
    public void removeAuthor(AuthorEditModel model)
    {
	if(!authorList.contains(model))
	    return;

	authorList.removeModel(model);
 	if(model != null)
	model.remove(this);
	
	//this.getPublication().removeAuthor(model.getAuthor());
	
	//	setAuthorsModified(true);
    }

    /**
     * Returns <code>TextEditModel<code> representing year of publication
     *
     * @return <code>TextEditModel<code> representing year of publication
     */
    public TextEditModel getYearTextModel()
    {
	if(year == null) {
	    /*
	    year = new DefaultTextEditModel(new RichText(new Text("",
								  TextField.DEFAULT_FIELD_STYLE.getTextStyle()),
							 TextField.DEFAULT_FIELD_STYLE));
	    */
	    year = getDefaultTextEditModel();
	}
    	return year;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing year of publication
     *
     * @param model <code>TextEditModel<code> to represent year of publication
     */
    public void setYearTextModel(TextEditModel model)
    {
	if(year == model)
	    return;

	year = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing ISBN/ISSN of publication
     *
     * @return <code>TextEditModel<code> representing ISBN/ISSN of publication
     */
    public TextEditModel getISXNTextModel()
    {
	return isxn;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing ISSN/ISBN of publication
     *
     * @param model <code>TextEditModel<code> to represent ISSN/ISBN of publication
     */
    public void setISXNTextModel(TextEditModel model)
    {
	if(isxn == model)
	    return;

	isxn = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing publisher of the publication
     *
     * @return <code>TextEditModel<code> representing publisher of publication
     */
    public TextEditModel getPublisherTextModel()
    {
	return publisher;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing publisher of publication
     *
     * @param model <code>TextEditModel<code> to represent publisher of publication
     */
    public void setPublisherTextModel(TextEditModel model)
    {
	if(publisher == model)
	    return;

	publisher = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing place of the publication
     *
     * @return <code>TextEditModel<code> representing place of publication
     */
    public TextEditModel getPlaceTextModel()
    {
	return place;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing place of publication
     *
     * @param model <code>TextEditModel<code> to represent place of publication
     */
    public void setPlaceTextModel(TextEditModel model)
    {
	if(place == model)
	    return;

	place = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing title of container <code>Publication</code>, 
     * e.g. journal title, edited book title (as container of chapters).
     *
     * @return <code>TextEditModel<code> representing title of container <code>Publication</code>
     */
    public TextEditModel getContainerTitleTextModel()
    {
	if (containerTitle == null)
	    setContainer();

	return containerTitle;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing containter <code>Publication</code>'s title
     *
     * @param model <code>TextEditModel<code> to represent containter <code>Publication</code>'s title
     */
    public void setContainerTitleTextModel(TextEditModel model)
    {
	if(containerTitle == model)
	    return;
	containerTitle = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing volume of the <code>Publication</code>
     *
     * @return <code>TextEditModel<code> representing volume of the <code>Publication</code>
     */
    public TextEditModel getVolumeTextModel()
    {
	return volume;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing volume of the <code>Publication</code>
     *
     * @param model <code>TextEditModel<code> to represent volume of the <code>Publication</code>
     */
    public void setVolumeTextModel(TextEditModel model)
    {
	if(volume == model)
	    return;

	volume = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing number of the <code>Publication</code>
     *
     * @return <code>TextEditModel<code> representing number of the <code>Publication</code>
     */
    public TextEditModel getNumberTextModel()
    {
	return number;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing number of the <code>Publication</code>
     *
     * @param model <code>TextEditModel<code> to represent number of the <code>Publication</code>
     */
    public void setNumberTextModel(TextEditModel model)
    {
	if(number == model)
	    return;

	number = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing first page of the <code>Publication</code>
     *
     * @return <code>TextEditModel<code> representing first page of the <code>Publication</code>
     */
    public TextEditModel getFirstPageTextModel()
    {
	return firstPage;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing first page of the <code>Publication</code>
     *
     * @param model <code>TextEditModel<code> to represent first page of the <code>Publication</code>
     */
    public void setFirstPageModelText(TextEditModel model)
    {
	if(firstPage == model)
	    return;

	firstPage = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing last page of the <code>Publication</code>
     *
     * @return <code>TextEditModel<code> representing last page of the <code>Publication</code>
     */
    public TextEditModel getLastPageTextModel()
    {
	return lastPage;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing last page of the <code>Publication</code>
     *
     * @param model <code>TextEditModel<code> to represent last page of the <code>Publication</code>
     */
    public void setLastPageModelText(TextEditModel model)
    {
	if(lastPage == model)
	    return;

	lastPage = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing notes of the <code>Publication</code>
     *
     * @return <code>TextEditModel<code> representing notes of the <code>Publication</code>
     */
    public TextEditModel getNotesTextModel()
    {
	return notes;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing notes of the <code>Publication</code>
     *
     * @param model <code>TextEditModel<code> to represent notes of the <code>Publication</code>
     */
    public void setNotesModelText(TextEditModel model)
    {
	if(notes == model)
	    return;

	notes = model;
    }

    /**
     * Returns <code>TextEditModel<code> representing title of the <code>Publication</code>,
     * e.g. title of the article
     *
     * @return <code>TextEditModel<code> representing title of the <code>Publication</code>,
     */
    public TextEditModel getContentsTitleTextModel()
    {
	return contentsTitle;
    }

    /**
     * Sets <code>model</code> as <code>TextEditModel<code> representing title of the <code>Publication</code>,
     *
     * @param model <code>TextEditModel<code> to represent title of the <code>Publication</code>,
     */
    public void setContentsTitleModelText(TextEditModel model)
    {
	if(contentsTitle == model)
	    return;

	contentsTitle = model;
    }

    /**
     * Returns <code>TextModel</code> containing summary of
     * the model object
     *
     * @return <code>TextModel</code> containing summary of the model object
     */
    public TextEditModel getContentsSummaryTextModel()
    {
	return publicationSummary;
    }


    /**
     * Sets <code>container</code> as container of this object,
     * e.g. book containing this <code>Publication</code> as
     * a chapter.
     *
     * @param container <code>PublicationEditModel</code> containing this object
     */
    public void setContainer(PublicationEditModel container)
    {
	if(this.container == container) {
	    if(this.container == null &&
	       containerTitle == null)
		containerTitle = getDefaultRichTextEditModel();
	    return;
	}

	if(this.container != null) {
	    this.container.removeContents(this);
	    containerTitle = null;
	}
	else if(containerTitle != null)
	    containerTitle.removeTextModelListener(this);
	    
	this.container = container;

	if(container != null) {
	    container.updateSummary();
	    containerTitle = container.summary;
	}
	else {
	    if (containerTitle == null)
		containerTitle = getDefaultRichTextEditModel();
	    containerTitle.addTextModelListener(this);
	}

	setModified(true);
    }

    /**
     * Sets <code>container</code> as container of this object,
     * e.g. book containing this <code>Publication</code> as
     * a chapter.
     */
    protected void setContainer()
    {
	Publication pub = getPublication().getContainer();

	PublicationEditModel pubModel = 
	    (PublicationEditModel)getEditModel(pub);

	setContainer(pubModel);
	/*
	if(pubModel != null)
	    pubModel.addContents(this);
	*/
    }

    /**
     * Gets <code>PublicationEditModel</code> containing this object,
     * e.g. book containing this <code>Publication</code> as a chapter.
     *
     * @return PublicationEditModel containing this object, or null
     */
    public PublicationEditModel getContainer()
    {
	return container;
    }

    /**
     * Adds a <code>PublicationEditModel</code> as contents
     *
     * @param publicationEditModel to be added to contents of this publication
     */
    public void addContents(PublicationEditModel publicationEditModel)
    {
	contents.addElement(publicationEditModel);
    }

    /**
     * Removes specified <code>PublicationEditModel</code> from contents
     *
     * @param publicationEditModel to be removed from contents of this publication
     */
    public boolean removeContents(PublicationEditModel publicationEditModel)
    {
	return contents.removeElement(publicationEditModel);
    }

    public Vector getContents()
    {
	return contents;
    }

    public void setContents(Vector contents)
    {
	if(this.contents == contents)
	    return;

	this.contents = contents;
    }

    /**
     * Creates <code>TextListModel</code> representing table of contents
     * Constructor of subclasses may override this method
     * to provide its own model
     */
    protected void createTableOfContentsModel()
    {
	tableOfContents = new PublicationListModel();
    }

    /**
     * Returns <code>TableModel</code> representing table of contents
     */
    public TableModel getTableOfContentsModel()
    {
	return tableOfContents;
    }

    /**
     * Sets <code>tableOfContents</code> as <code>TableModel</code> representing table of contents
     */
    public void setTableOfContentsModel(TableModel tableOfContents)
    {
	if(this.tableOfContents == tableOfContents)
	    return;

	this.tableOfContents = tableOfContents;
    }

    protected void createAuthorTextModel()
    {
	authorList = new AuthorListModel(/*isEditable()*/);
	authorsSummaryModel =  authorList.getAuthorsSummaryModel();
	authors = getDefaultTextEditModel();

	secondaryAuthorList = new AuthorListModel(/*isEditable()*/);
	//	secondaryAuthors = secondaryAuthorList.getAuthorsSummaryModel();
	secondaryAuthors = getDefaultTextEditModel();

	Publication p = getPublication();
	if(p != null) {
	    Vector v = p.getAuthors();
	    if(v != null)
		authorList.setObjects(v);
	}
    }

    protected void listenAuthorTextModel()
    {
	authorsSummaryModel.addTextModelListener(this);
    }

    protected void createAppearanceTextModel()
    {
	appearanceList = new AppearanceListModel(/*isEditable()*/);
	Publication p = getPublication();
	if(p == null)
	    return;
	Vector v = p.getAppearances();
	if(v == null)
	    return;
	appearanceList.setObjects(v);
    }

    public AppearanceListModel getAppearanceTextListModel()
    {
	return appearanceList;
    }

    public void setAppearanceTextListModel(AppearanceListModel model)
    {
	if(appearanceList == model)
	    return;
	
	appearanceList = model;
    }


    public void addAppearance(AppearanceEditModel appearanceEditModel)
    {
	if(appearanceList.contains(appearanceEditModel))
	    return;

	appearanceList.addModel(appearanceEditModel);

	if(appearanceEditModel != null) {
	    appearanceEditModel.setPublicationEditModel(this);
	    getPublication().addAppearance(appearanceEditModel.getAppearance());
	}

	//setAppearancesModified(true);
    }

    public void removeAppearance(AppearanceEditModel appearanceEditModel)
    {
	if(!appearanceList.contains(appearanceEditModel))
	    return;
	appearanceList.removeModel(appearanceEditModel);
	/*
	appearanceEditModel.setPublicationEditModel(null);
	getPublication().removeAppearance(appearanceEditModel.getAppearance());
	*/

	//setAppearancesModified(true);
    }

    public Vector getAppearanceEditModels()
    {
	return appearanceList.getModels();
    }

    public void setAppearanceEditModels(Vector appearanceEditModels)
    {
	if(getAppearanceEditModels() == appearanceEditModels)
	    return;

	appearanceList.setModels(appearanceEditModels);
	//	setAppearancesModified(true);
    }

    /**
     * Gets <code>Object</code> to be edited by this model.
     *
     */
    public Publication getPublication()
    {
	return (Publication)getObject();
    }

    /**
     * Gets <code>Object</code> to be edited by this model.
     *
     */
    public void setObject(Object object)
    {
	Publication pub = getPublication();
	if(pub == object)
	    return;

	if(object != null &&
	   !(object instanceof Publication))
	    throw new IllegalArgumentException(object.getClass().getName() +
					       " is not an instance of " + "Publication");
	super.setObject(object);

	//	setPublicationTextModels(object);    
    }

    public Object clone()
    {
	return new PublicationEditModel();
    }

    public void clear()
    {
    }

    /**
     * Loeds attributes of <code>Object</code> to the model
     */
    public void loadAttributes()
    {
	super.loadAttributes();

	Publication publication = getPublication();
	if(publication == null) {
	    publication = new Publication();
	    toBeEdited = publication;
	}

	setString(getContainerTitleTextModel(), publication.getCitationTitle());
	setString(getContentsTitleTextModel(), publication.getContentsTitle());
	setString(isxn, publication.getISXN());
	setString(year, publication.getYear());
	setString(volume, publication.getVolume());
	setString(number, publication.getNumber());
	setString(firstPage, publication.getFirstPage());
	setString(lastPage, publication.getLastPage());
	setString(notes, publication.getNotes());
	setString(publisher, publication.getPublisher());
	setString(place, publication.getPlace());

	setString(authors, publication.getAuthorNames());
	authorList.setObjects(publication.getAuthors());
	//getAuthorsSummaryTextModel().setRichText(new RichText(publication.getAuthorListSummary(), DEFAULT_FIELD_STYLE));
	//publication.getAffiliationsString();
	appearanceList.setObjects(publication.getAppearances());

    }

    /**
     * Saves attributes to the <code>Publication</code>
     */
    public void saveAttributes() {
	updateList();
	Publication publication = getPublication();
	if (publication == null)
	    publication = new Publication();
	publication.setAuthorNames(authors.getRichText().getText().toString());
	publication.setAuthorListSummary(getAuthorsSummaryTextModel().getRichText().getText().toString());
		//publication.setAffiliationsString();
	publication.setCitationTitle(this.getContainerTitleTextModel().getRichText().getText().toString());
	publication.setContentsTitle(this.getContentsTitleTextModel().getRichText().getText().toString());
	publication.setISXN(getISXNTextModel().getRichText().getText().toString());
	publication.setYear(this.getYearTextModel().getRichText().getText().toString());
	publication.setVolume(this.getVolumeTextModel().getRichText().getText().toString());
	publication.setNumber(this.getNumberTextModel().getRichText().getText().toString());
	publication.setFirstPage(getFirstPageTextModel().getRichText().getText().toString());
	publication.setLastPage(getLastPageTextModel().getRichText().getText().toString());
	publication.setNotes(this.getNotesTextModel().getRichText().getText().toString());
	publication.setPublisher(this.getPublisherTextModel().getRichText().getText().toString());
	publication.setPlace(this.getPlaceTextModel().getRichText().getText().toString());
//		publication.setReceived();
//		publication.setRevised();
//		publication.setAccepted();

//	if(isAuthorsModified()) {
	    publication.setAuthors(authorList.getNamedObjects());
	    setAuthorsModified(false);
//	}

	if(isAppearancesModified()) {
	    publication.setAppearances(appearanceList.getNamedObjects());
	    setAppearancesModified(false);
	}
    }

    /**
     * Gets summary contents in <code>Text</code>
     *
     * @return Text[] representing summary of the model
     */
    public Text[] getSummaryTextArray()
    {
	/*
	if(!getPublication().isNominal()) {
	    return new Text[] {
		getAuthorsSummaryTextModel().getRichText().getText(),
		this.year.getRichText().getText(),
		this.containerTitle.getRichText().getText()
	    };
	}

	return new Text[] {
	    getAuthorsSummaryTextModel().getRichText().getText(),
	    new Text(getPublication().getYear()),
	    new Text(getPublication().getCitationTitle())
	};
	*/
	if(getPublication().isNominal()) {
	    NamedObjectBroker.getInstance().getNamedObject(getPublication().getName());
	    boolean notification = isNotifyEnabled();
	    setNotifyEnabled(false);
	    loadAttributes();
	    setNotifyEnabled(notification);
	}

	return new Text[] {
	    getAuthorsSummaryTextModel().getRichText().getText(),
	    this.year.getRichText().getText(),
	    this.containerTitle.getRichText().getText()
	};
    }
    
    public void updateList() {
	// update lists.
	appearanceList.update();
	authorList.update();
    }

    public TextEditModel getAuthorsSummaryTextModel()
    {
	if(getPublication().isNominal()) {
	    NamedObjectBroker broker = NamedObjectBroker.getInstance();
	    broker.getNamedObjectEntity(getName());
	    authorList.setObjects(getPublication().getAuthors());
	}
	return authorList.getAuthorsSummaryModel();
    }

    public void updateSummary()
    {
	if(getPublication().isNominal()) {
	    NamedObjectBroker broker = NamedObjectBroker.getInstance();
	    broker.getNamedObjectEntity(getName());
	    authorList.setObjects(getPublication().getAuthors());
	}

	Text text = new Text(updateViewName().toString());

	//	String s = getContentsTitleTextModel().getRichText().getText().toString();
	String s = getContainerTitleTextModel().getRichText().getText().toString();
	if(s.length() != 0) {
	    if(text.length() != 0)
		text.append(": ");
	    text.append(s);
	}

	setText(summary, text);
    }

    public Text updateViewName()
    {
	Text view = (Text)authorsSummaryModel.getRichText().getText().clone();
	String s = year.getRichText().getText().toString(); 
	if(s.length() != 0) {
	    if(view.length() > 0)
		view.append(' ');
	    view.append("(");
	    view.append(s);
	    view.append(")");
	}

	setText(viewName, view);
	return view;
    }

    public final TextEditModel getViewNameModel()
    {
	return viewName;
    }

    /**
     * Creates and returns an instance of <CODE>NamedObjectEditModel</CODE>
     * representing <CODE>object</CODE>.
     * The subclass must provide this method creating its an instance.
     *
     * @param object an instance of <CODE>NamedObject</CODE> to be represened by the model
     *
     * @return NamedObjectEditModel representing <CODE>object</CODE>
     */
    NamedObjectEditModel createEditModel(NamedObject object)
    {
	if(!(object instanceof Publication))
	    throw new IllegalArgumentException(object.getClass().getName()
						+ " is not an instance of org.nomencurator.Publication");
	PublicationEditModel model =
	    new PublicationEditModel();
	model.setObject(object);
	return model;
	//	return new PublicationEditModel((Publication)object);
    }

    /**
     * Invoked when the text model has been changed.
     */
    public void textModelChanged(TextModelEvent event)
    {
	Object source = event.getSource();
	if (source == authorList) {
	    setAuthorsModified(true);
	}
	else if (source == appearanceList) {
	    setAppearancesModified(true);
	}
	else if(source == authorsSummaryModel) {
	    updateSummary();
	}
	else {
	    super.textModelChanged(event);
	}
    }

    public static NamedObjectEditModel getInstance()
    {
	if(template == null)
	    template = new PublicationEditModel();
	return template;
    }

    protected boolean isAuthorsModified()
    {
	return authorsModified;
    }

    protected void setAuthorsModified(boolean b)
    {
	authorsModified = b;
	if(b)
	    modified = b;
    }

    protected boolean isAppearancesModified()
    {
	return appearancesModified;
    }

    protected void setAppearancesModified(boolean b)
    {
	appearancesModified = b;
	if(b)
	    modified = b;
    }

}
